/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rekognition.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Summary information for required items of personal protective equipment (PPE) detected on persons by a call to
 * <a>DetectProtectiveEquipment</a>. You specify the required type of PPE in the <code>SummarizationAttributes</code>
 * (<a>ProtectiveEquipmentSummarizationAttributes</a>) input parameter. The summary includes which persons were detected
 * wearing the required personal protective equipment (<code>PersonsWithRequiredEquipment</code>), which persons were
 * detected as not wearing the required PPE (<code>PersonsWithoutRequiredEquipment</code>), and the persons in which a
 * determination could not be made (<code>PersonsIndeterminate</code>).
 * </p>
 * <p>
 * To get a total for each category, use the size of the field array. For example, to find out how many people were
 * detected as wearing the specified PPE, use the size of the <code>PersonsWithRequiredEquipment</code> array. If you
 * want to find out more about a person, such as the location (<a>BoundingBox</a>) of the person on the image, use the
 * person ID in each array element. Each person ID matches the ID field of a <a>ProtectiveEquipmentPerson</a> object
 * returned in the <code>Persons</code> array by <code>DetectProtectiveEquipment</code>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ProtectiveEquipmentSummary implements SdkPojo, Serializable,
        ToCopyableBuilder<ProtectiveEquipmentSummary.Builder, ProtectiveEquipmentSummary> {
    private static final SdkField<List<Integer>> PERSONS_WITH_REQUIRED_EQUIPMENT_FIELD = SdkField
            .<List<Integer>> builder(MarshallingType.LIST)
            .memberName("PersonsWithRequiredEquipment")
            .getter(getter(ProtectiveEquipmentSummary::personsWithRequiredEquipment))
            .setter(setter(Builder::personsWithRequiredEquipment))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PersonsWithRequiredEquipment")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Integer> builder(MarshallingType.INTEGER)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Integer>> PERSONS_WITHOUT_REQUIRED_EQUIPMENT_FIELD = SdkField
            .<List<Integer>> builder(MarshallingType.LIST)
            .memberName("PersonsWithoutRequiredEquipment")
            .getter(getter(ProtectiveEquipmentSummary::personsWithoutRequiredEquipment))
            .setter(setter(Builder::personsWithoutRequiredEquipment))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PersonsWithoutRequiredEquipment")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Integer> builder(MarshallingType.INTEGER)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Integer>> PERSONS_INDETERMINATE_FIELD = SdkField
            .<List<Integer>> builder(MarshallingType.LIST)
            .memberName("PersonsIndeterminate")
            .getter(getter(ProtectiveEquipmentSummary::personsIndeterminate))
            .setter(setter(Builder::personsIndeterminate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PersonsIndeterminate").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Integer> builder(MarshallingType.INTEGER)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            PERSONS_WITH_REQUIRED_EQUIPMENT_FIELD, PERSONS_WITHOUT_REQUIRED_EQUIPMENT_FIELD, PERSONS_INDETERMINATE_FIELD));

    private static final long serialVersionUID = 1L;

    private final List<Integer> personsWithRequiredEquipment;

    private final List<Integer> personsWithoutRequiredEquipment;

    private final List<Integer> personsIndeterminate;

    private ProtectiveEquipmentSummary(BuilderImpl builder) {
        this.personsWithRequiredEquipment = builder.personsWithRequiredEquipment;
        this.personsWithoutRequiredEquipment = builder.personsWithoutRequiredEquipment;
        this.personsIndeterminate = builder.personsIndeterminate;
    }

    /**
     * Returns true if the PersonsWithRequiredEquipment property was specified by the sender (it may be empty), or false
     * if the sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the
     * AWS service.
     */
    public boolean hasPersonsWithRequiredEquipment() {
        return personsWithRequiredEquipment != null && !(personsWithRequiredEquipment instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of IDs for persons who are wearing detected personal protective equipment.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasPersonsWithRequiredEquipment()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An array of IDs for persons who are wearing detected personal protective equipment.
     */
    public List<Integer> personsWithRequiredEquipment() {
        return personsWithRequiredEquipment;
    }

    /**
     * Returns true if the PersonsWithoutRequiredEquipment property was specified by the sender (it may be empty), or
     * false if the sender did not specify the value (it will be empty). For responses returned by the SDK, the sender
     * is the AWS service.
     */
    public boolean hasPersonsWithoutRequiredEquipment() {
        return personsWithoutRequiredEquipment != null && !(personsWithoutRequiredEquipment instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of IDs for persons who are not wearing all of the types of PPE specified in the RequiredEquipmentTypes
     * field of the detected personal protective equipment.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasPersonsWithoutRequiredEquipment()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An array of IDs for persons who are not wearing all of the types of PPE specified in the
     *         RequiredEquipmentTypes field of the detected personal protective equipment.
     */
    public List<Integer> personsWithoutRequiredEquipment() {
        return personsWithoutRequiredEquipment;
    }

    /**
     * Returns true if the PersonsIndeterminate property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasPersonsIndeterminate() {
        return personsIndeterminate != null && !(personsIndeterminate instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of IDs for persons where it was not possible to determine if they are wearing personal protective
     * equipment.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasPersonsIndeterminate()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An array of IDs for persons where it was not possible to determine if they are wearing personal
     *         protective equipment.
     */
    public List<Integer> personsIndeterminate() {
        return personsIndeterminate;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(personsWithRequiredEquipment());
        hashCode = 31 * hashCode + Objects.hashCode(personsWithoutRequiredEquipment());
        hashCode = 31 * hashCode + Objects.hashCode(personsIndeterminate());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ProtectiveEquipmentSummary)) {
            return false;
        }
        ProtectiveEquipmentSummary other = (ProtectiveEquipmentSummary) obj;
        return Objects.equals(personsWithRequiredEquipment(), other.personsWithRequiredEquipment())
                && Objects.equals(personsWithoutRequiredEquipment(), other.personsWithoutRequiredEquipment())
                && Objects.equals(personsIndeterminate(), other.personsIndeterminate());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("ProtectiveEquipmentSummary").add("PersonsWithRequiredEquipment", personsWithRequiredEquipment())
                .add("PersonsWithoutRequiredEquipment", personsWithoutRequiredEquipment())
                .add("PersonsIndeterminate", personsIndeterminate()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "PersonsWithRequiredEquipment":
            return Optional.ofNullable(clazz.cast(personsWithRequiredEquipment()));
        case "PersonsWithoutRequiredEquipment":
            return Optional.ofNullable(clazz.cast(personsWithoutRequiredEquipment()));
        case "PersonsIndeterminate":
            return Optional.ofNullable(clazz.cast(personsIndeterminate()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ProtectiveEquipmentSummary, T> g) {
        return obj -> g.apply((ProtectiveEquipmentSummary) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ProtectiveEquipmentSummary> {
        /**
         * <p>
         * An array of IDs for persons who are wearing detected personal protective equipment.
         * </p>
         * 
         * @param personsWithRequiredEquipment
         *        An array of IDs for persons who are wearing detected personal protective equipment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder personsWithRequiredEquipment(Collection<Integer> personsWithRequiredEquipment);

        /**
         * <p>
         * An array of IDs for persons who are wearing detected personal protective equipment.
         * </p>
         * 
         * @param personsWithRequiredEquipment
         *        An array of IDs for persons who are wearing detected personal protective equipment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder personsWithRequiredEquipment(Integer... personsWithRequiredEquipment);

        /**
         * <p>
         * An array of IDs for persons who are not wearing all of the types of PPE specified in the
         * RequiredEquipmentTypes field of the detected personal protective equipment.
         * </p>
         * 
         * @param personsWithoutRequiredEquipment
         *        An array of IDs for persons who are not wearing all of the types of PPE specified in the
         *        RequiredEquipmentTypes field of the detected personal protective equipment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder personsWithoutRequiredEquipment(Collection<Integer> personsWithoutRequiredEquipment);

        /**
         * <p>
         * An array of IDs for persons who are not wearing all of the types of PPE specified in the
         * RequiredEquipmentTypes field of the detected personal protective equipment.
         * </p>
         * 
         * @param personsWithoutRequiredEquipment
         *        An array of IDs for persons who are not wearing all of the types of PPE specified in the
         *        RequiredEquipmentTypes field of the detected personal protective equipment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder personsWithoutRequiredEquipment(Integer... personsWithoutRequiredEquipment);

        /**
         * <p>
         * An array of IDs for persons where it was not possible to determine if they are wearing personal protective
         * equipment.
         * </p>
         * 
         * @param personsIndeterminate
         *        An array of IDs for persons where it was not possible to determine if they are wearing personal
         *        protective equipment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder personsIndeterminate(Collection<Integer> personsIndeterminate);

        /**
         * <p>
         * An array of IDs for persons where it was not possible to determine if they are wearing personal protective
         * equipment.
         * </p>
         * 
         * @param personsIndeterminate
         *        An array of IDs for persons where it was not possible to determine if they are wearing personal
         *        protective equipment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder personsIndeterminate(Integer... personsIndeterminate);
    }

    static final class BuilderImpl implements Builder {
        private List<Integer> personsWithRequiredEquipment = DefaultSdkAutoConstructList.getInstance();

        private List<Integer> personsWithoutRequiredEquipment = DefaultSdkAutoConstructList.getInstance();

        private List<Integer> personsIndeterminate = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ProtectiveEquipmentSummary model) {
            personsWithRequiredEquipment(model.personsWithRequiredEquipment);
            personsWithoutRequiredEquipment(model.personsWithoutRequiredEquipment);
            personsIndeterminate(model.personsIndeterminate);
        }

        public final Collection<Integer> getPersonsWithRequiredEquipment() {
            return personsWithRequiredEquipment;
        }

        @Override
        public final Builder personsWithRequiredEquipment(Collection<Integer> personsWithRequiredEquipment) {
            this.personsWithRequiredEquipment = ProtectiveEquipmentPersonIdsCopier.copy(personsWithRequiredEquipment);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder personsWithRequiredEquipment(Integer... personsWithRequiredEquipment) {
            personsWithRequiredEquipment(Arrays.asList(personsWithRequiredEquipment));
            return this;
        }

        public final void setPersonsWithRequiredEquipment(Collection<Integer> personsWithRequiredEquipment) {
            this.personsWithRequiredEquipment = ProtectiveEquipmentPersonIdsCopier.copy(personsWithRequiredEquipment);
        }

        public final Collection<Integer> getPersonsWithoutRequiredEquipment() {
            return personsWithoutRequiredEquipment;
        }

        @Override
        public final Builder personsWithoutRequiredEquipment(Collection<Integer> personsWithoutRequiredEquipment) {
            this.personsWithoutRequiredEquipment = ProtectiveEquipmentPersonIdsCopier.copy(personsWithoutRequiredEquipment);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder personsWithoutRequiredEquipment(Integer... personsWithoutRequiredEquipment) {
            personsWithoutRequiredEquipment(Arrays.asList(personsWithoutRequiredEquipment));
            return this;
        }

        public final void setPersonsWithoutRequiredEquipment(Collection<Integer> personsWithoutRequiredEquipment) {
            this.personsWithoutRequiredEquipment = ProtectiveEquipmentPersonIdsCopier.copy(personsWithoutRequiredEquipment);
        }

        public final Collection<Integer> getPersonsIndeterminate() {
            return personsIndeterminate;
        }

        @Override
        public final Builder personsIndeterminate(Collection<Integer> personsIndeterminate) {
            this.personsIndeterminate = ProtectiveEquipmentPersonIdsCopier.copy(personsIndeterminate);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder personsIndeterminate(Integer... personsIndeterminate) {
            personsIndeterminate(Arrays.asList(personsIndeterminate));
            return this;
        }

        public final void setPersonsIndeterminate(Collection<Integer> personsIndeterminate) {
            this.personsIndeterminate = ProtectiveEquipmentPersonIdsCopier.copy(personsIndeterminate);
        }

        @Override
        public ProtectiveEquipmentSummary build() {
            return new ProtectiveEquipmentSummary(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
