/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rekognition.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Structure containing attributes of the face that the algorithm detected.
 * </p>
 * <p>
 * A <code>FaceDetail</code> object contains either the default facial attributes or all facial attributes. The default
 * attributes are <code>BoundingBox</code>, <code>Confidence</code>, <code>Landmarks</code>, <code>Pose</code>, and
 * <code>Quality</code>.
 * </p>
 * <p>
 * <a>GetFaceDetection</a> is the only Amazon Rekognition Video stored video operation that can return a
 * <code>FaceDetail</code> object with all attributes. To specify which attributes to return, use the
 * <code>FaceAttributes</code> input parameter for <a>StartFaceDetection</a>. The following Amazon Rekognition Video
 * operations return only the default attributes. The corresponding Start operations don't have a
 * <code>FaceAttributes</code> input parameter.
 * </p>
 * <ul>
 * <li>
 * <p>
 * GetCelebrityRecognition
 * </p>
 * </li>
 * <li>
 * <p>
 * GetPersonTracking
 * </p>
 * </li>
 * <li>
 * <p>
 * GetFaceSearch
 * </p>
 * </li>
 * </ul>
 * <p>
 * The Amazon Rekognition Image <a>DetectFaces</a> and <a>IndexFaces</a> operations can return all facial attributes. To
 * specify which attributes to return, use the <code>Attributes</code> input parameter for <code>DetectFaces</code>. For
 * <code>IndexFaces</code>, use the <code>DetectAttributes</code> input parameter.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class FaceDetail implements SdkPojo, Serializable, ToCopyableBuilder<FaceDetail.Builder, FaceDetail> {
    private static final SdkField<BoundingBox> BOUNDING_BOX_FIELD = SdkField.<BoundingBox> builder(MarshallingType.SDK_POJO)
            .memberName("BoundingBox").getter(getter(FaceDetail::boundingBox)).setter(setter(Builder::boundingBox))
            .constructor(BoundingBox::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BoundingBox").build()).build();

    private static final SdkField<AgeRange> AGE_RANGE_FIELD = SdkField.<AgeRange> builder(MarshallingType.SDK_POJO)
            .memberName("AgeRange").getter(getter(FaceDetail::ageRange)).setter(setter(Builder::ageRange))
            .constructor(AgeRange::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AgeRange").build()).build();

    private static final SdkField<Smile> SMILE_FIELD = SdkField.<Smile> builder(MarshallingType.SDK_POJO).memberName("Smile")
            .getter(getter(FaceDetail::smile)).setter(setter(Builder::smile)).constructor(Smile::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Smile").build()).build();

    private static final SdkField<Eyeglasses> EYEGLASSES_FIELD = SdkField.<Eyeglasses> builder(MarshallingType.SDK_POJO)
            .memberName("Eyeglasses").getter(getter(FaceDetail::eyeglasses)).setter(setter(Builder::eyeglasses))
            .constructor(Eyeglasses::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Eyeglasses").build()).build();

    private static final SdkField<Sunglasses> SUNGLASSES_FIELD = SdkField.<Sunglasses> builder(MarshallingType.SDK_POJO)
            .memberName("Sunglasses").getter(getter(FaceDetail::sunglasses)).setter(setter(Builder::sunglasses))
            .constructor(Sunglasses::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Sunglasses").build()).build();

    private static final SdkField<Gender> GENDER_FIELD = SdkField.<Gender> builder(MarshallingType.SDK_POJO).memberName("Gender")
            .getter(getter(FaceDetail::gender)).setter(setter(Builder::gender)).constructor(Gender::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Gender").build()).build();

    private static final SdkField<Beard> BEARD_FIELD = SdkField.<Beard> builder(MarshallingType.SDK_POJO).memberName("Beard")
            .getter(getter(FaceDetail::beard)).setter(setter(Builder::beard)).constructor(Beard::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Beard").build()).build();

    private static final SdkField<Mustache> MUSTACHE_FIELD = SdkField.<Mustache> builder(MarshallingType.SDK_POJO)
            .memberName("Mustache").getter(getter(FaceDetail::mustache)).setter(setter(Builder::mustache))
            .constructor(Mustache::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Mustache").build()).build();

    private static final SdkField<EyeOpen> EYES_OPEN_FIELD = SdkField.<EyeOpen> builder(MarshallingType.SDK_POJO)
            .memberName("EyesOpen").getter(getter(FaceDetail::eyesOpen)).setter(setter(Builder::eyesOpen))
            .constructor(EyeOpen::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EyesOpen").build()).build();

    private static final SdkField<MouthOpen> MOUTH_OPEN_FIELD = SdkField.<MouthOpen> builder(MarshallingType.SDK_POJO)
            .memberName("MouthOpen").getter(getter(FaceDetail::mouthOpen)).setter(setter(Builder::mouthOpen))
            .constructor(MouthOpen::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MouthOpen").build()).build();

    private static final SdkField<List<Emotion>> EMOTIONS_FIELD = SdkField
            .<List<Emotion>> builder(MarshallingType.LIST)
            .memberName("Emotions")
            .getter(getter(FaceDetail::emotions))
            .setter(setter(Builder::emotions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Emotions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Emotion> builder(MarshallingType.SDK_POJO)
                                            .constructor(Emotion::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Landmark>> LANDMARKS_FIELD = SdkField
            .<List<Landmark>> builder(MarshallingType.LIST)
            .memberName("Landmarks")
            .getter(getter(FaceDetail::landmarks))
            .setter(setter(Builder::landmarks))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Landmarks").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Landmark> builder(MarshallingType.SDK_POJO)
                                            .constructor(Landmark::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Pose> POSE_FIELD = SdkField.<Pose> builder(MarshallingType.SDK_POJO).memberName("Pose")
            .getter(getter(FaceDetail::pose)).setter(setter(Builder::pose)).constructor(Pose::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Pose").build()).build();

    private static final SdkField<ImageQuality> QUALITY_FIELD = SdkField.<ImageQuality> builder(MarshallingType.SDK_POJO)
            .memberName("Quality").getter(getter(FaceDetail::quality)).setter(setter(Builder::quality))
            .constructor(ImageQuality::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Quality").build()).build();

    private static final SdkField<Float> CONFIDENCE_FIELD = SdkField.<Float> builder(MarshallingType.FLOAT)
            .memberName("Confidence").getter(getter(FaceDetail::confidence)).setter(setter(Builder::confidence))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Confidence").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(BOUNDING_BOX_FIELD,
            AGE_RANGE_FIELD, SMILE_FIELD, EYEGLASSES_FIELD, SUNGLASSES_FIELD, GENDER_FIELD, BEARD_FIELD, MUSTACHE_FIELD,
            EYES_OPEN_FIELD, MOUTH_OPEN_FIELD, EMOTIONS_FIELD, LANDMARKS_FIELD, POSE_FIELD, QUALITY_FIELD, CONFIDENCE_FIELD));

    private static final long serialVersionUID = 1L;

    private final BoundingBox boundingBox;

    private final AgeRange ageRange;

    private final Smile smile;

    private final Eyeglasses eyeglasses;

    private final Sunglasses sunglasses;

    private final Gender gender;

    private final Beard beard;

    private final Mustache mustache;

    private final EyeOpen eyesOpen;

    private final MouthOpen mouthOpen;

    private final List<Emotion> emotions;

    private final List<Landmark> landmarks;

    private final Pose pose;

    private final ImageQuality quality;

    private final Float confidence;

    private FaceDetail(BuilderImpl builder) {
        this.boundingBox = builder.boundingBox;
        this.ageRange = builder.ageRange;
        this.smile = builder.smile;
        this.eyeglasses = builder.eyeglasses;
        this.sunglasses = builder.sunglasses;
        this.gender = builder.gender;
        this.beard = builder.beard;
        this.mustache = builder.mustache;
        this.eyesOpen = builder.eyesOpen;
        this.mouthOpen = builder.mouthOpen;
        this.emotions = builder.emotions;
        this.landmarks = builder.landmarks;
        this.pose = builder.pose;
        this.quality = builder.quality;
        this.confidence = builder.confidence;
    }

    /**
     * <p>
     * Bounding box of the face. Default attribute.
     * </p>
     * 
     * @return Bounding box of the face. Default attribute.
     */
    public final BoundingBox boundingBox() {
        return boundingBox;
    }

    /**
     * <p>
     * The estimated age range, in years, for the face. Low represents the lowest estimated age and High represents the
     * highest estimated age.
     * </p>
     * 
     * @return The estimated age range, in years, for the face. Low represents the lowest estimated age and High
     *         represents the highest estimated age.
     */
    public final AgeRange ageRange() {
        return ageRange;
    }

    /**
     * <p>
     * Indicates whether or not the face is smiling, and the confidence level in the determination.
     * </p>
     * 
     * @return Indicates whether or not the face is smiling, and the confidence level in the determination.
     */
    public final Smile smile() {
        return smile;
    }

    /**
     * <p>
     * Indicates whether or not the face is wearing eye glasses, and the confidence level in the determination.
     * </p>
     * 
     * @return Indicates whether or not the face is wearing eye glasses, and the confidence level in the determination.
     */
    public final Eyeglasses eyeglasses() {
        return eyeglasses;
    }

    /**
     * <p>
     * Indicates whether or not the face is wearing sunglasses, and the confidence level in the determination.
     * </p>
     * 
     * @return Indicates whether or not the face is wearing sunglasses, and the confidence level in the determination.
     */
    public final Sunglasses sunglasses() {
        return sunglasses;
    }

    /**
     * <p>
     * The predicted gender of a detected face.
     * </p>
     * 
     * @return The predicted gender of a detected face.
     */
    public final Gender gender() {
        return gender;
    }

    /**
     * <p>
     * Indicates whether or not the face has a beard, and the confidence level in the determination.
     * </p>
     * 
     * @return Indicates whether or not the face has a beard, and the confidence level in the determination.
     */
    public final Beard beard() {
        return beard;
    }

    /**
     * <p>
     * Indicates whether or not the face has a mustache, and the confidence level in the determination.
     * </p>
     * 
     * @return Indicates whether or not the face has a mustache, and the confidence level in the determination.
     */
    public final Mustache mustache() {
        return mustache;
    }

    /**
     * <p>
     * Indicates whether or not the eyes on the face are open, and the confidence level in the determination.
     * </p>
     * 
     * @return Indicates whether or not the eyes on the face are open, and the confidence level in the determination.
     */
    public final EyeOpen eyesOpen() {
        return eyesOpen;
    }

    /**
     * <p>
     * Indicates whether or not the mouth on the face is open, and the confidence level in the determination.
     * </p>
     * 
     * @return Indicates whether or not the mouth on the face is open, and the confidence level in the determination.
     */
    public final MouthOpen mouthOpen() {
        return mouthOpen;
    }

    /**
     * Returns true if the Emotions property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasEmotions() {
        return emotions != null && !(emotions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The emotions that appear to be expressed on the face, and the confidence level in the determination. The API is
     * only making a determination of the physical appearance of a person's face. It is not a determination of the
     * person’s internal emotional state and should not be used in such a way. For example, a person pretending to have
     * a sad face might not be sad emotionally.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasEmotions()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The emotions that appear to be expressed on the face, and the confidence level in the determination. The
     *         API is only making a determination of the physical appearance of a person's face. It is not a
     *         determination of the person’s internal emotional state and should not be used in such a way. For example,
     *         a person pretending to have a sad face might not be sad emotionally.
     */
    public final List<Emotion> emotions() {
        return emotions;
    }

    /**
     * Returns true if the Landmarks property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasLandmarks() {
        return landmarks != null && !(landmarks instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Indicates the location of landmarks on the face. Default attribute.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasLandmarks()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Indicates the location of landmarks on the face. Default attribute.
     */
    public final List<Landmark> landmarks() {
        return landmarks;
    }

    /**
     * <p>
     * Indicates the pose of the face as determined by its pitch, roll, and yaw. Default attribute.
     * </p>
     * 
     * @return Indicates the pose of the face as determined by its pitch, roll, and yaw. Default attribute.
     */
    public final Pose pose() {
        return pose;
    }

    /**
     * <p>
     * Identifies image brightness and sharpness. Default attribute.
     * </p>
     * 
     * @return Identifies image brightness and sharpness. Default attribute.
     */
    public final ImageQuality quality() {
        return quality;
    }

    /**
     * <p>
     * Confidence level that the bounding box contains a face (and not a different object such as a tree). Default
     * attribute.
     * </p>
     * 
     * @return Confidence level that the bounding box contains a face (and not a different object such as a tree).
     *         Default attribute.
     */
    public final Float confidence() {
        return confidence;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(boundingBox());
        hashCode = 31 * hashCode + Objects.hashCode(ageRange());
        hashCode = 31 * hashCode + Objects.hashCode(smile());
        hashCode = 31 * hashCode + Objects.hashCode(eyeglasses());
        hashCode = 31 * hashCode + Objects.hashCode(sunglasses());
        hashCode = 31 * hashCode + Objects.hashCode(gender());
        hashCode = 31 * hashCode + Objects.hashCode(beard());
        hashCode = 31 * hashCode + Objects.hashCode(mustache());
        hashCode = 31 * hashCode + Objects.hashCode(eyesOpen());
        hashCode = 31 * hashCode + Objects.hashCode(mouthOpen());
        hashCode = 31 * hashCode + Objects.hashCode(hasEmotions() ? emotions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasLandmarks() ? landmarks() : null);
        hashCode = 31 * hashCode + Objects.hashCode(pose());
        hashCode = 31 * hashCode + Objects.hashCode(quality());
        hashCode = 31 * hashCode + Objects.hashCode(confidence());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof FaceDetail)) {
            return false;
        }
        FaceDetail other = (FaceDetail) obj;
        return Objects.equals(boundingBox(), other.boundingBox()) && Objects.equals(ageRange(), other.ageRange())
                && Objects.equals(smile(), other.smile()) && Objects.equals(eyeglasses(), other.eyeglasses())
                && Objects.equals(sunglasses(), other.sunglasses()) && Objects.equals(gender(), other.gender())
                && Objects.equals(beard(), other.beard()) && Objects.equals(mustache(), other.mustache())
                && Objects.equals(eyesOpen(), other.eyesOpen()) && Objects.equals(mouthOpen(), other.mouthOpen())
                && hasEmotions() == other.hasEmotions() && Objects.equals(emotions(), other.emotions())
                && hasLandmarks() == other.hasLandmarks() && Objects.equals(landmarks(), other.landmarks())
                && Objects.equals(pose(), other.pose()) && Objects.equals(quality(), other.quality())
                && Objects.equals(confidence(), other.confidence());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("FaceDetail").add("BoundingBox", boundingBox()).add("AgeRange", ageRange()).add("Smile", smile())
                .add("Eyeglasses", eyeglasses()).add("Sunglasses", sunglasses()).add("Gender", gender()).add("Beard", beard())
                .add("Mustache", mustache()).add("EyesOpen", eyesOpen()).add("MouthOpen", mouthOpen())
                .add("Emotions", hasEmotions() ? emotions() : null).add("Landmarks", hasLandmarks() ? landmarks() : null)
                .add("Pose", pose()).add("Quality", quality()).add("Confidence", confidence()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "BoundingBox":
            return Optional.ofNullable(clazz.cast(boundingBox()));
        case "AgeRange":
            return Optional.ofNullable(clazz.cast(ageRange()));
        case "Smile":
            return Optional.ofNullable(clazz.cast(smile()));
        case "Eyeglasses":
            return Optional.ofNullable(clazz.cast(eyeglasses()));
        case "Sunglasses":
            return Optional.ofNullable(clazz.cast(sunglasses()));
        case "Gender":
            return Optional.ofNullable(clazz.cast(gender()));
        case "Beard":
            return Optional.ofNullable(clazz.cast(beard()));
        case "Mustache":
            return Optional.ofNullable(clazz.cast(mustache()));
        case "EyesOpen":
            return Optional.ofNullable(clazz.cast(eyesOpen()));
        case "MouthOpen":
            return Optional.ofNullable(clazz.cast(mouthOpen()));
        case "Emotions":
            return Optional.ofNullable(clazz.cast(emotions()));
        case "Landmarks":
            return Optional.ofNullable(clazz.cast(landmarks()));
        case "Pose":
            return Optional.ofNullable(clazz.cast(pose()));
        case "Quality":
            return Optional.ofNullable(clazz.cast(quality()));
        case "Confidence":
            return Optional.ofNullable(clazz.cast(confidence()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<FaceDetail, T> g) {
        return obj -> g.apply((FaceDetail) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, FaceDetail> {
        /**
         * <p>
         * Bounding box of the face. Default attribute.
         * </p>
         * 
         * @param boundingBox
         *        Bounding box of the face. Default attribute.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder boundingBox(BoundingBox boundingBox);

        /**
         * <p>
         * Bounding box of the face. Default attribute.
         * </p>
         * This is a convenience that creates an instance of the {@link BoundingBox.Builder} avoiding the need to create
         * one manually via {@link BoundingBox#builder()}.
         *
         * When the {@link Consumer} completes, {@link BoundingBox.Builder#build()} is called immediately and its result
         * is passed to {@link #boundingBox(BoundingBox)}.
         * 
         * @param boundingBox
         *        a consumer that will call methods on {@link BoundingBox.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #boundingBox(BoundingBox)
         */
        default Builder boundingBox(Consumer<BoundingBox.Builder> boundingBox) {
            return boundingBox(BoundingBox.builder().applyMutation(boundingBox).build());
        }

        /**
         * <p>
         * The estimated age range, in years, for the face. Low represents the lowest estimated age and High represents
         * the highest estimated age.
         * </p>
         * 
         * @param ageRange
         *        The estimated age range, in years, for the face. Low represents the lowest estimated age and High
         *        represents the highest estimated age.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ageRange(AgeRange ageRange);

        /**
         * <p>
         * The estimated age range, in years, for the face. Low represents the lowest estimated age and High represents
         * the highest estimated age.
         * </p>
         * This is a convenience that creates an instance of the {@link AgeRange.Builder} avoiding the need to create
         * one manually via {@link AgeRange#builder()}.
         *
         * When the {@link Consumer} completes, {@link AgeRange.Builder#build()} is called immediately and its result is
         * passed to {@link #ageRange(AgeRange)}.
         * 
         * @param ageRange
         *        a consumer that will call methods on {@link AgeRange.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #ageRange(AgeRange)
         */
        default Builder ageRange(Consumer<AgeRange.Builder> ageRange) {
            return ageRange(AgeRange.builder().applyMutation(ageRange).build());
        }

        /**
         * <p>
         * Indicates whether or not the face is smiling, and the confidence level in the determination.
         * </p>
         * 
         * @param smile
         *        Indicates whether or not the face is smiling, and the confidence level in the determination.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder smile(Smile smile);

        /**
         * <p>
         * Indicates whether or not the face is smiling, and the confidence level in the determination.
         * </p>
         * This is a convenience that creates an instance of the {@link Smile.Builder} avoiding the need to create one
         * manually via {@link Smile#builder()}.
         *
         * When the {@link Consumer} completes, {@link Smile.Builder#build()} is called immediately and its result is
         * passed to {@link #smile(Smile)}.
         * 
         * @param smile
         *        a consumer that will call methods on {@link Smile.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #smile(Smile)
         */
        default Builder smile(Consumer<Smile.Builder> smile) {
            return smile(Smile.builder().applyMutation(smile).build());
        }

        /**
         * <p>
         * Indicates whether or not the face is wearing eye glasses, and the confidence level in the determination.
         * </p>
         * 
         * @param eyeglasses
         *        Indicates whether or not the face is wearing eye glasses, and the confidence level in the
         *        determination.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eyeglasses(Eyeglasses eyeglasses);

        /**
         * <p>
         * Indicates whether or not the face is wearing eye glasses, and the confidence level in the determination.
         * </p>
         * This is a convenience that creates an instance of the {@link Eyeglasses.Builder} avoiding the need to create
         * one manually via {@link Eyeglasses#builder()}.
         *
         * When the {@link Consumer} completes, {@link Eyeglasses.Builder#build()} is called immediately and its result
         * is passed to {@link #eyeglasses(Eyeglasses)}.
         * 
         * @param eyeglasses
         *        a consumer that will call methods on {@link Eyeglasses.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #eyeglasses(Eyeglasses)
         */
        default Builder eyeglasses(Consumer<Eyeglasses.Builder> eyeglasses) {
            return eyeglasses(Eyeglasses.builder().applyMutation(eyeglasses).build());
        }

        /**
         * <p>
         * Indicates whether or not the face is wearing sunglasses, and the confidence level in the determination.
         * </p>
         * 
         * @param sunglasses
         *        Indicates whether or not the face is wearing sunglasses, and the confidence level in the
         *        determination.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sunglasses(Sunglasses sunglasses);

        /**
         * <p>
         * Indicates whether or not the face is wearing sunglasses, and the confidence level in the determination.
         * </p>
         * This is a convenience that creates an instance of the {@link Sunglasses.Builder} avoiding the need to create
         * one manually via {@link Sunglasses#builder()}.
         *
         * When the {@link Consumer} completes, {@link Sunglasses.Builder#build()} is called immediately and its result
         * is passed to {@link #sunglasses(Sunglasses)}.
         * 
         * @param sunglasses
         *        a consumer that will call methods on {@link Sunglasses.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #sunglasses(Sunglasses)
         */
        default Builder sunglasses(Consumer<Sunglasses.Builder> sunglasses) {
            return sunglasses(Sunglasses.builder().applyMutation(sunglasses).build());
        }

        /**
         * <p>
         * The predicted gender of a detected face.
         * </p>
         * 
         * @param gender
         *        The predicted gender of a detected face.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gender(Gender gender);

        /**
         * <p>
         * The predicted gender of a detected face.
         * </p>
         * This is a convenience that creates an instance of the {@link Gender.Builder} avoiding the need to create one
         * manually via {@link Gender#builder()}.
         *
         * When the {@link Consumer} completes, {@link Gender.Builder#build()} is called immediately and its result is
         * passed to {@link #gender(Gender)}.
         * 
         * @param gender
         *        a consumer that will call methods on {@link Gender.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #gender(Gender)
         */
        default Builder gender(Consumer<Gender.Builder> gender) {
            return gender(Gender.builder().applyMutation(gender).build());
        }

        /**
         * <p>
         * Indicates whether or not the face has a beard, and the confidence level in the determination.
         * </p>
         * 
         * @param beard
         *        Indicates whether or not the face has a beard, and the confidence level in the determination.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder beard(Beard beard);

        /**
         * <p>
         * Indicates whether or not the face has a beard, and the confidence level in the determination.
         * </p>
         * This is a convenience that creates an instance of the {@link Beard.Builder} avoiding the need to create one
         * manually via {@link Beard#builder()}.
         *
         * When the {@link Consumer} completes, {@link Beard.Builder#build()} is called immediately and its result is
         * passed to {@link #beard(Beard)}.
         * 
         * @param beard
         *        a consumer that will call methods on {@link Beard.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #beard(Beard)
         */
        default Builder beard(Consumer<Beard.Builder> beard) {
            return beard(Beard.builder().applyMutation(beard).build());
        }

        /**
         * <p>
         * Indicates whether or not the face has a mustache, and the confidence level in the determination.
         * </p>
         * 
         * @param mustache
         *        Indicates whether or not the face has a mustache, and the confidence level in the determination.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mustache(Mustache mustache);

        /**
         * <p>
         * Indicates whether or not the face has a mustache, and the confidence level in the determination.
         * </p>
         * This is a convenience that creates an instance of the {@link Mustache.Builder} avoiding the need to create
         * one manually via {@link Mustache#builder()}.
         *
         * When the {@link Consumer} completes, {@link Mustache.Builder#build()} is called immediately and its result is
         * passed to {@link #mustache(Mustache)}.
         * 
         * @param mustache
         *        a consumer that will call methods on {@link Mustache.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #mustache(Mustache)
         */
        default Builder mustache(Consumer<Mustache.Builder> mustache) {
            return mustache(Mustache.builder().applyMutation(mustache).build());
        }

        /**
         * <p>
         * Indicates whether or not the eyes on the face are open, and the confidence level in the determination.
         * </p>
         * 
         * @param eyesOpen
         *        Indicates whether or not the eyes on the face are open, and the confidence level in the determination.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eyesOpen(EyeOpen eyesOpen);

        /**
         * <p>
         * Indicates whether or not the eyes on the face are open, and the confidence level in the determination.
         * </p>
         * This is a convenience that creates an instance of the {@link EyeOpen.Builder} avoiding the need to create one
         * manually via {@link EyeOpen#builder()}.
         *
         * When the {@link Consumer} completes, {@link EyeOpen.Builder#build()} is called immediately and its result is
         * passed to {@link #eyesOpen(EyeOpen)}.
         * 
         * @param eyesOpen
         *        a consumer that will call methods on {@link EyeOpen.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #eyesOpen(EyeOpen)
         */
        default Builder eyesOpen(Consumer<EyeOpen.Builder> eyesOpen) {
            return eyesOpen(EyeOpen.builder().applyMutation(eyesOpen).build());
        }

        /**
         * <p>
         * Indicates whether or not the mouth on the face is open, and the confidence level in the determination.
         * </p>
         * 
         * @param mouthOpen
         *        Indicates whether or not the mouth on the face is open, and the confidence level in the determination.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mouthOpen(MouthOpen mouthOpen);

        /**
         * <p>
         * Indicates whether or not the mouth on the face is open, and the confidence level in the determination.
         * </p>
         * This is a convenience that creates an instance of the {@link MouthOpen.Builder} avoiding the need to create
         * one manually via {@link MouthOpen#builder()}.
         *
         * When the {@link Consumer} completes, {@link MouthOpen.Builder#build()} is called immediately and its result
         * is passed to {@link #mouthOpen(MouthOpen)}.
         * 
         * @param mouthOpen
         *        a consumer that will call methods on {@link MouthOpen.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #mouthOpen(MouthOpen)
         */
        default Builder mouthOpen(Consumer<MouthOpen.Builder> mouthOpen) {
            return mouthOpen(MouthOpen.builder().applyMutation(mouthOpen).build());
        }

        /**
         * <p>
         * The emotions that appear to be expressed on the face, and the confidence level in the determination. The API
         * is only making a determination of the physical appearance of a person's face. It is not a determination of
         * the person’s internal emotional state and should not be used in such a way. For example, a person pretending
         * to have a sad face might not be sad emotionally.
         * </p>
         * 
         * @param emotions
         *        The emotions that appear to be expressed on the face, and the confidence level in the determination.
         *        The API is only making a determination of the physical appearance of a person's face. It is not a
         *        determination of the person’s internal emotional state and should not be used in such a way. For
         *        example, a person pretending to have a sad face might not be sad emotionally.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder emotions(Collection<Emotion> emotions);

        /**
         * <p>
         * The emotions that appear to be expressed on the face, and the confidence level in the determination. The API
         * is only making a determination of the physical appearance of a person's face. It is not a determination of
         * the person’s internal emotional state and should not be used in such a way. For example, a person pretending
         * to have a sad face might not be sad emotionally.
         * </p>
         * 
         * @param emotions
         *        The emotions that appear to be expressed on the face, and the confidence level in the determination.
         *        The API is only making a determination of the physical appearance of a person's face. It is not a
         *        determination of the person’s internal emotional state and should not be used in such a way. For
         *        example, a person pretending to have a sad face might not be sad emotionally.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder emotions(Emotion... emotions);

        /**
         * <p>
         * The emotions that appear to be expressed on the face, and the confidence level in the determination. The API
         * is only making a determination of the physical appearance of a person's face. It is not a determination of
         * the person’s internal emotional state and should not be used in such a way. For example, a person pretending
         * to have a sad face might not be sad emotionally.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Emotion>.Builder} avoiding the need to
         * create one manually via {@link List<Emotion>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Emotion>.Builder#build()} is called immediately and its
         * result is passed to {@link #emotions(List<Emotion>)}.
         * 
         * @param emotions
         *        a consumer that will call methods on {@link List<Emotion>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #emotions(List<Emotion>)
         */
        Builder emotions(Consumer<Emotion.Builder>... emotions);

        /**
         * <p>
         * Indicates the location of landmarks on the face. Default attribute.
         * </p>
         * 
         * @param landmarks
         *        Indicates the location of landmarks on the face. Default attribute.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder landmarks(Collection<Landmark> landmarks);

        /**
         * <p>
         * Indicates the location of landmarks on the face. Default attribute.
         * </p>
         * 
         * @param landmarks
         *        Indicates the location of landmarks on the face. Default attribute.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder landmarks(Landmark... landmarks);

        /**
         * <p>
         * Indicates the location of landmarks on the face. Default attribute.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Landmark>.Builder} avoiding the need to
         * create one manually via {@link List<Landmark>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Landmark>.Builder#build()} is called immediately and its
         * result is passed to {@link #landmarks(List<Landmark>)}.
         * 
         * @param landmarks
         *        a consumer that will call methods on {@link List<Landmark>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #landmarks(List<Landmark>)
         */
        Builder landmarks(Consumer<Landmark.Builder>... landmarks);

        /**
         * <p>
         * Indicates the pose of the face as determined by its pitch, roll, and yaw. Default attribute.
         * </p>
         * 
         * @param pose
         *        Indicates the pose of the face as determined by its pitch, roll, and yaw. Default attribute.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder pose(Pose pose);

        /**
         * <p>
         * Indicates the pose of the face as determined by its pitch, roll, and yaw. Default attribute.
         * </p>
         * This is a convenience that creates an instance of the {@link Pose.Builder} avoiding the need to create one
         * manually via {@link Pose#builder()}.
         *
         * When the {@link Consumer} completes, {@link Pose.Builder#build()} is called immediately and its result is
         * passed to {@link #pose(Pose)}.
         * 
         * @param pose
         *        a consumer that will call methods on {@link Pose.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #pose(Pose)
         */
        default Builder pose(Consumer<Pose.Builder> pose) {
            return pose(Pose.builder().applyMutation(pose).build());
        }

        /**
         * <p>
         * Identifies image brightness and sharpness. Default attribute.
         * </p>
         * 
         * @param quality
         *        Identifies image brightness and sharpness. Default attribute.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder quality(ImageQuality quality);

        /**
         * <p>
         * Identifies image brightness and sharpness. Default attribute.
         * </p>
         * This is a convenience that creates an instance of the {@link ImageQuality.Builder} avoiding the need to
         * create one manually via {@link ImageQuality#builder()}.
         *
         * When the {@link Consumer} completes, {@link ImageQuality.Builder#build()} is called immediately and its
         * result is passed to {@link #quality(ImageQuality)}.
         * 
         * @param quality
         *        a consumer that will call methods on {@link ImageQuality.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #quality(ImageQuality)
         */
        default Builder quality(Consumer<ImageQuality.Builder> quality) {
            return quality(ImageQuality.builder().applyMutation(quality).build());
        }

        /**
         * <p>
         * Confidence level that the bounding box contains a face (and not a different object such as a tree). Default
         * attribute.
         * </p>
         * 
         * @param confidence
         *        Confidence level that the bounding box contains a face (and not a different object such as a tree).
         *        Default attribute.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder confidence(Float confidence);
    }

    static final class BuilderImpl implements Builder {
        private BoundingBox boundingBox;

        private AgeRange ageRange;

        private Smile smile;

        private Eyeglasses eyeglasses;

        private Sunglasses sunglasses;

        private Gender gender;

        private Beard beard;

        private Mustache mustache;

        private EyeOpen eyesOpen;

        private MouthOpen mouthOpen;

        private List<Emotion> emotions = DefaultSdkAutoConstructList.getInstance();

        private List<Landmark> landmarks = DefaultSdkAutoConstructList.getInstance();

        private Pose pose;

        private ImageQuality quality;

        private Float confidence;

        private BuilderImpl() {
        }

        private BuilderImpl(FaceDetail model) {
            boundingBox(model.boundingBox);
            ageRange(model.ageRange);
            smile(model.smile);
            eyeglasses(model.eyeglasses);
            sunglasses(model.sunglasses);
            gender(model.gender);
            beard(model.beard);
            mustache(model.mustache);
            eyesOpen(model.eyesOpen);
            mouthOpen(model.mouthOpen);
            emotions(model.emotions);
            landmarks(model.landmarks);
            pose(model.pose);
            quality(model.quality);
            confidence(model.confidence);
        }

        public final BoundingBox.Builder getBoundingBox() {
            return boundingBox != null ? boundingBox.toBuilder() : null;
        }

        public final void setBoundingBox(BoundingBox.BuilderImpl boundingBox) {
            this.boundingBox = boundingBox != null ? boundingBox.build() : null;
        }

        @Override
        @Transient
        public final Builder boundingBox(BoundingBox boundingBox) {
            this.boundingBox = boundingBox;
            return this;
        }

        public final AgeRange.Builder getAgeRange() {
            return ageRange != null ? ageRange.toBuilder() : null;
        }

        public final void setAgeRange(AgeRange.BuilderImpl ageRange) {
            this.ageRange = ageRange != null ? ageRange.build() : null;
        }

        @Override
        @Transient
        public final Builder ageRange(AgeRange ageRange) {
            this.ageRange = ageRange;
            return this;
        }

        public final Smile.Builder getSmile() {
            return smile != null ? smile.toBuilder() : null;
        }

        public final void setSmile(Smile.BuilderImpl smile) {
            this.smile = smile != null ? smile.build() : null;
        }

        @Override
        @Transient
        public final Builder smile(Smile smile) {
            this.smile = smile;
            return this;
        }

        public final Eyeglasses.Builder getEyeglasses() {
            return eyeglasses != null ? eyeglasses.toBuilder() : null;
        }

        public final void setEyeglasses(Eyeglasses.BuilderImpl eyeglasses) {
            this.eyeglasses = eyeglasses != null ? eyeglasses.build() : null;
        }

        @Override
        @Transient
        public final Builder eyeglasses(Eyeglasses eyeglasses) {
            this.eyeglasses = eyeglasses;
            return this;
        }

        public final Sunglasses.Builder getSunglasses() {
            return sunglasses != null ? sunglasses.toBuilder() : null;
        }

        public final void setSunglasses(Sunglasses.BuilderImpl sunglasses) {
            this.sunglasses = sunglasses != null ? sunglasses.build() : null;
        }

        @Override
        @Transient
        public final Builder sunglasses(Sunglasses sunglasses) {
            this.sunglasses = sunglasses;
            return this;
        }

        public final Gender.Builder getGender() {
            return gender != null ? gender.toBuilder() : null;
        }

        public final void setGender(Gender.BuilderImpl gender) {
            this.gender = gender != null ? gender.build() : null;
        }

        @Override
        @Transient
        public final Builder gender(Gender gender) {
            this.gender = gender;
            return this;
        }

        public final Beard.Builder getBeard() {
            return beard != null ? beard.toBuilder() : null;
        }

        public final void setBeard(Beard.BuilderImpl beard) {
            this.beard = beard != null ? beard.build() : null;
        }

        @Override
        @Transient
        public final Builder beard(Beard beard) {
            this.beard = beard;
            return this;
        }

        public final Mustache.Builder getMustache() {
            return mustache != null ? mustache.toBuilder() : null;
        }

        public final void setMustache(Mustache.BuilderImpl mustache) {
            this.mustache = mustache != null ? mustache.build() : null;
        }

        @Override
        @Transient
        public final Builder mustache(Mustache mustache) {
            this.mustache = mustache;
            return this;
        }

        public final EyeOpen.Builder getEyesOpen() {
            return eyesOpen != null ? eyesOpen.toBuilder() : null;
        }

        public final void setEyesOpen(EyeOpen.BuilderImpl eyesOpen) {
            this.eyesOpen = eyesOpen != null ? eyesOpen.build() : null;
        }

        @Override
        @Transient
        public final Builder eyesOpen(EyeOpen eyesOpen) {
            this.eyesOpen = eyesOpen;
            return this;
        }

        public final MouthOpen.Builder getMouthOpen() {
            return mouthOpen != null ? mouthOpen.toBuilder() : null;
        }

        public final void setMouthOpen(MouthOpen.BuilderImpl mouthOpen) {
            this.mouthOpen = mouthOpen != null ? mouthOpen.build() : null;
        }

        @Override
        @Transient
        public final Builder mouthOpen(MouthOpen mouthOpen) {
            this.mouthOpen = mouthOpen;
            return this;
        }

        public final List<Emotion.Builder> getEmotions() {
            List<Emotion.Builder> result = EmotionsCopier.copyToBuilder(this.emotions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setEmotions(Collection<Emotion.BuilderImpl> emotions) {
            this.emotions = EmotionsCopier.copyFromBuilder(emotions);
        }

        @Override
        @Transient
        public final Builder emotions(Collection<Emotion> emotions) {
            this.emotions = EmotionsCopier.copy(emotions);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder emotions(Emotion... emotions) {
            emotions(Arrays.asList(emotions));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder emotions(Consumer<Emotion.Builder>... emotions) {
            emotions(Stream.of(emotions).map(c -> Emotion.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final List<Landmark.Builder> getLandmarks() {
            List<Landmark.Builder> result = LandmarksCopier.copyToBuilder(this.landmarks);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setLandmarks(Collection<Landmark.BuilderImpl> landmarks) {
            this.landmarks = LandmarksCopier.copyFromBuilder(landmarks);
        }

        @Override
        @Transient
        public final Builder landmarks(Collection<Landmark> landmarks) {
            this.landmarks = LandmarksCopier.copy(landmarks);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder landmarks(Landmark... landmarks) {
            landmarks(Arrays.asList(landmarks));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder landmarks(Consumer<Landmark.Builder>... landmarks) {
            landmarks(Stream.of(landmarks).map(c -> Landmark.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Pose.Builder getPose() {
            return pose != null ? pose.toBuilder() : null;
        }

        public final void setPose(Pose.BuilderImpl pose) {
            this.pose = pose != null ? pose.build() : null;
        }

        @Override
        @Transient
        public final Builder pose(Pose pose) {
            this.pose = pose;
            return this;
        }

        public final ImageQuality.Builder getQuality() {
            return quality != null ? quality.toBuilder() : null;
        }

        public final void setQuality(ImageQuality.BuilderImpl quality) {
            this.quality = quality != null ? quality.build() : null;
        }

        @Override
        @Transient
        public final Builder quality(ImageQuality quality) {
            this.quality = quality;
            return this;
        }

        public final Float getConfidence() {
            return confidence;
        }

        public final void setConfidence(Float confidence) {
            this.confidence = confidence;
        }

        @Override
        @Transient
        public final Builder confidence(Float confidence) {
            this.confidence = confidence;
            return this;
        }

        @Override
        public FaceDetail build() {
            return new FaceDetail(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
