/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rekognition.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about a word or line of text detected by <a>DetectText</a>.
 * </p>
 * <p>
 * The <code>DetectedText</code> field contains the text that Amazon Rekognition detected in the image.
 * </p>
 * <p>
 * Every word and line has an identifier (<code>Id</code>). Each word belongs to a line and has a parent identifier (
 * <code>ParentId</code>) that identifies the line of text in which the word appears. The word <code>Id</code> is also
 * an index for the word within a line of words.
 * </p>
 * <p>
 * For more information, see Detecting Text in the Amazon Rekognition Developer Guide.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class TextDetection implements SdkPojo, Serializable, ToCopyableBuilder<TextDetection.Builder, TextDetection> {
    private static final SdkField<String> DETECTED_TEXT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DetectedText").getter(getter(TextDetection::detectedText)).setter(setter(Builder::detectedText))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DetectedText").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Type")
            .getter(getter(TextDetection::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type").build()).build();

    private static final SdkField<Integer> ID_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER).memberName("Id")
            .getter(getter(TextDetection::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").build()).build();

    private static final SdkField<Integer> PARENT_ID_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("ParentId").getter(getter(TextDetection::parentId)).setter(setter(Builder::parentId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ParentId").build()).build();

    private static final SdkField<Float> CONFIDENCE_FIELD = SdkField.<Float> builder(MarshallingType.FLOAT)
            .memberName("Confidence").getter(getter(TextDetection::confidence)).setter(setter(Builder::confidence))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Confidence").build()).build();

    private static final SdkField<Geometry> GEOMETRY_FIELD = SdkField.<Geometry> builder(MarshallingType.SDK_POJO)
            .memberName("Geometry").getter(getter(TextDetection::geometry)).setter(setter(Builder::geometry))
            .constructor(Geometry::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Geometry").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DETECTED_TEXT_FIELD,
            TYPE_FIELD, ID_FIELD, PARENT_ID_FIELD, CONFIDENCE_FIELD, GEOMETRY_FIELD));

    private static final long serialVersionUID = 1L;

    private final String detectedText;

    private final String type;

    private final Integer id;

    private final Integer parentId;

    private final Float confidence;

    private final Geometry geometry;

    private TextDetection(BuilderImpl builder) {
        this.detectedText = builder.detectedText;
        this.type = builder.type;
        this.id = builder.id;
        this.parentId = builder.parentId;
        this.confidence = builder.confidence;
        this.geometry = builder.geometry;
    }

    /**
     * <p>
     * The word or line of text recognized by Amazon Rekognition.
     * </p>
     * 
     * @return The word or line of text recognized by Amazon Rekognition.
     */
    public final String detectedText() {
        return detectedText;
    }

    /**
     * <p>
     * The type of text that was detected.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link TextTypes#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of text that was detected.
     * @see TextTypes
     */
    public final TextTypes type() {
        return TextTypes.fromValue(type);
    }

    /**
     * <p>
     * The type of text that was detected.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link TextTypes#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of text that was detected.
     * @see TextTypes
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * The identifier for the detected text. The identifier is only unique for a single call to <code>DetectText</code>.
     * </p>
     * 
     * @return The identifier for the detected text. The identifier is only unique for a single call to
     *         <code>DetectText</code>.
     */
    public final Integer id() {
        return id;
    }

    /**
     * <p>
     * The Parent identifier for the detected text identified by the value of <code>ID</code>. If the type of detected
     * text is <code>LINE</code>, the value of <code>ParentId</code> is <code>Null</code>.
     * </p>
     * 
     * @return The Parent identifier for the detected text identified by the value of <code>ID</code>. If the type of
     *         detected text is <code>LINE</code>, the value of <code>ParentId</code> is <code>Null</code>.
     */
    public final Integer parentId() {
        return parentId;
    }

    /**
     * <p>
     * The confidence that Amazon Rekognition has in the accuracy of the detected text and the accuracy of the geometry
     * points around the detected text.
     * </p>
     * 
     * @return The confidence that Amazon Rekognition has in the accuracy of the detected text and the accuracy of the
     *         geometry points around the detected text.
     */
    public final Float confidence() {
        return confidence;
    }

    /**
     * <p>
     * The location of the detected text on the image. Includes an axis aligned coarse bounding box surrounding the text
     * and a finer grain polygon for more accurate spatial information.
     * </p>
     * 
     * @return The location of the detected text on the image. Includes an axis aligned coarse bounding box surrounding
     *         the text and a finer grain polygon for more accurate spatial information.
     */
    public final Geometry geometry() {
        return geometry;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(detectedText());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(parentId());
        hashCode = 31 * hashCode + Objects.hashCode(confidence());
        hashCode = 31 * hashCode + Objects.hashCode(geometry());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TextDetection)) {
            return false;
        }
        TextDetection other = (TextDetection) obj;
        return Objects.equals(detectedText(), other.detectedText()) && Objects.equals(typeAsString(), other.typeAsString())
                && Objects.equals(id(), other.id()) && Objects.equals(parentId(), other.parentId())
                && Objects.equals(confidence(), other.confidence()) && Objects.equals(geometry(), other.geometry());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("TextDetection").add("DetectedText", detectedText()).add("Type", typeAsString()).add("Id", id())
                .add("ParentId", parentId()).add("Confidence", confidence()).add("Geometry", geometry()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DetectedText":
            return Optional.ofNullable(clazz.cast(detectedText()));
        case "Type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "ParentId":
            return Optional.ofNullable(clazz.cast(parentId()));
        case "Confidence":
            return Optional.ofNullable(clazz.cast(confidence()));
        case "Geometry":
            return Optional.ofNullable(clazz.cast(geometry()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<TextDetection, T> g) {
        return obj -> g.apply((TextDetection) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, TextDetection> {
        /**
         * <p>
         * The word or line of text recognized by Amazon Rekognition.
         * </p>
         * 
         * @param detectedText
         *        The word or line of text recognized by Amazon Rekognition.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder detectedText(String detectedText);

        /**
         * <p>
         * The type of text that was detected.
         * </p>
         * 
         * @param type
         *        The type of text that was detected.
         * @see TextTypes
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TextTypes
         */
        Builder type(String type);

        /**
         * <p>
         * The type of text that was detected.
         * </p>
         * 
         * @param type
         *        The type of text that was detected.
         * @see TextTypes
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TextTypes
         */
        Builder type(TextTypes type);

        /**
         * <p>
         * The identifier for the detected text. The identifier is only unique for a single call to
         * <code>DetectText</code>.
         * </p>
         * 
         * @param id
         *        The identifier for the detected text. The identifier is only unique for a single call to
         *        <code>DetectText</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(Integer id);

        /**
         * <p>
         * The Parent identifier for the detected text identified by the value of <code>ID</code>. If the type of
         * detected text is <code>LINE</code>, the value of <code>ParentId</code> is <code>Null</code>.
         * </p>
         * 
         * @param parentId
         *        The Parent identifier for the detected text identified by the value of <code>ID</code>. If the type of
         *        detected text is <code>LINE</code>, the value of <code>ParentId</code> is <code>Null</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parentId(Integer parentId);

        /**
         * <p>
         * The confidence that Amazon Rekognition has in the accuracy of the detected text and the accuracy of the
         * geometry points around the detected text.
         * </p>
         * 
         * @param confidence
         *        The confidence that Amazon Rekognition has in the accuracy of the detected text and the accuracy of
         *        the geometry points around the detected text.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder confidence(Float confidence);

        /**
         * <p>
         * The location of the detected text on the image. Includes an axis aligned coarse bounding box surrounding the
         * text and a finer grain polygon for more accurate spatial information.
         * </p>
         * 
         * @param geometry
         *        The location of the detected text on the image. Includes an axis aligned coarse bounding box
         *        surrounding the text and a finer grain polygon for more accurate spatial information.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder geometry(Geometry geometry);

        /**
         * <p>
         * The location of the detected text on the image. Includes an axis aligned coarse bounding box surrounding the
         * text and a finer grain polygon for more accurate spatial information.
         * </p>
         * This is a convenience that creates an instance of the {@link Geometry.Builder} avoiding the need to create
         * one manually via {@link Geometry#builder()}.
         *
         * When the {@link Consumer} completes, {@link Geometry.Builder#build()} is called immediately and its result is
         * passed to {@link #geometry(Geometry)}.
         * 
         * @param geometry
         *        a consumer that will call methods on {@link Geometry.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #geometry(Geometry)
         */
        default Builder geometry(Consumer<Geometry.Builder> geometry) {
            return geometry(Geometry.builder().applyMutation(geometry).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String detectedText;

        private String type;

        private Integer id;

        private Integer parentId;

        private Float confidence;

        private Geometry geometry;

        private BuilderImpl() {
        }

        private BuilderImpl(TextDetection model) {
            detectedText(model.detectedText);
            type(model.type);
            id(model.id);
            parentId(model.parentId);
            confidence(model.confidence);
            geometry(model.geometry);
        }

        public final String getDetectedText() {
            return detectedText;
        }

        public final void setDetectedText(String detectedText) {
            this.detectedText = detectedText;
        }

        @Override
        @Transient
        public final Builder detectedText(String detectedText) {
            this.detectedText = detectedText;
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        @Transient
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        @Transient
        public final Builder type(TextTypes type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final Integer getId() {
            return id;
        }

        public final void setId(Integer id) {
            this.id = id;
        }

        @Override
        @Transient
        public final Builder id(Integer id) {
            this.id = id;
            return this;
        }

        public final Integer getParentId() {
            return parentId;
        }

        public final void setParentId(Integer parentId) {
            this.parentId = parentId;
        }

        @Override
        @Transient
        public final Builder parentId(Integer parentId) {
            this.parentId = parentId;
            return this;
        }

        public final Float getConfidence() {
            return confidence;
        }

        public final void setConfidence(Float confidence) {
            this.confidence = confidence;
        }

        @Override
        @Transient
        public final Builder confidence(Float confidence) {
            this.confidence = confidence;
            return this;
        }

        public final Geometry.Builder getGeometry() {
            return geometry != null ? geometry.toBuilder() : null;
        }

        public final void setGeometry(Geometry.BuilderImpl geometry) {
            this.geometry = geometry != null ? geometry.build() : null;
        }

        @Override
        @Transient
        public final Builder geometry(Geometry geometry) {
            this.geometry = geometry;
            return this;
        }

        @Override
        public TextDetection build() {
            return new TextDetection(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
