/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rekognition.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about a person whose face matches a face(s) in an Amazon Rekognition collection. Includes information
 * about the faces in the Amazon Rekognition collection (<a>FaceMatch</a>), information about the person
 * (<a>PersonDetail</a>), and the time stamp for when the person was detected in a video. An array of
 * <code>PersonMatch</code> objects is returned by <a>GetFaceSearch</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PersonMatch implements SdkPojo, Serializable, ToCopyableBuilder<PersonMatch.Builder, PersonMatch> {
    private static final SdkField<Long> TIMESTAMP_FIELD = SdkField.<Long> builder(MarshallingType.LONG).memberName("Timestamp")
            .getter(getter(PersonMatch::timestamp)).setter(setter(Builder::timestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Timestamp").build()).build();

    private static final SdkField<PersonDetail> PERSON_FIELD = SdkField.<PersonDetail> builder(MarshallingType.SDK_POJO)
            .memberName("Person").getter(getter(PersonMatch::person)).setter(setter(Builder::person))
            .constructor(PersonDetail::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Person").build()).build();

    private static final SdkField<List<FaceMatch>> FACE_MATCHES_FIELD = SdkField
            .<List<FaceMatch>> builder(MarshallingType.LIST)
            .memberName("FaceMatches")
            .getter(getter(PersonMatch::faceMatches))
            .setter(setter(Builder::faceMatches))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FaceMatches").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<FaceMatch> builder(MarshallingType.SDK_POJO)
                                            .constructor(FaceMatch::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TIMESTAMP_FIELD, PERSON_FIELD,
            FACE_MATCHES_FIELD));

    private static final long serialVersionUID = 1L;

    private final Long timestamp;

    private final PersonDetail person;

    private final List<FaceMatch> faceMatches;

    private PersonMatch(BuilderImpl builder) {
        this.timestamp = builder.timestamp;
        this.person = builder.person;
        this.faceMatches = builder.faceMatches;
    }

    /**
     * <p>
     * The time, in milliseconds from the beginning of the video, that the person was matched in the video.
     * </p>
     * 
     * @return The time, in milliseconds from the beginning of the video, that the person was matched in the video.
     */
    public final Long timestamp() {
        return timestamp;
    }

    /**
     * <p>
     * Information about the matched person.
     * </p>
     * 
     * @return Information about the matched person.
     */
    public final PersonDetail person() {
        return person;
    }

    /**
     * For responses, this returns true if the service returned a value for the FaceMatches property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasFaceMatches() {
        return faceMatches != null && !(faceMatches instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Information about the faces in the input collection that match the face of a person in the video.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFaceMatches} method.
     * </p>
     * 
     * @return Information about the faces in the input collection that match the face of a person in the video.
     */
    public final List<FaceMatch> faceMatches() {
        return faceMatches;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(timestamp());
        hashCode = 31 * hashCode + Objects.hashCode(person());
        hashCode = 31 * hashCode + Objects.hashCode(hasFaceMatches() ? faceMatches() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PersonMatch)) {
            return false;
        }
        PersonMatch other = (PersonMatch) obj;
        return Objects.equals(timestamp(), other.timestamp()) && Objects.equals(person(), other.person())
                && hasFaceMatches() == other.hasFaceMatches() && Objects.equals(faceMatches(), other.faceMatches());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PersonMatch").add("Timestamp", timestamp()).add("Person", person())
                .add("FaceMatches", hasFaceMatches() ? faceMatches() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Timestamp":
            return Optional.ofNullable(clazz.cast(timestamp()));
        case "Person":
            return Optional.ofNullable(clazz.cast(person()));
        case "FaceMatches":
            return Optional.ofNullable(clazz.cast(faceMatches()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PersonMatch, T> g) {
        return obj -> g.apply((PersonMatch) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PersonMatch> {
        /**
         * <p>
         * The time, in milliseconds from the beginning of the video, that the person was matched in the video.
         * </p>
         * 
         * @param timestamp
         *        The time, in milliseconds from the beginning of the video, that the person was matched in the video.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timestamp(Long timestamp);

        /**
         * <p>
         * Information about the matched person.
         * </p>
         * 
         * @param person
         *        Information about the matched person.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder person(PersonDetail person);

        /**
         * <p>
         * Information about the matched person.
         * </p>
         * This is a convenience that creates an instance of the {@link PersonDetail.Builder} avoiding the need to
         * create one manually via {@link PersonDetail#builder()}.
         *
         * When the {@link Consumer} completes, {@link PersonDetail.Builder#build()} is called immediately and its
         * result is passed to {@link #person(PersonDetail)}.
         * 
         * @param person
         *        a consumer that will call methods on {@link PersonDetail.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #person(PersonDetail)
         */
        default Builder person(Consumer<PersonDetail.Builder> person) {
            return person(PersonDetail.builder().applyMutation(person).build());
        }

        /**
         * <p>
         * Information about the faces in the input collection that match the face of a person in the video.
         * </p>
         * 
         * @param faceMatches
         *        Information about the faces in the input collection that match the face of a person in the video.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder faceMatches(Collection<FaceMatch> faceMatches);

        /**
         * <p>
         * Information about the faces in the input collection that match the face of a person in the video.
         * </p>
         * 
         * @param faceMatches
         *        Information about the faces in the input collection that match the face of a person in the video.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder faceMatches(FaceMatch... faceMatches);

        /**
         * <p>
         * Information about the faces in the input collection that match the face of a person in the video.
         * </p>
         * This is a convenience that creates an instance of the {@link List<FaceMatch>.Builder} avoiding the need to
         * create one manually via {@link List<FaceMatch>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<FaceMatch>.Builder#build()} is called immediately and its
         * result is passed to {@link #faceMatches(List<FaceMatch>)}.
         * 
         * @param faceMatches
         *        a consumer that will call methods on {@link List<FaceMatch>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #faceMatches(List<FaceMatch>)
         */
        Builder faceMatches(Consumer<FaceMatch.Builder>... faceMatches);
    }

    static final class BuilderImpl implements Builder {
        private Long timestamp;

        private PersonDetail person;

        private List<FaceMatch> faceMatches = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(PersonMatch model) {
            timestamp(model.timestamp);
            person(model.person);
            faceMatches(model.faceMatches);
        }

        public final Long getTimestamp() {
            return timestamp;
        }

        public final void setTimestamp(Long timestamp) {
            this.timestamp = timestamp;
        }

        @Override
        @Transient
        public final Builder timestamp(Long timestamp) {
            this.timestamp = timestamp;
            return this;
        }

        public final PersonDetail.Builder getPerson() {
            return person != null ? person.toBuilder() : null;
        }

        public final void setPerson(PersonDetail.BuilderImpl person) {
            this.person = person != null ? person.build() : null;
        }

        @Override
        @Transient
        public final Builder person(PersonDetail person) {
            this.person = person;
            return this;
        }

        public final List<FaceMatch.Builder> getFaceMatches() {
            List<FaceMatch.Builder> result = FaceMatchListCopier.copyToBuilder(this.faceMatches);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setFaceMatches(Collection<FaceMatch.BuilderImpl> faceMatches) {
            this.faceMatches = FaceMatchListCopier.copyFromBuilder(faceMatches);
        }

        @Override
        @Transient
        public final Builder faceMatches(Collection<FaceMatch> faceMatches) {
            this.faceMatches = FaceMatchListCopier.copy(faceMatches);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder faceMatches(FaceMatch... faceMatches) {
            faceMatches(Arrays.asList(faceMatches));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder faceMatches(Consumer<FaceMatch.Builder>... faceMatches) {
            faceMatches(Stream.of(faceMatches).map(c -> FaceMatch.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public PersonMatch build() {
            return new PersonMatch(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
