/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rekognition.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.retry.backoff.BackoffStrategy;
import software.amazon.awssdk.core.retry.backoff.FixedDelayBackoffStrategy;
import software.amazon.awssdk.core.waiters.Waiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.services.rekognition.RekognitionClient;
import software.amazon.awssdk.services.rekognition.model.DescribeProjectVersionsRequest;
import software.amazon.awssdk.services.rekognition.model.DescribeProjectVersionsResponse;
import software.amazon.awssdk.services.rekognition.model.RekognitionRequest;
import software.amazon.awssdk.services.rekognition.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultRekognitionWaiter implements RekognitionWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private final RekognitionClient client;

    private final AttributeMap managedResources;

    private final Waiter<DescribeProjectVersionsResponse> projectVersionTrainingCompletedWaiter;

    private final Waiter<DescribeProjectVersionsResponse> projectVersionRunningWaiter;

    private DefaultRekognitionWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = RekognitionClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        managedResources = attributeMapBuilder.build();
        this.projectVersionTrainingCompletedWaiter = Waiter.builder(DescribeProjectVersionsResponse.class)
                .acceptors(projectVersionTrainingCompletedWaiterAcceptors())
                .overrideConfiguration(projectVersionTrainingCompletedWaiterConfig(builder.overrideConfiguration)).build();
        this.projectVersionRunningWaiter = Waiter.builder(DescribeProjectVersionsResponse.class)
                .acceptors(projectVersionRunningWaiterAcceptors())
                .overrideConfiguration(projectVersionRunningWaiterConfig(builder.overrideConfiguration)).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public WaiterResponse<DescribeProjectVersionsResponse> waitUntilProjectVersionRunning(
            DescribeProjectVersionsRequest describeProjectVersionsRequest) {
        return projectVersionRunningWaiter.run(() -> client
                .describeProjectVersions(applyWaitersUserAgent(describeProjectVersionsRequest)));
    }

    @Override
    public WaiterResponse<DescribeProjectVersionsResponse> waitUntilProjectVersionRunning(
            DescribeProjectVersionsRequest describeProjectVersionsRequest, WaiterOverrideConfiguration overrideConfig) {
        return projectVersionRunningWaiter.run(
                () -> client.describeProjectVersions(applyWaitersUserAgent(describeProjectVersionsRequest)),
                projectVersionRunningWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeProjectVersionsResponse> waitUntilProjectVersionTrainingCompleted(
            DescribeProjectVersionsRequest describeProjectVersionsRequest) {
        return projectVersionTrainingCompletedWaiter.run(() -> client
                .describeProjectVersions(applyWaitersUserAgent(describeProjectVersionsRequest)));
    }

    @Override
    public WaiterResponse<DescribeProjectVersionsResponse> waitUntilProjectVersionTrainingCompleted(
            DescribeProjectVersionsRequest describeProjectVersionsRequest, WaiterOverrideConfiguration overrideConfig) {
        return projectVersionTrainingCompletedWaiter.run(
                () -> client.describeProjectVersions(applyWaitersUserAgent(describeProjectVersionsRequest)),
                projectVersionTrainingCompletedWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super DescribeProjectVersionsResponse>> projectVersionTrainingCompletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeProjectVersionsResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("ProjectVersionDescriptions").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "TRAINING_COMPLETED"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("ProjectVersionDescriptions").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "TRAINING_FAILED"));
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeProjectVersionsResponse>> projectVersionRunningWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeProjectVersionsResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("ProjectVersionDescriptions").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "RUNNING"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("ProjectVersionDescriptions").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "FAILED"));
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration projectVersionTrainingCompletedWaiterConfig(
            WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(360);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(120)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration projectVersionRunningWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static RekognitionWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends RekognitionRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version("waiter").name("hll").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements RekognitionWaiter.Builder {
        private RekognitionClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private DefaultBuilder() {
        }

        @Override
        public RekognitionWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public RekognitionWaiter.Builder client(RekognitionClient client) {
            this.client = client;
            return this;
        }

        public RekognitionWaiter build() {
            return new DefaultRekognitionWaiter(this);
        }
    }
}
