/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rekognition.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A filter that allows you to control the black frame detection by specifying the black levels and pixel coverage of
 * black pixels in a frame. As videos can come from multiple sources, formats, and time periods, they may contain
 * different standards and varying noise levels for black frames that need to be accounted for. For more information,
 * see <a>StartSegmentDetection</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class BlackFrame implements SdkPojo, Serializable, ToCopyableBuilder<BlackFrame.Builder, BlackFrame> {
    private static final SdkField<Float> MAX_PIXEL_THRESHOLD_FIELD = SdkField.<Float> builder(MarshallingType.FLOAT)
            .memberName("MaxPixelThreshold").getter(getter(BlackFrame::maxPixelThreshold))
            .setter(setter(Builder::maxPixelThreshold))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxPixelThreshold").build()).build();

    private static final SdkField<Float> MIN_COVERAGE_PERCENTAGE_FIELD = SdkField.<Float> builder(MarshallingType.FLOAT)
            .memberName("MinCoveragePercentage").getter(getter(BlackFrame::minCoveragePercentage))
            .setter(setter(Builder::minCoveragePercentage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MinCoveragePercentage").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(MAX_PIXEL_THRESHOLD_FIELD,
            MIN_COVERAGE_PERCENTAGE_FIELD));

    private static final long serialVersionUID = 1L;

    private final Float maxPixelThreshold;

    private final Float minCoveragePercentage;

    private BlackFrame(BuilderImpl builder) {
        this.maxPixelThreshold = builder.maxPixelThreshold;
        this.minCoveragePercentage = builder.minCoveragePercentage;
    }

    /**
     * <p>
     * A threshold used to determine the maximum luminance value for a pixel to be considered black. In a full color
     * range video, luminance values range from 0-255. A pixel value of 0 is pure black, and the most strict filter. The
     * maximum black pixel value is computed as follows: max_black_pixel_value = minimum_luminance + MaxPixelThreshold
     * *luminance_range.
     * </p>
     * <p>
     * For example, for a full range video with BlackPixelThreshold = 0.1, max_black_pixel_value is 0 + 0.1 * (255-0) =
     * 25.5.
     * </p>
     * <p>
     * The default value of MaxPixelThreshold is 0.2, which maps to a max_black_pixel_value of 51 for a full range
     * video. You can lower this threshold to be more strict on black levels.
     * </p>
     * 
     * @return A threshold used to determine the maximum luminance value for a pixel to be considered black. In a full
     *         color range video, luminance values range from 0-255. A pixel value of 0 is pure black, and the most
     *         strict filter. The maximum black pixel value is computed as follows: max_black_pixel_value =
     *         minimum_luminance + MaxPixelThreshold *luminance_range. </p>
     *         <p>
     *         For example, for a full range video with BlackPixelThreshold = 0.1, max_black_pixel_value is 0 + 0.1 *
     *         (255-0) = 25.5.
     *         </p>
     *         <p>
     *         The default value of MaxPixelThreshold is 0.2, which maps to a max_black_pixel_value of 51 for a full
     *         range video. You can lower this threshold to be more strict on black levels.
     */
    public final Float maxPixelThreshold() {
        return maxPixelThreshold;
    }

    /**
     * <p>
     * The minimum percentage of pixels in a frame that need to have a luminance below the max_black_pixel_value for a
     * frame to be considered a black frame. Luminance is calculated using the BT.709 matrix.
     * </p>
     * <p>
     * The default value is 99, which means at least 99% of all pixels in the frame are black pixels as per the
     * <code>MaxPixelThreshold</code> set. You can reduce this value to allow more noise on the black frame.
     * </p>
     * 
     * @return The minimum percentage of pixels in a frame that need to have a luminance below the max_black_pixel_value
     *         for a frame to be considered a black frame. Luminance is calculated using the BT.709 matrix. </p>
     *         <p>
     *         The default value is 99, which means at least 99% of all pixels in the frame are black pixels as per the
     *         <code>MaxPixelThreshold</code> set. You can reduce this value to allow more noise on the black frame.
     */
    public final Float minCoveragePercentage() {
        return minCoveragePercentage;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(maxPixelThreshold());
        hashCode = 31 * hashCode + Objects.hashCode(minCoveragePercentage());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof BlackFrame)) {
            return false;
        }
        BlackFrame other = (BlackFrame) obj;
        return Objects.equals(maxPixelThreshold(), other.maxPixelThreshold())
                && Objects.equals(minCoveragePercentage(), other.minCoveragePercentage());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("BlackFrame").add("MaxPixelThreshold", maxPixelThreshold())
                .add("MinCoveragePercentage", minCoveragePercentage()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "MaxPixelThreshold":
            return Optional.ofNullable(clazz.cast(maxPixelThreshold()));
        case "MinCoveragePercentage":
            return Optional.ofNullable(clazz.cast(minCoveragePercentage()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<BlackFrame, T> g) {
        return obj -> g.apply((BlackFrame) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, BlackFrame> {
        /**
         * <p>
         * A threshold used to determine the maximum luminance value for a pixel to be considered black. In a full color
         * range video, luminance values range from 0-255. A pixel value of 0 is pure black, and the most strict filter.
         * The maximum black pixel value is computed as follows: max_black_pixel_value = minimum_luminance +
         * MaxPixelThreshold *luminance_range.
         * </p>
         * <p>
         * For example, for a full range video with BlackPixelThreshold = 0.1, max_black_pixel_value is 0 + 0.1 *
         * (255-0) = 25.5.
         * </p>
         * <p>
         * The default value of MaxPixelThreshold is 0.2, which maps to a max_black_pixel_value of 51 for a full range
         * video. You can lower this threshold to be more strict on black levels.
         * </p>
         * 
         * @param maxPixelThreshold
         *        A threshold used to determine the maximum luminance value for a pixel to be considered black. In a
         *        full color range video, luminance values range from 0-255. A pixel value of 0 is pure black, and the
         *        most strict filter. The maximum black pixel value is computed as follows: max_black_pixel_value =
         *        minimum_luminance + MaxPixelThreshold *luminance_range. </p>
         *        <p>
         *        For example, for a full range video with BlackPixelThreshold = 0.1, max_black_pixel_value is 0 + 0.1 *
         *        (255-0) = 25.5.
         *        </p>
         *        <p>
         *        The default value of MaxPixelThreshold is 0.2, which maps to a max_black_pixel_value of 51 for a full
         *        range video. You can lower this threshold to be more strict on black levels.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxPixelThreshold(Float maxPixelThreshold);

        /**
         * <p>
         * The minimum percentage of pixels in a frame that need to have a luminance below the max_black_pixel_value for
         * a frame to be considered a black frame. Luminance is calculated using the BT.709 matrix.
         * </p>
         * <p>
         * The default value is 99, which means at least 99% of all pixels in the frame are black pixels as per the
         * <code>MaxPixelThreshold</code> set. You can reduce this value to allow more noise on the black frame.
         * </p>
         * 
         * @param minCoveragePercentage
         *        The minimum percentage of pixels in a frame that need to have a luminance below the
         *        max_black_pixel_value for a frame to be considered a black frame. Luminance is calculated using the
         *        BT.709 matrix. </p>
         *        <p>
         *        The default value is 99, which means at least 99% of all pixels in the frame are black pixels as per
         *        the <code>MaxPixelThreshold</code> set. You can reduce this value to allow more noise on the black
         *        frame.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder minCoveragePercentage(Float minCoveragePercentage);
    }

    static final class BuilderImpl implements Builder {
        private Float maxPixelThreshold;

        private Float minCoveragePercentage;

        private BuilderImpl() {
        }

        private BuilderImpl(BlackFrame model) {
            maxPixelThreshold(model.maxPixelThreshold);
            minCoveragePercentage(model.minCoveragePercentage);
        }

        public final Float getMaxPixelThreshold() {
            return maxPixelThreshold;
        }

        public final void setMaxPixelThreshold(Float maxPixelThreshold) {
            this.maxPixelThreshold = maxPixelThreshold;
        }

        @Override
        @Transient
        public final Builder maxPixelThreshold(Float maxPixelThreshold) {
            this.maxPixelThreshold = maxPixelThreshold;
            return this;
        }

        public final Float getMinCoveragePercentage() {
            return minCoveragePercentage;
        }

        public final void setMinCoveragePercentage(Float minCoveragePercentage) {
            this.minCoveragePercentage = minCoveragePercentage;
        }

        @Override
        @Transient
        public final Builder minCoveragePercentage(Float minCoveragePercentage) {
            this.minCoveragePercentage = minCoveragePercentage;
            return this;
        }

        @Override
        public BlackFrame build() {
            return new BlackFrame(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
