/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rekognition.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the Amazon S3 bucket location of the validation data for a model training job.
 * </p>
 * <p>
 * The validation data includes error information for individual JSON Lines in the dataset. For more information, see
 * <i>Debugging a Failed Model Training</i> in the Amazon Rekognition Custom Labels Developer Guide.
 * </p>
 * <p>
 * You get the <code>ValidationData</code> object for the training dataset (<a>TrainingDataResult</a>) and the test
 * dataset (<a>TestingDataResult</a>) by calling <a>DescribeProjectVersions</a>.
 * </p>
 * <p>
 * The assets array contains a single <a>Asset</a> object. The <a>GroundTruthManifest</a> field of the Asset object
 * contains the S3 bucket location of the validation data.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ValidationData implements SdkPojo, Serializable, ToCopyableBuilder<ValidationData.Builder, ValidationData> {
    private static final SdkField<List<Asset>> ASSETS_FIELD = SdkField
            .<List<Asset>> builder(MarshallingType.LIST)
            .memberName("Assets")
            .getter(getter(ValidationData::assets))
            .setter(setter(Builder::assets))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Assets").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Asset> builder(MarshallingType.SDK_POJO)
                                            .constructor(Asset::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ASSETS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("Assets", ASSETS_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final List<Asset> assets;

    private ValidationData(BuilderImpl builder) {
        this.assets = builder.assets;
    }

    /**
     * For responses, this returns true if the service returned a value for the Assets property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasAssets() {
        return assets != null && !(assets instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The assets that comprise the validation data.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAssets} method.
     * </p>
     * 
     * @return The assets that comprise the validation data.
     */
    public final List<Asset> assets() {
        return assets;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasAssets() ? assets() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ValidationData)) {
            return false;
        }
        ValidationData other = (ValidationData) obj;
        return hasAssets() == other.hasAssets() && Objects.equals(assets(), other.assets());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ValidationData").add("Assets", hasAssets() ? assets() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Assets":
            return Optional.ofNullable(clazz.cast(assets()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<ValidationData, T> g) {
        return obj -> g.apply((ValidationData) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ValidationData> {
        /**
         * <p>
         * The assets that comprise the validation data.
         * </p>
         * 
         * @param assets
         *        The assets that comprise the validation data.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder assets(Collection<Asset> assets);

        /**
         * <p>
         * The assets that comprise the validation data.
         * </p>
         * 
         * @param assets
         *        The assets that comprise the validation data.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder assets(Asset... assets);

        /**
         * <p>
         * The assets that comprise the validation data.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.rekognition.model.Asset.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.rekognition.model.Asset#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.rekognition.model.Asset.Builder#build()} is called immediately and its
         * result is passed to {@link #assets(List<Asset>)}.
         * 
         * @param assets
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.rekognition.model.Asset.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #assets(java.util.Collection<Asset>)
         */
        Builder assets(Consumer<Asset.Builder>... assets);
    }

    static final class BuilderImpl implements Builder {
        private List<Asset> assets = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ValidationData model) {
            assets(model.assets);
        }

        public final List<Asset.Builder> getAssets() {
            List<Asset.Builder> result = AssetsCopier.copyToBuilder(this.assets);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setAssets(Collection<Asset.BuilderImpl> assets) {
            this.assets = AssetsCopier.copyFromBuilder(assets);
        }

        @Override
        public final Builder assets(Collection<Asset> assets) {
            this.assets = AssetsCopier.copy(assets);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder assets(Asset... assets) {
            assets(Arrays.asList(assets));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder assets(Consumer<Asset.Builder>... assets) {
            assets(Stream.of(assets).map(c -> Asset.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public ValidationData build() {
            return new ValidationData(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
