/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rekognition.paginators;

import java.util.Collections;
import java.util.Iterator;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.pagination.sync.PaginatedResponsesIterator;
import software.amazon.awssdk.core.pagination.sync.SdkIterable;
import software.amazon.awssdk.core.pagination.sync.SyncPageFetcher;
import software.amazon.awssdk.core.util.PaginatorUtils;
import software.amazon.awssdk.services.rekognition.RekognitionClient;
import software.amazon.awssdk.services.rekognition.model.GetLabelDetectionRequest;
import software.amazon.awssdk.services.rekognition.model.GetLabelDetectionResponse;

/**
 * <p>
 * Represents the output for the
 * {@link software.amazon.awssdk.services.rekognition.RekognitionClient#getLabelDetectionPaginator(software.amazon.awssdk.services.rekognition.model.GetLabelDetectionRequest)}
 * operation which is a paginated operation. This class is an iterable of
 * {@link software.amazon.awssdk.services.rekognition.model.GetLabelDetectionResponse} that can be used to iterate
 * through all the response pages of the operation.
 * </p>
 * <p>
 * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet and
 * so there is no guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily
 * loading response pages by making service calls until there are no pages left or your iteration stops. If there are
 * errors in your request, you will see the failures only after you start iterating through the iterable.
 * </p>
 *
 * <p>
 * The following are few ways to iterate through the response pages:
 * </p>
 * 1) Using a Stream
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.rekognition.paginators.GetLabelDetectionIterable responses = client.getLabelDetectionPaginator(request);
 * responses.stream().forEach(....);
 * }
 * </pre>
 *
 * 2) Using For loop
 * 
 * <pre>
 * {
 *     &#064;code
 *     software.amazon.awssdk.services.rekognition.paginators.GetLabelDetectionIterable responses = client
 *             .getLabelDetectionPaginator(request);
 *     for (software.amazon.awssdk.services.rekognition.model.GetLabelDetectionResponse response : responses) {
 *         // do something;
 *     }
 * }
 * </pre>
 *
 * 3) Use iterator directly
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.rekognition.paginators.GetLabelDetectionIterable responses = client.getLabelDetectionPaginator(request);
 * responses.iterator().forEachRemaining(....);
 * }
 * </pre>
 * <p>
 * <b>Note: If you prefer to have control on service calls, use the
 * {@link #getLabelDetection(software.amazon.awssdk.services.rekognition.model.GetLabelDetectionRequest)} operation.</b>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class GetLabelDetectionIterable implements SdkIterable<GetLabelDetectionResponse> {
    private final RekognitionClient client;

    private final GetLabelDetectionRequest firstRequest;

    private final SyncPageFetcher nextPageFetcher;

    public GetLabelDetectionIterable(RekognitionClient client, GetLabelDetectionRequest firstRequest) {
        this.client = client;
        this.firstRequest = firstRequest;
        this.nextPageFetcher = new GetLabelDetectionResponseFetcher();
    }

    @Override
    public Iterator<GetLabelDetectionResponse> iterator() {
        return PaginatedResponsesIterator.builder().nextPageFetcher(nextPageFetcher).build();
    }

    /**
     * <p>
     * A helper method to resume the pages in case of unexpected failures. The method takes the last successful response
     * page as input and returns an instance of {@link GetLabelDetectionIterable} that can be used to retrieve the
     * consecutive pages that follows the input page.
     * </p>
     */
    private final GetLabelDetectionIterable resume(GetLabelDetectionResponse lastSuccessfulPage) {
        if (nextPageFetcher.hasNextPage(lastSuccessfulPage)) {
            return new GetLabelDetectionIterable(client, firstRequest.toBuilder().nextToken(lastSuccessfulPage.nextToken())
                    .build());
        }
        return new GetLabelDetectionIterable(client, firstRequest) {
            @Override
            public Iterator<GetLabelDetectionResponse> iterator() {
                return Collections.emptyIterator();
            }
        };
    }

    private class GetLabelDetectionResponseFetcher implements SyncPageFetcher<GetLabelDetectionResponse> {
        @Override
        public boolean hasNextPage(GetLabelDetectionResponse previousPage) {
            return PaginatorUtils.isOutputTokenAvailable(previousPage.nextToken());
        }

        @Override
        public GetLabelDetectionResponse nextPage(GetLabelDetectionResponse previousPage) {
            if (previousPage == null) {
                return client.getLabelDetection(firstRequest);
            }
            return client.getLabelDetection(firstRequest.toBuilder().nextToken(previousPage.nextToken()).build());
        }
    }
}
