/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rekognition.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The Amazon S3 bucket location to which Amazon Rekognition publishes the detailed inference results of a video
 * analysis operation. These results include the name of the stream processor resource, the session ID of the stream
 * processing session, and labeled timestamps and bounding boxes for detected labels.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class S3Destination implements SdkPojo, Serializable, ToCopyableBuilder<S3Destination.Builder, S3Destination> {
    private static final SdkField<String> BUCKET_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Bucket")
            .getter(getter(S3Destination::bucket)).setter(setter(Builder::bucket))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Bucket").build()).build();

    private static final SdkField<String> KEY_PREFIX_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KeyPrefix").getter(getter(S3Destination::keyPrefix)).setter(setter(Builder::keyPrefix))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeyPrefix").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays
            .asList(BUCKET_FIELD, KEY_PREFIX_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String bucket;

    private final String keyPrefix;

    private S3Destination(BuilderImpl builder) {
        this.bucket = builder.bucket;
        this.keyPrefix = builder.keyPrefix;
    }

    /**
     * <p>
     * The name of the Amazon S3 bucket you want to associate with the streaming video project. You must be the owner of
     * the Amazon S3 bucket.
     * </p>
     * 
     * @return The name of the Amazon S3 bucket you want to associate with the streaming video project. You must be the
     *         owner of the Amazon S3 bucket.
     */
    public final String bucket() {
        return bucket;
    }

    /**
     * <p>
     * The prefix value of the location within the bucket that you want the information to be published to. For more
     * information, see <a href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/using-prefixes.html">Using
     * prefixes</a>.
     * </p>
     * 
     * @return The prefix value of the location within the bucket that you want the information to be published to. For
     *         more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/using-prefixes.html">Using prefixes</a>.
     */
    public final String keyPrefix() {
        return keyPrefix;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(bucket());
        hashCode = 31 * hashCode + Objects.hashCode(keyPrefix());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof S3Destination)) {
            return false;
        }
        S3Destination other = (S3Destination) obj;
        return Objects.equals(bucket(), other.bucket()) && Objects.equals(keyPrefix(), other.keyPrefix());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("S3Destination").add("Bucket", bucket()).add("KeyPrefix", keyPrefix()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Bucket":
            return Optional.ofNullable(clazz.cast(bucket()));
        case "KeyPrefix":
            return Optional.ofNullable(clazz.cast(keyPrefix()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Bucket", BUCKET_FIELD);
        map.put("KeyPrefix", KEY_PREFIX_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<S3Destination, T> g) {
        return obj -> g.apply((S3Destination) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, S3Destination> {
        /**
         * <p>
         * The name of the Amazon S3 bucket you want to associate with the streaming video project. You must be the
         * owner of the Amazon S3 bucket.
         * </p>
         * 
         * @param bucket
         *        The name of the Amazon S3 bucket you want to associate with the streaming video project. You must be
         *        the owner of the Amazon S3 bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bucket(String bucket);

        /**
         * <p>
         * The prefix value of the location within the bucket that you want the information to be published to. For more
         * information, see <a href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/using-prefixes.html">Using
         * prefixes</a>.
         * </p>
         * 
         * @param keyPrefix
         *        The prefix value of the location within the bucket that you want the information to be published to.
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/using-prefixes.html">Using prefixes</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keyPrefix(String keyPrefix);
    }

    static final class BuilderImpl implements Builder {
        private String bucket;

        private String keyPrefix;

        private BuilderImpl() {
        }

        private BuilderImpl(S3Destination model) {
            bucket(model.bucket);
            keyPrefix(model.keyPrefix);
        }

        public final String getBucket() {
            return bucket;
        }

        public final void setBucket(String bucket) {
            this.bucket = bucket;
        }

        @Override
        public final Builder bucket(String bucket) {
            this.bucket = bucket;
            return this;
        }

        public final String getKeyPrefix() {
            return keyPrefix;
        }

        public final void setKeyPrefix(String keyPrefix) {
            this.keyPrefix = keyPrefix;
        }

        @Override
        public final Builder keyPrefix(String keyPrefix) {
            this.keyPrefix = keyPrefix;
            return this;
        }

        @Override
        public S3Destination build() {
            return new S3Destination(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
