/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rekognition.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about a body part detected by <a>DetectProtectiveEquipment</a> that contains PPE. An array of
 * <code>ProtectiveEquipmentBodyPart</code> objects is returned for each person detected by
 * <code>DetectProtectiveEquipment</code>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ProtectiveEquipmentBodyPart implements SdkPojo, Serializable,
        ToCopyableBuilder<ProtectiveEquipmentBodyPart.Builder, ProtectiveEquipmentBodyPart> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(ProtectiveEquipmentBodyPart::nameAsString)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<Float> CONFIDENCE_FIELD = SdkField.<Float> builder(MarshallingType.FLOAT)
            .memberName("Confidence").getter(getter(ProtectiveEquipmentBodyPart::confidence)).setter(setter(Builder::confidence))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Confidence").build()).build();

    private static final SdkField<List<EquipmentDetection>> EQUIPMENT_DETECTIONS_FIELD = SdkField
            .<List<EquipmentDetection>> builder(MarshallingType.LIST)
            .memberName("EquipmentDetections")
            .getter(getter(ProtectiveEquipmentBodyPart::equipmentDetections))
            .setter(setter(Builder::equipmentDetections))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EquipmentDetections").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<EquipmentDetection> builder(MarshallingType.SDK_POJO)
                                            .constructor(EquipmentDetection::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, CONFIDENCE_FIELD,
            EQUIPMENT_DETECTIONS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String name;

    private final Float confidence;

    private final List<EquipmentDetection> equipmentDetections;

    private ProtectiveEquipmentBodyPart(BuilderImpl builder) {
        this.name = builder.name;
        this.confidence = builder.confidence;
        this.equipmentDetections = builder.equipmentDetections;
    }

    /**
     * <p>
     * The detected body part.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #name} will return
     * {@link BodyPart#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #nameAsString}.
     * </p>
     * 
     * @return The detected body part.
     * @see BodyPart
     */
    public final BodyPart name() {
        return BodyPart.fromValue(name);
    }

    /**
     * <p>
     * The detected body part.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #name} will return
     * {@link BodyPart#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #nameAsString}.
     * </p>
     * 
     * @return The detected body part.
     * @see BodyPart
     */
    public final String nameAsString() {
        return name;
    }

    /**
     * <p>
     * The confidence that Amazon Rekognition has in the detection accuracy of the detected body part.
     * </p>
     * 
     * @return The confidence that Amazon Rekognition has in the detection accuracy of the detected body part.
     */
    public final Float confidence() {
        return confidence;
    }

    /**
     * For responses, this returns true if the service returned a value for the EquipmentDetections property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasEquipmentDetections() {
        return equipmentDetections != null && !(equipmentDetections instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of Personal Protective Equipment items detected around a body part.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEquipmentDetections} method.
     * </p>
     * 
     * @return An array of Personal Protective Equipment items detected around a body part.
     */
    public final List<EquipmentDetection> equipmentDetections() {
        return equipmentDetections;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(nameAsString());
        hashCode = 31 * hashCode + Objects.hashCode(confidence());
        hashCode = 31 * hashCode + Objects.hashCode(hasEquipmentDetections() ? equipmentDetections() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ProtectiveEquipmentBodyPart)) {
            return false;
        }
        ProtectiveEquipmentBodyPart other = (ProtectiveEquipmentBodyPart) obj;
        return Objects.equals(nameAsString(), other.nameAsString()) && Objects.equals(confidence(), other.confidence())
                && hasEquipmentDetections() == other.hasEquipmentDetections()
                && Objects.equals(equipmentDetections(), other.equipmentDetections());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ProtectiveEquipmentBodyPart").add("Name", nameAsString()).add("Confidence", confidence())
                .add("EquipmentDetections", hasEquipmentDetections() ? equipmentDetections() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(nameAsString()));
        case "Confidence":
            return Optional.ofNullable(clazz.cast(confidence()));
        case "EquipmentDetections":
            return Optional.ofNullable(clazz.cast(equipmentDetections()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Name", NAME_FIELD);
        map.put("Confidence", CONFIDENCE_FIELD);
        map.put("EquipmentDetections", EQUIPMENT_DETECTIONS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ProtectiveEquipmentBodyPart, T> g) {
        return obj -> g.apply((ProtectiveEquipmentBodyPart) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ProtectiveEquipmentBodyPart> {
        /**
         * <p>
         * The detected body part.
         * </p>
         * 
         * @param name
         *        The detected body part.
         * @see BodyPart
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BodyPart
         */
        Builder name(String name);

        /**
         * <p>
         * The detected body part.
         * </p>
         * 
         * @param name
         *        The detected body part.
         * @see BodyPart
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BodyPart
         */
        Builder name(BodyPart name);

        /**
         * <p>
         * The confidence that Amazon Rekognition has in the detection accuracy of the detected body part.
         * </p>
         * 
         * @param confidence
         *        The confidence that Amazon Rekognition has in the detection accuracy of the detected body part.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder confidence(Float confidence);

        /**
         * <p>
         * An array of Personal Protective Equipment items detected around a body part.
         * </p>
         * 
         * @param equipmentDetections
         *        An array of Personal Protective Equipment items detected around a body part.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder equipmentDetections(Collection<EquipmentDetection> equipmentDetections);

        /**
         * <p>
         * An array of Personal Protective Equipment items detected around a body part.
         * </p>
         * 
         * @param equipmentDetections
         *        An array of Personal Protective Equipment items detected around a body part.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder equipmentDetections(EquipmentDetection... equipmentDetections);

        /**
         * <p>
         * An array of Personal Protective Equipment items detected around a body part.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.rekognition.model.EquipmentDetection.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.rekognition.model.EquipmentDetection#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.rekognition.model.EquipmentDetection.Builder#build()} is called
         * immediately and its result is passed to {@link #equipmentDetections(List<EquipmentDetection>)}.
         * 
         * @param equipmentDetections
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.rekognition.model.EquipmentDetection.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #equipmentDetections(java.util.Collection<EquipmentDetection>)
         */
        Builder equipmentDetections(Consumer<EquipmentDetection.Builder>... equipmentDetections);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private Float confidence;

        private List<EquipmentDetection> equipmentDetections = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ProtectiveEquipmentBodyPart model) {
            name(model.name);
            confidence(model.confidence);
            equipmentDetections(model.equipmentDetections);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        @Override
        public final Builder name(BodyPart name) {
            this.name(name == null ? null : name.toString());
            return this;
        }

        public final Float getConfidence() {
            return confidence;
        }

        public final void setConfidence(Float confidence) {
            this.confidence = confidence;
        }

        @Override
        public final Builder confidence(Float confidence) {
            this.confidence = confidence;
            return this;
        }

        public final List<EquipmentDetection.Builder> getEquipmentDetections() {
            List<EquipmentDetection.Builder> result = EquipmentDetectionsCopier.copyToBuilder(this.equipmentDetections);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setEquipmentDetections(Collection<EquipmentDetection.BuilderImpl> equipmentDetections) {
            this.equipmentDetections = EquipmentDetectionsCopier.copyFromBuilder(equipmentDetections);
        }

        @Override
        public final Builder equipmentDetections(Collection<EquipmentDetection> equipmentDetections) {
            this.equipmentDetections = EquipmentDetectionsCopier.copy(equipmentDetections);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder equipmentDetections(EquipmentDetection... equipmentDetections) {
            equipmentDetections(Arrays.asList(equipmentDetections));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder equipmentDetections(Consumer<EquipmentDetection.Builder>... equipmentDetections) {
            equipmentDetections(Stream.of(equipmentDetections).map(c -> EquipmentDetection.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public ProtectiveEquipmentBodyPart build() {
            return new ProtectiveEquipmentBodyPart(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
