/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.repostspace;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.repostspace.internal.RepostspaceServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.repostspace.model.AccessDeniedException;
import software.amazon.awssdk.services.repostspace.model.BatchAddRoleRequest;
import software.amazon.awssdk.services.repostspace.model.BatchAddRoleResponse;
import software.amazon.awssdk.services.repostspace.model.BatchRemoveRoleRequest;
import software.amazon.awssdk.services.repostspace.model.BatchRemoveRoleResponse;
import software.amazon.awssdk.services.repostspace.model.ConflictException;
import software.amazon.awssdk.services.repostspace.model.CreateSpaceRequest;
import software.amazon.awssdk.services.repostspace.model.CreateSpaceResponse;
import software.amazon.awssdk.services.repostspace.model.DeleteSpaceRequest;
import software.amazon.awssdk.services.repostspace.model.DeleteSpaceResponse;
import software.amazon.awssdk.services.repostspace.model.DeregisterAdminRequest;
import software.amazon.awssdk.services.repostspace.model.DeregisterAdminResponse;
import software.amazon.awssdk.services.repostspace.model.GetSpaceRequest;
import software.amazon.awssdk.services.repostspace.model.GetSpaceResponse;
import software.amazon.awssdk.services.repostspace.model.InternalServerException;
import software.amazon.awssdk.services.repostspace.model.ListSpacesRequest;
import software.amazon.awssdk.services.repostspace.model.ListSpacesResponse;
import software.amazon.awssdk.services.repostspace.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.repostspace.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.repostspace.model.RegisterAdminRequest;
import software.amazon.awssdk.services.repostspace.model.RegisterAdminResponse;
import software.amazon.awssdk.services.repostspace.model.RepostspaceException;
import software.amazon.awssdk.services.repostspace.model.ResourceNotFoundException;
import software.amazon.awssdk.services.repostspace.model.SendInvitesRequest;
import software.amazon.awssdk.services.repostspace.model.SendInvitesResponse;
import software.amazon.awssdk.services.repostspace.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.repostspace.model.TagResourceRequest;
import software.amazon.awssdk.services.repostspace.model.TagResourceResponse;
import software.amazon.awssdk.services.repostspace.model.ThrottlingException;
import software.amazon.awssdk.services.repostspace.model.UntagResourceRequest;
import software.amazon.awssdk.services.repostspace.model.UntagResourceResponse;
import software.amazon.awssdk.services.repostspace.model.UpdateSpaceRequest;
import software.amazon.awssdk.services.repostspace.model.UpdateSpaceResponse;
import software.amazon.awssdk.services.repostspace.model.ValidationException;
import software.amazon.awssdk.services.repostspace.transform.BatchAddRoleRequestMarshaller;
import software.amazon.awssdk.services.repostspace.transform.BatchRemoveRoleRequestMarshaller;
import software.amazon.awssdk.services.repostspace.transform.CreateSpaceRequestMarshaller;
import software.amazon.awssdk.services.repostspace.transform.DeleteSpaceRequestMarshaller;
import software.amazon.awssdk.services.repostspace.transform.DeregisterAdminRequestMarshaller;
import software.amazon.awssdk.services.repostspace.transform.GetSpaceRequestMarshaller;
import software.amazon.awssdk.services.repostspace.transform.ListSpacesRequestMarshaller;
import software.amazon.awssdk.services.repostspace.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.repostspace.transform.RegisterAdminRequestMarshaller;
import software.amazon.awssdk.services.repostspace.transform.SendInvitesRequestMarshaller;
import software.amazon.awssdk.services.repostspace.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.repostspace.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.repostspace.transform.UpdateSpaceRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link RepostspaceAsyncClient}.
 *
 * @see RepostspaceAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultRepostspaceAsyncClient implements RepostspaceAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultRepostspaceAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultRepostspaceAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Add role to multiple users or groups in a private re:Post.
     * </p>
     *
     * @param batchAddRoleRequest
     * @return A Java Future containing the result of the BatchAddRole operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RepostspaceException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RepostspaceAsyncClient.BatchAddRole
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/repostspace-2022-05-13/BatchAddRole" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<BatchAddRoleResponse> batchAddRole(BatchAddRoleRequest batchAddRoleRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchAddRoleRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchAddRoleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "repostspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchAddRole");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchAddRoleResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    BatchAddRoleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchAddRoleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchAddRoleRequest, BatchAddRoleResponse>()
                            .withOperationName("BatchAddRole").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new BatchAddRoleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(batchAddRoleRequest));
            CompletableFuture<BatchAddRoleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Remove role from multiple users or groups in a private re:Post.
     * </p>
     *
     * @param batchRemoveRoleRequest
     * @return A Java Future containing the result of the BatchRemoveRole operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RepostspaceException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RepostspaceAsyncClient.BatchRemoveRole
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/repostspace-2022-05-13/BatchRemoveRole" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchRemoveRoleResponse> batchRemoveRole(BatchRemoveRoleRequest batchRemoveRoleRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchRemoveRoleRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchRemoveRoleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "repostspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchRemoveRole");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchRemoveRoleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchRemoveRoleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchRemoveRoleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchRemoveRoleRequest, BatchRemoveRoleResponse>()
                            .withOperationName("BatchRemoveRole").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new BatchRemoveRoleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(batchRemoveRoleRequest));
            CompletableFuture<BatchRemoveRoleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an AWS re:Post Private private re:Post.
     * </p>
     *
     * @param createSpaceRequest
     * @return A Java Future containing the result of the CreateSpace operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException Request would cause a service quota to be exceeded.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RepostspaceException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RepostspaceAsyncClient.CreateSpace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/repostspace-2022-05-13/CreateSpace" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateSpaceResponse> createSpace(CreateSpaceRequest createSpaceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createSpaceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSpaceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "repostspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSpace");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateSpaceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateSpaceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateSpaceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateSpaceRequest, CreateSpaceResponse>()
                            .withOperationName("CreateSpace").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateSpaceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createSpaceRequest));
            CompletableFuture<CreateSpaceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an AWS re:Post Private private re:Post.
     * </p>
     *
     * @param deleteSpaceRequest
     * @return A Java Future containing the result of the DeleteSpace operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RepostspaceException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RepostspaceAsyncClient.DeleteSpace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/repostspace-2022-05-13/DeleteSpace" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteSpaceResponse> deleteSpace(DeleteSpaceRequest deleteSpaceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteSpaceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSpaceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "repostspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSpace");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteSpaceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteSpaceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteSpaceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteSpaceRequest, DeleteSpaceResponse>()
                            .withOperationName("DeleteSpace").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteSpaceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteSpaceRequest));
            CompletableFuture<DeleteSpaceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the user or group from the list of administrators of the private re:Post.
     * </p>
     *
     * @param deregisterAdminRequest
     * @return A Java Future containing the result of the DeregisterAdmin operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RepostspaceException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RepostspaceAsyncClient.DeregisterAdmin
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/repostspace-2022-05-13/DeregisterAdmin" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeregisterAdminResponse> deregisterAdmin(DeregisterAdminRequest deregisterAdminRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deregisterAdminRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deregisterAdminRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "repostspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeregisterAdmin");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeregisterAdminResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeregisterAdminResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeregisterAdminResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeregisterAdminRequest, DeregisterAdminResponse>()
                            .withOperationName("DeregisterAdmin").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeregisterAdminRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deregisterAdminRequest));
            CompletableFuture<DeregisterAdminResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Displays information about the AWS re:Post Private private re:Post.
     * </p>
     *
     * @param getSpaceRequest
     * @return A Java Future containing the result of the GetSpace operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RepostspaceException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RepostspaceAsyncClient.GetSpace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/repostspace-2022-05-13/GetSpace" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetSpaceResponse> getSpace(GetSpaceRequest getSpaceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getSpaceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSpaceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "repostspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSpace");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSpaceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetSpaceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSpaceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSpaceRequest, GetSpaceResponse>().withOperationName("GetSpace")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetSpaceRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(getSpaceRequest));
            CompletableFuture<GetSpaceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of AWS re:Post Private private re:Posts in the account with some information about each private
     * re:Post.
     * </p>
     *
     * @param listSpacesRequest
     * @return A Java Future containing the result of the ListSpaces operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RepostspaceException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RepostspaceAsyncClient.ListSpaces
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/repostspace-2022-05-13/ListSpaces" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListSpacesResponse> listSpaces(ListSpacesRequest listSpacesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listSpacesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSpacesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "repostspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSpaces");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSpacesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListSpacesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListSpacesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSpacesRequest, ListSpacesResponse>().withOperationName("ListSpaces")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListSpacesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listSpacesRequest));
            CompletableFuture<ListSpacesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the tags that are associated with the AWS re:Post Private resource specified by the resourceArn. The only
     * resource that can be tagged is a private re:Post.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RepostspaceException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RepostspaceAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/repostspace-2022-05-13/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "repostspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds a user or group to the list of administrators of the private re:Post.
     * </p>
     *
     * @param registerAdminRequest
     * @return A Java Future containing the result of the RegisterAdmin operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RepostspaceException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RepostspaceAsyncClient.RegisterAdmin
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/repostspace-2022-05-13/RegisterAdmin" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<RegisterAdminResponse> registerAdmin(RegisterAdminRequest registerAdminRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(registerAdminRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, registerAdminRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "repostspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RegisterAdmin");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RegisterAdminResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    RegisterAdminResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RegisterAdminResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RegisterAdminRequest, RegisterAdminResponse>()
                            .withOperationName("RegisterAdmin").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new RegisterAdminRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(registerAdminRequest));
            CompletableFuture<RegisterAdminResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Sends an invitation email to selected users and groups.
     * </p>
     *
     * @param sendInvitesRequest
     * @return A Java Future containing the result of the SendInvites operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RepostspaceException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RepostspaceAsyncClient.SendInvites
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/repostspace-2022-05-13/SendInvites" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<SendInvitesResponse> sendInvites(SendInvitesRequest sendInvitesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(sendInvitesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, sendInvitesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "repostspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SendInvites");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SendInvitesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    SendInvitesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SendInvitesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SendInvitesRequest, SendInvitesResponse>()
                            .withOperationName("SendInvites").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new SendInvitesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(sendInvitesRequest));
            CompletableFuture<SendInvitesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Associates tags with an AWS re:Post Private resource. Currently, the only resource that can be tagged is the
     * private re:Post. If you specify a new tag key for the resource, the tag is appended to the list of tags that are
     * associated with the resource. If you specify a tag key that’s already associated with the resource, the new tag
     * value that you specify replaces the previous value for that tag.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RepostspaceException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RepostspaceAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/repostspace-2022-05-13/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "repostspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the association of the tag with the AWS re:Post Private resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RepostspaceException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RepostspaceAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/repostspace-2022-05-13/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "repostspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Modifies an existing AWS re:Post Private private re:Post.
     * </p>
     *
     * @param updateSpaceRequest
     * @return A Java Future containing the result of the UpdateSpace operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RepostspaceException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RepostspaceAsyncClient.UpdateSpace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/repostspace-2022-05-13/UpdateSpace" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateSpaceResponse> updateSpace(UpdateSpaceRequest updateSpaceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateSpaceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateSpaceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "repostspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSpace");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateSpaceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateSpaceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateSpaceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateSpaceRequest, UpdateSpaceResponse>()
                            .withOperationName("UpdateSpace").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateSpaceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateSpaceRequest));
            CompletableFuture<UpdateSpaceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final RepostspaceServiceClientConfiguration serviceClientConfiguration() {
        return new RepostspaceServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(RepostspaceException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        RepostspaceServiceClientConfigurationBuilder serviceConfigBuilder = new RepostspaceServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
