/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.resiliencehub.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Defines a resiliency policy.
 * </p>
 * <note>
 * <p>
 * Resilience Hub allows you to provide a value of zero for <code>rtoInSecs</code> and <code>rpoInSecs</code> of your
 * resiliency policy. But, while assessing your application, the lowest possible assessment result is near zero. Hence,
 * if you provide value zero for <code>rtoInSecs</code> and <code>rpoInSecs</code>, the estimated workload RTO and
 * estimated workload RPO result will be near zero and the <b>Compliance status</b> for your application will be set to
 * <b>Policy breached</b>.
 * </p>
 * </note>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ResiliencyPolicy implements SdkPojo, Serializable,
        ToCopyableBuilder<ResiliencyPolicy.Builder, ResiliencyPolicy> {
    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("creationTime").getter(getter(ResiliencyPolicy::creationTime)).setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("creationTime").build()).build();

    private static final SdkField<String> DATA_LOCATION_CONSTRAINT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("dataLocationConstraint").getter(getter(ResiliencyPolicy::dataLocationConstraintAsString))
            .setter(setter(Builder::dataLocationConstraint))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("dataLocationConstraint").build())
            .build();

    private static final SdkField<String> ESTIMATED_COST_TIER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("estimatedCostTier").getter(getter(ResiliencyPolicy::estimatedCostTierAsString))
            .setter(setter(Builder::estimatedCostTier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("estimatedCostTier").build()).build();

    private static final SdkField<Map<String, FailurePolicy>> POLICY_FIELD = SdkField
            .<Map<String, FailurePolicy>> builder(MarshallingType.MAP)
            .memberName("policy")
            .getter(getter(ResiliencyPolicy::policyAsStrings))
            .setter(setter(Builder::policyWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("policy").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<FailurePolicy> builder(MarshallingType.SDK_POJO)
                                            .constructor(FailurePolicy::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> POLICY_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("policyArn").getter(getter(ResiliencyPolicy::policyArn)).setter(setter(Builder::policyArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("policyArn").build()).build();

    private static final SdkField<String> POLICY_DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("policyDescription").getter(getter(ResiliencyPolicy::policyDescription))
            .setter(setter(Builder::policyDescription))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("policyDescription").build()).build();

    private static final SdkField<String> POLICY_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("policyName").getter(getter(ResiliencyPolicy::policyName)).setter(setter(Builder::policyName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("policyName").build()).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("tags")
            .getter(getter(ResiliencyPolicy::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> TIER_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("tier")
            .getter(getter(ResiliencyPolicy::tierAsString)).setter(setter(Builder::tier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tier").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CREATION_TIME_FIELD,
            DATA_LOCATION_CONSTRAINT_FIELD, ESTIMATED_COST_TIER_FIELD, POLICY_FIELD, POLICY_ARN_FIELD, POLICY_DESCRIPTION_FIELD,
            POLICY_NAME_FIELD, TAGS_FIELD, TIER_FIELD));

    private static final long serialVersionUID = 1L;

    private final Instant creationTime;

    private final String dataLocationConstraint;

    private final String estimatedCostTier;

    private final Map<String, FailurePolicy> policy;

    private final String policyArn;

    private final String policyDescription;

    private final String policyName;

    private final Map<String, String> tags;

    private final String tier;

    private ResiliencyPolicy(BuilderImpl builder) {
        this.creationTime = builder.creationTime;
        this.dataLocationConstraint = builder.dataLocationConstraint;
        this.estimatedCostTier = builder.estimatedCostTier;
        this.policy = builder.policy;
        this.policyArn = builder.policyArn;
        this.policyDescription = builder.policyDescription;
        this.policyName = builder.policyName;
        this.tags = builder.tags;
        this.tier = builder.tier;
    }

    /**
     * <p>
     * Date and time when the resiliency policy was created.
     * </p>
     * 
     * @return Date and time when the resiliency policy was created.
     */
    public final Instant creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * Specifies a high-level geographical location constraint for where your resilience policy data can be stored.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #dataLocationConstraint} will return {@link DataLocationConstraint#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #dataLocationConstraintAsString}.
     * </p>
     * 
     * @return Specifies a high-level geographical location constraint for where your resilience policy data can be
     *         stored.
     * @see DataLocationConstraint
     */
    public final DataLocationConstraint dataLocationConstraint() {
        return DataLocationConstraint.fromValue(dataLocationConstraint);
    }

    /**
     * <p>
     * Specifies a high-level geographical location constraint for where your resilience policy data can be stored.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #dataLocationConstraint} will return {@link DataLocationConstraint#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #dataLocationConstraintAsString}.
     * </p>
     * 
     * @return Specifies a high-level geographical location constraint for where your resilience policy data can be
     *         stored.
     * @see DataLocationConstraint
     */
    public final String dataLocationConstraintAsString() {
        return dataLocationConstraint;
    }

    /**
     * <p>
     * Specifies the estimated cost tier of the resiliency policy.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #estimatedCostTier}
     * will return {@link EstimatedCostTier#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #estimatedCostTierAsString}.
     * </p>
     * 
     * @return Specifies the estimated cost tier of the resiliency policy.
     * @see EstimatedCostTier
     */
    public final EstimatedCostTier estimatedCostTier() {
        return EstimatedCostTier.fromValue(estimatedCostTier);
    }

    /**
     * <p>
     * Specifies the estimated cost tier of the resiliency policy.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #estimatedCostTier}
     * will return {@link EstimatedCostTier#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #estimatedCostTierAsString}.
     * </p>
     * 
     * @return Specifies the estimated cost tier of the resiliency policy.
     * @see EstimatedCostTier
     */
    public final String estimatedCostTierAsString() {
        return estimatedCostTier;
    }

    /**
     * <p>
     * The resiliency policy.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPolicy} method.
     * </p>
     * 
     * @return The resiliency policy.
     */
    public final Map<DisruptionType, FailurePolicy> policy() {
        return DisruptionPolicyCopier.copyStringToEnum(policy);
    }

    /**
     * For responses, this returns true if the service returned a value for the Policy property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasPolicy() {
        return policy != null && !(policy instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The resiliency policy.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPolicy} method.
     * </p>
     * 
     * @return The resiliency policy.
     */
    public final Map<String, FailurePolicy> policyAsStrings() {
        return policy;
    }

    /**
     * <p>
     * Amazon Resource Name (ARN) of the resiliency policy. The format for this ARN is: arn:<code>partition</code>
     * :resiliencehub:<code>region</code>:<code>account</code>:resiliency-policy/<code>policy-id</code>. For more
     * information about ARNs, see <a href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">
     * Amazon Resource Names (ARNs)</a> in the <i>Amazon Web Services General Reference</i> guide.
     * </p>
     * 
     * @return Amazon Resource Name (ARN) of the resiliency policy. The format for this ARN is: arn:
     *         <code>partition</code>:resiliencehub:<code>region</code>:<code>account</code>:resiliency-policy/
     *         <code>policy-id</code>. For more information about ARNs, see <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html"> Amazon Resource Names
     *         (ARNs)</a> in the <i>Amazon Web Services General Reference</i> guide.
     */
    public final String policyArn() {
        return policyArn;
    }

    /**
     * <p>
     * The description for the policy.
     * </p>
     * 
     * @return The description for the policy.
     */
    public final String policyDescription() {
        return policyDescription;
    }

    /**
     * <p>
     * The name of the policy
     * </p>
     * 
     * @return The name of the policy
     */
    public final String policyName() {
        return policyName;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Tags assigned to the resource. A tag is a label that you assign to an Amazon Web Services resource. Each tag
     * consists of a key/value pair.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return Tags assigned to the resource. A tag is a label that you assign to an Amazon Web Services resource. Each
     *         tag consists of a key/value pair.
     */
    public final Map<String, String> tags() {
        return tags;
    }

    /**
     * <p>
     * The tier for this resiliency policy, ranging from the highest severity (<code>MissionCritical</code>) to lowest (
     * <code>NonCritical</code>).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #tier} will return
     * {@link ResiliencyPolicyTier#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #tierAsString}.
     * </p>
     * 
     * @return The tier for this resiliency policy, ranging from the highest severity (<code>MissionCritical</code>) to
     *         lowest (<code>NonCritical</code>).
     * @see ResiliencyPolicyTier
     */
    public final ResiliencyPolicyTier tier() {
        return ResiliencyPolicyTier.fromValue(tier);
    }

    /**
     * <p>
     * The tier for this resiliency policy, ranging from the highest severity (<code>MissionCritical</code>) to lowest (
     * <code>NonCritical</code>).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #tier} will return
     * {@link ResiliencyPolicyTier#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #tierAsString}.
     * </p>
     * 
     * @return The tier for this resiliency policy, ranging from the highest severity (<code>MissionCritical</code>) to
     *         lowest (<code>NonCritical</code>).
     * @see ResiliencyPolicyTier
     */
    public final String tierAsString() {
        return tier;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(dataLocationConstraintAsString());
        hashCode = 31 * hashCode + Objects.hashCode(estimatedCostTierAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasPolicy() ? policyAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(policyArn());
        hashCode = 31 * hashCode + Objects.hashCode(policyDescription());
        hashCode = 31 * hashCode + Objects.hashCode(policyName());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(tierAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ResiliencyPolicy)) {
            return false;
        }
        ResiliencyPolicy other = (ResiliencyPolicy) obj;
        return Objects.equals(creationTime(), other.creationTime())
                && Objects.equals(dataLocationConstraintAsString(), other.dataLocationConstraintAsString())
                && Objects.equals(estimatedCostTierAsString(), other.estimatedCostTierAsString())
                && hasPolicy() == other.hasPolicy() && Objects.equals(policyAsStrings(), other.policyAsStrings())
                && Objects.equals(policyArn(), other.policyArn())
                && Objects.equals(policyDescription(), other.policyDescription())
                && Objects.equals(policyName(), other.policyName()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && Objects.equals(tierAsString(), other.tierAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ResiliencyPolicy").add("CreationTime", creationTime())
                .add("DataLocationConstraint", dataLocationConstraintAsString())
                .add("EstimatedCostTier", estimatedCostTierAsString()).add("Policy", hasPolicy() ? policyAsStrings() : null)
                .add("PolicyArn", policyArn()).add("PolicyDescription", policyDescription()).add("PolicyName", policyName())
                .add("Tags", tags() == null ? null : "*** Sensitive Data Redacted ***").add("Tier", tierAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "creationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "dataLocationConstraint":
            return Optional.ofNullable(clazz.cast(dataLocationConstraintAsString()));
        case "estimatedCostTier":
            return Optional.ofNullable(clazz.cast(estimatedCostTierAsString()));
        case "policy":
            return Optional.ofNullable(clazz.cast(policyAsStrings()));
        case "policyArn":
            return Optional.ofNullable(clazz.cast(policyArn()));
        case "policyDescription":
            return Optional.ofNullable(clazz.cast(policyDescription()));
        case "policyName":
            return Optional.ofNullable(clazz.cast(policyName()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "tier":
            return Optional.ofNullable(clazz.cast(tierAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ResiliencyPolicy, T> g) {
        return obj -> g.apply((ResiliencyPolicy) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ResiliencyPolicy> {
        /**
         * <p>
         * Date and time when the resiliency policy was created.
         * </p>
         * 
         * @param creationTime
         *        Date and time when the resiliency policy was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * Specifies a high-level geographical location constraint for where your resilience policy data can be stored.
         * </p>
         * 
         * @param dataLocationConstraint
         *        Specifies a high-level geographical location constraint for where your resilience policy data can be
         *        stored.
         * @see DataLocationConstraint
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DataLocationConstraint
         */
        Builder dataLocationConstraint(String dataLocationConstraint);

        /**
         * <p>
         * Specifies a high-level geographical location constraint for where your resilience policy data can be stored.
         * </p>
         * 
         * @param dataLocationConstraint
         *        Specifies a high-level geographical location constraint for where your resilience policy data can be
         *        stored.
         * @see DataLocationConstraint
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DataLocationConstraint
         */
        Builder dataLocationConstraint(DataLocationConstraint dataLocationConstraint);

        /**
         * <p>
         * Specifies the estimated cost tier of the resiliency policy.
         * </p>
         * 
         * @param estimatedCostTier
         *        Specifies the estimated cost tier of the resiliency policy.
         * @see EstimatedCostTier
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EstimatedCostTier
         */
        Builder estimatedCostTier(String estimatedCostTier);

        /**
         * <p>
         * Specifies the estimated cost tier of the resiliency policy.
         * </p>
         * 
         * @param estimatedCostTier
         *        Specifies the estimated cost tier of the resiliency policy.
         * @see EstimatedCostTier
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EstimatedCostTier
         */
        Builder estimatedCostTier(EstimatedCostTier estimatedCostTier);

        /**
         * <p>
         * The resiliency policy.
         * </p>
         * 
         * @param policy
         *        The resiliency policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder policyWithStrings(Map<String, FailurePolicy> policy);

        /**
         * <p>
         * The resiliency policy.
         * </p>
         * 
         * @param policy
         *        The resiliency policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder policy(Map<DisruptionType, FailurePolicy> policy);

        /**
         * <p>
         * Amazon Resource Name (ARN) of the resiliency policy. The format for this ARN is: arn:<code>partition</code>
         * :resiliencehub:<code>region</code>:<code>account</code>:resiliency-policy/<code>policy-id</code>. For more
         * information about ARNs, see <a
         * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html"> Amazon Resource Names
         * (ARNs)</a> in the <i>Amazon Web Services General Reference</i> guide.
         * </p>
         * 
         * @param policyArn
         *        Amazon Resource Name (ARN) of the resiliency policy. The format for this ARN is: arn:
         *        <code>partition</code>:resiliencehub:<code>region</code>:<code>account</code>:resiliency-policy/
         *        <code>policy-id</code>. For more information about ARNs, see <a
         *        href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html"> Amazon Resource
         *        Names (ARNs)</a> in the <i>Amazon Web Services General Reference</i> guide.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder policyArn(String policyArn);

        /**
         * <p>
         * The description for the policy.
         * </p>
         * 
         * @param policyDescription
         *        The description for the policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder policyDescription(String policyDescription);

        /**
         * <p>
         * The name of the policy
         * </p>
         * 
         * @param policyName
         *        The name of the policy
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder policyName(String policyName);

        /**
         * <p>
         * Tags assigned to the resource. A tag is a label that you assign to an Amazon Web Services resource. Each tag
         * consists of a key/value pair.
         * </p>
         * 
         * @param tags
         *        Tags assigned to the resource. A tag is a label that you assign to an Amazon Web Services resource.
         *        Each tag consists of a key/value pair.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        /**
         * <p>
         * The tier for this resiliency policy, ranging from the highest severity (<code>MissionCritical</code>) to
         * lowest (<code>NonCritical</code>).
         * </p>
         * 
         * @param tier
         *        The tier for this resiliency policy, ranging from the highest severity (<code>MissionCritical</code>)
         *        to lowest (<code>NonCritical</code>).
         * @see ResiliencyPolicyTier
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResiliencyPolicyTier
         */
        Builder tier(String tier);

        /**
         * <p>
         * The tier for this resiliency policy, ranging from the highest severity (<code>MissionCritical</code>) to
         * lowest (<code>NonCritical</code>).
         * </p>
         * 
         * @param tier
         *        The tier for this resiliency policy, ranging from the highest severity (<code>MissionCritical</code>)
         *        to lowest (<code>NonCritical</code>).
         * @see ResiliencyPolicyTier
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResiliencyPolicyTier
         */
        Builder tier(ResiliencyPolicyTier tier);
    }

    static final class BuilderImpl implements Builder {
        private Instant creationTime;

        private String dataLocationConstraint;

        private String estimatedCostTier;

        private Map<String, FailurePolicy> policy = DefaultSdkAutoConstructMap.getInstance();

        private String policyArn;

        private String policyDescription;

        private String policyName;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private String tier;

        private BuilderImpl() {
        }

        private BuilderImpl(ResiliencyPolicy model) {
            creationTime(model.creationTime);
            dataLocationConstraint(model.dataLocationConstraint);
            estimatedCostTier(model.estimatedCostTier);
            policyWithStrings(model.policy);
            policyArn(model.policyArn);
            policyDescription(model.policyDescription);
            policyName(model.policyName);
            tags(model.tags);
            tier(model.tier);
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final String getDataLocationConstraint() {
            return dataLocationConstraint;
        }

        public final void setDataLocationConstraint(String dataLocationConstraint) {
            this.dataLocationConstraint = dataLocationConstraint;
        }

        @Override
        public final Builder dataLocationConstraint(String dataLocationConstraint) {
            this.dataLocationConstraint = dataLocationConstraint;
            return this;
        }

        @Override
        public final Builder dataLocationConstraint(DataLocationConstraint dataLocationConstraint) {
            this.dataLocationConstraint(dataLocationConstraint == null ? null : dataLocationConstraint.toString());
            return this;
        }

        public final String getEstimatedCostTier() {
            return estimatedCostTier;
        }

        public final void setEstimatedCostTier(String estimatedCostTier) {
            this.estimatedCostTier = estimatedCostTier;
        }

        @Override
        public final Builder estimatedCostTier(String estimatedCostTier) {
            this.estimatedCostTier = estimatedCostTier;
            return this;
        }

        @Override
        public final Builder estimatedCostTier(EstimatedCostTier estimatedCostTier) {
            this.estimatedCostTier(estimatedCostTier == null ? null : estimatedCostTier.toString());
            return this;
        }

        public final Map<String, FailurePolicy.Builder> getPolicy() {
            Map<String, FailurePolicy.Builder> result = DisruptionPolicyCopier.copyToBuilder(this.policy);
            if (result instanceof SdkAutoConstructMap) {
                return null;
            }
            return result;
        }

        public final void setPolicy(Map<String, FailurePolicy.BuilderImpl> policy) {
            this.policy = DisruptionPolicyCopier.copyFromBuilder(policy);
        }

        @Override
        public final Builder policyWithStrings(Map<String, FailurePolicy> policy) {
            this.policy = DisruptionPolicyCopier.copy(policy);
            return this;
        }

        @Override
        public final Builder policy(Map<DisruptionType, FailurePolicy> policy) {
            this.policy = DisruptionPolicyCopier.copyEnumToString(policy);
            return this;
        }

        public final String getPolicyArn() {
            return policyArn;
        }

        public final void setPolicyArn(String policyArn) {
            this.policyArn = policyArn;
        }

        @Override
        public final Builder policyArn(String policyArn) {
            this.policyArn = policyArn;
            return this;
        }

        public final String getPolicyDescription() {
            return policyDescription;
        }

        public final void setPolicyDescription(String policyDescription) {
            this.policyDescription = policyDescription;
        }

        @Override
        public final Builder policyDescription(String policyDescription) {
            this.policyDescription = policyDescription;
            return this;
        }

        public final String getPolicyName() {
            return policyName;
        }

        public final void setPolicyName(String policyName) {
            this.policyName = policyName;
        }

        @Override
        public final Builder policyName(String policyName) {
            this.policyName = policyName;
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
            return this;
        }

        public final String getTier() {
            return tier;
        }

        public final void setTier(String tier) {
            this.tier = tier;
        }

        @Override
        public final Builder tier(String tier) {
            this.tier = tier;
            return this;
        }

        @Override
        public final Builder tier(ResiliencyPolicyTier tier) {
            this.tier(tier == null ? null : tier.toString());
            return this;
        }

        @Override
        public ResiliencyPolicy build() {
            return new ResiliencyPolicy(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
