/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.resiliencehub.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Creates a new grouping recommendation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class GroupingRecommendation implements SdkPojo, Serializable,
        ToCopyableBuilder<GroupingRecommendation.Builder, GroupingRecommendation> {
    private static final SdkField<String> CONFIDENCE_LEVEL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("confidenceLevel").getter(getter(GroupingRecommendation::confidenceLevelAsString))
            .setter(setter(Builder::confidenceLevel))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("confidenceLevel").build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("creationTime").getter(getter(GroupingRecommendation::creationTime))
            .setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("creationTime").build()).build();

    private static final SdkField<GroupingAppComponent> GROUPING_APP_COMPONENT_FIELD = SdkField
            .<GroupingAppComponent> builder(MarshallingType.SDK_POJO).memberName("groupingAppComponent")
            .getter(getter(GroupingRecommendation::groupingAppComponent)).setter(setter(Builder::groupingAppComponent))
            .constructor(GroupingAppComponent::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("groupingAppComponent").build())
            .build();

    private static final SdkField<String> GROUPING_RECOMMENDATION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("groupingRecommendationId").getter(getter(GroupingRecommendation::groupingRecommendationId))
            .setter(setter(Builder::groupingRecommendationId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("groupingRecommendationId").build())
            .build();

    private static final SdkField<List<String>> RECOMMENDATION_REASONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("recommendationReasons")
            .getter(getter(GroupingRecommendation::recommendationReasons))
            .setter(setter(Builder::recommendationReasons))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("recommendationReasons").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> REJECTION_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("rejectionReason").getter(getter(GroupingRecommendation::rejectionReasonAsString))
            .setter(setter(Builder::rejectionReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("rejectionReason").build()).build();

    private static final SdkField<List<GroupingResource>> RESOURCES_FIELD = SdkField
            .<List<GroupingResource>> builder(MarshallingType.LIST)
            .memberName("resources")
            .getter(getter(GroupingRecommendation::resources))
            .setter(setter(Builder::resources))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resources").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<GroupingResource> builder(MarshallingType.SDK_POJO)
                                            .constructor(GroupingResource::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Double> SCORE_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE).memberName("score")
            .getter(getter(GroupingRecommendation::score)).setter(setter(Builder::score))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("score").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(GroupingRecommendation::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CONFIDENCE_LEVEL_FIELD,
            CREATION_TIME_FIELD, GROUPING_APP_COMPONENT_FIELD, GROUPING_RECOMMENDATION_ID_FIELD, RECOMMENDATION_REASONS_FIELD,
            REJECTION_REASON_FIELD, RESOURCES_FIELD, SCORE_FIELD, STATUS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String confidenceLevel;

    private final Instant creationTime;

    private final GroupingAppComponent groupingAppComponent;

    private final String groupingRecommendationId;

    private final List<String> recommendationReasons;

    private final String rejectionReason;

    private final List<GroupingResource> resources;

    private final Double score;

    private final String status;

    private GroupingRecommendation(BuilderImpl builder) {
        this.confidenceLevel = builder.confidenceLevel;
        this.creationTime = builder.creationTime;
        this.groupingAppComponent = builder.groupingAppComponent;
        this.groupingRecommendationId = builder.groupingRecommendationId;
        this.recommendationReasons = builder.recommendationReasons;
        this.rejectionReason = builder.rejectionReason;
        this.resources = builder.resources;
        this.score = builder.score;
        this.status = builder.status;
    }

    /**
     * <p>
     * Indicates the confidence level of Resilience Hub on the grouping recommendation.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #confidenceLevel}
     * will return {@link GroupingRecommendationConfidenceLevel#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #confidenceLevelAsString}.
     * </p>
     * 
     * @return Indicates the confidence level of Resilience Hub on the grouping recommendation.
     * @see GroupingRecommendationConfidenceLevel
     */
    public final GroupingRecommendationConfidenceLevel confidenceLevel() {
        return GroupingRecommendationConfidenceLevel.fromValue(confidenceLevel);
    }

    /**
     * <p>
     * Indicates the confidence level of Resilience Hub on the grouping recommendation.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #confidenceLevel}
     * will return {@link GroupingRecommendationConfidenceLevel#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #confidenceLevelAsString}.
     * </p>
     * 
     * @return Indicates the confidence level of Resilience Hub on the grouping recommendation.
     * @see GroupingRecommendationConfidenceLevel
     */
    public final String confidenceLevelAsString() {
        return confidenceLevel;
    }

    /**
     * <p>
     * Indicates the creation time of the grouping recommendation.
     * </p>
     * 
     * @return Indicates the creation time of the grouping recommendation.
     */
    public final Instant creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * Indicates the name of the recommended Application Component (AppComponent).
     * </p>
     * 
     * @return Indicates the name of the recommended Application Component (AppComponent).
     */
    public final GroupingAppComponent groupingAppComponent() {
        return groupingAppComponent;
    }

    /**
     * <p>
     * Indicates all the reasons available for rejecting a grouping recommendation.
     * </p>
     * 
     * @return Indicates all the reasons available for rejecting a grouping recommendation.
     */
    public final String groupingRecommendationId() {
        return groupingRecommendationId;
    }

    /**
     * For responses, this returns true if the service returned a value for the RecommendationReasons property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasRecommendationReasons() {
        return recommendationReasons != null && !(recommendationReasons instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Indicates all the reasons available for rejecting a grouping recommendation.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRecommendationReasons} method.
     * </p>
     * 
     * @return Indicates all the reasons available for rejecting a grouping recommendation.
     */
    public final List<String> recommendationReasons() {
        return recommendationReasons;
    }

    /**
     * <p>
     * Indicates the reason you had selected while rejecting a grouping recommendation.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #rejectionReason}
     * will return {@link GroupingRecommendationRejectionReason#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #rejectionReasonAsString}.
     * </p>
     * 
     * @return Indicates the reason you had selected while rejecting a grouping recommendation.
     * @see GroupingRecommendationRejectionReason
     */
    public final GroupingRecommendationRejectionReason rejectionReason() {
        return GroupingRecommendationRejectionReason.fromValue(rejectionReason);
    }

    /**
     * <p>
     * Indicates the reason you had selected while rejecting a grouping recommendation.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #rejectionReason}
     * will return {@link GroupingRecommendationRejectionReason#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #rejectionReasonAsString}.
     * </p>
     * 
     * @return Indicates the reason you had selected while rejecting a grouping recommendation.
     * @see GroupingRecommendationRejectionReason
     */
    public final String rejectionReasonAsString() {
        return rejectionReason;
    }

    /**
     * For responses, this returns true if the service returned a value for the Resources property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasResources() {
        return resources != null && !(resources instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Indicates the resources that are grouped in a recommended AppComponent.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResources} method.
     * </p>
     * 
     * @return Indicates the resources that are grouped in a recommended AppComponent.
     */
    public final List<GroupingResource> resources() {
        return resources;
    }

    /**
     * <p>
     * Indicates the confidence level of the grouping recommendation.
     * </p>
     * 
     * @return Indicates the confidence level of the grouping recommendation.
     */
    public final Double score() {
        return score;
    }

    /**
     * <p>
     * Indicates the status of grouping resources into AppComponents.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link GroupingRecommendationStatusType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #statusAsString}.
     * </p>
     * 
     * @return Indicates the status of grouping resources into AppComponents.
     * @see GroupingRecommendationStatusType
     */
    public final GroupingRecommendationStatusType status() {
        return GroupingRecommendationStatusType.fromValue(status);
    }

    /**
     * <p>
     * Indicates the status of grouping resources into AppComponents.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link GroupingRecommendationStatusType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #statusAsString}.
     * </p>
     * 
     * @return Indicates the status of grouping resources into AppComponents.
     * @see GroupingRecommendationStatusType
     */
    public final String statusAsString() {
        return status;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(confidenceLevelAsString());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(groupingAppComponent());
        hashCode = 31 * hashCode + Objects.hashCode(groupingRecommendationId());
        hashCode = 31 * hashCode + Objects.hashCode(hasRecommendationReasons() ? recommendationReasons() : null);
        hashCode = 31 * hashCode + Objects.hashCode(rejectionReasonAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasResources() ? resources() : null);
        hashCode = 31 * hashCode + Objects.hashCode(score());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GroupingRecommendation)) {
            return false;
        }
        GroupingRecommendation other = (GroupingRecommendation) obj;
        return Objects.equals(confidenceLevelAsString(), other.confidenceLevelAsString())
                && Objects.equals(creationTime(), other.creationTime())
                && Objects.equals(groupingAppComponent(), other.groupingAppComponent())
                && Objects.equals(groupingRecommendationId(), other.groupingRecommendationId())
                && hasRecommendationReasons() == other.hasRecommendationReasons()
                && Objects.equals(recommendationReasons(), other.recommendationReasons())
                && Objects.equals(rejectionReasonAsString(), other.rejectionReasonAsString())
                && hasResources() == other.hasResources() && Objects.equals(resources(), other.resources())
                && Objects.equals(score(), other.score()) && Objects.equals(statusAsString(), other.statusAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("GroupingRecommendation").add("ConfidenceLevel", confidenceLevelAsString())
                .add("CreationTime", creationTime()).add("GroupingAppComponent", groupingAppComponent())
                .add("GroupingRecommendationId", groupingRecommendationId())
                .add("RecommendationReasons", hasRecommendationReasons() ? recommendationReasons() : null)
                .add("RejectionReason", rejectionReasonAsString()).add("Resources", hasResources() ? resources() : null)
                .add("Score", score()).add("Status", statusAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "confidenceLevel":
            return Optional.ofNullable(clazz.cast(confidenceLevelAsString()));
        case "creationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "groupingAppComponent":
            return Optional.ofNullable(clazz.cast(groupingAppComponent()));
        case "groupingRecommendationId":
            return Optional.ofNullable(clazz.cast(groupingRecommendationId()));
        case "recommendationReasons":
            return Optional.ofNullable(clazz.cast(recommendationReasons()));
        case "rejectionReason":
            return Optional.ofNullable(clazz.cast(rejectionReasonAsString()));
        case "resources":
            return Optional.ofNullable(clazz.cast(resources()));
        case "score":
            return Optional.ofNullable(clazz.cast(score()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<GroupingRecommendation, T> g) {
        return obj -> g.apply((GroupingRecommendation) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, GroupingRecommendation> {
        /**
         * <p>
         * Indicates the confidence level of Resilience Hub on the grouping recommendation.
         * </p>
         * 
         * @param confidenceLevel
         *        Indicates the confidence level of Resilience Hub on the grouping recommendation.
         * @see GroupingRecommendationConfidenceLevel
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GroupingRecommendationConfidenceLevel
         */
        Builder confidenceLevel(String confidenceLevel);

        /**
         * <p>
         * Indicates the confidence level of Resilience Hub on the grouping recommendation.
         * </p>
         * 
         * @param confidenceLevel
         *        Indicates the confidence level of Resilience Hub on the grouping recommendation.
         * @see GroupingRecommendationConfidenceLevel
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GroupingRecommendationConfidenceLevel
         */
        Builder confidenceLevel(GroupingRecommendationConfidenceLevel confidenceLevel);

        /**
         * <p>
         * Indicates the creation time of the grouping recommendation.
         * </p>
         * 
         * @param creationTime
         *        Indicates the creation time of the grouping recommendation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * Indicates the name of the recommended Application Component (AppComponent).
         * </p>
         * 
         * @param groupingAppComponent
         *        Indicates the name of the recommended Application Component (AppComponent).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder groupingAppComponent(GroupingAppComponent groupingAppComponent);

        /**
         * <p>
         * Indicates the name of the recommended Application Component (AppComponent).
         * </p>
         * This is a convenience method that creates an instance of the {@link GroupingAppComponent.Builder} avoiding
         * the need to create one manually via {@link GroupingAppComponent#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link GroupingAppComponent.Builder#build()} is called immediately and
         * its result is passed to {@link #groupingAppComponent(GroupingAppComponent)}.
         * 
         * @param groupingAppComponent
         *        a consumer that will call methods on {@link GroupingAppComponent.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #groupingAppComponent(GroupingAppComponent)
         */
        default Builder groupingAppComponent(Consumer<GroupingAppComponent.Builder> groupingAppComponent) {
            return groupingAppComponent(GroupingAppComponent.builder().applyMutation(groupingAppComponent).build());
        }

        /**
         * <p>
         * Indicates all the reasons available for rejecting a grouping recommendation.
         * </p>
         * 
         * @param groupingRecommendationId
         *        Indicates all the reasons available for rejecting a grouping recommendation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder groupingRecommendationId(String groupingRecommendationId);

        /**
         * <p>
         * Indicates all the reasons available for rejecting a grouping recommendation.
         * </p>
         * 
         * @param recommendationReasons
         *        Indicates all the reasons available for rejecting a grouping recommendation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recommendationReasons(Collection<String> recommendationReasons);

        /**
         * <p>
         * Indicates all the reasons available for rejecting a grouping recommendation.
         * </p>
         * 
         * @param recommendationReasons
         *        Indicates all the reasons available for rejecting a grouping recommendation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recommendationReasons(String... recommendationReasons);

        /**
         * <p>
         * Indicates the reason you had selected while rejecting a grouping recommendation.
         * </p>
         * 
         * @param rejectionReason
         *        Indicates the reason you had selected while rejecting a grouping recommendation.
         * @see GroupingRecommendationRejectionReason
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GroupingRecommendationRejectionReason
         */
        Builder rejectionReason(String rejectionReason);

        /**
         * <p>
         * Indicates the reason you had selected while rejecting a grouping recommendation.
         * </p>
         * 
         * @param rejectionReason
         *        Indicates the reason you had selected while rejecting a grouping recommendation.
         * @see GroupingRecommendationRejectionReason
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GroupingRecommendationRejectionReason
         */
        Builder rejectionReason(GroupingRecommendationRejectionReason rejectionReason);

        /**
         * <p>
         * Indicates the resources that are grouped in a recommended AppComponent.
         * </p>
         * 
         * @param resources
         *        Indicates the resources that are grouped in a recommended AppComponent.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resources(Collection<GroupingResource> resources);

        /**
         * <p>
         * Indicates the resources that are grouped in a recommended AppComponent.
         * </p>
         * 
         * @param resources
         *        Indicates the resources that are grouped in a recommended AppComponent.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resources(GroupingResource... resources);

        /**
         * <p>
         * Indicates the resources that are grouped in a recommended AppComponent.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.resiliencehub.model.GroupingResource.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.resiliencehub.model.GroupingResource#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.resiliencehub.model.GroupingResource.Builder#build()} is called
         * immediately and its result is passed to {@link #resources(List<GroupingResource>)}.
         * 
         * @param resources
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.resiliencehub.model.GroupingResource.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resources(java.util.Collection<GroupingResource>)
         */
        Builder resources(Consumer<GroupingResource.Builder>... resources);

        /**
         * <p>
         * Indicates the confidence level of the grouping recommendation.
         * </p>
         * 
         * @param score
         *        Indicates the confidence level of the grouping recommendation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder score(Double score);

        /**
         * <p>
         * Indicates the status of grouping resources into AppComponents.
         * </p>
         * 
         * @param status
         *        Indicates the status of grouping resources into AppComponents.
         * @see GroupingRecommendationStatusType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GroupingRecommendationStatusType
         */
        Builder status(String status);

        /**
         * <p>
         * Indicates the status of grouping resources into AppComponents.
         * </p>
         * 
         * @param status
         *        Indicates the status of grouping resources into AppComponents.
         * @see GroupingRecommendationStatusType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GroupingRecommendationStatusType
         */
        Builder status(GroupingRecommendationStatusType status);
    }

    static final class BuilderImpl implements Builder {
        private String confidenceLevel;

        private Instant creationTime;

        private GroupingAppComponent groupingAppComponent;

        private String groupingRecommendationId;

        private List<String> recommendationReasons = DefaultSdkAutoConstructList.getInstance();

        private String rejectionReason;

        private List<GroupingResource> resources = DefaultSdkAutoConstructList.getInstance();

        private Double score;

        private String status;

        private BuilderImpl() {
        }

        private BuilderImpl(GroupingRecommendation model) {
            confidenceLevel(model.confidenceLevel);
            creationTime(model.creationTime);
            groupingAppComponent(model.groupingAppComponent);
            groupingRecommendationId(model.groupingRecommendationId);
            recommendationReasons(model.recommendationReasons);
            rejectionReason(model.rejectionReason);
            resources(model.resources);
            score(model.score);
            status(model.status);
        }

        public final String getConfidenceLevel() {
            return confidenceLevel;
        }

        public final void setConfidenceLevel(String confidenceLevel) {
            this.confidenceLevel = confidenceLevel;
        }

        @Override
        public final Builder confidenceLevel(String confidenceLevel) {
            this.confidenceLevel = confidenceLevel;
            return this;
        }

        @Override
        public final Builder confidenceLevel(GroupingRecommendationConfidenceLevel confidenceLevel) {
            this.confidenceLevel(confidenceLevel == null ? null : confidenceLevel.toString());
            return this;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final GroupingAppComponent.Builder getGroupingAppComponent() {
            return groupingAppComponent != null ? groupingAppComponent.toBuilder() : null;
        }

        public final void setGroupingAppComponent(GroupingAppComponent.BuilderImpl groupingAppComponent) {
            this.groupingAppComponent = groupingAppComponent != null ? groupingAppComponent.build() : null;
        }

        @Override
        public final Builder groupingAppComponent(GroupingAppComponent groupingAppComponent) {
            this.groupingAppComponent = groupingAppComponent;
            return this;
        }

        public final String getGroupingRecommendationId() {
            return groupingRecommendationId;
        }

        public final void setGroupingRecommendationId(String groupingRecommendationId) {
            this.groupingRecommendationId = groupingRecommendationId;
        }

        @Override
        public final Builder groupingRecommendationId(String groupingRecommendationId) {
            this.groupingRecommendationId = groupingRecommendationId;
            return this;
        }

        public final Collection<String> getRecommendationReasons() {
            if (recommendationReasons instanceof SdkAutoConstructList) {
                return null;
            }
            return recommendationReasons;
        }

        public final void setRecommendationReasons(Collection<String> recommendationReasons) {
            this.recommendationReasons = String255ListCopier.copy(recommendationReasons);
        }

        @Override
        public final Builder recommendationReasons(Collection<String> recommendationReasons) {
            this.recommendationReasons = String255ListCopier.copy(recommendationReasons);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder recommendationReasons(String... recommendationReasons) {
            recommendationReasons(Arrays.asList(recommendationReasons));
            return this;
        }

        public final String getRejectionReason() {
            return rejectionReason;
        }

        public final void setRejectionReason(String rejectionReason) {
            this.rejectionReason = rejectionReason;
        }

        @Override
        public final Builder rejectionReason(String rejectionReason) {
            this.rejectionReason = rejectionReason;
            return this;
        }

        @Override
        public final Builder rejectionReason(GroupingRecommendationRejectionReason rejectionReason) {
            this.rejectionReason(rejectionReason == null ? null : rejectionReason.toString());
            return this;
        }

        public final List<GroupingResource.Builder> getResources() {
            List<GroupingResource.Builder> result = GroupingResourceListCopier.copyToBuilder(this.resources);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setResources(Collection<GroupingResource.BuilderImpl> resources) {
            this.resources = GroupingResourceListCopier.copyFromBuilder(resources);
        }

        @Override
        public final Builder resources(Collection<GroupingResource> resources) {
            this.resources = GroupingResourceListCopier.copy(resources);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resources(GroupingResource... resources) {
            resources(Arrays.asList(resources));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resources(Consumer<GroupingResource.Builder>... resources) {
            resources(Stream.of(resources).map(c -> GroupingResource.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Double getScore() {
            return score;
        }

        public final void setScore(Double score) {
            this.score = score;
        }

        @Override
        public final Builder score(Double score) {
            this.score = score;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(GroupingRecommendationStatusType status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        @Override
        public GroupingRecommendation build() {
            return new GroupingRecommendation(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
