/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.resiliencehub.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Indicates a specific risk identified in the Resilience Hub assessment and the corresponding recommendation provided
 * to address that risk.
 * </p>
 * <note>
 * <p>
 * The assessment summary generated by large language models (LLMs) on Amazon Bedrock are only suggestions. The current
 * level of generative AI technology is not perfect and LLMs are not infallible. Bias and incorrect answers, although
 * rare, should be expected. Review each recommendation in the assessment summary before you use the output from an LLM.
 * </p>
 * </note> <note>
 * <p>
 * This property is available only in the US East (N. Virginia) Region.
 * </p>
 * </note>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AssessmentRiskRecommendation implements SdkPojo, Serializable,
        ToCopyableBuilder<AssessmentRiskRecommendation.Builder, AssessmentRiskRecommendation> {
    private static final SdkField<List<String>> APP_COMPONENTS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("appComponents")
            .getter(getter(AssessmentRiskRecommendation::appComponents))
            .setter(setter(Builder::appComponents))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("appComponents").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> RECOMMENDATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("recommendation").getter(getter(AssessmentRiskRecommendation::recommendation))
            .setter(setter(Builder::recommendation))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("recommendation").build()).build();

    private static final SdkField<String> RISK_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("risk")
            .getter(getter(AssessmentRiskRecommendation::risk)).setter(setter(Builder::risk))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("risk").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(APP_COMPONENTS_FIELD,
            RECOMMENDATION_FIELD, RISK_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final List<String> appComponents;

    private final String recommendation;

    private final String risk;

    private AssessmentRiskRecommendation(BuilderImpl builder) {
        this.appComponents = builder.appComponents;
        this.recommendation = builder.recommendation;
        this.risk = builder.risk;
    }

    /**
     * For responses, this returns true if the service returned a value for the AppComponents property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAppComponents() {
        return appComponents != null && !(appComponents instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Indicates the Application Components (AppComponents) that were assessed as part of the assessment and are
     * associated with the identified risk and recommendation.
     * </p>
     * <note>
     * <p>
     * This property is available only in the US East (N. Virginia) Region.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAppComponents} method.
     * </p>
     * 
     * @return Indicates the Application Components (AppComponents) that were assessed as part of the assessment and are
     *         associated with the identified risk and recommendation.</p> <note>
     *         <p>
     *         This property is available only in the US East (N. Virginia) Region.
     *         </p>
     */
    public final List<String> appComponents() {
        return appComponents;
    }

    /**
     * <p>
     * Indicates the recommendation provided by the Resilience Hub to address the identified risks in the application.
     * </p>
     * <note>
     * <p>
     * This property is available only in the US East (N. Virginia) Region.
     * </p>
     * </note>
     * 
     * @return Indicates the recommendation provided by the Resilience Hub to address the identified risks in the
     *         application.</p> <note>
     *         <p>
     *         This property is available only in the US East (N. Virginia) Region.
     *         </p>
     */
    public final String recommendation() {
        return recommendation;
    }

    /**
     * <p>
     * Indicates the description of the potential risk identified in the application as part of the Resilience Hub
     * assessment.
     * </p>
     * <note>
     * <p>
     * This property is available only in the US East (N. Virginia) Region.
     * </p>
     * </note>
     * 
     * @return Indicates the description of the potential risk identified in the application as part of the Resilience
     *         Hub assessment.</p> <note>
     *         <p>
     *         This property is available only in the US East (N. Virginia) Region.
     *         </p>
     */
    public final String risk() {
        return risk;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasAppComponents() ? appComponents() : null);
        hashCode = 31 * hashCode + Objects.hashCode(recommendation());
        hashCode = 31 * hashCode + Objects.hashCode(risk());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AssessmentRiskRecommendation)) {
            return false;
        }
        AssessmentRiskRecommendation other = (AssessmentRiskRecommendation) obj;
        return hasAppComponents() == other.hasAppComponents() && Objects.equals(appComponents(), other.appComponents())
                && Objects.equals(recommendation(), other.recommendation()) && Objects.equals(risk(), other.risk());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AssessmentRiskRecommendation").add("AppComponents", hasAppComponents() ? appComponents() : null)
                .add("Recommendation", recommendation()).add("Risk", risk()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "appComponents":
            return Optional.ofNullable(clazz.cast(appComponents()));
        case "recommendation":
            return Optional.ofNullable(clazz.cast(recommendation()));
        case "risk":
            return Optional.ofNullable(clazz.cast(risk()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("appComponents", APP_COMPONENTS_FIELD);
        map.put("recommendation", RECOMMENDATION_FIELD);
        map.put("risk", RISK_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<AssessmentRiskRecommendation, T> g) {
        return obj -> g.apply((AssessmentRiskRecommendation) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AssessmentRiskRecommendation> {
        /**
         * <p>
         * Indicates the Application Components (AppComponents) that were assessed as part of the assessment and are
         * associated with the identified risk and recommendation.
         * </p>
         * <note>
         * <p>
         * This property is available only in the US East (N. Virginia) Region.
         * </p>
         * </note>
         * 
         * @param appComponents
         *        Indicates the Application Components (AppComponents) that were assessed as part of the assessment and
         *        are associated with the identified risk and recommendation.</p> <note>
         *        <p>
         *        This property is available only in the US East (N. Virginia) Region.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder appComponents(Collection<String> appComponents);

        /**
         * <p>
         * Indicates the Application Components (AppComponents) that were assessed as part of the assessment and are
         * associated with the identified risk and recommendation.
         * </p>
         * <note>
         * <p>
         * This property is available only in the US East (N. Virginia) Region.
         * </p>
         * </note>
         * 
         * @param appComponents
         *        Indicates the Application Components (AppComponents) that were assessed as part of the assessment and
         *        are associated with the identified risk and recommendation.</p> <note>
         *        <p>
         *        This property is available only in the US East (N. Virginia) Region.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder appComponents(String... appComponents);

        /**
         * <p>
         * Indicates the recommendation provided by the Resilience Hub to address the identified risks in the
         * application.
         * </p>
         * <note>
         * <p>
         * This property is available only in the US East (N. Virginia) Region.
         * </p>
         * </note>
         * 
         * @param recommendation
         *        Indicates the recommendation provided by the Resilience Hub to address the identified risks in the
         *        application.</p> <note>
         *        <p>
         *        This property is available only in the US East (N. Virginia) Region.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recommendation(String recommendation);

        /**
         * <p>
         * Indicates the description of the potential risk identified in the application as part of the Resilience Hub
         * assessment.
         * </p>
         * <note>
         * <p>
         * This property is available only in the US East (N. Virginia) Region.
         * </p>
         * </note>
         * 
         * @param risk
         *        Indicates the description of the potential risk identified in the application as part of the
         *        Resilience Hub assessment.</p> <note>
         *        <p>
         *        This property is available only in the US East (N. Virginia) Region.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder risk(String risk);
    }

    static final class BuilderImpl implements Builder {
        private List<String> appComponents = DefaultSdkAutoConstructList.getInstance();

        private String recommendation;

        private String risk;

        private BuilderImpl() {
        }

        private BuilderImpl(AssessmentRiskRecommendation model) {
            appComponents(model.appComponents);
            recommendation(model.recommendation);
            risk(model.risk);
        }

        public final Collection<String> getAppComponents() {
            if (appComponents instanceof SdkAutoConstructList) {
                return null;
            }
            return appComponents;
        }

        public final void setAppComponents(Collection<String> appComponents) {
            this.appComponents = AppComponentNameListCopier.copy(appComponents);
        }

        @Override
        public final Builder appComponents(Collection<String> appComponents) {
            this.appComponents = AppComponentNameListCopier.copy(appComponents);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder appComponents(String... appComponents) {
            appComponents(Arrays.asList(appComponents));
            return this;
        }

        public final String getRecommendation() {
            return recommendation;
        }

        public final void setRecommendation(String recommendation) {
            this.recommendation = recommendation;
        }

        @Override
        public final Builder recommendation(String recommendation) {
            this.recommendation = recommendation;
            return this;
        }

        public final String getRisk() {
            return risk;
        }

        public final void setRisk(String risk) {
            this.risk = risk;
        }

        @Override
        public final Builder risk(String risk) {
            this.risk = risk;
            return this;
        }

        @Override
        public AssessmentRiskRecommendation build() {
            return new AssessmentRiskRecommendation(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
