/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.resiliencehub.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The input source of the Amazon Elastic Kubernetes Service cluster.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class EksSource implements SdkPojo, Serializable, ToCopyableBuilder<EksSource.Builder, EksSource> {
    private static final SdkField<String> EKS_CLUSTER_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("eksClusterArn").getter(getter(EksSource::eksClusterArn)).setter(setter(Builder::eksClusterArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("eksClusterArn").build()).build();

    private static final SdkField<List<String>> NAMESPACES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("namespaces")
            .getter(getter(EksSource::namespaces))
            .setter(setter(Builder::namespaces))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("namespaces").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(EKS_CLUSTER_ARN_FIELD,
            NAMESPACES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String eksClusterArn;

    private final List<String> namespaces;

    private EksSource(BuilderImpl builder) {
        this.eksClusterArn = builder.eksClusterArn;
        this.namespaces = builder.namespaces;
    }

    /**
     * <p>
     * Amazon Resource Name (ARN) of the Amazon Elastic Kubernetes Service cluster. The format for this ARN is: arn:
     * <code>aws</code>:eks:<code>region</code>:<code>account-id</code>:cluster/<code>cluster-name</code>. For more
     * information about ARNs, see <a href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">
     * Amazon Resource Names (ARNs)</a> in the <i>Amazon Web Services General Reference</i> guide.
     * </p>
     * 
     * @return Amazon Resource Name (ARN) of the Amazon Elastic Kubernetes Service cluster. The format for this ARN is:
     *         arn:<code>aws</code>:eks:<code>region</code>:<code>account-id</code>:cluster/<code>cluster-name</code>.
     *         For more information about ARNs, see <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html"> Amazon Resource Names
     *         (ARNs)</a> in the <i>Amazon Web Services General Reference</i> guide.
     */
    public final String eksClusterArn() {
        return eksClusterArn;
    }

    /**
     * For responses, this returns true if the service returned a value for the Namespaces property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasNamespaces() {
        return namespaces != null && !(namespaces instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of namespaces located on your Amazon Elastic Kubernetes Service cluster.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasNamespaces} method.
     * </p>
     * 
     * @return The list of namespaces located on your Amazon Elastic Kubernetes Service cluster.
     */
    public final List<String> namespaces() {
        return namespaces;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(eksClusterArn());
        hashCode = 31 * hashCode + Objects.hashCode(hasNamespaces() ? namespaces() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EksSource)) {
            return false;
        }
        EksSource other = (EksSource) obj;
        return Objects.equals(eksClusterArn(), other.eksClusterArn()) && hasNamespaces() == other.hasNamespaces()
                && Objects.equals(namespaces(), other.namespaces());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("EksSource").add("EksClusterArn", eksClusterArn())
                .add("Namespaces", hasNamespaces() ? namespaces() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "eksClusterArn":
            return Optional.ofNullable(clazz.cast(eksClusterArn()));
        case "namespaces":
            return Optional.ofNullable(clazz.cast(namespaces()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("eksClusterArn", EKS_CLUSTER_ARN_FIELD);
        map.put("namespaces", NAMESPACES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<EksSource, T> g) {
        return obj -> g.apply((EksSource) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, EksSource> {
        /**
         * <p>
         * Amazon Resource Name (ARN) of the Amazon Elastic Kubernetes Service cluster. The format for this ARN is: arn:
         * <code>aws</code>:eks:<code>region</code>:<code>account-id</code>:cluster/<code>cluster-name</code>. For more
         * information about ARNs, see <a
         * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html"> Amazon Resource Names
         * (ARNs)</a> in the <i>Amazon Web Services General Reference</i> guide.
         * </p>
         * 
         * @param eksClusterArn
         *        Amazon Resource Name (ARN) of the Amazon Elastic Kubernetes Service cluster. The format for this ARN
         *        is: arn:<code>aws</code>:eks:<code>region</code>:<code>account-id</code>:cluster/
         *        <code>cluster-name</code>. For more information about ARNs, see <a
         *        href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html"> Amazon Resource
         *        Names (ARNs)</a> in the <i>Amazon Web Services General Reference</i> guide.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eksClusterArn(String eksClusterArn);

        /**
         * <p>
         * The list of namespaces located on your Amazon Elastic Kubernetes Service cluster.
         * </p>
         * 
         * @param namespaces
         *        The list of namespaces located on your Amazon Elastic Kubernetes Service cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder namespaces(Collection<String> namespaces);

        /**
         * <p>
         * The list of namespaces located on your Amazon Elastic Kubernetes Service cluster.
         * </p>
         * 
         * @param namespaces
         *        The list of namespaces located on your Amazon Elastic Kubernetes Service cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder namespaces(String... namespaces);
    }

    static final class BuilderImpl implements Builder {
        private String eksClusterArn;

        private List<String> namespaces = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(EksSource model) {
            eksClusterArn(model.eksClusterArn);
            namespaces(model.namespaces);
        }

        public final String getEksClusterArn() {
            return eksClusterArn;
        }

        public final void setEksClusterArn(String eksClusterArn) {
            this.eksClusterArn = eksClusterArn;
        }

        @Override
        public final Builder eksClusterArn(String eksClusterArn) {
            this.eksClusterArn = eksClusterArn;
            return this;
        }

        public final Collection<String> getNamespaces() {
            if (namespaces instanceof SdkAutoConstructList) {
                return null;
            }
            return namespaces;
        }

        public final void setNamespaces(Collection<String> namespaces) {
            this.namespaces = EksNamespaceListCopier.copy(namespaces);
        }

        @Override
        public final Builder namespaces(Collection<String> namespaces) {
            this.namespaces = EksNamespaceListCopier.copy(namespaces);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder namespaces(String... namespaces) {
            namespaces(Arrays.asList(namespaces));
            return this;
        }

        @Override
        public EksSource build() {
            return new EksSource(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
