/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.resourcegroups;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.resourcegroups.model.BadRequestException;
import software.amazon.awssdk.services.resourcegroups.model.CreateGroupRequest;
import software.amazon.awssdk.services.resourcegroups.model.CreateGroupResponse;
import software.amazon.awssdk.services.resourcegroups.model.DeleteGroupRequest;
import software.amazon.awssdk.services.resourcegroups.model.DeleteGroupResponse;
import software.amazon.awssdk.services.resourcegroups.model.ForbiddenException;
import software.amazon.awssdk.services.resourcegroups.model.GetGroupQueryRequest;
import software.amazon.awssdk.services.resourcegroups.model.GetGroupQueryResponse;
import software.amazon.awssdk.services.resourcegroups.model.GetGroupRequest;
import software.amazon.awssdk.services.resourcegroups.model.GetGroupResponse;
import software.amazon.awssdk.services.resourcegroups.model.GetTagsRequest;
import software.amazon.awssdk.services.resourcegroups.model.GetTagsResponse;
import software.amazon.awssdk.services.resourcegroups.model.InternalServerErrorException;
import software.amazon.awssdk.services.resourcegroups.model.ListGroupResourcesRequest;
import software.amazon.awssdk.services.resourcegroups.model.ListGroupResourcesResponse;
import software.amazon.awssdk.services.resourcegroups.model.ListGroupsRequest;
import software.amazon.awssdk.services.resourcegroups.model.ListGroupsResponse;
import software.amazon.awssdk.services.resourcegroups.model.MethodNotAllowedException;
import software.amazon.awssdk.services.resourcegroups.model.NotFoundException;
import software.amazon.awssdk.services.resourcegroups.model.ResourceGroupsException;
import software.amazon.awssdk.services.resourcegroups.model.ResourceGroupsRequest;
import software.amazon.awssdk.services.resourcegroups.model.SearchResourcesRequest;
import software.amazon.awssdk.services.resourcegroups.model.SearchResourcesResponse;
import software.amazon.awssdk.services.resourcegroups.model.TagRequest;
import software.amazon.awssdk.services.resourcegroups.model.TagResponse;
import software.amazon.awssdk.services.resourcegroups.model.TooManyRequestsException;
import software.amazon.awssdk.services.resourcegroups.model.UnauthorizedException;
import software.amazon.awssdk.services.resourcegroups.model.UntagRequest;
import software.amazon.awssdk.services.resourcegroups.model.UntagResponse;
import software.amazon.awssdk.services.resourcegroups.model.UpdateGroupQueryRequest;
import software.amazon.awssdk.services.resourcegroups.model.UpdateGroupQueryResponse;
import software.amazon.awssdk.services.resourcegroups.model.UpdateGroupRequest;
import software.amazon.awssdk.services.resourcegroups.model.UpdateGroupResponse;
import software.amazon.awssdk.services.resourcegroups.paginators.ListGroupResourcesPublisher;
import software.amazon.awssdk.services.resourcegroups.paginators.ListGroupsPublisher;
import software.amazon.awssdk.services.resourcegroups.paginators.SearchResourcesPublisher;
import software.amazon.awssdk.services.resourcegroups.transform.CreateGroupRequestMarshaller;
import software.amazon.awssdk.services.resourcegroups.transform.DeleteGroupRequestMarshaller;
import software.amazon.awssdk.services.resourcegroups.transform.GetGroupQueryRequestMarshaller;
import software.amazon.awssdk.services.resourcegroups.transform.GetGroupRequestMarshaller;
import software.amazon.awssdk.services.resourcegroups.transform.GetTagsRequestMarshaller;
import software.amazon.awssdk.services.resourcegroups.transform.ListGroupResourcesRequestMarshaller;
import software.amazon.awssdk.services.resourcegroups.transform.ListGroupsRequestMarshaller;
import software.amazon.awssdk.services.resourcegroups.transform.SearchResourcesRequestMarshaller;
import software.amazon.awssdk.services.resourcegroups.transform.TagRequestMarshaller;
import software.amazon.awssdk.services.resourcegroups.transform.UntagRequestMarshaller;
import software.amazon.awssdk.services.resourcegroups.transform.UpdateGroupQueryRequestMarshaller;
import software.amazon.awssdk.services.resourcegroups.transform.UpdateGroupRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link ResourceGroupsAsyncClient}.
 *
 * @see ResourceGroupsAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultResourceGroupsAsyncClient implements ResourceGroupsAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultResourceGroupsAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultResourceGroupsAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates a group with a specified name, description, and resource query.
     * </p>
     *
     * @param createGroupRequest
     * @return A Java Future containing the result of the CreateGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request does not comply with validation rules that are defined for the
     *         request parameters.</li>
     *         <li>ForbiddenException The caller is not authorized to make the request.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method which is not allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException The caller has exceeded throttling limits.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.CreateGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/CreateGroup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateGroupResponse> createGroup(CreateGroupRequest createGroupRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateGroupRequest, CreateGroupResponse>()
                            .withOperationName("CreateGroup").withMarshaller(new CreateGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createGroupRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a specified resource group. Deleting a resource group does not delete resources that are members of the
     * group; it only deletes the group structure.
     * </p>
     *
     * @param deleteGroupRequest
     * @return A Java Future containing the result of the DeleteGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request does not comply with validation rules that are defined for the
     *         request parameters.</li>
     *         <li>ForbiddenException The caller is not authorized to make the request.</li>
     *         <li>NotFoundException One or more resources specified in the request do not exist.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method which is not allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException The caller has exceeded throttling limits.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.DeleteGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/DeleteGroup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteGroupResponse> deleteGroup(DeleteGroupRequest deleteGroupRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteGroupRequest, DeleteGroupResponse>()
                            .withOperationName("DeleteGroup").withMarshaller(new DeleteGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteGroupRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about a specified resource group.
     * </p>
     *
     * @param getGroupRequest
     * @return A Java Future containing the result of the GetGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request does not comply with validation rules that are defined for the
     *         request parameters.</li>
     *         <li>ForbiddenException The caller is not authorized to make the request.</li>
     *         <li>NotFoundException One or more resources specified in the request do not exist.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method which is not allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException The caller has exceeded throttling limits.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.GetGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/GetGroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetGroupResponse> getGroup(GetGroupRequest getGroupRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetGroupRequest, GetGroupResponse>().withOperationName("GetGroup")
                            .withMarshaller(new GetGroupRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getGroupRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the resource query associated with the specified resource group.
     * </p>
     *
     * @param getGroupQueryRequest
     * @return A Java Future containing the result of the GetGroupQuery operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request does not comply with validation rules that are defined for the
     *         request parameters.</li>
     *         <li>ForbiddenException The caller is not authorized to make the request.</li>
     *         <li>NotFoundException One or more resources specified in the request do not exist.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method which is not allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException The caller has exceeded throttling limits.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.GetGroupQuery
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/GetGroupQuery" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetGroupQueryResponse> getGroupQuery(GetGroupQueryRequest getGroupQueryRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetGroupQueryResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetGroupQueryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetGroupQueryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetGroupQueryRequest, GetGroupQueryResponse>()
                            .withOperationName("GetGroupQuery")
                            .withMarshaller(new GetGroupQueryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getGroupQueryRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of tags that are associated with a resource group, specified by an ARN.
     * </p>
     *
     * @param getTagsRequest
     * @return A Java Future containing the result of the GetTags operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request does not comply with validation rules that are defined for the
     *         request parameters.</li>
     *         <li>ForbiddenException The caller is not authorized to make the request.</li>
     *         <li>NotFoundException One or more resources specified in the request do not exist.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method which is not allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException The caller has exceeded throttling limits.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.GetTags
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/GetTags" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetTagsResponse> getTags(GetTagsRequest getTagsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetTagsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetTagsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetTagsRequest, GetTagsResponse>().withOperationName("GetTags")
                            .withMarshaller(new GetTagsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getTagsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of ARNs of resources that are members of a specified resource group.
     * </p>
     *
     * @param listGroupResourcesRequest
     * @return A Java Future containing the result of the ListGroupResources operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException The request has not been applied because it lacks valid authentication
     *         credentials for the target resource.</li>
     *         <li>BadRequestException The request does not comply with validation rules that are defined for the
     *         request parameters.</li>
     *         <li>ForbiddenException The caller is not authorized to make the request.</li>
     *         <li>NotFoundException One or more resources specified in the request do not exist.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method which is not allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException The caller has exceeded throttling limits.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.ListGroupResources
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/ListGroupResources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListGroupResourcesResponse> listGroupResources(ListGroupResourcesRequest listGroupResourcesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListGroupResourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListGroupResourcesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListGroupResourcesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListGroupResourcesRequest, ListGroupResourcesResponse>()
                            .withOperationName("ListGroupResources")
                            .withMarshaller(new ListGroupResourcesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listGroupResourcesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of ARNs of resources that are members of a specified resource group.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listGroupResources(software.amazon.awssdk.services.resourcegroups.model.ListGroupResourcesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.resourcegroups.paginators.ListGroupResourcesPublisher publisher = client.listGroupResourcesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.resourcegroups.paginators.ListGroupResourcesPublisher publisher = client.listGroupResourcesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.resourcegroups.model.ListGroupResourcesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.resourcegroups.model.ListGroupResourcesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listGroupResources(software.amazon.awssdk.services.resourcegroups.model.ListGroupResourcesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listGroupResourcesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException The request has not been applied because it lacks valid authentication
     *         credentials for the target resource.</li>
     *         <li>BadRequestException The request does not comply with validation rules that are defined for the
     *         request parameters.</li>
     *         <li>ForbiddenException The caller is not authorized to make the request.</li>
     *         <li>NotFoundException One or more resources specified in the request do not exist.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method which is not allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException The caller has exceeded throttling limits.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.ListGroupResources
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/ListGroupResources"
     *      target="_top">AWS API Documentation</a>
     */
    public ListGroupResourcesPublisher listGroupResourcesPaginator(ListGroupResourcesRequest listGroupResourcesRequest) {
        return new ListGroupResourcesPublisher(this, applyPaginatorUserAgent(listGroupResourcesRequest));
    }

    /**
     * <p>
     * Returns a list of existing resource groups in your account.
     * </p>
     *
     * @param listGroupsRequest
     * @return A Java Future containing the result of the ListGroups operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request does not comply with validation rules that are defined for the
     *         request parameters.</li>
     *         <li>ForbiddenException The caller is not authorized to make the request.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method which is not allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException The caller has exceeded throttling limits.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.ListGroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/ListGroups" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListGroupsResponse> listGroups(ListGroupsRequest listGroupsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListGroupsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListGroupsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListGroupsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListGroupsRequest, ListGroupsResponse>().withOperationName("ListGroups")
                            .withMarshaller(new ListGroupsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listGroupsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of existing resource groups in your account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listGroups(software.amazon.awssdk.services.resourcegroups.model.ListGroupsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.resourcegroups.paginators.ListGroupsPublisher publisher = client.listGroupsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.resourcegroups.paginators.ListGroupsPublisher publisher = client.listGroupsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.resourcegroups.model.ListGroupsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.resourcegroups.model.ListGroupsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listGroups(software.amazon.awssdk.services.resourcegroups.model.ListGroupsRequest)} operation.</b>
     * </p>
     *
     * @param listGroupsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request does not comply with validation rules that are defined for the
     *         request parameters.</li>
     *         <li>ForbiddenException The caller is not authorized to make the request.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method which is not allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException The caller has exceeded throttling limits.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.ListGroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/ListGroups" target="_top">AWS API
     *      Documentation</a>
     */
    public ListGroupsPublisher listGroupsPaginator(ListGroupsRequest listGroupsRequest) {
        return new ListGroupsPublisher(this, applyPaginatorUserAgent(listGroupsRequest));
    }

    /**
     * <p>
     * Returns a list of AWS resource identifiers that matches a specified query. The query uses the same format as a
     * resource query in a CreateGroup or UpdateGroupQuery operation.
     * </p>
     *
     * @param searchResourcesRequest
     * @return A Java Future containing the result of the SearchResources operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException The request has not been applied because it lacks valid authentication
     *         credentials for the target resource.</li>
     *         <li>BadRequestException The request does not comply with validation rules that are defined for the
     *         request parameters.</li>
     *         <li>ForbiddenException The caller is not authorized to make the request.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method which is not allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException The caller has exceeded throttling limits.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.SearchResources
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/SearchResources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<SearchResourcesResponse> searchResources(SearchResourcesRequest searchResourcesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SearchResourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, SearchResourcesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SearchResourcesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SearchResourcesRequest, SearchResourcesResponse>()
                            .withOperationName("SearchResources")
                            .withMarshaller(new SearchResourcesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(searchResourcesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of AWS resource identifiers that matches a specified query. The query uses the same format as a
     * resource query in a CreateGroup or UpdateGroupQuery operation.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #searchResources(software.amazon.awssdk.services.resourcegroups.model.SearchResourcesRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.resourcegroups.paginators.SearchResourcesPublisher publisher = client.searchResourcesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.resourcegroups.paginators.SearchResourcesPublisher publisher = client.searchResourcesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.resourcegroups.model.SearchResourcesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.resourcegroups.model.SearchResourcesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #searchResources(software.amazon.awssdk.services.resourcegroups.model.SearchResourcesRequest)}
     * operation.</b>
     * </p>
     *
     * @param searchResourcesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException The request has not been applied because it lacks valid authentication
     *         credentials for the target resource.</li>
     *         <li>BadRequestException The request does not comply with validation rules that are defined for the
     *         request parameters.</li>
     *         <li>ForbiddenException The caller is not authorized to make the request.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method which is not allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException The caller has exceeded throttling limits.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.SearchResources
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/SearchResources"
     *      target="_top">AWS API Documentation</a>
     */
    public SearchResourcesPublisher searchResourcesPaginator(SearchResourcesRequest searchResourcesRequest) {
        return new SearchResourcesPublisher(this, applyPaginatorUserAgent(searchResourcesRequest));
    }

    /**
     * <p>
     * Adds tags to a resource group with the specified ARN. Existing tags on a resource group are not changed if they
     * are not specified in the request parameters.
     * </p>
     *
     * @param tagRequest
     * @return A Java Future containing the result of the Tag operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request does not comply with validation rules that are defined for the
     *         request parameters.</li>
     *         <li>ForbiddenException The caller is not authorized to make the request.</li>
     *         <li>NotFoundException One or more resources specified in the request do not exist.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method which is not allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException The caller has exceeded throttling limits.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.Tag
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/Tag" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResponse> tag(TagRequest tagRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagRequest, TagResponse>().withOperationName("Tag")
                            .withMarshaller(new TagRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(tagRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes specified tags from a specified resource.
     * </p>
     *
     * @param untagRequest
     * @return A Java Future containing the result of the Untag operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request does not comply with validation rules that are defined for the
     *         request parameters.</li>
     *         <li>ForbiddenException The caller is not authorized to make the request.</li>
     *         <li>NotFoundException One or more resources specified in the request do not exist.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method which is not allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException The caller has exceeded throttling limits.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.Untag
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/Untag" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResponse> untag(UntagRequest untagRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagRequest, UntagResponse>().withOperationName("Untag")
                            .withMarshaller(new UntagRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(untagRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an existing group with a new or changed description. You cannot update the name of a resource group.
     * </p>
     *
     * @param updateGroupRequest
     * @return A Java Future containing the result of the UpdateGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request does not comply with validation rules that are defined for the
     *         request parameters.</li>
     *         <li>ForbiddenException The caller is not authorized to make the request.</li>
     *         <li>NotFoundException One or more resources specified in the request do not exist.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method which is not allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException The caller has exceeded throttling limits.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.UpdateGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/UpdateGroup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateGroupResponse> updateGroup(UpdateGroupRequest updateGroupRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateGroupRequest, UpdateGroupResponse>()
                            .withOperationName("UpdateGroup").withMarshaller(new UpdateGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateGroupRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the resource query of a group.
     * </p>
     *
     * @param updateGroupQueryRequest
     * @return A Java Future containing the result of the UpdateGroupQuery operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request does not comply with validation rules that are defined for the
     *         request parameters.</li>
     *         <li>ForbiddenException The caller is not authorized to make the request.</li>
     *         <li>NotFoundException One or more resources specified in the request do not exist.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method which is not allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException The caller has exceeded throttling limits.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.UpdateGroupQuery
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/UpdateGroupQuery"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateGroupQueryResponse> updateGroupQuery(UpdateGroupQueryRequest updateGroupQueryRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateGroupQueryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateGroupQueryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateGroupQueryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateGroupQueryRequest, UpdateGroupQueryResponse>()
                            .withOperationName("UpdateGroupQuery")
                            .withMarshaller(new UpdateGroupQueryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateGroupQueryRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ResourceGroupsException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnauthorizedException")
                                .exceptionBuilderSupplier(UnauthorizedException::builder).httpStatusCode(401).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ForbiddenException")
                                .exceptionBuilderSupplier(ForbiddenException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MethodNotAllowedException")
                                .exceptionBuilderSupplier(MethodNotAllowedException::builder).httpStatusCode(405).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerErrorException")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).httpStatusCode(500).build());
    }

    private <T extends ResourceGroupsRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
