/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.resourcegroups;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.resourcegroups.model.BadRequestException;
import software.amazon.awssdk.services.resourcegroups.model.CreateGroupRequest;
import software.amazon.awssdk.services.resourcegroups.model.CreateGroupResponse;
import software.amazon.awssdk.services.resourcegroups.model.DeleteGroupRequest;
import software.amazon.awssdk.services.resourcegroups.model.DeleteGroupResponse;
import software.amazon.awssdk.services.resourcegroups.model.ForbiddenException;
import software.amazon.awssdk.services.resourcegroups.model.GetGroupConfigurationRequest;
import software.amazon.awssdk.services.resourcegroups.model.GetGroupConfigurationResponse;
import software.amazon.awssdk.services.resourcegroups.model.GetGroupQueryRequest;
import software.amazon.awssdk.services.resourcegroups.model.GetGroupQueryResponse;
import software.amazon.awssdk.services.resourcegroups.model.GetGroupRequest;
import software.amazon.awssdk.services.resourcegroups.model.GetGroupResponse;
import software.amazon.awssdk.services.resourcegroups.model.GetTagsRequest;
import software.amazon.awssdk.services.resourcegroups.model.GetTagsResponse;
import software.amazon.awssdk.services.resourcegroups.model.GroupResourcesRequest;
import software.amazon.awssdk.services.resourcegroups.model.GroupResourcesResponse;
import software.amazon.awssdk.services.resourcegroups.model.InternalServerErrorException;
import software.amazon.awssdk.services.resourcegroups.model.ListGroupResourcesRequest;
import software.amazon.awssdk.services.resourcegroups.model.ListGroupResourcesResponse;
import software.amazon.awssdk.services.resourcegroups.model.ListGroupsRequest;
import software.amazon.awssdk.services.resourcegroups.model.ListGroupsResponse;
import software.amazon.awssdk.services.resourcegroups.model.MethodNotAllowedException;
import software.amazon.awssdk.services.resourcegroups.model.NotFoundException;
import software.amazon.awssdk.services.resourcegroups.model.ResourceGroupsException;
import software.amazon.awssdk.services.resourcegroups.model.ResourceGroupsRequest;
import software.amazon.awssdk.services.resourcegroups.model.SearchResourcesRequest;
import software.amazon.awssdk.services.resourcegroups.model.SearchResourcesResponse;
import software.amazon.awssdk.services.resourcegroups.model.TagRequest;
import software.amazon.awssdk.services.resourcegroups.model.TagResponse;
import software.amazon.awssdk.services.resourcegroups.model.TooManyRequestsException;
import software.amazon.awssdk.services.resourcegroups.model.UnauthorizedException;
import software.amazon.awssdk.services.resourcegroups.model.UngroupResourcesRequest;
import software.amazon.awssdk.services.resourcegroups.model.UngroupResourcesResponse;
import software.amazon.awssdk.services.resourcegroups.model.UntagRequest;
import software.amazon.awssdk.services.resourcegroups.model.UntagResponse;
import software.amazon.awssdk.services.resourcegroups.model.UpdateGroupQueryRequest;
import software.amazon.awssdk.services.resourcegroups.model.UpdateGroupQueryResponse;
import software.amazon.awssdk.services.resourcegroups.model.UpdateGroupRequest;
import software.amazon.awssdk.services.resourcegroups.model.UpdateGroupResponse;
import software.amazon.awssdk.services.resourcegroups.paginators.ListGroupResourcesIterable;
import software.amazon.awssdk.services.resourcegroups.paginators.ListGroupsIterable;
import software.amazon.awssdk.services.resourcegroups.paginators.SearchResourcesIterable;
import software.amazon.awssdk.services.resourcegroups.transform.CreateGroupRequestMarshaller;
import software.amazon.awssdk.services.resourcegroups.transform.DeleteGroupRequestMarshaller;
import software.amazon.awssdk.services.resourcegroups.transform.GetGroupConfigurationRequestMarshaller;
import software.amazon.awssdk.services.resourcegroups.transform.GetGroupQueryRequestMarshaller;
import software.amazon.awssdk.services.resourcegroups.transform.GetGroupRequestMarshaller;
import software.amazon.awssdk.services.resourcegroups.transform.GetTagsRequestMarshaller;
import software.amazon.awssdk.services.resourcegroups.transform.GroupResourcesRequestMarshaller;
import software.amazon.awssdk.services.resourcegroups.transform.ListGroupResourcesRequestMarshaller;
import software.amazon.awssdk.services.resourcegroups.transform.ListGroupsRequestMarshaller;
import software.amazon.awssdk.services.resourcegroups.transform.SearchResourcesRequestMarshaller;
import software.amazon.awssdk.services.resourcegroups.transform.TagRequestMarshaller;
import software.amazon.awssdk.services.resourcegroups.transform.UngroupResourcesRequestMarshaller;
import software.amazon.awssdk.services.resourcegroups.transform.UntagRequestMarshaller;
import software.amazon.awssdk.services.resourcegroups.transform.UpdateGroupQueryRequestMarshaller;
import software.amazon.awssdk.services.resourcegroups.transform.UpdateGroupRequestMarshaller;

/**
 * Internal implementation of {@link ResourceGroupsClient}.
 *
 * @see ResourceGroupsClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultResourceGroupsClient implements ResourceGroupsClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultResourceGroupsClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates a resource group with the specified name and description. You can optionally include a resource query, or
     * a service configuration.
     * </p>
     *
     * @param createGroupRequest
     * @return Result of the CreateGroup operation returned by the service.
     * @throws BadRequestException
     *         The request includes one or more parameters that violate validation rules.
     * @throws ForbiddenException
     *         The caller isn't authorized to make the request. Check permissions.
     * @throws MethodNotAllowedException
     *         The request uses an HTTP method that isn't allowed for the specified resource.
     * @throws TooManyRequestsException
     *         You've exceeded throttling limits by making too many requests in a period of time.
     * @throws InternalServerErrorException
     *         An internal error occurred while processing the request. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ResourceGroupsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ResourceGroupsClient.CreateGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/CreateGroup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateGroupResponse createGroup(CreateGroupRequest createGroupRequest) throws BadRequestException, ForbiddenException,
            MethodNotAllowedException, TooManyRequestsException, InternalServerErrorException, AwsServiceException,
            SdkClientException, ResourceGroupsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Groups");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateGroup");

            return clientHandler.execute(new ClientExecutionParams<CreateGroupRequest, CreateGroupResponse>()
                    .withOperationName("CreateGroup").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createGroupRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified resource group. Deleting a resource group does not delete any resources that are members of
     * the group; it only deletes the group structure.
     * </p>
     *
     * @param deleteGroupRequest
     * @return Result of the DeleteGroup operation returned by the service.
     * @throws BadRequestException
     *         The request includes one or more parameters that violate validation rules.
     * @throws ForbiddenException
     *         The caller isn't authorized to make the request. Check permissions.
     * @throws NotFoundException
     *         One or more of the specified resources don't exist.
     * @throws MethodNotAllowedException
     *         The request uses an HTTP method that isn't allowed for the specified resource.
     * @throws TooManyRequestsException
     *         You've exceeded throttling limits by making too many requests in a period of time.
     * @throws InternalServerErrorException
     *         An internal error occurred while processing the request. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ResourceGroupsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ResourceGroupsClient.DeleteGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/DeleteGroup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteGroupResponse deleteGroup(DeleteGroupRequest deleteGroupRequest) throws BadRequestException, ForbiddenException,
            NotFoundException, MethodNotAllowedException, TooManyRequestsException, InternalServerErrorException,
            AwsServiceException, SdkClientException, ResourceGroupsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Groups");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteGroup");

            return clientHandler.execute(new ClientExecutionParams<DeleteGroupRequest, DeleteGroupResponse>()
                    .withOperationName("DeleteGroup").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteGroupRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about a specified resource group.
     * </p>
     *
     * @param getGroupRequest
     * @return Result of the GetGroup operation returned by the service.
     * @throws BadRequestException
     *         The request includes one or more parameters that violate validation rules.
     * @throws ForbiddenException
     *         The caller isn't authorized to make the request. Check permissions.
     * @throws NotFoundException
     *         One or more of the specified resources don't exist.
     * @throws MethodNotAllowedException
     *         The request uses an HTTP method that isn't allowed for the specified resource.
     * @throws TooManyRequestsException
     *         You've exceeded throttling limits by making too many requests in a period of time.
     * @throws InternalServerErrorException
     *         An internal error occurred while processing the request. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ResourceGroupsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ResourceGroupsClient.GetGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/GetGroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetGroupResponse getGroup(GetGroupRequest getGroupRequest) throws BadRequestException, ForbiddenException,
            NotFoundException, MethodNotAllowedException, TooManyRequestsException, InternalServerErrorException,
            AwsServiceException, SdkClientException, ResourceGroupsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Groups");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetGroup");

            return clientHandler.execute(new ClientExecutionParams<GetGroupRequest, GetGroupResponse>()
                    .withOperationName("GetGroup").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getGroupRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new GetGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the service configuration associated with the specified resource group. AWS Resource Groups supports
     * configurations for the following resource group types:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AWS::EC2::CapacityReservationPool</code> - Amazon EC2 capacity reservation pools. For more information, see
     * <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/capacity-reservations-using.html#create-cr-group">
     * Working with capacity reservation groups</a> in the <i>EC2 Users Guide</i>.
     * </p>
     * </li>
     * </ul>
     *
     * @param getGroupConfigurationRequest
     * @return Result of the GetGroupConfiguration operation returned by the service.
     * @throws BadRequestException
     *         The request includes one or more parameters that violate validation rules.
     * @throws ForbiddenException
     *         The caller isn't authorized to make the request. Check permissions.
     * @throws NotFoundException
     *         One or more of the specified resources don't exist.
     * @throws MethodNotAllowedException
     *         The request uses an HTTP method that isn't allowed for the specified resource.
     * @throws TooManyRequestsException
     *         You've exceeded throttling limits by making too many requests in a period of time.
     * @throws InternalServerErrorException
     *         An internal error occurred while processing the request. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ResourceGroupsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ResourceGroupsClient.GetGroupConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/GetGroupConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetGroupConfigurationResponse getGroupConfiguration(GetGroupConfigurationRequest getGroupConfigurationRequest)
            throws BadRequestException, ForbiddenException, NotFoundException, MethodNotAllowedException,
            TooManyRequestsException, InternalServerErrorException, AwsServiceException, SdkClientException,
            ResourceGroupsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetGroupConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetGroupConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getGroupConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Groups");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetGroupConfiguration");

            return clientHandler.execute(new ClientExecutionParams<GetGroupConfigurationRequest, GetGroupConfigurationResponse>()
                    .withOperationName("GetGroupConfiguration").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getGroupConfigurationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetGroupConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the resource query associated with the specified resource group.
     * </p>
     *
     * @param getGroupQueryRequest
     * @return Result of the GetGroupQuery operation returned by the service.
     * @throws BadRequestException
     *         The request includes one or more parameters that violate validation rules.
     * @throws ForbiddenException
     *         The caller isn't authorized to make the request. Check permissions.
     * @throws NotFoundException
     *         One or more of the specified resources don't exist.
     * @throws MethodNotAllowedException
     *         The request uses an HTTP method that isn't allowed for the specified resource.
     * @throws TooManyRequestsException
     *         You've exceeded throttling limits by making too many requests in a period of time.
     * @throws InternalServerErrorException
     *         An internal error occurred while processing the request. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ResourceGroupsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ResourceGroupsClient.GetGroupQuery
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/GetGroupQuery" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetGroupQueryResponse getGroupQuery(GetGroupQueryRequest getGroupQueryRequest) throws BadRequestException,
            ForbiddenException, NotFoundException, MethodNotAllowedException, TooManyRequestsException,
            InternalServerErrorException, AwsServiceException, SdkClientException, ResourceGroupsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetGroupQueryResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetGroupQueryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getGroupQueryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Groups");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetGroupQuery");

            return clientHandler.execute(new ClientExecutionParams<GetGroupQueryRequest, GetGroupQueryResponse>()
                    .withOperationName("GetGroupQuery").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getGroupQueryRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetGroupQueryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of tags that are associated with a resource group, specified by an ARN.
     * </p>
     *
     * @param getTagsRequest
     * @return Result of the GetTags operation returned by the service.
     * @throws BadRequestException
     *         The request includes one or more parameters that violate validation rules.
     * @throws ForbiddenException
     *         The caller isn't authorized to make the request. Check permissions.
     * @throws NotFoundException
     *         One or more of the specified resources don't exist.
     * @throws MethodNotAllowedException
     *         The request uses an HTTP method that isn't allowed for the specified resource.
     * @throws TooManyRequestsException
     *         You've exceeded throttling limits by making too many requests in a period of time.
     * @throws InternalServerErrorException
     *         An internal error occurred while processing the request. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ResourceGroupsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ResourceGroupsClient.GetTags
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/GetTags" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetTagsResponse getTags(GetTagsRequest getTagsRequest) throws BadRequestException, ForbiddenException,
            NotFoundException, MethodNotAllowedException, TooManyRequestsException, InternalServerErrorException,
            AwsServiceException, SdkClientException, ResourceGroupsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetTagsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Groups");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTags");

            return clientHandler.execute(new ClientExecutionParams<GetTagsRequest, GetTagsResponse>()
                    .withOperationName("GetTags").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getTagsRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new GetTagsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds the specified resources to the specified group.
     * </p>
     *
     * @param groupResourcesRequest
     * @return Result of the GroupResources operation returned by the service.
     * @throws BadRequestException
     *         The request includes one or more parameters that violate validation rules.
     * @throws ForbiddenException
     *         The caller isn't authorized to make the request. Check permissions.
     * @throws NotFoundException
     *         One or more of the specified resources don't exist.
     * @throws MethodNotAllowedException
     *         The request uses an HTTP method that isn't allowed for the specified resource.
     * @throws TooManyRequestsException
     *         You've exceeded throttling limits by making too many requests in a period of time.
     * @throws InternalServerErrorException
     *         An internal error occurred while processing the request. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ResourceGroupsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ResourceGroupsClient.GroupResources
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/GroupResources" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GroupResourcesResponse groupResources(GroupResourcesRequest groupResourcesRequest) throws BadRequestException,
            ForbiddenException, NotFoundException, MethodNotAllowedException, TooManyRequestsException,
            InternalServerErrorException, AwsServiceException, SdkClientException, ResourceGroupsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GroupResourcesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GroupResourcesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, groupResourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Groups");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GroupResources");

            return clientHandler.execute(new ClientExecutionParams<GroupResourcesRequest, GroupResourcesResponse>()
                    .withOperationName("GroupResources").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(groupResourcesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GroupResourcesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of ARNs of the resources that are members of a specified resource group.
     * </p>
     *
     * @param listGroupResourcesRequest
     * @return Result of the ListGroupResources operation returned by the service.
     * @throws UnauthorizedException
     *         The request was rejected because it doesn't have valid credentials for the target resource.
     * @throws BadRequestException
     *         The request includes one or more parameters that violate validation rules.
     * @throws ForbiddenException
     *         The caller isn't authorized to make the request. Check permissions.
     * @throws NotFoundException
     *         One or more of the specified resources don't exist.
     * @throws MethodNotAllowedException
     *         The request uses an HTTP method that isn't allowed for the specified resource.
     * @throws TooManyRequestsException
     *         You've exceeded throttling limits by making too many requests in a period of time.
     * @throws InternalServerErrorException
     *         An internal error occurred while processing the request. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ResourceGroupsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ResourceGroupsClient.ListGroupResources
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/ListGroupResources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListGroupResourcesResponse listGroupResources(ListGroupResourcesRequest listGroupResourcesRequest)
            throws UnauthorizedException, BadRequestException, ForbiddenException, NotFoundException, MethodNotAllowedException,
            TooManyRequestsException, InternalServerErrorException, AwsServiceException, SdkClientException,
            ResourceGroupsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListGroupResourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListGroupResourcesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listGroupResourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Groups");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListGroupResources");

            return clientHandler.execute(new ClientExecutionParams<ListGroupResourcesRequest, ListGroupResourcesResponse>()
                    .withOperationName("ListGroupResources").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listGroupResourcesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListGroupResourcesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of ARNs of the resources that are members of a specified resource group.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listGroupResources(software.amazon.awssdk.services.resourcegroups.model.ListGroupResourcesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.resourcegroups.paginators.ListGroupResourcesIterable responses = client.listGroupResourcesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.resourcegroups.paginators.ListGroupResourcesIterable responses = client
     *             .listGroupResourcesPaginator(request);
     *     for (software.amazon.awssdk.services.resourcegroups.model.ListGroupResourcesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.resourcegroups.paginators.ListGroupResourcesIterable responses = client.listGroupResourcesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listGroupResources(software.amazon.awssdk.services.resourcegroups.model.ListGroupResourcesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listGroupResourcesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws UnauthorizedException
     *         The request was rejected because it doesn't have valid credentials for the target resource.
     * @throws BadRequestException
     *         The request includes one or more parameters that violate validation rules.
     * @throws ForbiddenException
     *         The caller isn't authorized to make the request. Check permissions.
     * @throws NotFoundException
     *         One or more of the specified resources don't exist.
     * @throws MethodNotAllowedException
     *         The request uses an HTTP method that isn't allowed for the specified resource.
     * @throws TooManyRequestsException
     *         You've exceeded throttling limits by making too many requests in a period of time.
     * @throws InternalServerErrorException
     *         An internal error occurred while processing the request. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ResourceGroupsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ResourceGroupsClient.ListGroupResources
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/ListGroupResources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListGroupResourcesIterable listGroupResourcesPaginator(ListGroupResourcesRequest listGroupResourcesRequest)
            throws UnauthorizedException, BadRequestException, ForbiddenException, NotFoundException, MethodNotAllowedException,
            TooManyRequestsException, InternalServerErrorException, AwsServiceException, SdkClientException,
            ResourceGroupsException {
        return new ListGroupResourcesIterable(this, applyPaginatorUserAgent(listGroupResourcesRequest));
    }

    /**
     * <p>
     * Returns a list of existing resource groups in your account.
     * </p>
     *
     * @param listGroupsRequest
     * @return Result of the ListGroups operation returned by the service.
     * @throws BadRequestException
     *         The request includes one or more parameters that violate validation rules.
     * @throws ForbiddenException
     *         The caller isn't authorized to make the request. Check permissions.
     * @throws MethodNotAllowedException
     *         The request uses an HTTP method that isn't allowed for the specified resource.
     * @throws TooManyRequestsException
     *         You've exceeded throttling limits by making too many requests in a period of time.
     * @throws InternalServerErrorException
     *         An internal error occurred while processing the request. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ResourceGroupsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ResourceGroupsClient.ListGroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/ListGroups" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListGroupsResponse listGroups(ListGroupsRequest listGroupsRequest) throws BadRequestException, ForbiddenException,
            MethodNotAllowedException, TooManyRequestsException, InternalServerErrorException, AwsServiceException,
            SdkClientException, ResourceGroupsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListGroupsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListGroupsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listGroupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Groups");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListGroups");

            return clientHandler
                    .execute(new ClientExecutionParams<ListGroupsRequest, ListGroupsResponse>().withOperationName("ListGroups")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listGroupsRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListGroupsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of existing resource groups in your account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listGroups(software.amazon.awssdk.services.resourcegroups.model.ListGroupsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.resourcegroups.paginators.ListGroupsIterable responses = client.listGroupsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.resourcegroups.paginators.ListGroupsIterable responses = client.listGroupsPaginator(request);
     *     for (software.amazon.awssdk.services.resourcegroups.model.ListGroupsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.resourcegroups.paginators.ListGroupsIterable responses = client.listGroupsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listGroups(software.amazon.awssdk.services.resourcegroups.model.ListGroupsRequest)} operation.</b>
     * </p>
     *
     * @param listGroupsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws BadRequestException
     *         The request includes one or more parameters that violate validation rules.
     * @throws ForbiddenException
     *         The caller isn't authorized to make the request. Check permissions.
     * @throws MethodNotAllowedException
     *         The request uses an HTTP method that isn't allowed for the specified resource.
     * @throws TooManyRequestsException
     *         You've exceeded throttling limits by making too many requests in a period of time.
     * @throws InternalServerErrorException
     *         An internal error occurred while processing the request. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ResourceGroupsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ResourceGroupsClient.ListGroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/ListGroups" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListGroupsIterable listGroupsPaginator(ListGroupsRequest listGroupsRequest) throws BadRequestException,
            ForbiddenException, MethodNotAllowedException, TooManyRequestsException, InternalServerErrorException,
            AwsServiceException, SdkClientException, ResourceGroupsException {
        return new ListGroupsIterable(this, applyPaginatorUserAgent(listGroupsRequest));
    }

    /**
     * <p>
     * Returns a list of AWS resource identifiers that matches tne specified query. The query uses the same format as a
     * resource query in a CreateGroup or UpdateGroupQuery operation.
     * </p>
     *
     * @param searchResourcesRequest
     * @return Result of the SearchResources operation returned by the service.
     * @throws UnauthorizedException
     *         The request was rejected because it doesn't have valid credentials for the target resource.
     * @throws BadRequestException
     *         The request includes one or more parameters that violate validation rules.
     * @throws ForbiddenException
     *         The caller isn't authorized to make the request. Check permissions.
     * @throws MethodNotAllowedException
     *         The request uses an HTTP method that isn't allowed for the specified resource.
     * @throws TooManyRequestsException
     *         You've exceeded throttling limits by making too many requests in a period of time.
     * @throws InternalServerErrorException
     *         An internal error occurred while processing the request. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ResourceGroupsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ResourceGroupsClient.SearchResources
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/SearchResources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public SearchResourcesResponse searchResources(SearchResourcesRequest searchResourcesRequest) throws UnauthorizedException,
            BadRequestException, ForbiddenException, MethodNotAllowedException, TooManyRequestsException,
            InternalServerErrorException, AwsServiceException, SdkClientException, ResourceGroupsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SearchResourcesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                SearchResourcesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, searchResourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Groups");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SearchResources");

            return clientHandler.execute(new ClientExecutionParams<SearchResourcesRequest, SearchResourcesResponse>()
                    .withOperationName("SearchResources").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(searchResourcesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new SearchResourcesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of AWS resource identifiers that matches tne specified query. The query uses the same format as a
     * resource query in a CreateGroup or UpdateGroupQuery operation.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #searchResources(software.amazon.awssdk.services.resourcegroups.model.SearchResourcesRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.resourcegroups.paginators.SearchResourcesIterable responses = client.searchResourcesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.resourcegroups.paginators.SearchResourcesIterable responses = client
     *             .searchResourcesPaginator(request);
     *     for (software.amazon.awssdk.services.resourcegroups.model.SearchResourcesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.resourcegroups.paginators.SearchResourcesIterable responses = client.searchResourcesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #searchResources(software.amazon.awssdk.services.resourcegroups.model.SearchResourcesRequest)}
     * operation.</b>
     * </p>
     *
     * @param searchResourcesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws UnauthorizedException
     *         The request was rejected because it doesn't have valid credentials for the target resource.
     * @throws BadRequestException
     *         The request includes one or more parameters that violate validation rules.
     * @throws ForbiddenException
     *         The caller isn't authorized to make the request. Check permissions.
     * @throws MethodNotAllowedException
     *         The request uses an HTTP method that isn't allowed for the specified resource.
     * @throws TooManyRequestsException
     *         You've exceeded throttling limits by making too many requests in a period of time.
     * @throws InternalServerErrorException
     *         An internal error occurred while processing the request. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ResourceGroupsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ResourceGroupsClient.SearchResources
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/SearchResources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public SearchResourcesIterable searchResourcesPaginator(SearchResourcesRequest searchResourcesRequest)
            throws UnauthorizedException, BadRequestException, ForbiddenException, MethodNotAllowedException,
            TooManyRequestsException, InternalServerErrorException, AwsServiceException, SdkClientException,
            ResourceGroupsException {
        return new SearchResourcesIterable(this, applyPaginatorUserAgent(searchResourcesRequest));
    }

    /**
     * <p>
     * Adds tags to a resource group with the specified ARN. Existing tags on a resource group are not changed if they
     * are not specified in the request parameters.
     * </p>
     * <important>
     * <p>
     * Do not store personally identifiable information (PII) or other confidential or sensitive information in tags. We
     * use tags to provide you with billing and administration services. Tags are not intended to be used for private or
     * sensitive data.
     * </p>
     * </important>
     *
     * @param tagRequest
     * @return Result of the Tag operation returned by the service.
     * @throws BadRequestException
     *         The request includes one or more parameters that violate validation rules.
     * @throws ForbiddenException
     *         The caller isn't authorized to make the request. Check permissions.
     * @throws NotFoundException
     *         One or more of the specified resources don't exist.
     * @throws MethodNotAllowedException
     *         The request uses an HTTP method that isn't allowed for the specified resource.
     * @throws TooManyRequestsException
     *         You've exceeded throttling limits by making too many requests in a period of time.
     * @throws InternalServerErrorException
     *         An internal error occurred while processing the request. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ResourceGroupsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ResourceGroupsClient.Tag
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/Tag" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResponse tag(TagRequest tagRequest) throws BadRequestException, ForbiddenException, NotFoundException,
            MethodNotAllowedException, TooManyRequestsException, InternalServerErrorException, AwsServiceException,
            SdkClientException, ResourceGroupsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagRequest.overrideConfiguration()
                .orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Groups");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "Tag");

            return clientHandler.execute(new ClientExecutionParams<TagRequest, TagResponse>().withOperationName("Tag")
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(tagRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new TagRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes the specified resources from the specified group.
     * </p>
     *
     * @param ungroupResourcesRequest
     * @return Result of the UngroupResources operation returned by the service.
     * @throws BadRequestException
     *         The request includes one or more parameters that violate validation rules.
     * @throws ForbiddenException
     *         The caller isn't authorized to make the request. Check permissions.
     * @throws NotFoundException
     *         One or more of the specified resources don't exist.
     * @throws MethodNotAllowedException
     *         The request uses an HTTP method that isn't allowed for the specified resource.
     * @throws TooManyRequestsException
     *         You've exceeded throttling limits by making too many requests in a period of time.
     * @throws InternalServerErrorException
     *         An internal error occurred while processing the request. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ResourceGroupsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ResourceGroupsClient.UngroupResources
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/UngroupResources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UngroupResourcesResponse ungroupResources(UngroupResourcesRequest ungroupResourcesRequest) throws BadRequestException,
            ForbiddenException, NotFoundException, MethodNotAllowedException, TooManyRequestsException,
            InternalServerErrorException, AwsServiceException, SdkClientException, ResourceGroupsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UngroupResourcesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UngroupResourcesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, ungroupResourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Groups");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UngroupResources");

            return clientHandler.execute(new ClientExecutionParams<UngroupResourcesRequest, UngroupResourcesResponse>()
                    .withOperationName("UngroupResources").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(ungroupResourcesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UngroupResourcesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes tags from a specified resource group.
     * </p>
     *
     * @param untagRequest
     * @return Result of the Untag operation returned by the service.
     * @throws BadRequestException
     *         The request includes one or more parameters that violate validation rules.
     * @throws ForbiddenException
     *         The caller isn't authorized to make the request. Check permissions.
     * @throws NotFoundException
     *         One or more of the specified resources don't exist.
     * @throws MethodNotAllowedException
     *         The request uses an HTTP method that isn't allowed for the specified resource.
     * @throws TooManyRequestsException
     *         You've exceeded throttling limits by making too many requests in a period of time.
     * @throws InternalServerErrorException
     *         An internal error occurred while processing the request. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ResourceGroupsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ResourceGroupsClient.Untag
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/Untag" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResponse untag(UntagRequest untagRequest) throws BadRequestException, ForbiddenException, NotFoundException,
            MethodNotAllowedException, TooManyRequestsException, InternalServerErrorException, AwsServiceException,
            SdkClientException, ResourceGroupsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Groups");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "Untag");

            return clientHandler.execute(new ClientExecutionParams<UntagRequest, UntagResponse>().withOperationName("Untag")
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(untagRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new UntagRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the description for an existing group. You cannot update the name of a resource group.
     * </p>
     *
     * @param updateGroupRequest
     * @return Result of the UpdateGroup operation returned by the service.
     * @throws BadRequestException
     *         The request includes one or more parameters that violate validation rules.
     * @throws ForbiddenException
     *         The caller isn't authorized to make the request. Check permissions.
     * @throws NotFoundException
     *         One or more of the specified resources don't exist.
     * @throws MethodNotAllowedException
     *         The request uses an HTTP method that isn't allowed for the specified resource.
     * @throws TooManyRequestsException
     *         You've exceeded throttling limits by making too many requests in a period of time.
     * @throws InternalServerErrorException
     *         An internal error occurred while processing the request. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ResourceGroupsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ResourceGroupsClient.UpdateGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/UpdateGroup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateGroupResponse updateGroup(UpdateGroupRequest updateGroupRequest) throws BadRequestException, ForbiddenException,
            NotFoundException, MethodNotAllowedException, TooManyRequestsException, InternalServerErrorException,
            AwsServiceException, SdkClientException, ResourceGroupsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Groups");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateGroup");

            return clientHandler.execute(new ClientExecutionParams<UpdateGroupRequest, UpdateGroupResponse>()
                    .withOperationName("UpdateGroup").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateGroupRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the resource query of a group.
     * </p>
     *
     * @param updateGroupQueryRequest
     * @return Result of the UpdateGroupQuery operation returned by the service.
     * @throws BadRequestException
     *         The request includes one or more parameters that violate validation rules.
     * @throws ForbiddenException
     *         The caller isn't authorized to make the request. Check permissions.
     * @throws NotFoundException
     *         One or more of the specified resources don't exist.
     * @throws MethodNotAllowedException
     *         The request uses an HTTP method that isn't allowed for the specified resource.
     * @throws TooManyRequestsException
     *         You've exceeded throttling limits by making too many requests in a period of time.
     * @throws InternalServerErrorException
     *         An internal error occurred while processing the request. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ResourceGroupsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ResourceGroupsClient.UpdateGroupQuery
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/UpdateGroupQuery"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateGroupQueryResponse updateGroupQuery(UpdateGroupQueryRequest updateGroupQueryRequest) throws BadRequestException,
            ForbiddenException, NotFoundException, MethodNotAllowedException, TooManyRequestsException,
            InternalServerErrorException, AwsServiceException, SdkClientException, ResourceGroupsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateGroupQueryResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateGroupQueryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateGroupQueryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Groups");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateGroupQuery");

            return clientHandler.execute(new ClientExecutionParams<UpdateGroupQueryRequest, UpdateGroupQueryResponse>()
                    .withOperationName("UpdateGroupQuery").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateGroupQueryRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateGroupQueryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ResourceGroupsException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnauthorizedException")
                                .exceptionBuilderSupplier(UnauthorizedException::builder).httpStatusCode(401).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ForbiddenException")
                                .exceptionBuilderSupplier(ForbiddenException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MethodNotAllowedException")
                                .exceptionBuilderSupplier(MethodNotAllowedException::builder).httpStatusCode(405).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerErrorException")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).httpStatusCode(500).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends ResourceGroupsRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
