/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.resourcegroups.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The query that is used to define a resource group or a search for resources. A query specifies both a query type and
 * a query string as a JSON object. See the examples section for example JSON strings.
 * </p>
 * <p>
 * The examples that follow are shown as standard JSON strings. If you include such a string as a parameter to the AWS
 * CLI or an SDK API, you might need to 'escape' the string into a single line. For example, see the <a
 * href="https://docs.aws.amazon.com/cli/latest/userguide/cli-usage-parameters-quoting-strings.html">Quoting strings</a>
 * in the <i>AWS CLI User Guide</i>.
 * </p>
 * <p>
 * <b>Example 1</b>
 * </p>
 * <p>
 * The following generic example shows a resource query JSON string that includes only resources that meet the following
 * criteria:
 * </p>
 * <ul>
 * <li>
 * <p>
 * The resource type must be either <code>resource_type1</code> or <code>resource_type2</code>.
 * </p>
 * </li>
 * <li>
 * <p>
 * The resource must have a tag <code>Key1</code> with a value of either <code>ValueA</code> or <code>ValueB</code>.
 * </p>
 * </li>
 * <li>
 * <p>
 * The resource must have a tag <code>Key2</code> with a value of either <code>ValueC</code> or <code>ValueD</code>.
 * </p>
 * </li>
 * </ul>
 * <p>
 * <code>{ "Type": "TAG_FILTERS_1_0", "Query": { "ResourceTypeFilters": [ "resource_type1", "resource_type2"], "TagFilters": [ { "Key": "Key1", "Values": ["ValueA","ValueB"] }, { "Key":"Key2", "Values":["ValueC","ValueD"] } ] } }</code>
 * </p>
 * <p>
 * This has the equivalent "shortcut" syntax of the following:
 * </p>
 * <p>
 * <code>{ "Type": "TAG_FILTERS_1_0", "Query": { "ResourceTypeFilters": [ "resource_type1", "resource_type2"], "TagFilters": [ { "Key1": ["ValueA","ValueB"] }, { "Key2": ["ValueC","ValueD"] } ] } }</code>
 * </p>
 * <p>
 * <b>Example 2</b>
 * </p>
 * <p>
 * The following example shows a resource query JSON string that includes only Amazon EC2 instances that are tagged
 * <code>Stage</code> with a value of <code>Test</code>.
 * </p>
 * <p>
 * <code>{ "Type": "TAG_FILTERS_1_0", "Query": "{ "ResourceTypeFilters": "AWS::EC2::Instance", "TagFilters": { "Stage": "Test" } } }</code>
 * </p>
 * <p>
 * <b>Example 3</b>
 * </p>
 * <p>
 * The following example shows a resource query JSON string that includes resource of any supported type as long as it
 * is tagged <code>Stage</code> with a value of <code>Prod</code>.
 * </p>
 * <p>
 * <code>{ "Type": "TAG_FILTERS_1_0", "Query": { "ResourceTypeFilters": "AWS::AllSupported", "TagFilters": { "Stage": "Prod" } } }</code>
 * </p>
 * <p>
 * <b>Example 4</b>
 * </p>
 * <p>
 * The following example shows a resource query JSON string that includes only Amazon EC2 instances and Amazon S3
 * buckets that are part of the specified AWS CloudFormation stack.
 * </p>
 * <p>
 * <code>{ "Type": "CLOUDFORMATION_STACK_1_0", "Query": { "ResourceTypeFilters": [ "AWS::EC2::Instance", "AWS::S3::Bucket" ], "StackIdentifier": "arn:aws:cloudformation:us-west-2:123456789012:stack/AWStestuseraccount/fb0d5000-aba8-00e8-aa9e-50d5cEXAMPLE" } }</code>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ResourceQuery implements SdkPojo, Serializable, ToCopyableBuilder<ResourceQuery.Builder, ResourceQuery> {
    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ResourceQuery::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type").build()).build();

    private static final SdkField<String> QUERY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ResourceQuery::query)).setter(setter(Builder::query))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Query").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TYPE_FIELD, QUERY_FIELD));

    private static final long serialVersionUID = 1L;

    private final String type;

    private final String query;

    private ResourceQuery(BuilderImpl builder) {
        this.type = builder.type;
        this.query = builder.query;
    }

    /**
     * <p>
     * The type of the query. You can use the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <i> <code>CLOUDFORMATION_STACK_1_0:</code> </i>Specifies that the <code>Query</code> contains an ARN for a
     * CloudFormation stack.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i> <code>TAG_FILTERS_1_0:</code> </i>Specifies that the <code>Query</code> parameter contains a JSON string that
     * represents a collection of simple tag filters for resource types and tags. The JSON string uses a syntax similar
     * to the
     * <code> <a href="https://docs.aws.amazon.com/resourcegroupstagging/latest/APIReference/API_GetResources.html">GetResources</a> </code>
     * operation, but uses only the
     * <code> <a href="https://docs.aws.amazon.com/resourcegroupstagging/latest/APIReference/API_GetResources.html#resourcegrouptagging-GetResources-request-ResourceTypeFilters"> ResourceTypeFilters</a> </code>
     * and
     * <code> <a href="https://docs.aws.amazon.com/resourcegroupstagging/latest/APIReference/API_GetResources.html#resourcegrouptagging-GetResources-request-TagFiltersTagFilters">TagFilters</a> </code>
     * fields. If you specify more than one tag key, only resources that match all tag keys, and at least one value of
     * each specified tag key, are returned in your query. If you specify more than one value for a tag key, a resource
     * matches the filter if it has a tag key value that matches <i>any</i> of the specified values.
     * </p>
     * <p>
     * For example, consider the following sample query for resources that have two tags, <code>Stage</code> and
     * <code>Version</code>, with two values each:
     * </p>
     * <p>
     * <code>[{"Stage":["Test","Deploy"]},{"Version":["1","2"]}]</code>
     * </p>
     * <p>
     * The results of this query could include the following.
     * </p>
     * <ul>
     * <li>
     * <p>
     * An EC2 instance that has the following two tags: <code>{"Stage":"Deploy"}</code>, and
     * <code>{"Version":"2"}</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * An S3 bucket that has the following two tags: <code>{"Stage":"Test"}</code>, and <code>{"Version":"1"}</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * The query would not include the following items in the results, however.
     * </p>
     * <ul>
     * <li>
     * <p>
     * An EC2 instance that has only the following tag: <code>{"Stage":"Deploy"}</code>.
     * </p>
     * <p>
     * The instance does not have <b>all</b> of the tag keys specified in the filter, so it is excluded from the
     * results.
     * </p>
     * </li>
     * <li>
     * <p>
     * An RDS database that has the following two tags: <code>{"Stage":"Archived"}</code> and
     * <code>{"Version":"4"}</code>
     * </p>
     * <p>
     * The database has all of the tag keys, but none of those keys has an associated value that matches at least one of
     * the specified values in the filter.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link QueryType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of the query. You can use the following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <i> <code>CLOUDFORMATION_STACK_1_0:</code> </i>Specifies that the <code>Query</code> contains an ARN for
     *         a CloudFormation stack.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i> <code>TAG_FILTERS_1_0:</code> </i>Specifies that the <code>Query</code> parameter contains a JSON
     *         string that represents a collection of simple tag filters for resource types and tags. The JSON string
     *         uses a syntax similar to the
     *         <code> <a href="https://docs.aws.amazon.com/resourcegroupstagging/latest/APIReference/API_GetResources.html">GetResources</a> </code>
     *         operation, but uses only the
     *         <code> <a href="https://docs.aws.amazon.com/resourcegroupstagging/latest/APIReference/API_GetResources.html#resourcegrouptagging-GetResources-request-ResourceTypeFilters"> ResourceTypeFilters</a> </code>
     *         and
     *         <code> <a href="https://docs.aws.amazon.com/resourcegroupstagging/latest/APIReference/API_GetResources.html#resourcegrouptagging-GetResources-request-TagFiltersTagFilters">TagFilters</a> </code>
     *         fields. If you specify more than one tag key, only resources that match all tag keys, and at least one
     *         value of each specified tag key, are returned in your query. If you specify more than one value for a tag
     *         key, a resource matches the filter if it has a tag key value that matches <i>any</i> of the specified
     *         values.
     *         </p>
     *         <p>
     *         For example, consider the following sample query for resources that have two tags, <code>Stage</code> and
     *         <code>Version</code>, with two values each:
     *         </p>
     *         <p>
     *         <code>[{"Stage":["Test","Deploy"]},{"Version":["1","2"]}]</code>
     *         </p>
     *         <p>
     *         The results of this query could include the following.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         An EC2 instance that has the following two tags: <code>{"Stage":"Deploy"}</code>, and
     *         <code>{"Version":"2"}</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         An S3 bucket that has the following two tags: <code>{"Stage":"Test"}</code>, and
     *         <code>{"Version":"1"}</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         The query would not include the following items in the results, however.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         An EC2 instance that has only the following tag: <code>{"Stage":"Deploy"}</code>.
     *         </p>
     *         <p>
     *         The instance does not have <b>all</b> of the tag keys specified in the filter, so it is excluded from the
     *         results.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         An RDS database that has the following two tags: <code>{"Stage":"Archived"}</code> and
     *         <code>{"Version":"4"}</code>
     *         </p>
     *         <p>
     *         The database has all of the tag keys, but none of those keys has an associated value that matches at
     *         least one of the specified values in the filter.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     * @see QueryType
     */
    public QueryType type() {
        return QueryType.fromValue(type);
    }

    /**
     * <p>
     * The type of the query. You can use the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <i> <code>CLOUDFORMATION_STACK_1_0:</code> </i>Specifies that the <code>Query</code> contains an ARN for a
     * CloudFormation stack.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i> <code>TAG_FILTERS_1_0:</code> </i>Specifies that the <code>Query</code> parameter contains a JSON string that
     * represents a collection of simple tag filters for resource types and tags. The JSON string uses a syntax similar
     * to the
     * <code> <a href="https://docs.aws.amazon.com/resourcegroupstagging/latest/APIReference/API_GetResources.html">GetResources</a> </code>
     * operation, but uses only the
     * <code> <a href="https://docs.aws.amazon.com/resourcegroupstagging/latest/APIReference/API_GetResources.html#resourcegrouptagging-GetResources-request-ResourceTypeFilters"> ResourceTypeFilters</a> </code>
     * and
     * <code> <a href="https://docs.aws.amazon.com/resourcegroupstagging/latest/APIReference/API_GetResources.html#resourcegrouptagging-GetResources-request-TagFiltersTagFilters">TagFilters</a> </code>
     * fields. If you specify more than one tag key, only resources that match all tag keys, and at least one value of
     * each specified tag key, are returned in your query. If you specify more than one value for a tag key, a resource
     * matches the filter if it has a tag key value that matches <i>any</i> of the specified values.
     * </p>
     * <p>
     * For example, consider the following sample query for resources that have two tags, <code>Stage</code> and
     * <code>Version</code>, with two values each:
     * </p>
     * <p>
     * <code>[{"Stage":["Test","Deploy"]},{"Version":["1","2"]}]</code>
     * </p>
     * <p>
     * The results of this query could include the following.
     * </p>
     * <ul>
     * <li>
     * <p>
     * An EC2 instance that has the following two tags: <code>{"Stage":"Deploy"}</code>, and
     * <code>{"Version":"2"}</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * An S3 bucket that has the following two tags: <code>{"Stage":"Test"}</code>, and <code>{"Version":"1"}</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * The query would not include the following items in the results, however.
     * </p>
     * <ul>
     * <li>
     * <p>
     * An EC2 instance that has only the following tag: <code>{"Stage":"Deploy"}</code>.
     * </p>
     * <p>
     * The instance does not have <b>all</b> of the tag keys specified in the filter, so it is excluded from the
     * results.
     * </p>
     * </li>
     * <li>
     * <p>
     * An RDS database that has the following two tags: <code>{"Stage":"Archived"}</code> and
     * <code>{"Version":"4"}</code>
     * </p>
     * <p>
     * The database has all of the tag keys, but none of those keys has an associated value that matches at least one of
     * the specified values in the filter.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link QueryType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of the query. You can use the following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <i> <code>CLOUDFORMATION_STACK_1_0:</code> </i>Specifies that the <code>Query</code> contains an ARN for
     *         a CloudFormation stack.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i> <code>TAG_FILTERS_1_0:</code> </i>Specifies that the <code>Query</code> parameter contains a JSON
     *         string that represents a collection of simple tag filters for resource types and tags. The JSON string
     *         uses a syntax similar to the
     *         <code> <a href="https://docs.aws.amazon.com/resourcegroupstagging/latest/APIReference/API_GetResources.html">GetResources</a> </code>
     *         operation, but uses only the
     *         <code> <a href="https://docs.aws.amazon.com/resourcegroupstagging/latest/APIReference/API_GetResources.html#resourcegrouptagging-GetResources-request-ResourceTypeFilters"> ResourceTypeFilters</a> </code>
     *         and
     *         <code> <a href="https://docs.aws.amazon.com/resourcegroupstagging/latest/APIReference/API_GetResources.html#resourcegrouptagging-GetResources-request-TagFiltersTagFilters">TagFilters</a> </code>
     *         fields. If you specify more than one tag key, only resources that match all tag keys, and at least one
     *         value of each specified tag key, are returned in your query. If you specify more than one value for a tag
     *         key, a resource matches the filter if it has a tag key value that matches <i>any</i> of the specified
     *         values.
     *         </p>
     *         <p>
     *         For example, consider the following sample query for resources that have two tags, <code>Stage</code> and
     *         <code>Version</code>, with two values each:
     *         </p>
     *         <p>
     *         <code>[{"Stage":["Test","Deploy"]},{"Version":["1","2"]}]</code>
     *         </p>
     *         <p>
     *         The results of this query could include the following.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         An EC2 instance that has the following two tags: <code>{"Stage":"Deploy"}</code>, and
     *         <code>{"Version":"2"}</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         An S3 bucket that has the following two tags: <code>{"Stage":"Test"}</code>, and
     *         <code>{"Version":"1"}</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         The query would not include the following items in the results, however.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         An EC2 instance that has only the following tag: <code>{"Stage":"Deploy"}</code>.
     *         </p>
     *         <p>
     *         The instance does not have <b>all</b> of the tag keys specified in the filter, so it is excluded from the
     *         results.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         An RDS database that has the following two tags: <code>{"Stage":"Archived"}</code> and
     *         <code>{"Version":"4"}</code>
     *         </p>
     *         <p>
     *         The database has all of the tag keys, but none of those keys has an associated value that matches at
     *         least one of the specified values in the filter.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     * @see QueryType
     */
    public String typeAsString() {
        return type;
    }

    /**
     * <p>
     * The query that defines a group or a search.
     * </p>
     * 
     * @return The query that defines a group or a search.
     */
    public String query() {
        return query;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(query());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ResourceQuery)) {
            return false;
        }
        ResourceQuery other = (ResourceQuery) obj;
        return Objects.equals(typeAsString(), other.typeAsString()) && Objects.equals(query(), other.query());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("ResourceQuery").add("Type", typeAsString()).add("Query", query()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "Query":
            return Optional.ofNullable(clazz.cast(query()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ResourceQuery, T> g) {
        return obj -> g.apply((ResourceQuery) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ResourceQuery> {
        /**
         * <p>
         * The type of the query. You can use the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <i> <code>CLOUDFORMATION_STACK_1_0:</code> </i>Specifies that the <code>Query</code> contains an ARN for a
         * CloudFormation stack.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i> <code>TAG_FILTERS_1_0:</code> </i>Specifies that the <code>Query</code> parameter contains a JSON string
         * that represents a collection of simple tag filters for resource types and tags. The JSON string uses a syntax
         * similar to the
         * <code> <a href="https://docs.aws.amazon.com/resourcegroupstagging/latest/APIReference/API_GetResources.html">GetResources</a> </code>
         * operation, but uses only the
         * <code> <a href="https://docs.aws.amazon.com/resourcegroupstagging/latest/APIReference/API_GetResources.html#resourcegrouptagging-GetResources-request-ResourceTypeFilters"> ResourceTypeFilters</a> </code>
         * and
         * <code> <a href="https://docs.aws.amazon.com/resourcegroupstagging/latest/APIReference/API_GetResources.html#resourcegrouptagging-GetResources-request-TagFiltersTagFilters">TagFilters</a> </code>
         * fields. If you specify more than one tag key, only resources that match all tag keys, and at least one value
         * of each specified tag key, are returned in your query. If you specify more than one value for a tag key, a
         * resource matches the filter if it has a tag key value that matches <i>any</i> of the specified values.
         * </p>
         * <p>
         * For example, consider the following sample query for resources that have two tags, <code>Stage</code> and
         * <code>Version</code>, with two values each:
         * </p>
         * <p>
         * <code>[{"Stage":["Test","Deploy"]},{"Version":["1","2"]}]</code>
         * </p>
         * <p>
         * The results of this query could include the following.
         * </p>
         * <ul>
         * <li>
         * <p>
         * An EC2 instance that has the following two tags: <code>{"Stage":"Deploy"}</code>, and
         * <code>{"Version":"2"}</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * An S3 bucket that has the following two tags: <code>{"Stage":"Test"}</code>, and <code>{"Version":"1"}</code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * The query would not include the following items in the results, however.
         * </p>
         * <ul>
         * <li>
         * <p>
         * An EC2 instance that has only the following tag: <code>{"Stage":"Deploy"}</code>.
         * </p>
         * <p>
         * The instance does not have <b>all</b> of the tag keys specified in the filter, so it is excluded from the
         * results.
         * </p>
         * </li>
         * <li>
         * <p>
         * An RDS database that has the following two tags: <code>{"Stage":"Archived"}</code> and
         * <code>{"Version":"4"}</code>
         * </p>
         * <p>
         * The database has all of the tag keys, but none of those keys has an associated value that matches at least
         * one of the specified values in the filter.
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * 
         * @param type
         *        The type of the query. You can use the following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <i> <code>CLOUDFORMATION_STACK_1_0:</code> </i>Specifies that the <code>Query</code> contains an ARN
         *        for a CloudFormation stack.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i> <code>TAG_FILTERS_1_0:</code> </i>Specifies that the <code>Query</code> parameter contains a JSON
         *        string that represents a collection of simple tag filters for resource types and tags. The JSON string
         *        uses a syntax similar to the
         *        <code> <a href="https://docs.aws.amazon.com/resourcegroupstagging/latest/APIReference/API_GetResources.html">GetResources</a> </code>
         *        operation, but uses only the
         *        <code> <a href="https://docs.aws.amazon.com/resourcegroupstagging/latest/APIReference/API_GetResources.html#resourcegrouptagging-GetResources-request-ResourceTypeFilters"> ResourceTypeFilters</a> </code>
         *        and
         *        <code> <a href="https://docs.aws.amazon.com/resourcegroupstagging/latest/APIReference/API_GetResources.html#resourcegrouptagging-GetResources-request-TagFiltersTagFilters">TagFilters</a> </code>
         *        fields. If you specify more than one tag key, only resources that match all tag keys, and at least one
         *        value of each specified tag key, are returned in your query. If you specify more than one value for a
         *        tag key, a resource matches the filter if it has a tag key value that matches <i>any</i> of the
         *        specified values.
         *        </p>
         *        <p>
         *        For example, consider the following sample query for resources that have two tags, <code>Stage</code>
         *        and <code>Version</code>, with two values each:
         *        </p>
         *        <p>
         *        <code>[{"Stage":["Test","Deploy"]},{"Version":["1","2"]}]</code>
         *        </p>
         *        <p>
         *        The results of this query could include the following.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        An EC2 instance that has the following two tags: <code>{"Stage":"Deploy"}</code>, and
         *        <code>{"Version":"2"}</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        An S3 bucket that has the following two tags: <code>{"Stage":"Test"}</code>, and
         *        <code>{"Version":"1"}</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        The query would not include the following items in the results, however.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        An EC2 instance that has only the following tag: <code>{"Stage":"Deploy"}</code>.
         *        </p>
         *        <p>
         *        The instance does not have <b>all</b> of the tag keys specified in the filter, so it is excluded from
         *        the results.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        An RDS database that has the following two tags: <code>{"Stage":"Archived"}</code> and
         *        <code>{"Version":"4"}</code>
         *        </p>
         *        <p>
         *        The database has all of the tag keys, but none of those keys has an associated value that matches at
         *        least one of the specified values in the filter.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         * @see QueryType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see QueryType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of the query. You can use the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <i> <code>CLOUDFORMATION_STACK_1_0:</code> </i>Specifies that the <code>Query</code> contains an ARN for a
         * CloudFormation stack.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i> <code>TAG_FILTERS_1_0:</code> </i>Specifies that the <code>Query</code> parameter contains a JSON string
         * that represents a collection of simple tag filters for resource types and tags. The JSON string uses a syntax
         * similar to the
         * <code> <a href="https://docs.aws.amazon.com/resourcegroupstagging/latest/APIReference/API_GetResources.html">GetResources</a> </code>
         * operation, but uses only the
         * <code> <a href="https://docs.aws.amazon.com/resourcegroupstagging/latest/APIReference/API_GetResources.html#resourcegrouptagging-GetResources-request-ResourceTypeFilters"> ResourceTypeFilters</a> </code>
         * and
         * <code> <a href="https://docs.aws.amazon.com/resourcegroupstagging/latest/APIReference/API_GetResources.html#resourcegrouptagging-GetResources-request-TagFiltersTagFilters">TagFilters</a> </code>
         * fields. If you specify more than one tag key, only resources that match all tag keys, and at least one value
         * of each specified tag key, are returned in your query. If you specify more than one value for a tag key, a
         * resource matches the filter if it has a tag key value that matches <i>any</i> of the specified values.
         * </p>
         * <p>
         * For example, consider the following sample query for resources that have two tags, <code>Stage</code> and
         * <code>Version</code>, with two values each:
         * </p>
         * <p>
         * <code>[{"Stage":["Test","Deploy"]},{"Version":["1","2"]}]</code>
         * </p>
         * <p>
         * The results of this query could include the following.
         * </p>
         * <ul>
         * <li>
         * <p>
         * An EC2 instance that has the following two tags: <code>{"Stage":"Deploy"}</code>, and
         * <code>{"Version":"2"}</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * An S3 bucket that has the following two tags: <code>{"Stage":"Test"}</code>, and <code>{"Version":"1"}</code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * The query would not include the following items in the results, however.
         * </p>
         * <ul>
         * <li>
         * <p>
         * An EC2 instance that has only the following tag: <code>{"Stage":"Deploy"}</code>.
         * </p>
         * <p>
         * The instance does not have <b>all</b> of the tag keys specified in the filter, so it is excluded from the
         * results.
         * </p>
         * </li>
         * <li>
         * <p>
         * An RDS database that has the following two tags: <code>{"Stage":"Archived"}</code> and
         * <code>{"Version":"4"}</code>
         * </p>
         * <p>
         * The database has all of the tag keys, but none of those keys has an associated value that matches at least
         * one of the specified values in the filter.
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * 
         * @param type
         *        The type of the query. You can use the following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <i> <code>CLOUDFORMATION_STACK_1_0:</code> </i>Specifies that the <code>Query</code> contains an ARN
         *        for a CloudFormation stack.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i> <code>TAG_FILTERS_1_0:</code> </i>Specifies that the <code>Query</code> parameter contains a JSON
         *        string that represents a collection of simple tag filters for resource types and tags. The JSON string
         *        uses a syntax similar to the
         *        <code> <a href="https://docs.aws.amazon.com/resourcegroupstagging/latest/APIReference/API_GetResources.html">GetResources</a> </code>
         *        operation, but uses only the
         *        <code> <a href="https://docs.aws.amazon.com/resourcegroupstagging/latest/APIReference/API_GetResources.html#resourcegrouptagging-GetResources-request-ResourceTypeFilters"> ResourceTypeFilters</a> </code>
         *        and
         *        <code> <a href="https://docs.aws.amazon.com/resourcegroupstagging/latest/APIReference/API_GetResources.html#resourcegrouptagging-GetResources-request-TagFiltersTagFilters">TagFilters</a> </code>
         *        fields. If you specify more than one tag key, only resources that match all tag keys, and at least one
         *        value of each specified tag key, are returned in your query. If you specify more than one value for a
         *        tag key, a resource matches the filter if it has a tag key value that matches <i>any</i> of the
         *        specified values.
         *        </p>
         *        <p>
         *        For example, consider the following sample query for resources that have two tags, <code>Stage</code>
         *        and <code>Version</code>, with two values each:
         *        </p>
         *        <p>
         *        <code>[{"Stage":["Test","Deploy"]},{"Version":["1","2"]}]</code>
         *        </p>
         *        <p>
         *        The results of this query could include the following.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        An EC2 instance that has the following two tags: <code>{"Stage":"Deploy"}</code>, and
         *        <code>{"Version":"2"}</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        An S3 bucket that has the following two tags: <code>{"Stage":"Test"}</code>, and
         *        <code>{"Version":"1"}</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        The query would not include the following items in the results, however.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        An EC2 instance that has only the following tag: <code>{"Stage":"Deploy"}</code>.
         *        </p>
         *        <p>
         *        The instance does not have <b>all</b> of the tag keys specified in the filter, so it is excluded from
         *        the results.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        An RDS database that has the following two tags: <code>{"Stage":"Archived"}</code> and
         *        <code>{"Version":"4"}</code>
         *        </p>
         *        <p>
         *        The database has all of the tag keys, but none of those keys has an associated value that matches at
         *        least one of the specified values in the filter.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         * @see QueryType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see QueryType
         */
        Builder type(QueryType type);

        /**
         * <p>
         * The query that defines a group or a search.
         * </p>
         * 
         * @param query
         *        The query that defines a group or a search.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder query(String query);
    }

    static final class BuilderImpl implements Builder {
        private String type;

        private String query;

        private BuilderImpl() {
        }

        private BuilderImpl(ResourceQuery model) {
            type(model.type);
            query(model.query);
        }

        public final String getType() {
            return type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(QueryType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final void setType(String type) {
            this.type = type;
        }

        public final String getQuery() {
            return query;
        }

        @Override
        public final Builder query(String query) {
            this.query = query;
            return this;
        }

        public final void setQuery(String query) {
            this.query = query;
        }

        @Override
        public ResourceQuery build() {
            return new ResourceQuery(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
