/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.resourcegroups.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An item in a group configuration. A group configuration can have one or more items.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class GroupConfigurationItem implements SdkPojo, Serializable,
        ToCopyableBuilder<GroupConfigurationItem.Builder, GroupConfigurationItem> {
    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GroupConfigurationItem::type)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type").build()).build();

    private static final SdkField<List<GroupConfigurationParameter>> PARAMETERS_FIELD = SdkField
            .<List<GroupConfigurationParameter>> builder(MarshallingType.LIST)
            .getter(getter(GroupConfigurationItem::parameters))
            .setter(setter(Builder::parameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Parameters").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<GroupConfigurationParameter> builder(MarshallingType.SDK_POJO)
                                            .constructor(GroupConfigurationParameter::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TYPE_FIELD, PARAMETERS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String type;

    private final List<GroupConfigurationParameter> parameters;

    private GroupConfigurationItem(BuilderImpl builder) {
        this.type = builder.type;
        this.parameters = builder.parameters;
    }

    /**
     * <p>
     * Specifies the type of group configuration item. Each item must have a unique value for <code>type</code>.
     * </p>
     * <p>
     * You can specify the following string values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AWS::EC2::CapacityReservationPool</code>
     * </p>
     * <p>
     * For more information about EC2 capacity reservation groups, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/capacity-reservations-using.html#create-cr-group"
     * >Working with capacity reservation groups</a> in the <i>EC2 Users Guide</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AWS::ResourceGroups::Generic</code> - Supports parameters that configure the behavior of resource groups of
     * any type.
     * </p>
     * </li>
     * </ul>
     * 
     * @return Specifies the type of group configuration item. Each item must have a unique value for <code>type</code>
     *         .</p>
     *         <p>
     *         You can specify the following string values:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>AWS::EC2::CapacityReservationPool</code>
     *         </p>
     *         <p>
     *         For more information about EC2 capacity reservation groups, see <a href=
     *         "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/capacity-reservations-using.html#create-cr-group"
     *         >Working with capacity reservation groups</a> in the <i>EC2 Users Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>AWS::ResourceGroups::Generic</code> - Supports parameters that configure the behavior of resource
     *         groups of any type.
     *         </p>
     *         </li>
     */
    public String type() {
        return type;
    }

    /**
     * Returns true if the Parameters property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasParameters() {
        return parameters != null && !(parameters instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A collection of parameters for this group configuration item.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasParameters()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A collection of parameters for this group configuration item.
     */
    public List<GroupConfigurationParameter> parameters() {
        return parameters;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(type());
        hashCode = 31 * hashCode + Objects.hashCode(parameters());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GroupConfigurationItem)) {
            return false;
        }
        GroupConfigurationItem other = (GroupConfigurationItem) obj;
        return Objects.equals(type(), other.type()) && Objects.equals(parameters(), other.parameters());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("GroupConfigurationItem").add("Type", type()).add("Parameters", parameters()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Type":
            return Optional.ofNullable(clazz.cast(type()));
        case "Parameters":
            return Optional.ofNullable(clazz.cast(parameters()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<GroupConfigurationItem, T> g) {
        return obj -> g.apply((GroupConfigurationItem) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, GroupConfigurationItem> {
        /**
         * <p>
         * Specifies the type of group configuration item. Each item must have a unique value for <code>type</code>.
         * </p>
         * <p>
         * You can specify the following string values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>AWS::EC2::CapacityReservationPool</code>
         * </p>
         * <p>
         * For more information about EC2 capacity reservation groups, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/capacity-reservations-using.html#create-cr-group"
         * >Working with capacity reservation groups</a> in the <i>EC2 Users Guide</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>AWS::ResourceGroups::Generic</code> - Supports parameters that configure the behavior of resource
         * groups of any type.
         * </p>
         * </li>
         * </ul>
         * 
         * @param type
         *        Specifies the type of group configuration item. Each item must have a unique value for
         *        <code>type</code>.</p>
         *        <p>
         *        You can specify the following string values:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>AWS::EC2::CapacityReservationPool</code>
         *        </p>
         *        <p>
         *        For more information about EC2 capacity reservation groups, see <a href=
         *        "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/capacity-reservations-using.html#create-cr-group"
         *        >Working with capacity reservation groups</a> in the <i>EC2 Users Guide</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>AWS::ResourceGroups::Generic</code> - Supports parameters that configure the behavior of
         *        resource groups of any type.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder type(String type);

        /**
         * <p>
         * A collection of parameters for this group configuration item.
         * </p>
         * 
         * @param parameters
         *        A collection of parameters for this group configuration item.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(Collection<GroupConfigurationParameter> parameters);

        /**
         * <p>
         * A collection of parameters for this group configuration item.
         * </p>
         * 
         * @param parameters
         *        A collection of parameters for this group configuration item.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(GroupConfigurationParameter... parameters);

        /**
         * <p>
         * A collection of parameters for this group configuration item.
         * </p>
         * This is a convenience that creates an instance of the {@link List<GroupConfigurationParameter>.Builder}
         * avoiding the need to create one manually via {@link List<GroupConfigurationParameter>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<GroupConfigurationParameter>.Builder#build()} is called
         * immediately and its result is passed to {@link #parameters(List<GroupConfigurationParameter>)}.
         * 
         * @param parameters
         *        a consumer that will call methods on {@link List<GroupConfigurationParameter>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #parameters(List<GroupConfigurationParameter>)
         */
        Builder parameters(Consumer<GroupConfigurationParameter.Builder>... parameters);
    }

    static final class BuilderImpl implements Builder {
        private String type;

        private List<GroupConfigurationParameter> parameters = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(GroupConfigurationItem model) {
            type(model.type);
            parameters(model.parameters);
        }

        public final String getType() {
            return type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        public final void setType(String type) {
            this.type = type;
        }

        public final Collection<GroupConfigurationParameter.Builder> getParameters() {
            return parameters != null ? parameters.stream().map(GroupConfigurationParameter::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder parameters(Collection<GroupConfigurationParameter> parameters) {
            this.parameters = GroupParameterListCopier.copy(parameters);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder parameters(GroupConfigurationParameter... parameters) {
            parameters(Arrays.asList(parameters));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder parameters(Consumer<GroupConfigurationParameter.Builder>... parameters) {
            parameters(Stream.of(parameters).map(c -> GroupConfigurationParameter.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setParameters(Collection<GroupConfigurationParameter.BuilderImpl> parameters) {
            this.parameters = GroupParameterListCopier.copyFromBuilder(parameters);
        }

        @Override
        public GroupConfigurationItem build() {
            return new GroupConfigurationItem(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
