/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.resourcegroups;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsClient;
import software.amazon.awssdk.services.resourcegroups.model.CreateGroupRequest;
import software.amazon.awssdk.services.resourcegroups.model.CreateGroupResponse;
import software.amazon.awssdk.services.resourcegroups.model.DeleteGroupRequest;
import software.amazon.awssdk.services.resourcegroups.model.DeleteGroupResponse;
import software.amazon.awssdk.services.resourcegroups.model.GetAccountSettingsRequest;
import software.amazon.awssdk.services.resourcegroups.model.GetAccountSettingsResponse;
import software.amazon.awssdk.services.resourcegroups.model.GetGroupConfigurationRequest;
import software.amazon.awssdk.services.resourcegroups.model.GetGroupConfigurationResponse;
import software.amazon.awssdk.services.resourcegroups.model.GetGroupQueryRequest;
import software.amazon.awssdk.services.resourcegroups.model.GetGroupQueryResponse;
import software.amazon.awssdk.services.resourcegroups.model.GetGroupRequest;
import software.amazon.awssdk.services.resourcegroups.model.GetGroupResponse;
import software.amazon.awssdk.services.resourcegroups.model.GetTagsRequest;
import software.amazon.awssdk.services.resourcegroups.model.GetTagsResponse;
import software.amazon.awssdk.services.resourcegroups.model.GroupResourcesRequest;
import software.amazon.awssdk.services.resourcegroups.model.GroupResourcesResponse;
import software.amazon.awssdk.services.resourcegroups.model.ListGroupResourcesRequest;
import software.amazon.awssdk.services.resourcegroups.model.ListGroupResourcesResponse;
import software.amazon.awssdk.services.resourcegroups.model.ListGroupsRequest;
import software.amazon.awssdk.services.resourcegroups.model.ListGroupsResponse;
import software.amazon.awssdk.services.resourcegroups.model.PutGroupConfigurationRequest;
import software.amazon.awssdk.services.resourcegroups.model.PutGroupConfigurationResponse;
import software.amazon.awssdk.services.resourcegroups.model.SearchResourcesRequest;
import software.amazon.awssdk.services.resourcegroups.model.SearchResourcesResponse;
import software.amazon.awssdk.services.resourcegroups.model.TagRequest;
import software.amazon.awssdk.services.resourcegroups.model.TagResponse;
import software.amazon.awssdk.services.resourcegroups.model.UngroupResourcesRequest;
import software.amazon.awssdk.services.resourcegroups.model.UngroupResourcesResponse;
import software.amazon.awssdk.services.resourcegroups.model.UntagRequest;
import software.amazon.awssdk.services.resourcegroups.model.UntagResponse;
import software.amazon.awssdk.services.resourcegroups.model.UpdateAccountSettingsRequest;
import software.amazon.awssdk.services.resourcegroups.model.UpdateAccountSettingsResponse;
import software.amazon.awssdk.services.resourcegroups.model.UpdateGroupQueryRequest;
import software.amazon.awssdk.services.resourcegroups.model.UpdateGroupQueryResponse;
import software.amazon.awssdk.services.resourcegroups.model.UpdateGroupRequest;
import software.amazon.awssdk.services.resourcegroups.model.UpdateGroupResponse;
import software.amazon.awssdk.services.resourcegroups.paginators.ListGroupResourcesPublisher;
import software.amazon.awssdk.services.resourcegroups.paginators.ListGroupsPublisher;
import software.amazon.awssdk.services.resourcegroups.paginators.SearchResourcesPublisher;

/**
 * Service client for accessing Resource Groups asynchronously. This can be created using the static {@link #builder()}
 * method.The asynchronous client performs non-blocking I/O when configured with any {@link SdkAsyncHttpClient}
 * supported in the SDK. However, full non-blocking is not guaranteed as the async client may perform blocking calls in
 * some cases such as credentials retrieval and endpoint discovery as part of the async API call.
 *
 * <p>
 * Resource Groups lets you organize Amazon Web Services resources such as Amazon Elastic Compute Cloud instances,
 * Amazon Relational Database Service databases, and Amazon Simple Storage Service buckets into groups using criteria
 * that you define as tags. A resource group is a collection of resources that match the resource types specified in a
 * query, and share one or more tags or portions of tags. You can create a group of resources based on their roles in
 * your cloud infrastructure, lifecycle stages, regions, application layers, or virtually any criteria. Resource Groups
 * enable you to automate management tasks, such as those in Amazon Web Services Systems Manager Automation documents,
 * on tag-related resources in Amazon Web Services Systems Manager. Groups of tagged resources also let you quickly view
 * a custom console in Amazon Web Services Systems Manager that shows Config compliance and other monitoring data about
 * member resources.
 * </p>
 * <p>
 * To create a resource group, build a resource query, and specify tags that identify the criteria that members of the
 * group have in common. Tags are key-value pairs.
 * </p>
 * <p>
 * For more information about Resource Groups, see the <a
 * href="https://docs.aws.amazon.com/ARG/latest/userguide/welcome.html">Resource Groups User Guide</a>.
 * </p>
 * <p>
 * Resource Groups uses a REST-compliant API that you can use to perform the following types of operations.
 * </p>
 * <ul>
 * <li>
 * <p>
 * Create, Read, Update, and Delete (CRUD) operations on resource groups and resource query entities
 * </p>
 * </li>
 * <li>
 * <p>
 * Applying, editing, and removing tags from resource groups
 * </p>
 * </li>
 * <li>
 * <p>
 * Resolving resource group member ARNs so they can be returned as search results
 * </p>
 * </li>
 * <li>
 * <p>
 * Getting data about resources that are members of a group
 * </p>
 * </li>
 * <li>
 * <p>
 * Searching Amazon Web Services resources based on a resource query
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface ResourceGroupsAsyncClient extends AwsClient {
    String SERVICE_NAME = "resource-groups";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "resource-groups";

    /**
     * <p>
     * Creates a resource group with the specified name and description. You can optionally include either a resource
     * query or a service configuration. For more information about constructing a resource query, see <a
     * href="https://docs.aws.amazon.com/ARG/latest/userguide/getting_started-query.html">Build queries and groups in
     * Resource Groups</a> in the <i>Resource Groups User Guide</i>. For more information about service-linked groups
     * and service configurations, see <a
     * href="https://docs.aws.amazon.com/ARG/latest/APIReference/about-slg.html">Service configurations for Resource
     * Groups</a>.
     * </p>
     * <p>
     * <b>Minimum permissions</b>
     * </p>
     * <p>
     * To run this command, you must have the following permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>resource-groups:CreateGroup</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param createGroupRequest
     * @return A Java Future containing the result of the CreateGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request includes one or more parameters that violate validation rules.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request. Check permissions.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method that isn't allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException You've exceeded throttling limits by making too many requests in a period of
     *         time.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.CreateGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/CreateGroup" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<CreateGroupResponse> createGroup(CreateGroupRequest createGroupRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates a resource group with the specified name and description. You can optionally include either a resource
     * query or a service configuration. For more information about constructing a resource query, see <a
     * href="https://docs.aws.amazon.com/ARG/latest/userguide/getting_started-query.html">Build queries and groups in
     * Resource Groups</a> in the <i>Resource Groups User Guide</i>. For more information about service-linked groups
     * and service configurations, see <a
     * href="https://docs.aws.amazon.com/ARG/latest/APIReference/about-slg.html">Service configurations for Resource
     * Groups</a>.
     * </p>
     * <p>
     * <b>Minimum permissions</b>
     * </p>
     * <p>
     * To run this command, you must have the following permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>resource-groups:CreateGroup</code>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateGroupRequest.Builder} avoiding the need to
     * create one manually via {@link CreateGroupRequest#builder()}
     * </p>
     *
     * @param createGroupRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.resourcegroups.model.CreateGroupRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the CreateGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request includes one or more parameters that violate validation rules.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request. Check permissions.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method that isn't allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException You've exceeded throttling limits by making too many requests in a period of
     *         time.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.CreateGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/CreateGroup" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<CreateGroupResponse> createGroup(Consumer<CreateGroupRequest.Builder> createGroupRequest) {
        return createGroup(CreateGroupRequest.builder().applyMutation(createGroupRequest).build());
    }

    /**
     * <p>
     * Deletes the specified resource group. Deleting a resource group does not delete any resources that are members of
     * the group; it only deletes the group structure.
     * </p>
     * <p>
     * <b>Minimum permissions</b>
     * </p>
     * <p>
     * To run this command, you must have the following permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>resource-groups:DeleteGroup</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param deleteGroupRequest
     * @return A Java Future containing the result of the DeleteGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request includes one or more parameters that violate validation rules.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request. Check permissions.</li>
     *         <li>NotFoundException One or more of the specified resources don't exist.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method that isn't allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException You've exceeded throttling limits by making too many requests in a period of
     *         time.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.DeleteGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/DeleteGroup" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<DeleteGroupResponse> deleteGroup(DeleteGroupRequest deleteGroupRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes the specified resource group. Deleting a resource group does not delete any resources that are members of
     * the group; it only deletes the group structure.
     * </p>
     * <p>
     * <b>Minimum permissions</b>
     * </p>
     * <p>
     * To run this command, you must have the following permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>resource-groups:DeleteGroup</code>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteGroupRequest.Builder} avoiding the need to
     * create one manually via {@link DeleteGroupRequest#builder()}
     * </p>
     *
     * @param deleteGroupRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.resourcegroups.model.DeleteGroupRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the DeleteGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request includes one or more parameters that violate validation rules.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request. Check permissions.</li>
     *         <li>NotFoundException One or more of the specified resources don't exist.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method that isn't allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException You've exceeded throttling limits by making too many requests in a period of
     *         time.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.DeleteGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/DeleteGroup" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<DeleteGroupResponse> deleteGroup(Consumer<DeleteGroupRequest.Builder> deleteGroupRequest) {
        return deleteGroup(DeleteGroupRequest.builder().applyMutation(deleteGroupRequest).build());
    }

    /**
     * <p>
     * Retrieves the current status of optional features in Resource Groups.
     * </p>
     *
     * @param getAccountSettingsRequest
     * @return A Java Future containing the result of the GetAccountSettings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request includes one or more parameters that violate validation rules.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request. Check permissions.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method that isn't allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException You've exceeded throttling limits by making too many requests in a period of
     *         time.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.GetAccountSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/GetAccountSettings"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<GetAccountSettingsResponse> getAccountSettings(GetAccountSettingsRequest getAccountSettingsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieves the current status of optional features in Resource Groups.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetAccountSettingsRequest.Builder} avoiding the
     * need to create one manually via {@link GetAccountSettingsRequest#builder()}
     * </p>
     *
     * @param getAccountSettingsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.resourcegroups.model.GetAccountSettingsRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the GetAccountSettings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request includes one or more parameters that violate validation rules.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request. Check permissions.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method that isn't allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException You've exceeded throttling limits by making too many requests in a period of
     *         time.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.GetAccountSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/GetAccountSettings"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<GetAccountSettingsResponse> getAccountSettings(
            Consumer<GetAccountSettingsRequest.Builder> getAccountSettingsRequest) {
        return getAccountSettings(GetAccountSettingsRequest.builder().applyMutation(getAccountSettingsRequest).build());
    }

    /**
     * <p>
     * Returns information about a specified resource group.
     * </p>
     * <p>
     * <b>Minimum permissions</b>
     * </p>
     * <p>
     * To run this command, you must have the following permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>resource-groups:GetGroup</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param getGroupRequest
     * @return A Java Future containing the result of the GetGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request includes one or more parameters that violate validation rules.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request. Check permissions.</li>
     *         <li>NotFoundException One or more of the specified resources don't exist.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method that isn't allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException You've exceeded throttling limits by making too many requests in a period of
     *         time.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.GetGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/GetGroup" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<GetGroupResponse> getGroup(GetGroupRequest getGroupRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns information about a specified resource group.
     * </p>
     * <p>
     * <b>Minimum permissions</b>
     * </p>
     * <p>
     * To run this command, you must have the following permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>resource-groups:GetGroup</code>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetGroupRequest.Builder} avoiding the need to
     * create one manually via {@link GetGroupRequest#builder()}
     * </p>
     *
     * @param getGroupRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.resourcegroups.model.GetGroupRequest.Builder} to create a request.
     * @return A Java Future containing the result of the GetGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request includes one or more parameters that violate validation rules.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request. Check permissions.</li>
     *         <li>NotFoundException One or more of the specified resources don't exist.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method that isn't allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException You've exceeded throttling limits by making too many requests in a period of
     *         time.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.GetGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/GetGroup" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<GetGroupResponse> getGroup(Consumer<GetGroupRequest.Builder> getGroupRequest) {
        return getGroup(GetGroupRequest.builder().applyMutation(getGroupRequest).build());
    }

    /**
     * <p>
     * Retrieves the service configuration associated with the specified resource group. For details about the service
     * configuration syntax, see <a href="https://docs.aws.amazon.com/ARG/latest/APIReference/about-slg.html">Service
     * configurations for Resource Groups</a>.
     * </p>
     * <p>
     * <b>Minimum permissions</b>
     * </p>
     * <p>
     * To run this command, you must have the following permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>resource-groups:GetGroupConfiguration</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param getGroupConfigurationRequest
     * @return A Java Future containing the result of the GetGroupConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request includes one or more parameters that violate validation rules.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request. Check permissions.</li>
     *         <li>NotFoundException One or more of the specified resources don't exist.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method that isn't allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException You've exceeded throttling limits by making too many requests in a period of
     *         time.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.GetGroupConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/GetGroupConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<GetGroupConfigurationResponse> getGroupConfiguration(
            GetGroupConfigurationRequest getGroupConfigurationRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieves the service configuration associated with the specified resource group. For details about the service
     * configuration syntax, see <a href="https://docs.aws.amazon.com/ARG/latest/APIReference/about-slg.html">Service
     * configurations for Resource Groups</a>.
     * </p>
     * <p>
     * <b>Minimum permissions</b>
     * </p>
     * <p>
     * To run this command, you must have the following permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>resource-groups:GetGroupConfiguration</code>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetGroupConfigurationRequest.Builder} avoiding the
     * need to create one manually via {@link GetGroupConfigurationRequest#builder()}
     * </p>
     *
     * @param getGroupConfigurationRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.resourcegroups.model.GetGroupConfigurationRequest.Builder} to
     *        create a request.
     * @return A Java Future containing the result of the GetGroupConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request includes one or more parameters that violate validation rules.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request. Check permissions.</li>
     *         <li>NotFoundException One or more of the specified resources don't exist.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method that isn't allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException You've exceeded throttling limits by making too many requests in a period of
     *         time.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.GetGroupConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/GetGroupConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<GetGroupConfigurationResponse> getGroupConfiguration(
            Consumer<GetGroupConfigurationRequest.Builder> getGroupConfigurationRequest) {
        return getGroupConfiguration(GetGroupConfigurationRequest.builder().applyMutation(getGroupConfigurationRequest).build());
    }

    /**
     * <p>
     * Retrieves the resource query associated with the specified resource group. For more information about resource
     * queries, see <a
     * href="https://docs.aws.amazon.com/ARG/latest/userguide/gettingstarted-query.html#gettingstarted-query-cli-tag"
     * >Create a tag-based group in Resource Groups</a>.
     * </p>
     * <p>
     * <b>Minimum permissions</b>
     * </p>
     * <p>
     * To run this command, you must have the following permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>resource-groups:GetGroupQuery</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param getGroupQueryRequest
     * @return A Java Future containing the result of the GetGroupQuery operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request includes one or more parameters that violate validation rules.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request. Check permissions.</li>
     *         <li>NotFoundException One or more of the specified resources don't exist.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method that isn't allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException You've exceeded throttling limits by making too many requests in a period of
     *         time.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.GetGroupQuery
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/GetGroupQuery" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<GetGroupQueryResponse> getGroupQuery(GetGroupQueryRequest getGroupQueryRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieves the resource query associated with the specified resource group. For more information about resource
     * queries, see <a
     * href="https://docs.aws.amazon.com/ARG/latest/userguide/gettingstarted-query.html#gettingstarted-query-cli-tag"
     * >Create a tag-based group in Resource Groups</a>.
     * </p>
     * <p>
     * <b>Minimum permissions</b>
     * </p>
     * <p>
     * To run this command, you must have the following permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>resource-groups:GetGroupQuery</code>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetGroupQueryRequest.Builder} avoiding the need to
     * create one manually via {@link GetGroupQueryRequest#builder()}
     * </p>
     *
     * @param getGroupQueryRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.resourcegroups.model.GetGroupQueryRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the GetGroupQuery operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request includes one or more parameters that violate validation rules.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request. Check permissions.</li>
     *         <li>NotFoundException One or more of the specified resources don't exist.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method that isn't allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException You've exceeded throttling limits by making too many requests in a period of
     *         time.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.GetGroupQuery
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/GetGroupQuery" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<GetGroupQueryResponse> getGroupQuery(Consumer<GetGroupQueryRequest.Builder> getGroupQueryRequest) {
        return getGroupQuery(GetGroupQueryRequest.builder().applyMutation(getGroupQueryRequest).build());
    }

    /**
     * <p>
     * Returns a list of tags that are associated with a resource group, specified by an ARN.
     * </p>
     * <p>
     * <b>Minimum permissions</b>
     * </p>
     * <p>
     * To run this command, you must have the following permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>resource-groups:GetTags</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param getTagsRequest
     * @return A Java Future containing the result of the GetTags operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request includes one or more parameters that violate validation rules.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request. Check permissions.</li>
     *         <li>NotFoundException One or more of the specified resources don't exist.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method that isn't allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException You've exceeded throttling limits by making too many requests in a period of
     *         time.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.GetTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/GetTags" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<GetTagsResponse> getTags(GetTagsRequest getTagsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns a list of tags that are associated with a resource group, specified by an ARN.
     * </p>
     * <p>
     * <b>Minimum permissions</b>
     * </p>
     * <p>
     * To run this command, you must have the following permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>resource-groups:GetTags</code>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetTagsRequest.Builder} avoiding the need to create
     * one manually via {@link GetTagsRequest#builder()}
     * </p>
     *
     * @param getTagsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.resourcegroups.model.GetTagsRequest.Builder} to create a request.
     * @return A Java Future containing the result of the GetTags operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request includes one or more parameters that violate validation rules.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request. Check permissions.</li>
     *         <li>NotFoundException One or more of the specified resources don't exist.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method that isn't allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException You've exceeded throttling limits by making too many requests in a period of
     *         time.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.GetTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/GetTags" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<GetTagsResponse> getTags(Consumer<GetTagsRequest.Builder> getTagsRequest) {
        return getTags(GetTagsRequest.builder().applyMutation(getTagsRequest).build());
    }

    /**
     * <p>
     * Adds the specified resources to the specified group.
     * </p>
     * <important>
     * <p>
     * You can use this operation with only resource groups that are configured with the following types:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AWS::EC2::HostManagement</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AWS::EC2::CapacityReservationPool</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * Other resource group type and resource types aren't currently supported by this operation.
     * </p>
     * </important>
     * <p>
     * <b>Minimum permissions</b>
     * </p>
     * <p>
     * To run this command, you must have the following permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>resource-groups:GroupResources</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param groupResourcesRequest
     * @return A Java Future containing the result of the GroupResources operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request includes one or more parameters that violate validation rules.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request. Check permissions.</li>
     *         <li>NotFoundException One or more of the specified resources don't exist.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method that isn't allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException You've exceeded throttling limits by making too many requests in a period of
     *         time.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.GroupResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/GroupResources"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<GroupResourcesResponse> groupResources(GroupResourcesRequest groupResourcesRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Adds the specified resources to the specified group.
     * </p>
     * <important>
     * <p>
     * You can use this operation with only resource groups that are configured with the following types:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AWS::EC2::HostManagement</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AWS::EC2::CapacityReservationPool</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * Other resource group type and resource types aren't currently supported by this operation.
     * </p>
     * </important>
     * <p>
     * <b>Minimum permissions</b>
     * </p>
     * <p>
     * To run this command, you must have the following permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>resource-groups:GroupResources</code>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GroupResourcesRequest.Builder} avoiding the need to
     * create one manually via {@link GroupResourcesRequest#builder()}
     * </p>
     *
     * @param groupResourcesRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.resourcegroups.model.GroupResourcesRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the GroupResources operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request includes one or more parameters that violate validation rules.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request. Check permissions.</li>
     *         <li>NotFoundException One or more of the specified resources don't exist.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method that isn't allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException You've exceeded throttling limits by making too many requests in a period of
     *         time.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.GroupResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/GroupResources"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<GroupResourcesResponse> groupResources(Consumer<GroupResourcesRequest.Builder> groupResourcesRequest) {
        return groupResources(GroupResourcesRequest.builder().applyMutation(groupResourcesRequest).build());
    }

    /**
     * <p>
     * Returns a list of ARNs of the resources that are members of a specified resource group.
     * </p>
     * <p>
     * <b>Minimum permissions</b>
     * </p>
     * <p>
     * To run this command, you must have the following permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>resource-groups:ListGroupResources</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>cloudformation:DescribeStacks</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>cloudformation:ListStackResources</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>tag:GetResources</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param listGroupResourcesRequest
     * @return A Java Future containing the result of the ListGroupResources operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnauthorizedException The request was rejected because it doesn't have valid credentials for the
     *         target resource.</li>
     *         <li>BadRequestException The request includes one or more parameters that violate validation rules.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request. Check permissions.</li>
     *         <li>NotFoundException One or more of the specified resources don't exist.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method that isn't allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException You've exceeded throttling limits by making too many requests in a period of
     *         time.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.ListGroupResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/ListGroupResources"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<ListGroupResourcesResponse> listGroupResources(ListGroupResourcesRequest listGroupResourcesRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns a list of ARNs of the resources that are members of a specified resource group.
     * </p>
     * <p>
     * <b>Minimum permissions</b>
     * </p>
     * <p>
     * To run this command, you must have the following permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>resource-groups:ListGroupResources</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>cloudformation:DescribeStacks</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>cloudformation:ListStackResources</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>tag:GetResources</code>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListGroupResourcesRequest.Builder} avoiding the
     * need to create one manually via {@link ListGroupResourcesRequest#builder()}
     * </p>
     *
     * @param listGroupResourcesRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.resourcegroups.model.ListGroupResourcesRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the ListGroupResources operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnauthorizedException The request was rejected because it doesn't have valid credentials for the
     *         target resource.</li>
     *         <li>BadRequestException The request includes one or more parameters that violate validation rules.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request. Check permissions.</li>
     *         <li>NotFoundException One or more of the specified resources don't exist.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method that isn't allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException You've exceeded throttling limits by making too many requests in a period of
     *         time.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.ListGroupResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/ListGroupResources"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<ListGroupResourcesResponse> listGroupResources(
            Consumer<ListGroupResourcesRequest.Builder> listGroupResourcesRequest) {
        return listGroupResources(ListGroupResourcesRequest.builder().applyMutation(listGroupResourcesRequest).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listGroupResources(software.amazon.awssdk.services.resourcegroups.model.ListGroupResourcesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.resourcegroups.paginators.ListGroupResourcesPublisher publisher = client.listGroupResourcesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.resourcegroups.paginators.ListGroupResourcesPublisher publisher = client.listGroupResourcesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.resourcegroups.model.ListGroupResourcesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.resourcegroups.model.ListGroupResourcesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listGroupResources(software.amazon.awssdk.services.resourcegroups.model.ListGroupResourcesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listGroupResourcesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnauthorizedException The request was rejected because it doesn't have valid credentials for the
     *         target resource.</li>
     *         <li>BadRequestException The request includes one or more parameters that violate validation rules.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request. Check permissions.</li>
     *         <li>NotFoundException One or more of the specified resources don't exist.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method that isn't allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException You've exceeded throttling limits by making too many requests in a period of
     *         time.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.ListGroupResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/ListGroupResources"
     *      target="_top">AWS API Documentation</a>
     */
    default ListGroupResourcesPublisher listGroupResourcesPaginator(ListGroupResourcesRequest listGroupResourcesRequest) {
        return new ListGroupResourcesPublisher(this, listGroupResourcesRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listGroupResources(software.amazon.awssdk.services.resourcegroups.model.ListGroupResourcesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.resourcegroups.paginators.ListGroupResourcesPublisher publisher = client.listGroupResourcesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.resourcegroups.paginators.ListGroupResourcesPublisher publisher = client.listGroupResourcesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.resourcegroups.model.ListGroupResourcesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.resourcegroups.model.ListGroupResourcesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listGroupResources(software.amazon.awssdk.services.resourcegroups.model.ListGroupResourcesRequest)}
     * operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListGroupResourcesRequest.Builder} avoiding the
     * need to create one manually via {@link ListGroupResourcesRequest#builder()}
     * </p>
     *
     * @param listGroupResourcesRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.resourcegroups.model.ListGroupResourcesRequest.Builder} to create a
     *        request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnauthorizedException The request was rejected because it doesn't have valid credentials for the
     *         target resource.</li>
     *         <li>BadRequestException The request includes one or more parameters that violate validation rules.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request. Check permissions.</li>
     *         <li>NotFoundException One or more of the specified resources don't exist.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method that isn't allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException You've exceeded throttling limits by making too many requests in a period of
     *         time.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.ListGroupResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/ListGroupResources"
     *      target="_top">AWS API Documentation</a>
     */
    default ListGroupResourcesPublisher listGroupResourcesPaginator(
            Consumer<ListGroupResourcesRequest.Builder> listGroupResourcesRequest) {
        return listGroupResourcesPaginator(ListGroupResourcesRequest.builder().applyMutation(listGroupResourcesRequest).build());
    }

    /**
     * <p>
     * Returns a list of existing Resource Groups in your account.
     * </p>
     * <p>
     * <b>Minimum permissions</b>
     * </p>
     * <p>
     * To run this command, you must have the following permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>resource-groups:ListGroups</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param listGroupsRequest
     * @return A Java Future containing the result of the ListGroups operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request includes one or more parameters that violate validation rules.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request. Check permissions.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method that isn't allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException You've exceeded throttling limits by making too many requests in a period of
     *         time.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.ListGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/ListGroups" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<ListGroupsResponse> listGroups(ListGroupsRequest listGroupsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns a list of existing Resource Groups in your account.
     * </p>
     * <p>
     * <b>Minimum permissions</b>
     * </p>
     * <p>
     * To run this command, you must have the following permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>resource-groups:ListGroups</code>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListGroupsRequest.Builder} avoiding the need to
     * create one manually via {@link ListGroupsRequest#builder()}
     * </p>
     *
     * @param listGroupsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.resourcegroups.model.ListGroupsRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the ListGroups operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request includes one or more parameters that violate validation rules.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request. Check permissions.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method that isn't allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException You've exceeded throttling limits by making too many requests in a period of
     *         time.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.ListGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/ListGroups" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<ListGroupsResponse> listGroups(Consumer<ListGroupsRequest.Builder> listGroupsRequest) {
        return listGroups(ListGroupsRequest.builder().applyMutation(listGroupsRequest).build());
    }

    /**
     * <p>
     * Returns a list of existing Resource Groups in your account.
     * </p>
     * <p>
     * <b>Minimum permissions</b>
     * </p>
     * <p>
     * To run this command, you must have the following permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>resource-groups:ListGroups</code>
     * </p>
     * </li>
     * </ul>
     *
     * @return A Java Future containing the result of the ListGroups operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request includes one or more parameters that violate validation rules.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request. Check permissions.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method that isn't allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException You've exceeded throttling limits by making too many requests in a period of
     *         time.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.ListGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/ListGroups" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<ListGroupsResponse> listGroups() {
        return listGroups(ListGroupsRequest.builder().build());
    }

    /**
     * <p>
     * This is a variant of {@link #listGroups(software.amazon.awssdk.services.resourcegroups.model.ListGroupsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.resourcegroups.paginators.ListGroupsPublisher publisher = client.listGroupsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.resourcegroups.paginators.ListGroupsPublisher publisher = client.listGroupsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.resourcegroups.model.ListGroupsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.resourcegroups.model.ListGroupsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listGroups(software.amazon.awssdk.services.resourcegroups.model.ListGroupsRequest)} operation.</b>
     * </p>
     *
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request includes one or more parameters that violate validation rules.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request. Check permissions.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method that isn't allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException You've exceeded throttling limits by making too many requests in a period of
     *         time.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.ListGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/ListGroups" target="_top">AWS
     *      API Documentation</a>
     */
    default ListGroupsPublisher listGroupsPaginator() {
        return listGroupsPaginator(ListGroupsRequest.builder().build());
    }

    /**
     * <p>
     * This is a variant of {@link #listGroups(software.amazon.awssdk.services.resourcegroups.model.ListGroupsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.resourcegroups.paginators.ListGroupsPublisher publisher = client.listGroupsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.resourcegroups.paginators.ListGroupsPublisher publisher = client.listGroupsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.resourcegroups.model.ListGroupsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.resourcegroups.model.ListGroupsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listGroups(software.amazon.awssdk.services.resourcegroups.model.ListGroupsRequest)} operation.</b>
     * </p>
     *
     * @param listGroupsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request includes one or more parameters that violate validation rules.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request. Check permissions.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method that isn't allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException You've exceeded throttling limits by making too many requests in a period of
     *         time.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.ListGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/ListGroups" target="_top">AWS
     *      API Documentation</a>
     */
    default ListGroupsPublisher listGroupsPaginator(ListGroupsRequest listGroupsRequest) {
        return new ListGroupsPublisher(this, listGroupsRequest);
    }

    /**
     * <p>
     * This is a variant of {@link #listGroups(software.amazon.awssdk.services.resourcegroups.model.ListGroupsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.resourcegroups.paginators.ListGroupsPublisher publisher = client.listGroupsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.resourcegroups.paginators.ListGroupsPublisher publisher = client.listGroupsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.resourcegroups.model.ListGroupsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.resourcegroups.model.ListGroupsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listGroups(software.amazon.awssdk.services.resourcegroups.model.ListGroupsRequest)} operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListGroupsRequest.Builder} avoiding the need to
     * create one manually via {@link ListGroupsRequest#builder()}
     * </p>
     *
     * @param listGroupsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.resourcegroups.model.ListGroupsRequest.Builder} to create a
     *        request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request includes one or more parameters that violate validation rules.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request. Check permissions.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method that isn't allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException You've exceeded throttling limits by making too many requests in a period of
     *         time.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.ListGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/ListGroups" target="_top">AWS
     *      API Documentation</a>
     */
    default ListGroupsPublisher listGroupsPaginator(Consumer<ListGroupsRequest.Builder> listGroupsRequest) {
        return listGroupsPaginator(ListGroupsRequest.builder().applyMutation(listGroupsRequest).build());
    }

    /**
     * <p>
     * Attaches a service configuration to the specified group. This occurs asynchronously, and can take time to
     * complete. You can use <a>GetGroupConfiguration</a> to check the status of the update.
     * </p>
     * <p>
     * <b>Minimum permissions</b>
     * </p>
     * <p>
     * To run this command, you must have the following permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>resource-groups:PutGroupConfiguration</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param putGroupConfigurationRequest
     * @return A Java Future containing the result of the PutGroupConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request includes one or more parameters that violate validation rules.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request. Check permissions.</li>
     *         <li>NotFoundException One or more of the specified resources don't exist.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method that isn't allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException You've exceeded throttling limits by making too many requests in a period of
     *         time.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.PutGroupConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/PutGroupConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<PutGroupConfigurationResponse> putGroupConfiguration(
            PutGroupConfigurationRequest putGroupConfigurationRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Attaches a service configuration to the specified group. This occurs asynchronously, and can take time to
     * complete. You can use <a>GetGroupConfiguration</a> to check the status of the update.
     * </p>
     * <p>
     * <b>Minimum permissions</b>
     * </p>
     * <p>
     * To run this command, you must have the following permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>resource-groups:PutGroupConfiguration</code>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link PutGroupConfigurationRequest.Builder} avoiding the
     * need to create one manually via {@link PutGroupConfigurationRequest#builder()}
     * </p>
     *
     * @param putGroupConfigurationRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.resourcegroups.model.PutGroupConfigurationRequest.Builder} to
     *        create a request.
     * @return A Java Future containing the result of the PutGroupConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request includes one or more parameters that violate validation rules.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request. Check permissions.</li>
     *         <li>NotFoundException One or more of the specified resources don't exist.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method that isn't allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException You've exceeded throttling limits by making too many requests in a period of
     *         time.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.PutGroupConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/PutGroupConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<PutGroupConfigurationResponse> putGroupConfiguration(
            Consumer<PutGroupConfigurationRequest.Builder> putGroupConfigurationRequest) {
        return putGroupConfiguration(PutGroupConfigurationRequest.builder().applyMutation(putGroupConfigurationRequest).build());
    }

    /**
     * <p>
     * Returns a list of Amazon Web Services resource identifiers that matches the specified query. The query uses the
     * same format as a resource query in a <a>CreateGroup</a> or <a>UpdateGroupQuery</a> operation.
     * </p>
     * <p>
     * <b>Minimum permissions</b>
     * </p>
     * <p>
     * To run this command, you must have the following permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>resource-groups:SearchResources</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>cloudformation:DescribeStacks</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>cloudformation:ListStackResources</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>tag:GetResources</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param searchResourcesRequest
     * @return A Java Future containing the result of the SearchResources operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnauthorizedException The request was rejected because it doesn't have valid credentials for the
     *         target resource.</li>
     *         <li>BadRequestException The request includes one or more parameters that violate validation rules.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request. Check permissions.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method that isn't allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException You've exceeded throttling limits by making too many requests in a period of
     *         time.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.SearchResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/SearchResources"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<SearchResourcesResponse> searchResources(SearchResourcesRequest searchResourcesRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns a list of Amazon Web Services resource identifiers that matches the specified query. The query uses the
     * same format as a resource query in a <a>CreateGroup</a> or <a>UpdateGroupQuery</a> operation.
     * </p>
     * <p>
     * <b>Minimum permissions</b>
     * </p>
     * <p>
     * To run this command, you must have the following permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>resource-groups:SearchResources</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>cloudformation:DescribeStacks</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>cloudformation:ListStackResources</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>tag:GetResources</code>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link SearchResourcesRequest.Builder} avoiding the need
     * to create one manually via {@link SearchResourcesRequest#builder()}
     * </p>
     *
     * @param searchResourcesRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.resourcegroups.model.SearchResourcesRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the SearchResources operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnauthorizedException The request was rejected because it doesn't have valid credentials for the
     *         target resource.</li>
     *         <li>BadRequestException The request includes one or more parameters that violate validation rules.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request. Check permissions.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method that isn't allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException You've exceeded throttling limits by making too many requests in a period of
     *         time.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.SearchResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/SearchResources"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<SearchResourcesResponse> searchResources(
            Consumer<SearchResourcesRequest.Builder> searchResourcesRequest) {
        return searchResources(SearchResourcesRequest.builder().applyMutation(searchResourcesRequest).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #searchResources(software.amazon.awssdk.services.resourcegroups.model.SearchResourcesRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.resourcegroups.paginators.SearchResourcesPublisher publisher = client.searchResourcesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.resourcegroups.paginators.SearchResourcesPublisher publisher = client.searchResourcesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.resourcegroups.model.SearchResourcesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.resourcegroups.model.SearchResourcesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #searchResources(software.amazon.awssdk.services.resourcegroups.model.SearchResourcesRequest)}
     * operation.</b>
     * </p>
     *
     * @param searchResourcesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnauthorizedException The request was rejected because it doesn't have valid credentials for the
     *         target resource.</li>
     *         <li>BadRequestException The request includes one or more parameters that violate validation rules.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request. Check permissions.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method that isn't allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException You've exceeded throttling limits by making too many requests in a period of
     *         time.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.SearchResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/SearchResources"
     *      target="_top">AWS API Documentation</a>
     */
    default SearchResourcesPublisher searchResourcesPaginator(SearchResourcesRequest searchResourcesRequest) {
        return new SearchResourcesPublisher(this, searchResourcesRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #searchResources(software.amazon.awssdk.services.resourcegroups.model.SearchResourcesRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.resourcegroups.paginators.SearchResourcesPublisher publisher = client.searchResourcesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.resourcegroups.paginators.SearchResourcesPublisher publisher = client.searchResourcesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.resourcegroups.model.SearchResourcesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.resourcegroups.model.SearchResourcesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #searchResources(software.amazon.awssdk.services.resourcegroups.model.SearchResourcesRequest)}
     * operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link SearchResourcesRequest.Builder} avoiding the need
     * to create one manually via {@link SearchResourcesRequest#builder()}
     * </p>
     *
     * @param searchResourcesRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.resourcegroups.model.SearchResourcesRequest.Builder} to create a
     *        request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnauthorizedException The request was rejected because it doesn't have valid credentials for the
     *         target resource.</li>
     *         <li>BadRequestException The request includes one or more parameters that violate validation rules.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request. Check permissions.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method that isn't allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException You've exceeded throttling limits by making too many requests in a period of
     *         time.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.SearchResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/SearchResources"
     *      target="_top">AWS API Documentation</a>
     */
    default SearchResourcesPublisher searchResourcesPaginator(Consumer<SearchResourcesRequest.Builder> searchResourcesRequest) {
        return searchResourcesPaginator(SearchResourcesRequest.builder().applyMutation(searchResourcesRequest).build());
    }

    /**
     * <p>
     * Adds tags to a resource group with the specified ARN. Existing tags on a resource group are not changed if they
     * are not specified in the request parameters.
     * </p>
     * <important>
     * <p>
     * Do not store personally identifiable information (PII) or other confidential or sensitive information in tags. We
     * use tags to provide you with billing and administration services. Tags are not intended to be used for private or
     * sensitive data.
     * </p>
     * </important>
     * <p>
     * <b>Minimum permissions</b>
     * </p>
     * <p>
     * To run this command, you must have the following permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>resource-groups:Tag</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param tagRequest
     * @return A Java Future containing the result of the Tag operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request includes one or more parameters that violate validation rules.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request. Check permissions.</li>
     *         <li>NotFoundException One or more of the specified resources don't exist.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method that isn't allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException You've exceeded throttling limits by making too many requests in a period of
     *         time.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.Tag
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/Tag" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<TagResponse> tag(TagRequest tagRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Adds tags to a resource group with the specified ARN. Existing tags on a resource group are not changed if they
     * are not specified in the request parameters.
     * </p>
     * <important>
     * <p>
     * Do not store personally identifiable information (PII) or other confidential or sensitive information in tags. We
     * use tags to provide you with billing and administration services. Tags are not intended to be used for private or
     * sensitive data.
     * </p>
     * </important>
     * <p>
     * <b>Minimum permissions</b>
     * </p>
     * <p>
     * To run this command, you must have the following permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>resource-groups:Tag</code>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link TagRequest.Builder} avoiding the need to create one
     * manually via {@link TagRequest#builder()}
     * </p>
     *
     * @param tagRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.resourcegroups.model.TagRequest.Builder} to create a request.
     * @return A Java Future containing the result of the Tag operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request includes one or more parameters that violate validation rules.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request. Check permissions.</li>
     *         <li>NotFoundException One or more of the specified resources don't exist.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method that isn't allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException You've exceeded throttling limits by making too many requests in a period of
     *         time.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.Tag
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/Tag" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<TagResponse> tag(Consumer<TagRequest.Builder> tagRequest) {
        return tag(TagRequest.builder().applyMutation(tagRequest).build());
    }

    /**
     * <p>
     * Removes the specified resources from the specified group. This operation works only with static groups that you
     * populated using the <a>GroupResources</a> operation. It doesn't work with any resource groups that are
     * automatically populated by tag-based or CloudFormation stack-based queries.
     * </p>
     * <p>
     * <b>Minimum permissions</b>
     * </p>
     * <p>
     * To run this command, you must have the following permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>resource-groups:UngroupResources</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param ungroupResourcesRequest
     * @return A Java Future containing the result of the UngroupResources operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request includes one or more parameters that violate validation rules.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request. Check permissions.</li>
     *         <li>NotFoundException One or more of the specified resources don't exist.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method that isn't allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException You've exceeded throttling limits by making too many requests in a period of
     *         time.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.UngroupResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/UngroupResources"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<UngroupResourcesResponse> ungroupResources(UngroupResourcesRequest ungroupResourcesRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Removes the specified resources from the specified group. This operation works only with static groups that you
     * populated using the <a>GroupResources</a> operation. It doesn't work with any resource groups that are
     * automatically populated by tag-based or CloudFormation stack-based queries.
     * </p>
     * <p>
     * <b>Minimum permissions</b>
     * </p>
     * <p>
     * To run this command, you must have the following permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>resource-groups:UngroupResources</code>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UngroupResourcesRequest.Builder} avoiding the need
     * to create one manually via {@link UngroupResourcesRequest#builder()}
     * </p>
     *
     * @param ungroupResourcesRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.resourcegroups.model.UngroupResourcesRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the UngroupResources operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request includes one or more parameters that violate validation rules.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request. Check permissions.</li>
     *         <li>NotFoundException One or more of the specified resources don't exist.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method that isn't allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException You've exceeded throttling limits by making too many requests in a period of
     *         time.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.UngroupResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/UngroupResources"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<UngroupResourcesResponse> ungroupResources(
            Consumer<UngroupResourcesRequest.Builder> ungroupResourcesRequest) {
        return ungroupResources(UngroupResourcesRequest.builder().applyMutation(ungroupResourcesRequest).build());
    }

    /**
     * <p>
     * Deletes tags from a specified resource group.
     * </p>
     * <p>
     * <b>Minimum permissions</b>
     * </p>
     * <p>
     * To run this command, you must have the following permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>resource-groups:Untag</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param untagRequest
     * @return A Java Future containing the result of the Untag operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request includes one or more parameters that violate validation rules.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request. Check permissions.</li>
     *         <li>NotFoundException One or more of the specified resources don't exist.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method that isn't allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException You've exceeded throttling limits by making too many requests in a period of
     *         time.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.Untag
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/Untag" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<UntagResponse> untag(UntagRequest untagRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes tags from a specified resource group.
     * </p>
     * <p>
     * <b>Minimum permissions</b>
     * </p>
     * <p>
     * To run this command, you must have the following permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>resource-groups:Untag</code>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UntagRequest.Builder} avoiding the need to create
     * one manually via {@link UntagRequest#builder()}
     * </p>
     *
     * @param untagRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.resourcegroups.model.UntagRequest.Builder} to create a request.
     * @return A Java Future containing the result of the Untag operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request includes one or more parameters that violate validation rules.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request. Check permissions.</li>
     *         <li>NotFoundException One or more of the specified resources don't exist.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method that isn't allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException You've exceeded throttling limits by making too many requests in a period of
     *         time.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.Untag
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/Untag" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<UntagResponse> untag(Consumer<UntagRequest.Builder> untagRequest) {
        return untag(UntagRequest.builder().applyMutation(untagRequest).build());
    }

    /**
     * <p>
     * Turns on or turns off optional features in Resource Groups.
     * </p>
     * <p>
     * The preceding example shows that the request to turn on group lifecycle events is <code>IN_PROGRESS</code>. You
     * can call the <a>GetAccountSettings</a> operation to check for completion by looking for
     * <code>GroupLifecycleEventsStatus</code> to change to <code>ACTIVE</code>.
     * </p>
     *
     * @param updateAccountSettingsRequest
     * @return A Java Future containing the result of the UpdateAccountSettings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request includes one or more parameters that violate validation rules.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request. Check permissions.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method that isn't allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException You've exceeded throttling limits by making too many requests in a period of
     *         time.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.UpdateAccountSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/UpdateAccountSettings"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<UpdateAccountSettingsResponse> updateAccountSettings(
            UpdateAccountSettingsRequest updateAccountSettingsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Turns on or turns off optional features in Resource Groups.
     * </p>
     * <p>
     * The preceding example shows that the request to turn on group lifecycle events is <code>IN_PROGRESS</code>. You
     * can call the <a>GetAccountSettings</a> operation to check for completion by looking for
     * <code>GroupLifecycleEventsStatus</code> to change to <code>ACTIVE</code>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateAccountSettingsRequest.Builder} avoiding the
     * need to create one manually via {@link UpdateAccountSettingsRequest#builder()}
     * </p>
     *
     * @param updateAccountSettingsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.resourcegroups.model.UpdateAccountSettingsRequest.Builder} to
     *        create a request.
     * @return A Java Future containing the result of the UpdateAccountSettings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request includes one or more parameters that violate validation rules.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request. Check permissions.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method that isn't allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException You've exceeded throttling limits by making too many requests in a period of
     *         time.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.UpdateAccountSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/UpdateAccountSettings"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<UpdateAccountSettingsResponse> updateAccountSettings(
            Consumer<UpdateAccountSettingsRequest.Builder> updateAccountSettingsRequest) {
        return updateAccountSettings(UpdateAccountSettingsRequest.builder().applyMutation(updateAccountSettingsRequest).build());
    }

    /**
     * <p>
     * Updates the description for an existing group. You cannot update the name of a resource group.
     * </p>
     * <p>
     * <b>Minimum permissions</b>
     * </p>
     * <p>
     * To run this command, you must have the following permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>resource-groups:UpdateGroup</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param updateGroupRequest
     * @return A Java Future containing the result of the UpdateGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request includes one or more parameters that violate validation rules.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request. Check permissions.</li>
     *         <li>NotFoundException One or more of the specified resources don't exist.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method that isn't allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException You've exceeded throttling limits by making too many requests in a period of
     *         time.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.UpdateGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/UpdateGroup" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<UpdateGroupResponse> updateGroup(UpdateGroupRequest updateGroupRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Updates the description for an existing group. You cannot update the name of a resource group.
     * </p>
     * <p>
     * <b>Minimum permissions</b>
     * </p>
     * <p>
     * To run this command, you must have the following permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>resource-groups:UpdateGroup</code>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateGroupRequest.Builder} avoiding the need to
     * create one manually via {@link UpdateGroupRequest#builder()}
     * </p>
     *
     * @param updateGroupRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.resourcegroups.model.UpdateGroupRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the UpdateGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request includes one or more parameters that violate validation rules.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request. Check permissions.</li>
     *         <li>NotFoundException One or more of the specified resources don't exist.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method that isn't allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException You've exceeded throttling limits by making too many requests in a period of
     *         time.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.UpdateGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/UpdateGroup" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<UpdateGroupResponse> updateGroup(Consumer<UpdateGroupRequest.Builder> updateGroupRequest) {
        return updateGroup(UpdateGroupRequest.builder().applyMutation(updateGroupRequest).build());
    }

    /**
     * <p>
     * Updates the resource query of a group. For more information about resource queries, see <a
     * href="https://docs.aws.amazon.com/ARG/latest/userguide/gettingstarted-query.html#gettingstarted-query-cli-tag"
     * >Create a tag-based group in Resource Groups</a>.
     * </p>
     * <p>
     * <b>Minimum permissions</b>
     * </p>
     * <p>
     * To run this command, you must have the following permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>resource-groups:UpdateGroupQuery</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param updateGroupQueryRequest
     * @return A Java Future containing the result of the UpdateGroupQuery operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request includes one or more parameters that violate validation rules.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request. Check permissions.</li>
     *         <li>NotFoundException One or more of the specified resources don't exist.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method that isn't allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException You've exceeded throttling limits by making too many requests in a period of
     *         time.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.UpdateGroupQuery
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/UpdateGroupQuery"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<UpdateGroupQueryResponse> updateGroupQuery(UpdateGroupQueryRequest updateGroupQueryRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Updates the resource query of a group. For more information about resource queries, see <a
     * href="https://docs.aws.amazon.com/ARG/latest/userguide/gettingstarted-query.html#gettingstarted-query-cli-tag"
     * >Create a tag-based group in Resource Groups</a>.
     * </p>
     * <p>
     * <b>Minimum permissions</b>
     * </p>
     * <p>
     * To run this command, you must have the following permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>resource-groups:UpdateGroupQuery</code>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateGroupQueryRequest.Builder} avoiding the need
     * to create one manually via {@link UpdateGroupQueryRequest#builder()}
     * </p>
     *
     * @param updateGroupQueryRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.resourcegroups.model.UpdateGroupQueryRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the UpdateGroupQuery operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request includes one or more parameters that violate validation rules.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request. Check permissions.</li>
     *         <li>NotFoundException One or more of the specified resources don't exist.</li>
     *         <li>MethodNotAllowedException The request uses an HTTP method that isn't allowed for the specified
     *         resource.</li>
     *         <li>TooManyRequestsException You've exceeded throttling limits by making too many requests in a period of
     *         time.</li>
     *         <li>InternalServerErrorException An internal error occurred while processing the request. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ResourceGroupsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ResourceGroupsAsyncClient.UpdateGroupQuery
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/UpdateGroupQuery"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<UpdateGroupQueryResponse> updateGroupQuery(
            Consumer<UpdateGroupQueryRequest.Builder> updateGroupQueryRequest) {
        return updateGroupQuery(UpdateGroupQueryRequest.builder().applyMutation(updateGroupQueryRequest).build());
    }

    @Override
    default ResourceGroupsServiceClientConfiguration serviceClientConfiguration() {
        throw new UnsupportedOperationException();
    }

    /**
     * Create a {@link ResourceGroupsAsyncClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static ResourceGroupsAsyncClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link ResourceGroupsAsyncClient}.
     */
    static ResourceGroupsAsyncClientBuilder builder() {
        return new DefaultResourceGroupsAsyncClientBuilder();
    }
}
