/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.resourcegroups.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A service configuration associated with a resource group. The configuration options are determined by the Amazon Web
 * Services service that defines the <code>Type</code>, and specifies which resources can be included in the group. You
 * can add a service configuration when you create the group by using <a>CreateGroup</a>, or later by using the
 * <a>PutGroupConfiguration</a> operation. For details about group service configuration syntax, see <a
 * href="https://docs.aws.amazon.com/ARG/latest/APIReference/about-slg.html">Service configurations for resource
 * groups</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class GroupConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<GroupConfiguration.Builder, GroupConfiguration> {
    private static final SdkField<List<GroupConfigurationItem>> CONFIGURATION_FIELD = SdkField
            .<List<GroupConfigurationItem>> builder(MarshallingType.LIST)
            .memberName("Configuration")
            .getter(getter(GroupConfiguration::configuration))
            .setter(setter(Builder::configuration))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Configuration").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<GroupConfigurationItem> builder(MarshallingType.SDK_POJO)
                                            .constructor(GroupConfigurationItem::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<GroupConfigurationItem>> PROPOSED_CONFIGURATION_FIELD = SdkField
            .<List<GroupConfigurationItem>> builder(MarshallingType.LIST)
            .memberName("ProposedConfiguration")
            .getter(getter(GroupConfiguration::proposedConfiguration))
            .setter(setter(Builder::proposedConfiguration))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProposedConfiguration").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<GroupConfigurationItem> builder(MarshallingType.SDK_POJO)
                                            .constructor(GroupConfigurationItem::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(GroupConfiguration::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> FAILURE_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FailureReason").getter(getter(GroupConfiguration::failureReason)).setter(setter(Builder::failureReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FailureReason").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CONFIGURATION_FIELD,
            PROPOSED_CONFIGURATION_FIELD, STATUS_FIELD, FAILURE_REASON_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final List<GroupConfigurationItem> configuration;

    private final List<GroupConfigurationItem> proposedConfiguration;

    private final String status;

    private final String failureReason;

    private GroupConfiguration(BuilderImpl builder) {
        this.configuration = builder.configuration;
        this.proposedConfiguration = builder.proposedConfiguration;
        this.status = builder.status;
        this.failureReason = builder.failureReason;
    }

    /**
     * For responses, this returns true if the service returned a value for the Configuration property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasConfiguration() {
        return configuration != null && !(configuration instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The configuration currently associated with the group and in effect.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasConfiguration} method.
     * </p>
     * 
     * @return The configuration currently associated with the group and in effect.
     */
    public final List<GroupConfigurationItem> configuration() {
        return configuration;
    }

    /**
     * For responses, this returns true if the service returned a value for the ProposedConfiguration property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasProposedConfiguration() {
        return proposedConfiguration != null && !(proposedConfiguration instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * If present, the new configuration that is in the process of being applied to the group.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasProposedConfiguration} method.
     * </p>
     * 
     * @return If present, the new configuration that is in the process of being applied to the group.
     */
    public final List<GroupConfigurationItem> proposedConfiguration() {
        return proposedConfiguration;
    }

    /**
     * <p>
     * The current status of an attempt to update the group configuration.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link GroupConfigurationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of an attempt to update the group configuration.
     * @see GroupConfigurationStatus
     */
    public final GroupConfigurationStatus status() {
        return GroupConfigurationStatus.fromValue(status);
    }

    /**
     * <p>
     * The current status of an attempt to update the group configuration.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link GroupConfigurationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of an attempt to update the group configuration.
     * @see GroupConfigurationStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * If present, the reason why a request to update the group configuration failed.
     * </p>
     * 
     * @return If present, the reason why a request to update the group configuration failed.
     */
    public final String failureReason() {
        return failureReason;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasConfiguration() ? configuration() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasProposedConfiguration() ? proposedConfiguration() : null);
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(failureReason());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GroupConfiguration)) {
            return false;
        }
        GroupConfiguration other = (GroupConfiguration) obj;
        return hasConfiguration() == other.hasConfiguration() && Objects.equals(configuration(), other.configuration())
                && hasProposedConfiguration() == other.hasProposedConfiguration()
                && Objects.equals(proposedConfiguration(), other.proposedConfiguration())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(failureReason(), other.failureReason());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("GroupConfiguration").add("Configuration", hasConfiguration() ? configuration() : null)
                .add("ProposedConfiguration", hasProposedConfiguration() ? proposedConfiguration() : null)
                .add("Status", statusAsString()).add("FailureReason", failureReason()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Configuration":
            return Optional.ofNullable(clazz.cast(configuration()));
        case "ProposedConfiguration":
            return Optional.ofNullable(clazz.cast(proposedConfiguration()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "FailureReason":
            return Optional.ofNullable(clazz.cast(failureReason()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Configuration", CONFIGURATION_FIELD);
        map.put("ProposedConfiguration", PROPOSED_CONFIGURATION_FIELD);
        map.put("Status", STATUS_FIELD);
        map.put("FailureReason", FAILURE_REASON_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<GroupConfiguration, T> g) {
        return obj -> g.apply((GroupConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, GroupConfiguration> {
        /**
         * <p>
         * The configuration currently associated with the group and in effect.
         * </p>
         * 
         * @param configuration
         *        The configuration currently associated with the group and in effect.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configuration(Collection<GroupConfigurationItem> configuration);

        /**
         * <p>
         * The configuration currently associated with the group and in effect.
         * </p>
         * 
         * @param configuration
         *        The configuration currently associated with the group and in effect.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configuration(GroupConfigurationItem... configuration);

        /**
         * <p>
         * The configuration currently associated with the group and in effect.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.resourcegroups.model.GroupConfigurationItem.Builder} avoiding the need
         * to create one manually via
         * {@link software.amazon.awssdk.services.resourcegroups.model.GroupConfigurationItem#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.resourcegroups.model.GroupConfigurationItem.Builder#build()} is called
         * immediately and its result is passed to {@link #configuration(List<GroupConfigurationItem>)}.
         * 
         * @param configuration
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.resourcegroups.model.GroupConfigurationItem.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #configuration(java.util.Collection<GroupConfigurationItem>)
         */
        Builder configuration(Consumer<GroupConfigurationItem.Builder>... configuration);

        /**
         * <p>
         * If present, the new configuration that is in the process of being applied to the group.
         * </p>
         * 
         * @param proposedConfiguration
         *        If present, the new configuration that is in the process of being applied to the group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder proposedConfiguration(Collection<GroupConfigurationItem> proposedConfiguration);

        /**
         * <p>
         * If present, the new configuration that is in the process of being applied to the group.
         * </p>
         * 
         * @param proposedConfiguration
         *        If present, the new configuration that is in the process of being applied to the group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder proposedConfiguration(GroupConfigurationItem... proposedConfiguration);

        /**
         * <p>
         * If present, the new configuration that is in the process of being applied to the group.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.resourcegroups.model.GroupConfigurationItem.Builder} avoiding the need
         * to create one manually via
         * {@link software.amazon.awssdk.services.resourcegroups.model.GroupConfigurationItem#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.resourcegroups.model.GroupConfigurationItem.Builder#build()} is called
         * immediately and its result is passed to {@link #proposedConfiguration(List<GroupConfigurationItem>)}.
         * 
         * @param proposedConfiguration
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.resourcegroups.model.GroupConfigurationItem.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #proposedConfiguration(java.util.Collection<GroupConfigurationItem>)
         */
        Builder proposedConfiguration(Consumer<GroupConfigurationItem.Builder>... proposedConfiguration);

        /**
         * <p>
         * The current status of an attempt to update the group configuration.
         * </p>
         * 
         * @param status
         *        The current status of an attempt to update the group configuration.
         * @see GroupConfigurationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GroupConfigurationStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The current status of an attempt to update the group configuration.
         * </p>
         * 
         * @param status
         *        The current status of an attempt to update the group configuration.
         * @see GroupConfigurationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GroupConfigurationStatus
         */
        Builder status(GroupConfigurationStatus status);

        /**
         * <p>
         * If present, the reason why a request to update the group configuration failed.
         * </p>
         * 
         * @param failureReason
         *        If present, the reason why a request to update the group configuration failed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureReason(String failureReason);
    }

    static final class BuilderImpl implements Builder {
        private List<GroupConfigurationItem> configuration = DefaultSdkAutoConstructList.getInstance();

        private List<GroupConfigurationItem> proposedConfiguration = DefaultSdkAutoConstructList.getInstance();

        private String status;

        private String failureReason;

        private BuilderImpl() {
        }

        private BuilderImpl(GroupConfiguration model) {
            configuration(model.configuration);
            proposedConfiguration(model.proposedConfiguration);
            status(model.status);
            failureReason(model.failureReason);
        }

        public final List<GroupConfigurationItem.Builder> getConfiguration() {
            List<GroupConfigurationItem.Builder> result = GroupConfigurationListCopier.copyToBuilder(this.configuration);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setConfiguration(Collection<GroupConfigurationItem.BuilderImpl> configuration) {
            this.configuration = GroupConfigurationListCopier.copyFromBuilder(configuration);
        }

        @Override
        public final Builder configuration(Collection<GroupConfigurationItem> configuration) {
            this.configuration = GroupConfigurationListCopier.copy(configuration);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder configuration(GroupConfigurationItem... configuration) {
            configuration(Arrays.asList(configuration));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder configuration(Consumer<GroupConfigurationItem.Builder>... configuration) {
            configuration(Stream.of(configuration).map(c -> GroupConfigurationItem.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<GroupConfigurationItem.Builder> getProposedConfiguration() {
            List<GroupConfigurationItem.Builder> result = GroupConfigurationListCopier.copyToBuilder(this.proposedConfiguration);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setProposedConfiguration(Collection<GroupConfigurationItem.BuilderImpl> proposedConfiguration) {
            this.proposedConfiguration = GroupConfigurationListCopier.copyFromBuilder(proposedConfiguration);
        }

        @Override
        public final Builder proposedConfiguration(Collection<GroupConfigurationItem> proposedConfiguration) {
            this.proposedConfiguration = GroupConfigurationListCopier.copy(proposedConfiguration);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder proposedConfiguration(GroupConfigurationItem... proposedConfiguration) {
            proposedConfiguration(Arrays.asList(proposedConfiguration));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder proposedConfiguration(Consumer<GroupConfigurationItem.Builder>... proposedConfiguration) {
            proposedConfiguration(Stream.of(proposedConfiguration)
                    .map(c -> GroupConfigurationItem.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(GroupConfigurationStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getFailureReason() {
            return failureReason;
        }

        public final void setFailureReason(String failureReason) {
            this.failureReason = failureReason;
        }

        @Override
        public final Builder failureReason(String failureReason) {
            this.failureReason = failureReason;
            return this;
        }

        @Override
        public GroupConfiguration build() {
            return new GroupConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
