/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.resourcegroups.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An item in a group configuration. A group service configuration can have one or more items. For details about group
 * service configuration syntax, see <a
 * href="https://docs.aws.amazon.com/ARG/latest/APIReference/about-slg.html">Service configurations for resource
 * groups</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class GroupConfigurationItem implements SdkPojo, Serializable,
        ToCopyableBuilder<GroupConfigurationItem.Builder, GroupConfigurationItem> {
    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Type")
            .getter(getter(GroupConfigurationItem::type)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type").build()).build();

    private static final SdkField<List<GroupConfigurationParameter>> PARAMETERS_FIELD = SdkField
            .<List<GroupConfigurationParameter>> builder(MarshallingType.LIST)
            .memberName("Parameters")
            .getter(getter(GroupConfigurationItem::parameters))
            .setter(setter(Builder::parameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Parameters").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<GroupConfigurationParameter> builder(MarshallingType.SDK_POJO)
                                            .constructor(GroupConfigurationParameter::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TYPE_FIELD, PARAMETERS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String type;

    private final List<GroupConfigurationParameter> parameters;

    private GroupConfigurationItem(BuilderImpl builder) {
        this.type = builder.type;
        this.parameters = builder.parameters;
    }

    /**
     * <p>
     * Specifies the type of group configuration item. Each item must have a unique value for <code>type</code>. For the
     * list of types that you can specify for a configuration item, see <a
     * href="https://docs.aws.amazon.com/ARG/latest/APIReference/about-slg.html#about-slg-types">Supported resource
     * types and parameters</a>.
     * </p>
     * 
     * @return Specifies the type of group configuration item. Each item must have a unique value for <code>type</code>.
     *         For the list of types that you can specify for a configuration item, see <a
     *         href="https://docs.aws.amazon.com/ARG/latest/APIReference/about-slg.html#about-slg-types">Supported
     *         resource types and parameters</a>.
     */
    public final String type() {
        return type;
    }

    /**
     * For responses, this returns true if the service returned a value for the Parameters property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasParameters() {
        return parameters != null && !(parameters instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A collection of parameters for this group configuration item. For the list of parameters that you can use with
     * each configuration item type, see <a
     * href="https://docs.aws.amazon.com/ARG/latest/APIReference/about-slg.html#about-slg-types">Supported resource
     * types and parameters</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasParameters} method.
     * </p>
     * 
     * @return A collection of parameters for this group configuration item. For the list of parameters that you can use
     *         with each configuration item type, see <a
     *         href="https://docs.aws.amazon.com/ARG/latest/APIReference/about-slg.html#about-slg-types">Supported
     *         resource types and parameters</a>.
     */
    public final List<GroupConfigurationParameter> parameters() {
        return parameters;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(type());
        hashCode = 31 * hashCode + Objects.hashCode(hasParameters() ? parameters() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GroupConfigurationItem)) {
            return false;
        }
        GroupConfigurationItem other = (GroupConfigurationItem) obj;
        return Objects.equals(type(), other.type()) && hasParameters() == other.hasParameters()
                && Objects.equals(parameters(), other.parameters());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("GroupConfigurationItem").add("Type", type())
                .add("Parameters", hasParameters() ? parameters() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Type":
            return Optional.ofNullable(clazz.cast(type()));
        case "Parameters":
            return Optional.ofNullable(clazz.cast(parameters()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Type", TYPE_FIELD);
        map.put("Parameters", PARAMETERS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<GroupConfigurationItem, T> g) {
        return obj -> g.apply((GroupConfigurationItem) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, GroupConfigurationItem> {
        /**
         * <p>
         * Specifies the type of group configuration item. Each item must have a unique value for <code>type</code>. For
         * the list of types that you can specify for a configuration item, see <a
         * href="https://docs.aws.amazon.com/ARG/latest/APIReference/about-slg.html#about-slg-types">Supported resource
         * types and parameters</a>.
         * </p>
         * 
         * @param type
         *        Specifies the type of group configuration item. Each item must have a unique value for
         *        <code>type</code>. For the list of types that you can specify for a configuration item, see <a
         *        href="https://docs.aws.amazon.com/ARG/latest/APIReference/about-slg.html#about-slg-types">Supported
         *        resource types and parameters</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder type(String type);

        /**
         * <p>
         * A collection of parameters for this group configuration item. For the list of parameters that you can use
         * with each configuration item type, see <a
         * href="https://docs.aws.amazon.com/ARG/latest/APIReference/about-slg.html#about-slg-types">Supported resource
         * types and parameters</a>.
         * </p>
         * 
         * @param parameters
         *        A collection of parameters for this group configuration item. For the list of parameters that you can
         *        use with each configuration item type, see <a
         *        href="https://docs.aws.amazon.com/ARG/latest/APIReference/about-slg.html#about-slg-types">Supported
         *        resource types and parameters</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(Collection<GroupConfigurationParameter> parameters);

        /**
         * <p>
         * A collection of parameters for this group configuration item. For the list of parameters that you can use
         * with each configuration item type, see <a
         * href="https://docs.aws.amazon.com/ARG/latest/APIReference/about-slg.html#about-slg-types">Supported resource
         * types and parameters</a>.
         * </p>
         * 
         * @param parameters
         *        A collection of parameters for this group configuration item. For the list of parameters that you can
         *        use with each configuration item type, see <a
         *        href="https://docs.aws.amazon.com/ARG/latest/APIReference/about-slg.html#about-slg-types">Supported
         *        resource types and parameters</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(GroupConfigurationParameter... parameters);

        /**
         * <p>
         * A collection of parameters for this group configuration item. For the list of parameters that you can use
         * with each configuration item type, see <a
         * href="https://docs.aws.amazon.com/ARG/latest/APIReference/about-slg.html#about-slg-types">Supported resource
         * types and parameters</a>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.resourcegroups.model.GroupConfigurationParameter.Builder} avoiding the
         * need to create one manually via
         * {@link software.amazon.awssdk.services.resourcegroups.model.GroupConfigurationParameter#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.resourcegroups.model.GroupConfigurationParameter.Builder#build()} is
         * called immediately and its result is passed to {@link #parameters(List<GroupConfigurationParameter>)}.
         * 
         * @param parameters
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.resourcegroups.model.GroupConfigurationParameter.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #parameters(java.util.Collection<GroupConfigurationParameter>)
         */
        Builder parameters(Consumer<GroupConfigurationParameter.Builder>... parameters);
    }

    static final class BuilderImpl implements Builder {
        private String type;

        private List<GroupConfigurationParameter> parameters = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(GroupConfigurationItem model) {
            type(model.type);
            parameters(model.parameters);
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        public final List<GroupConfigurationParameter.Builder> getParameters() {
            List<GroupConfigurationParameter.Builder> result = GroupParameterListCopier.copyToBuilder(this.parameters);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setParameters(Collection<GroupConfigurationParameter.BuilderImpl> parameters) {
            this.parameters = GroupParameterListCopier.copyFromBuilder(parameters);
        }

        @Override
        public final Builder parameters(Collection<GroupConfigurationParameter> parameters) {
            this.parameters = GroupParameterListCopier.copy(parameters);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder parameters(GroupConfigurationParameter... parameters) {
            parameters(Arrays.asList(parameters));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder parameters(Consumer<GroupConfigurationParameter.Builder>... parameters) {
            parameters(Stream.of(parameters).map(c -> GroupConfigurationParameter.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public GroupConfigurationItem build() {
            return new GroupConfigurationItem(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
