/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.robomaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Configuration information for a deployment launch.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DeploymentLaunchConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<DeploymentLaunchConfig.Builder, DeploymentLaunchConfig> {
    private static final SdkField<String> PACKAGE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("packageName").getter(getter(DeploymentLaunchConfig::packageName)).setter(setter(Builder::packageName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("packageName").build()).build();

    private static final SdkField<String> PRE_LAUNCH_FILE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("preLaunchFile").getter(getter(DeploymentLaunchConfig::preLaunchFile))
            .setter(setter(Builder::preLaunchFile))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("preLaunchFile").build()).build();

    private static final SdkField<String> LAUNCH_FILE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("launchFile").getter(getter(DeploymentLaunchConfig::launchFile)).setter(setter(Builder::launchFile))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("launchFile").build()).build();

    private static final SdkField<String> POST_LAUNCH_FILE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("postLaunchFile").getter(getter(DeploymentLaunchConfig::postLaunchFile))
            .setter(setter(Builder::postLaunchFile))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("postLaunchFile").build()).build();

    private static final SdkField<Map<String, String>> ENVIRONMENT_VARIABLES_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("environmentVariables")
            .getter(getter(DeploymentLaunchConfig::environmentVariables))
            .setter(setter(Builder::environmentVariables))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("environmentVariables").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PACKAGE_NAME_FIELD,
            PRE_LAUNCH_FILE_FIELD, LAUNCH_FILE_FIELD, POST_LAUNCH_FILE_FIELD, ENVIRONMENT_VARIABLES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String packageName;

    private final String preLaunchFile;

    private final String launchFile;

    private final String postLaunchFile;

    private final Map<String, String> environmentVariables;

    private DeploymentLaunchConfig(BuilderImpl builder) {
        this.packageName = builder.packageName;
        this.preLaunchFile = builder.preLaunchFile;
        this.launchFile = builder.launchFile;
        this.postLaunchFile = builder.postLaunchFile;
        this.environmentVariables = builder.environmentVariables;
    }

    /**
     * <p>
     * The package name.
     * </p>
     * 
     * @return The package name.
     */
    public String packageName() {
        return packageName;
    }

    /**
     * <p>
     * The deployment pre-launch file. This file will be executed prior to the launch file.
     * </p>
     * 
     * @return The deployment pre-launch file. This file will be executed prior to the launch file.
     */
    public String preLaunchFile() {
        return preLaunchFile;
    }

    /**
     * <p>
     * The launch file name.
     * </p>
     * 
     * @return The launch file name.
     */
    public String launchFile() {
        return launchFile;
    }

    /**
     * <p>
     * The deployment post-launch file. This file will be executed after the launch file.
     * </p>
     * 
     * @return The deployment post-launch file. This file will be executed after the launch file.
     */
    public String postLaunchFile() {
        return postLaunchFile;
    }

    /**
     * Returns true if the EnvironmentVariables property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasEnvironmentVariables() {
        return environmentVariables != null && !(environmentVariables instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * An array of key/value pairs specifying environment variables for the robot application
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasEnvironmentVariables()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An array of key/value pairs specifying environment variables for the robot application
     */
    public Map<String, String> environmentVariables() {
        return environmentVariables;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(packageName());
        hashCode = 31 * hashCode + Objects.hashCode(preLaunchFile());
        hashCode = 31 * hashCode + Objects.hashCode(launchFile());
        hashCode = 31 * hashCode + Objects.hashCode(postLaunchFile());
        hashCode = 31 * hashCode + Objects.hashCode(hasEnvironmentVariables() ? environmentVariables() : null);
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DeploymentLaunchConfig)) {
            return false;
        }
        DeploymentLaunchConfig other = (DeploymentLaunchConfig) obj;
        return Objects.equals(packageName(), other.packageName()) && Objects.equals(preLaunchFile(), other.preLaunchFile())
                && Objects.equals(launchFile(), other.launchFile()) && Objects.equals(postLaunchFile(), other.postLaunchFile())
                && hasEnvironmentVariables() == other.hasEnvironmentVariables()
                && Objects.equals(environmentVariables(), other.environmentVariables());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("DeploymentLaunchConfig").add("PackageName", packageName()).add("PreLaunchFile", preLaunchFile())
                .add("LaunchFile", launchFile()).add("PostLaunchFile", postLaunchFile())
                .add("EnvironmentVariables", hasEnvironmentVariables() ? environmentVariables() : null).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "packageName":
            return Optional.ofNullable(clazz.cast(packageName()));
        case "preLaunchFile":
            return Optional.ofNullable(clazz.cast(preLaunchFile()));
        case "launchFile":
            return Optional.ofNullable(clazz.cast(launchFile()));
        case "postLaunchFile":
            return Optional.ofNullable(clazz.cast(postLaunchFile()));
        case "environmentVariables":
            return Optional.ofNullable(clazz.cast(environmentVariables()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DeploymentLaunchConfig, T> g) {
        return obj -> g.apply((DeploymentLaunchConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DeploymentLaunchConfig> {
        /**
         * <p>
         * The package name.
         * </p>
         * 
         * @param packageName
         *        The package name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder packageName(String packageName);

        /**
         * <p>
         * The deployment pre-launch file. This file will be executed prior to the launch file.
         * </p>
         * 
         * @param preLaunchFile
         *        The deployment pre-launch file. This file will be executed prior to the launch file.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder preLaunchFile(String preLaunchFile);

        /**
         * <p>
         * The launch file name.
         * </p>
         * 
         * @param launchFile
         *        The launch file name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder launchFile(String launchFile);

        /**
         * <p>
         * The deployment post-launch file. This file will be executed after the launch file.
         * </p>
         * 
         * @param postLaunchFile
         *        The deployment post-launch file. This file will be executed after the launch file.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder postLaunchFile(String postLaunchFile);

        /**
         * <p>
         * An array of key/value pairs specifying environment variables for the robot application
         * </p>
         * 
         * @param environmentVariables
         *        An array of key/value pairs specifying environment variables for the robot application
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environmentVariables(Map<String, String> environmentVariables);
    }

    static final class BuilderImpl implements Builder {
        private String packageName;

        private String preLaunchFile;

        private String launchFile;

        private String postLaunchFile;

        private Map<String, String> environmentVariables = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(DeploymentLaunchConfig model) {
            packageName(model.packageName);
            preLaunchFile(model.preLaunchFile);
            launchFile(model.launchFile);
            postLaunchFile(model.postLaunchFile);
            environmentVariables(model.environmentVariables);
        }

        public final String getPackageName() {
            return packageName;
        }

        @Override
        public final Builder packageName(String packageName) {
            this.packageName = packageName;
            return this;
        }

        public final void setPackageName(String packageName) {
            this.packageName = packageName;
        }

        public final String getPreLaunchFile() {
            return preLaunchFile;
        }

        @Override
        public final Builder preLaunchFile(String preLaunchFile) {
            this.preLaunchFile = preLaunchFile;
            return this;
        }

        public final void setPreLaunchFile(String preLaunchFile) {
            this.preLaunchFile = preLaunchFile;
        }

        public final String getLaunchFile() {
            return launchFile;
        }

        @Override
        public final Builder launchFile(String launchFile) {
            this.launchFile = launchFile;
            return this;
        }

        public final void setLaunchFile(String launchFile) {
            this.launchFile = launchFile;
        }

        public final String getPostLaunchFile() {
            return postLaunchFile;
        }

        @Override
        public final Builder postLaunchFile(String postLaunchFile) {
            this.postLaunchFile = postLaunchFile;
            return this;
        }

        public final void setPostLaunchFile(String postLaunchFile) {
            this.postLaunchFile = postLaunchFile;
        }

        public final Map<String, String> getEnvironmentVariables() {
            if (environmentVariables instanceof SdkAutoConstructMap) {
                return null;
            }
            return environmentVariables;
        }

        @Override
        public final Builder environmentVariables(Map<String, String> environmentVariables) {
            this.environmentVariables = EnvironmentVariableMapCopier.copy(environmentVariables);
            return this;
        }

        public final void setEnvironmentVariables(Map<String, String> environmentVariables) {
            this.environmentVariables = EnvironmentVariableMapCopier.copy(environmentVariables);
        }

        @Override
        public DeploymentLaunchConfig build() {
            return new DeploymentLaunchConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
