/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.robomaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about a launch configuration.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class LaunchConfig implements SdkPojo, Serializable, ToCopyableBuilder<LaunchConfig.Builder, LaunchConfig> {
    private static final SdkField<String> PACKAGE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("packageName").getter(getter(LaunchConfig::packageName)).setter(setter(Builder::packageName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("packageName").build()).build();

    private static final SdkField<String> LAUNCH_FILE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("launchFile").getter(getter(LaunchConfig::launchFile)).setter(setter(Builder::launchFile))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("launchFile").build()).build();

    private static final SdkField<Map<String, String>> ENVIRONMENT_VARIABLES_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("environmentVariables")
            .getter(getter(LaunchConfig::environmentVariables))
            .setter(setter(Builder::environmentVariables))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("environmentVariables").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<PortForwardingConfig> PORT_FORWARDING_CONFIG_FIELD = SdkField
            .<PortForwardingConfig> builder(MarshallingType.SDK_POJO).memberName("portForwardingConfig")
            .getter(getter(LaunchConfig::portForwardingConfig)).setter(setter(Builder::portForwardingConfig))
            .constructor(PortForwardingConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("portForwardingConfig").build())
            .build();

    private static final SdkField<Boolean> STREAM_UI_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("streamUI").getter(getter(LaunchConfig::streamUI)).setter(setter(Builder::streamUI))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("streamUI").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PACKAGE_NAME_FIELD,
            LAUNCH_FILE_FIELD, ENVIRONMENT_VARIABLES_FIELD, PORT_FORWARDING_CONFIG_FIELD, STREAM_UI_FIELD));

    private static final long serialVersionUID = 1L;

    private final String packageName;

    private final String launchFile;

    private final Map<String, String> environmentVariables;

    private final PortForwardingConfig portForwardingConfig;

    private final Boolean streamUI;

    private LaunchConfig(BuilderImpl builder) {
        this.packageName = builder.packageName;
        this.launchFile = builder.launchFile;
        this.environmentVariables = builder.environmentVariables;
        this.portForwardingConfig = builder.portForwardingConfig;
        this.streamUI = builder.streamUI;
    }

    /**
     * <p>
     * The package name.
     * </p>
     * 
     * @return The package name.
     */
    public final String packageName() {
        return packageName;
    }

    /**
     * <p>
     * The launch file name.
     * </p>
     * 
     * @return The launch file name.
     */
    public final String launchFile() {
        return launchFile;
    }

    /**
     * Returns true if the EnvironmentVariables property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasEnvironmentVariables() {
        return environmentVariables != null && !(environmentVariables instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The environment variables for the application launch.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasEnvironmentVariables()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The environment variables for the application launch.
     */
    public final Map<String, String> environmentVariables() {
        return environmentVariables;
    }

    /**
     * <p>
     * The port forwarding configuration.
     * </p>
     * 
     * @return The port forwarding configuration.
     */
    public final PortForwardingConfig portForwardingConfig() {
        return portForwardingConfig;
    }

    /**
     * <p>
     * Boolean indicating whether a streaming session will be configured for the application. If <code>True</code>, AWS
     * RoboMaker will configure a connection so you can interact with your application as it is running in the
     * simulation. You must configure and launch the component. It must have a graphical user interface.
     * </p>
     * 
     * @return Boolean indicating whether a streaming session will be configured for the application. If
     *         <code>True</code>, AWS RoboMaker will configure a connection so you can interact with your application as
     *         it is running in the simulation. You must configure and launch the component. It must have a graphical
     *         user interface.
     */
    public final Boolean streamUI() {
        return streamUI;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(packageName());
        hashCode = 31 * hashCode + Objects.hashCode(launchFile());
        hashCode = 31 * hashCode + Objects.hashCode(hasEnvironmentVariables() ? environmentVariables() : null);
        hashCode = 31 * hashCode + Objects.hashCode(portForwardingConfig());
        hashCode = 31 * hashCode + Objects.hashCode(streamUI());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof LaunchConfig)) {
            return false;
        }
        LaunchConfig other = (LaunchConfig) obj;
        return Objects.equals(packageName(), other.packageName()) && Objects.equals(launchFile(), other.launchFile())
                && hasEnvironmentVariables() == other.hasEnvironmentVariables()
                && Objects.equals(environmentVariables(), other.environmentVariables())
                && Objects.equals(portForwardingConfig(), other.portForwardingConfig())
                && Objects.equals(streamUI(), other.streamUI());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("LaunchConfig").add("PackageName", packageName()).add("LaunchFile", launchFile())
                .add("EnvironmentVariables", hasEnvironmentVariables() ? environmentVariables() : null)
                .add("PortForwardingConfig", portForwardingConfig()).add("StreamUI", streamUI()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "packageName":
            return Optional.ofNullable(clazz.cast(packageName()));
        case "launchFile":
            return Optional.ofNullable(clazz.cast(launchFile()));
        case "environmentVariables":
            return Optional.ofNullable(clazz.cast(environmentVariables()));
        case "portForwardingConfig":
            return Optional.ofNullable(clazz.cast(portForwardingConfig()));
        case "streamUI":
            return Optional.ofNullable(clazz.cast(streamUI()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<LaunchConfig, T> g) {
        return obj -> g.apply((LaunchConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, LaunchConfig> {
        /**
         * <p>
         * The package name.
         * </p>
         * 
         * @param packageName
         *        The package name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder packageName(String packageName);

        /**
         * <p>
         * The launch file name.
         * </p>
         * 
         * @param launchFile
         *        The launch file name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder launchFile(String launchFile);

        /**
         * <p>
         * The environment variables for the application launch.
         * </p>
         * 
         * @param environmentVariables
         *        The environment variables for the application launch.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environmentVariables(Map<String, String> environmentVariables);

        /**
         * <p>
         * The port forwarding configuration.
         * </p>
         * 
         * @param portForwardingConfig
         *        The port forwarding configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder portForwardingConfig(PortForwardingConfig portForwardingConfig);

        /**
         * <p>
         * The port forwarding configuration.
         * </p>
         * This is a convenience that creates an instance of the {@link PortForwardingConfig.Builder} avoiding the need
         * to create one manually via {@link PortForwardingConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link PortForwardingConfig.Builder#build()} is called immediately and
         * its result is passed to {@link #portForwardingConfig(PortForwardingConfig)}.
         * 
         * @param portForwardingConfig
         *        a consumer that will call methods on {@link PortForwardingConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #portForwardingConfig(PortForwardingConfig)
         */
        default Builder portForwardingConfig(Consumer<PortForwardingConfig.Builder> portForwardingConfig) {
            return portForwardingConfig(PortForwardingConfig.builder().applyMutation(portForwardingConfig).build());
        }

        /**
         * <p>
         * Boolean indicating whether a streaming session will be configured for the application. If <code>True</code>,
         * AWS RoboMaker will configure a connection so you can interact with your application as it is running in the
         * simulation. You must configure and launch the component. It must have a graphical user interface.
         * </p>
         * 
         * @param streamUI
         *        Boolean indicating whether a streaming session will be configured for the application. If
         *        <code>True</code>, AWS RoboMaker will configure a connection so you can interact with your application
         *        as it is running in the simulation. You must configure and launch the component. It must have a
         *        graphical user interface.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder streamUI(Boolean streamUI);
    }

    static final class BuilderImpl implements Builder {
        private String packageName;

        private String launchFile;

        private Map<String, String> environmentVariables = DefaultSdkAutoConstructMap.getInstance();

        private PortForwardingConfig portForwardingConfig;

        private Boolean streamUI;

        private BuilderImpl() {
        }

        private BuilderImpl(LaunchConfig model) {
            packageName(model.packageName);
            launchFile(model.launchFile);
            environmentVariables(model.environmentVariables);
            portForwardingConfig(model.portForwardingConfig);
            streamUI(model.streamUI);
        }

        public final String getPackageName() {
            return packageName;
        }

        @Override
        public final Builder packageName(String packageName) {
            this.packageName = packageName;
            return this;
        }

        public final void setPackageName(String packageName) {
            this.packageName = packageName;
        }

        public final String getLaunchFile() {
            return launchFile;
        }

        @Override
        public final Builder launchFile(String launchFile) {
            this.launchFile = launchFile;
            return this;
        }

        public final void setLaunchFile(String launchFile) {
            this.launchFile = launchFile;
        }

        public final Map<String, String> getEnvironmentVariables() {
            if (environmentVariables instanceof SdkAutoConstructMap) {
                return null;
            }
            return environmentVariables;
        }

        @Override
        public final Builder environmentVariables(Map<String, String> environmentVariables) {
            this.environmentVariables = EnvironmentVariableMapCopier.copy(environmentVariables);
            return this;
        }

        public final void setEnvironmentVariables(Map<String, String> environmentVariables) {
            this.environmentVariables = EnvironmentVariableMapCopier.copy(environmentVariables);
        }

        public final PortForwardingConfig.Builder getPortForwardingConfig() {
            return portForwardingConfig != null ? portForwardingConfig.toBuilder() : null;
        }

        @Override
        public final Builder portForwardingConfig(PortForwardingConfig portForwardingConfig) {
            this.portForwardingConfig = portForwardingConfig;
            return this;
        }

        public final void setPortForwardingConfig(PortForwardingConfig.BuilderImpl portForwardingConfig) {
            this.portForwardingConfig = portForwardingConfig != null ? portForwardingConfig.build() : null;
        }

        public final Boolean getStreamUI() {
            return streamUI;
        }

        @Override
        public final Builder streamUI(Boolean streamUI) {
            this.streamUI = streamUI;
            return this;
        }

        public final void setStreamUI(Boolean streamUI) {
            this.streamUI = streamUI;
        }

        @Override
        public LaunchConfig build() {
            return new LaunchConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
