/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.robomaker.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about a data source.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DataSourceConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<DataSourceConfig.Builder, DataSourceConfig> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(DataSourceConfig::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> S3_BUCKET_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("s3Bucket").getter(getter(DataSourceConfig::s3Bucket)).setter(setter(Builder::s3Bucket))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("s3Bucket").build()).build();

    private static final SdkField<List<String>> S3_KEYS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("s3Keys")
            .getter(getter(DataSourceConfig::s3Keys))
            .setter(setter(Builder::s3Keys))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("s3Keys").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("type")
            .getter(getter(DataSourceConfig::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final SdkField<String> DESTINATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("destination").getter(getter(DataSourceConfig::destination)).setter(setter(Builder::destination))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("destination").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, S3_BUCKET_FIELD,
            S3_KEYS_FIELD, TYPE_FIELD, DESTINATION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String s3Bucket;

    private final List<String> s3Keys;

    private final String type;

    private final String destination;

    private DataSourceConfig(BuilderImpl builder) {
        this.name = builder.name;
        this.s3Bucket = builder.s3Bucket;
        this.s3Keys = builder.s3Keys;
        this.type = builder.type;
        this.destination = builder.destination;
    }

    /**
     * <p>
     * The name of the data source.
     * </p>
     * 
     * @return The name of the data source.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The S3 bucket where the data files are located.
     * </p>
     * 
     * @return The S3 bucket where the data files are located.
     */
    public final String s3Bucket() {
        return s3Bucket;
    }

    /**
     * For responses, this returns true if the service returned a value for the S3Keys property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasS3Keys() {
        return s3Keys != null && !(s3Keys instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of S3 keys identifying the data source files.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasS3Keys} method.
     * </p>
     * 
     * @return The list of S3 keys identifying the data source files.
     */
    public final List<String> s3Keys() {
        return s3Keys;
    }

    /**
     * <p>
     * The data type for the data source that you're using for your container image or simulation job. You can use this
     * field to specify whether your data source is an Archive, an Amazon S3 prefix, or a file.
     * </p>
     * <p>
     * If you don't specify a field, the default value is <code>File</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link DataSourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The data type for the data source that you're using for your container image or simulation job. You can
     *         use this field to specify whether your data source is an Archive, an Amazon S3 prefix, or a file.</p>
     *         <p>
     *         If you don't specify a field, the default value is <code>File</code>.
     * @see DataSourceType
     */
    public final DataSourceType type() {
        return DataSourceType.fromValue(type);
    }

    /**
     * <p>
     * The data type for the data source that you're using for your container image or simulation job. You can use this
     * field to specify whether your data source is an Archive, an Amazon S3 prefix, or a file.
     * </p>
     * <p>
     * If you don't specify a field, the default value is <code>File</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link DataSourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The data type for the data source that you're using for your container image or simulation job. You can
     *         use this field to specify whether your data source is an Archive, an Amazon S3 prefix, or a file.</p>
     *         <p>
     *         If you don't specify a field, the default value is <code>File</code>.
     * @see DataSourceType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * The location where your files are mounted in the container image.
     * </p>
     * <p>
     * If you've specified the <code>type</code> of the data source as an <code>Archive</code>, you must provide an
     * Amazon S3 object key to your archive. The object key must point to either a <code>.zip</code> or
     * <code>.tar.gz</code> file.
     * </p>
     * <p>
     * If you've specified the <code>type</code> of the data source as a <code>Prefix</code>, you provide the Amazon S3
     * prefix that points to the files that you are using for your data source.
     * </p>
     * <p>
     * If you've specified the <code>type</code> of the data source as a <code>File</code>, you provide the Amazon S3
     * path to the file that you're using as your data source.
     * </p>
     * 
     * @return The location where your files are mounted in the container image.</p>
     *         <p>
     *         If you've specified the <code>type</code> of the data source as an <code>Archive</code>, you must provide
     *         an Amazon S3 object key to your archive. The object key must point to either a <code>.zip</code> or
     *         <code>.tar.gz</code> file.
     *         </p>
     *         <p>
     *         If you've specified the <code>type</code> of the data source as a <code>Prefix</code>, you provide the
     *         Amazon S3 prefix that points to the files that you are using for your data source.
     *         </p>
     *         <p>
     *         If you've specified the <code>type</code> of the data source as a <code>File</code>, you provide the
     *         Amazon S3 path to the file that you're using as your data source.
     */
    public final String destination() {
        return destination;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(s3Bucket());
        hashCode = 31 * hashCode + Objects.hashCode(hasS3Keys() ? s3Keys() : null);
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(destination());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DataSourceConfig)) {
            return false;
        }
        DataSourceConfig other = (DataSourceConfig) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(s3Bucket(), other.s3Bucket())
                && hasS3Keys() == other.hasS3Keys() && Objects.equals(s3Keys(), other.s3Keys())
                && Objects.equals(typeAsString(), other.typeAsString()) && Objects.equals(destination(), other.destination());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DataSourceConfig").add("Name", name()).add("S3Bucket", s3Bucket())
                .add("S3Keys", hasS3Keys() ? s3Keys() : null).add("Type", typeAsString()).add("Destination", destination())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "s3Bucket":
            return Optional.ofNullable(clazz.cast(s3Bucket()));
        case "s3Keys":
            return Optional.ofNullable(clazz.cast(s3Keys()));
        case "type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "destination":
            return Optional.ofNullable(clazz.cast(destination()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DataSourceConfig, T> g) {
        return obj -> g.apply((DataSourceConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DataSourceConfig> {
        /**
         * <p>
         * The name of the data source.
         * </p>
         * 
         * @param name
         *        The name of the data source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The S3 bucket where the data files are located.
         * </p>
         * 
         * @param s3Bucket
         *        The S3 bucket where the data files are located.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3Bucket(String s3Bucket);

        /**
         * <p>
         * The list of S3 keys identifying the data source files.
         * </p>
         * 
         * @param s3Keys
         *        The list of S3 keys identifying the data source files.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3Keys(Collection<String> s3Keys);

        /**
         * <p>
         * The list of S3 keys identifying the data source files.
         * </p>
         * 
         * @param s3Keys
         *        The list of S3 keys identifying the data source files.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3Keys(String... s3Keys);

        /**
         * <p>
         * The data type for the data source that you're using for your container image or simulation job. You can use
         * this field to specify whether your data source is an Archive, an Amazon S3 prefix, or a file.
         * </p>
         * <p>
         * If you don't specify a field, the default value is <code>File</code>.
         * </p>
         * 
         * @param type
         *        The data type for the data source that you're using for your container image or simulation job. You
         *        can use this field to specify whether your data source is an Archive, an Amazon S3 prefix, or a
         *        file.</p>
         *        <p>
         *        If you don't specify a field, the default value is <code>File</code>.
         * @see DataSourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DataSourceType
         */
        Builder type(String type);

        /**
         * <p>
         * The data type for the data source that you're using for your container image or simulation job. You can use
         * this field to specify whether your data source is an Archive, an Amazon S3 prefix, or a file.
         * </p>
         * <p>
         * If you don't specify a field, the default value is <code>File</code>.
         * </p>
         * 
         * @param type
         *        The data type for the data source that you're using for your container image or simulation job. You
         *        can use this field to specify whether your data source is an Archive, an Amazon S3 prefix, or a
         *        file.</p>
         *        <p>
         *        If you don't specify a field, the default value is <code>File</code>.
         * @see DataSourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DataSourceType
         */
        Builder type(DataSourceType type);

        /**
         * <p>
         * The location where your files are mounted in the container image.
         * </p>
         * <p>
         * If you've specified the <code>type</code> of the data source as an <code>Archive</code>, you must provide an
         * Amazon S3 object key to your archive. The object key must point to either a <code>.zip</code> or
         * <code>.tar.gz</code> file.
         * </p>
         * <p>
         * If you've specified the <code>type</code> of the data source as a <code>Prefix</code>, you provide the Amazon
         * S3 prefix that points to the files that you are using for your data source.
         * </p>
         * <p>
         * If you've specified the <code>type</code> of the data source as a <code>File</code>, you provide the Amazon
         * S3 path to the file that you're using as your data source.
         * </p>
         * 
         * @param destination
         *        The location where your files are mounted in the container image.</p>
         *        <p>
         *        If you've specified the <code>type</code> of the data source as an <code>Archive</code>, you must
         *        provide an Amazon S3 object key to your archive. The object key must point to either a
         *        <code>.zip</code> or <code>.tar.gz</code> file.
         *        </p>
         *        <p>
         *        If you've specified the <code>type</code> of the data source as a <code>Prefix</code>, you provide the
         *        Amazon S3 prefix that points to the files that you are using for your data source.
         *        </p>
         *        <p>
         *        If you've specified the <code>type</code> of the data source as a <code>File</code>, you provide the
         *        Amazon S3 path to the file that you're using as your data source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destination(String destination);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String s3Bucket;

        private List<String> s3Keys = DefaultSdkAutoConstructList.getInstance();

        private String type;

        private String destination;

        private BuilderImpl() {
        }

        private BuilderImpl(DataSourceConfig model) {
            name(model.name);
            s3Bucket(model.s3Bucket);
            s3Keys(model.s3Keys);
            type(model.type);
            destination(model.destination);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        @Transient
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getS3Bucket() {
            return s3Bucket;
        }

        public final void setS3Bucket(String s3Bucket) {
            this.s3Bucket = s3Bucket;
        }

        @Override
        @Transient
        public final Builder s3Bucket(String s3Bucket) {
            this.s3Bucket = s3Bucket;
            return this;
        }

        public final Collection<String> getS3Keys() {
            if (s3Keys instanceof SdkAutoConstructList) {
                return null;
            }
            return s3Keys;
        }

        public final void setS3Keys(Collection<String> s3Keys) {
            this.s3Keys = S3KeysOrPrefixesCopier.copy(s3Keys);
        }

        @Override
        @Transient
        public final Builder s3Keys(Collection<String> s3Keys) {
            this.s3Keys = S3KeysOrPrefixesCopier.copy(s3Keys);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder s3Keys(String... s3Keys) {
            s3Keys(Arrays.asList(s3Keys));
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        @Transient
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        @Transient
        public final Builder type(DataSourceType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final String getDestination() {
            return destination;
        }

        public final void setDestination(String destination) {
            this.destination = destination;
        }

        @Override
        @Transient
        public final Builder destination(String destination) {
            this.destination = destination;
            return this;
        }

        @Override
        public DataSourceConfig build() {
            return new DataSourceConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
