/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rolesanywhere;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.rolesanywhere.internal.RolesAnywhereServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.rolesanywhere.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.rolesanywhere.model.AccessDeniedException;
import software.amazon.awssdk.services.rolesanywhere.model.CreateProfileRequest;
import software.amazon.awssdk.services.rolesanywhere.model.CreateProfileResponse;
import software.amazon.awssdk.services.rolesanywhere.model.CreateTrustAnchorRequest;
import software.amazon.awssdk.services.rolesanywhere.model.CreateTrustAnchorResponse;
import software.amazon.awssdk.services.rolesanywhere.model.DeleteAttributeMappingRequest;
import software.amazon.awssdk.services.rolesanywhere.model.DeleteAttributeMappingResponse;
import software.amazon.awssdk.services.rolesanywhere.model.DeleteCrlRequest;
import software.amazon.awssdk.services.rolesanywhere.model.DeleteCrlResponse;
import software.amazon.awssdk.services.rolesanywhere.model.DeleteProfileRequest;
import software.amazon.awssdk.services.rolesanywhere.model.DeleteProfileResponse;
import software.amazon.awssdk.services.rolesanywhere.model.DeleteTrustAnchorRequest;
import software.amazon.awssdk.services.rolesanywhere.model.DeleteTrustAnchorResponse;
import software.amazon.awssdk.services.rolesanywhere.model.DisableCrlRequest;
import software.amazon.awssdk.services.rolesanywhere.model.DisableCrlResponse;
import software.amazon.awssdk.services.rolesanywhere.model.DisableProfileRequest;
import software.amazon.awssdk.services.rolesanywhere.model.DisableProfileResponse;
import software.amazon.awssdk.services.rolesanywhere.model.DisableTrustAnchorRequest;
import software.amazon.awssdk.services.rolesanywhere.model.DisableTrustAnchorResponse;
import software.amazon.awssdk.services.rolesanywhere.model.EnableCrlRequest;
import software.amazon.awssdk.services.rolesanywhere.model.EnableCrlResponse;
import software.amazon.awssdk.services.rolesanywhere.model.EnableProfileRequest;
import software.amazon.awssdk.services.rolesanywhere.model.EnableProfileResponse;
import software.amazon.awssdk.services.rolesanywhere.model.EnableTrustAnchorRequest;
import software.amazon.awssdk.services.rolesanywhere.model.EnableTrustAnchorResponse;
import software.amazon.awssdk.services.rolesanywhere.model.GetCrlRequest;
import software.amazon.awssdk.services.rolesanywhere.model.GetCrlResponse;
import software.amazon.awssdk.services.rolesanywhere.model.GetProfileRequest;
import software.amazon.awssdk.services.rolesanywhere.model.GetProfileResponse;
import software.amazon.awssdk.services.rolesanywhere.model.GetSubjectRequest;
import software.amazon.awssdk.services.rolesanywhere.model.GetSubjectResponse;
import software.amazon.awssdk.services.rolesanywhere.model.GetTrustAnchorRequest;
import software.amazon.awssdk.services.rolesanywhere.model.GetTrustAnchorResponse;
import software.amazon.awssdk.services.rolesanywhere.model.ImportCrlRequest;
import software.amazon.awssdk.services.rolesanywhere.model.ImportCrlResponse;
import software.amazon.awssdk.services.rolesanywhere.model.ListCrlsRequest;
import software.amazon.awssdk.services.rolesanywhere.model.ListCrlsResponse;
import software.amazon.awssdk.services.rolesanywhere.model.ListProfilesRequest;
import software.amazon.awssdk.services.rolesanywhere.model.ListProfilesResponse;
import software.amazon.awssdk.services.rolesanywhere.model.ListSubjectsRequest;
import software.amazon.awssdk.services.rolesanywhere.model.ListSubjectsResponse;
import software.amazon.awssdk.services.rolesanywhere.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.rolesanywhere.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.rolesanywhere.model.ListTrustAnchorsRequest;
import software.amazon.awssdk.services.rolesanywhere.model.ListTrustAnchorsResponse;
import software.amazon.awssdk.services.rolesanywhere.model.PutAttributeMappingRequest;
import software.amazon.awssdk.services.rolesanywhere.model.PutAttributeMappingResponse;
import software.amazon.awssdk.services.rolesanywhere.model.PutNotificationSettingsRequest;
import software.amazon.awssdk.services.rolesanywhere.model.PutNotificationSettingsResponse;
import software.amazon.awssdk.services.rolesanywhere.model.ResetNotificationSettingsRequest;
import software.amazon.awssdk.services.rolesanywhere.model.ResetNotificationSettingsResponse;
import software.amazon.awssdk.services.rolesanywhere.model.ResourceNotFoundException;
import software.amazon.awssdk.services.rolesanywhere.model.RolesAnywhereException;
import software.amazon.awssdk.services.rolesanywhere.model.TagResourceRequest;
import software.amazon.awssdk.services.rolesanywhere.model.TagResourceResponse;
import software.amazon.awssdk.services.rolesanywhere.model.TooManyTagsException;
import software.amazon.awssdk.services.rolesanywhere.model.UntagResourceRequest;
import software.amazon.awssdk.services.rolesanywhere.model.UntagResourceResponse;
import software.amazon.awssdk.services.rolesanywhere.model.UpdateCrlRequest;
import software.amazon.awssdk.services.rolesanywhere.model.UpdateCrlResponse;
import software.amazon.awssdk.services.rolesanywhere.model.UpdateProfileRequest;
import software.amazon.awssdk.services.rolesanywhere.model.UpdateProfileResponse;
import software.amazon.awssdk.services.rolesanywhere.model.UpdateTrustAnchorRequest;
import software.amazon.awssdk.services.rolesanywhere.model.UpdateTrustAnchorResponse;
import software.amazon.awssdk.services.rolesanywhere.model.ValidationException;
import software.amazon.awssdk.services.rolesanywhere.transform.CreateProfileRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.CreateTrustAnchorRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.DeleteAttributeMappingRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.DeleteCrlRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.DeleteProfileRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.DeleteTrustAnchorRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.DisableCrlRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.DisableProfileRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.DisableTrustAnchorRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.EnableCrlRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.EnableProfileRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.EnableTrustAnchorRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.GetCrlRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.GetProfileRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.GetSubjectRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.GetTrustAnchorRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.ImportCrlRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.ListCrlsRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.ListProfilesRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.ListSubjectsRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.ListTrustAnchorsRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.PutAttributeMappingRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.PutNotificationSettingsRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.ResetNotificationSettingsRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.UpdateCrlRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.UpdateProfileRequestMarshaller;
import software.amazon.awssdk.services.rolesanywhere.transform.UpdateTrustAnchorRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link RolesAnywhereClient}.
 *
 * @see RolesAnywhereClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultRolesAnywhereClient implements RolesAnywhereClient {
    private static final Logger log = Logger.loggerFor(DefaultRolesAnywhereClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultRolesAnywhereClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "RolesAnywhere" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a <i>profile</i>, a list of the roles that Roles Anywhere service is trusted to assume. You use profiles
     * to intersect permissions with IAM managed policies.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:CreateProfile</code>.
     * </p>
     *
     * @param createProfileRequest
     * @return Result of the CreateProfile operation returned by the service.
     * @throws ValidationException
     *         Validation exception error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RolesAnywhereException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RolesAnywhereClient.CreateProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/CreateProfile" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateProfileResponse createProfile(CreateProfileRequest createProfileRequest) throws ValidationException,
            AccessDeniedException, AwsServiceException, SdkClientException, RolesAnywhereException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateProfileResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateProfileResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createProfileRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateProfile");

            return clientHandler.execute(new ClientExecutionParams<CreateProfileRequest, CreateProfileResponse>()
                    .withOperationName("CreateProfile").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createProfileRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateProfileRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a trust anchor to establish trust between IAM Roles Anywhere and your certificate authority (CA). You can
     * define a trust anchor as a reference to an Private Certificate Authority (Private CA) or by uploading a CA
     * certificate. Your Amazon Web Services workloads can authenticate with the trust anchor using certificates issued
     * by the CA in exchange for temporary Amazon Web Services credentials.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:CreateTrustAnchor</code>.
     * </p>
     *
     * @param createTrustAnchorRequest
     * @return Result of the CreateTrustAnchor operation returned by the service.
     * @throws ValidationException
     *         Validation exception error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RolesAnywhereException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RolesAnywhereClient.CreateTrustAnchor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/CreateTrustAnchor"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateTrustAnchorResponse createTrustAnchor(CreateTrustAnchorRequest createTrustAnchorRequest)
            throws ValidationException, AccessDeniedException, AwsServiceException, SdkClientException, RolesAnywhereException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateTrustAnchorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateTrustAnchorResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createTrustAnchorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createTrustAnchorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateTrustAnchor");

            return clientHandler.execute(new ClientExecutionParams<CreateTrustAnchorRequest, CreateTrustAnchorResponse>()
                    .withOperationName("CreateTrustAnchor").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createTrustAnchorRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateTrustAnchorRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Delete an entry from the attribute mapping rules enforced by a given profile.
     * </p>
     *
     * @param deleteAttributeMappingRequest
     * @return Result of the DeleteAttributeMapping operation returned by the service.
     * @throws ValidationException
     *         Validation exception error.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RolesAnywhereException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RolesAnywhereClient.DeleteAttributeMapping
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/DeleteAttributeMapping"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteAttributeMappingResponse deleteAttributeMapping(DeleteAttributeMappingRequest deleteAttributeMappingRequest)
            throws ValidationException, ResourceNotFoundException, AccessDeniedException, AwsServiceException,
            SdkClientException, RolesAnywhereException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteAttributeMappingResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteAttributeMappingResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteAttributeMappingRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAttributeMappingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAttributeMapping");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteAttributeMappingRequest, DeleteAttributeMappingResponse>()
                            .withOperationName("DeleteAttributeMapping").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deleteAttributeMappingRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteAttributeMappingRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a certificate revocation list (CRL).
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:DeleteCrl</code>.
     * </p>
     *
     * @param deleteCrlRequest
     * @return Result of the DeleteCrl operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RolesAnywhereException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RolesAnywhereClient.DeleteCrl
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/DeleteCrl" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteCrlResponse deleteCrl(DeleteCrlRequest deleteCrlRequest) throws ResourceNotFoundException,
            AccessDeniedException, AwsServiceException, SdkClientException, RolesAnywhereException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteCrlResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteCrlResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteCrlRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteCrlRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteCrl");

            return clientHandler.execute(new ClientExecutionParams<DeleteCrlRequest, DeleteCrlResponse>()
                    .withOperationName("DeleteCrl").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(deleteCrlRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteCrlRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a profile.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:DeleteProfile</code>.
     * </p>
     *
     * @param deleteProfileRequest
     * @return Result of the DeleteProfile operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RolesAnywhereException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RolesAnywhereClient.DeleteProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/DeleteProfile" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteProfileResponse deleteProfile(DeleteProfileRequest deleteProfileRequest) throws ResourceNotFoundException,
            AccessDeniedException, AwsServiceException, SdkClientException, RolesAnywhereException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteProfileResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteProfileResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteProfileRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteProfile");

            return clientHandler.execute(new ClientExecutionParams<DeleteProfileRequest, DeleteProfileResponse>()
                    .withOperationName("DeleteProfile").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteProfileRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteProfileRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a trust anchor.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:DeleteTrustAnchor</code>.
     * </p>
     *
     * @param deleteTrustAnchorRequest
     * @return Result of the DeleteTrustAnchor operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RolesAnywhereException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RolesAnywhereClient.DeleteTrustAnchor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/DeleteTrustAnchor"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteTrustAnchorResponse deleteTrustAnchor(DeleteTrustAnchorRequest deleteTrustAnchorRequest)
            throws ResourceNotFoundException, AccessDeniedException, AwsServiceException, SdkClientException,
            RolesAnywhereException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteTrustAnchorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteTrustAnchorResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteTrustAnchorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTrustAnchorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTrustAnchor");

            return clientHandler.execute(new ClientExecutionParams<DeleteTrustAnchorRequest, DeleteTrustAnchorResponse>()
                    .withOperationName("DeleteTrustAnchor").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteTrustAnchorRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteTrustAnchorRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Disables a certificate revocation list (CRL).
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:DisableCrl</code>.
     * </p>
     *
     * @param disableCrlRequest
     * @return Result of the DisableCrl operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RolesAnywhereException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RolesAnywhereClient.DisableCrl
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/DisableCrl" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DisableCrlResponse disableCrl(DisableCrlRequest disableCrlRequest) throws ResourceNotFoundException,
            AccessDeniedException, AwsServiceException, SdkClientException, RolesAnywhereException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisableCrlResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DisableCrlResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disableCrlRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disableCrlRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisableCrl");

            return clientHandler.execute(new ClientExecutionParams<DisableCrlRequest, DisableCrlResponse>()
                    .withOperationName("DisableCrl").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(disableCrlRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DisableCrlRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Disables a profile. When disabled, temporary credential requests with this profile fail.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:DisableProfile</code>.
     * </p>
     *
     * @param disableProfileRequest
     * @return Result of the DisableProfile operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RolesAnywhereException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RolesAnywhereClient.DisableProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/DisableProfile" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DisableProfileResponse disableProfile(DisableProfileRequest disableProfileRequest) throws ResourceNotFoundException,
            AccessDeniedException, AwsServiceException, SdkClientException, RolesAnywhereException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisableProfileResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DisableProfileResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disableProfileRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disableProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisableProfile");

            return clientHandler.execute(new ClientExecutionParams<DisableProfileRequest, DisableProfileResponse>()
                    .withOperationName("DisableProfile").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(disableProfileRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DisableProfileRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Disables a trust anchor. When disabled, temporary credential requests specifying this trust anchor are
     * unauthorized.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:DisableTrustAnchor</code>.
     * </p>
     *
     * @param disableTrustAnchorRequest
     * @return Result of the DisableTrustAnchor operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RolesAnywhereException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RolesAnywhereClient.DisableTrustAnchor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/DisableTrustAnchor"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisableTrustAnchorResponse disableTrustAnchor(DisableTrustAnchorRequest disableTrustAnchorRequest)
            throws ResourceNotFoundException, AccessDeniedException, AwsServiceException, SdkClientException,
            RolesAnywhereException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisableTrustAnchorResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisableTrustAnchorResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disableTrustAnchorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disableTrustAnchorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisableTrustAnchor");

            return clientHandler.execute(new ClientExecutionParams<DisableTrustAnchorRequest, DisableTrustAnchorResponse>()
                    .withOperationName("DisableTrustAnchor").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(disableTrustAnchorRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DisableTrustAnchorRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Enables a certificate revocation list (CRL). When enabled, certificates stored in the CRL are unauthorized to
     * receive session credentials.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:EnableCrl</code>.
     * </p>
     *
     * @param enableCrlRequest
     * @return Result of the EnableCrl operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RolesAnywhereException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RolesAnywhereClient.EnableCrl
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/EnableCrl" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public EnableCrlResponse enableCrl(EnableCrlRequest enableCrlRequest) throws ResourceNotFoundException,
            AccessDeniedException, AwsServiceException, SdkClientException, RolesAnywhereException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<EnableCrlResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                EnableCrlResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(enableCrlRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, enableCrlRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "EnableCrl");

            return clientHandler.execute(new ClientExecutionParams<EnableCrlRequest, EnableCrlResponse>()
                    .withOperationName("EnableCrl").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(enableCrlRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new EnableCrlRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Enables temporary credential requests for a profile.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:EnableProfile</code>.
     * </p>
     *
     * @param enableProfileRequest
     * @return Result of the EnableProfile operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RolesAnywhereException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RolesAnywhereClient.EnableProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/EnableProfile" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public EnableProfileResponse enableProfile(EnableProfileRequest enableProfileRequest) throws ResourceNotFoundException,
            AccessDeniedException, AwsServiceException, SdkClientException, RolesAnywhereException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<EnableProfileResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                EnableProfileResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(enableProfileRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, enableProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "EnableProfile");

            return clientHandler.execute(new ClientExecutionParams<EnableProfileRequest, EnableProfileResponse>()
                    .withOperationName("EnableProfile").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(enableProfileRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new EnableProfileRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Enables a trust anchor. When enabled, certificates in the trust anchor chain are authorized for trust validation.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:EnableTrustAnchor</code>.
     * </p>
     *
     * @param enableTrustAnchorRequest
     * @return Result of the EnableTrustAnchor operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RolesAnywhereException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RolesAnywhereClient.EnableTrustAnchor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/EnableTrustAnchor"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public EnableTrustAnchorResponse enableTrustAnchor(EnableTrustAnchorRequest enableTrustAnchorRequest)
            throws ResourceNotFoundException, AccessDeniedException, AwsServiceException, SdkClientException,
            RolesAnywhereException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<EnableTrustAnchorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                EnableTrustAnchorResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(enableTrustAnchorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, enableTrustAnchorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "EnableTrustAnchor");

            return clientHandler.execute(new ClientExecutionParams<EnableTrustAnchorRequest, EnableTrustAnchorResponse>()
                    .withOperationName("EnableTrustAnchor").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(enableTrustAnchorRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new EnableTrustAnchorRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets a certificate revocation list (CRL).
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:GetCrl</code>.
     * </p>
     *
     * @param getCrlRequest
     * @return Result of the GetCrl operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RolesAnywhereException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RolesAnywhereClient.GetCrl
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/GetCrl" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetCrlResponse getCrl(GetCrlRequest getCrlRequest) throws ResourceNotFoundException, AwsServiceException,
            SdkClientException, RolesAnywhereException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetCrlResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetCrlResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getCrlRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getCrlRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetCrl");

            return clientHandler.execute(new ClientExecutionParams<GetCrlRequest, GetCrlResponse>().withOperationName("GetCrl")
                    .withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getCrlRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetCrlRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets a profile.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:GetProfile</code>.
     * </p>
     *
     * @param getProfileRequest
     * @return Result of the GetProfile operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RolesAnywhereException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RolesAnywhereClient.GetProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/GetProfile" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetProfileResponse getProfile(GetProfileRequest getProfileRequest) throws ResourceNotFoundException,
            AccessDeniedException, AwsServiceException, SdkClientException, RolesAnywhereException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetProfileResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetProfileResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getProfileRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetProfile");

            return clientHandler.execute(new ClientExecutionParams<GetProfileRequest, GetProfileResponse>()
                    .withOperationName("GetProfile").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getProfileRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetProfileRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets a <i>subject</i>, which associates a certificate identity with authentication attempts. The subject stores
     * auditing information such as the status of the last authentication attempt, the certificate data used in the
     * attempt, and the last time the associated identity attempted authentication.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:GetSubject</code>.
     * </p>
     *
     * @param getSubjectRequest
     * @return Result of the GetSubject operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RolesAnywhereException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RolesAnywhereClient.GetSubject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/GetSubject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetSubjectResponse getSubject(GetSubjectRequest getSubjectRequest) throws ResourceNotFoundException,
            AccessDeniedException, AwsServiceException, SdkClientException, RolesAnywhereException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetSubjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetSubjectResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getSubjectRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSubjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSubject");

            return clientHandler.execute(new ClientExecutionParams<GetSubjectRequest, GetSubjectResponse>()
                    .withOperationName("GetSubject").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getSubjectRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetSubjectRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets a trust anchor.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:GetTrustAnchor</code>.
     * </p>
     *
     * @param getTrustAnchorRequest
     * @return Result of the GetTrustAnchor operation returned by the service.
     * @throws ValidationException
     *         Validation exception error.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RolesAnywhereException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RolesAnywhereClient.GetTrustAnchor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/GetTrustAnchor" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetTrustAnchorResponse getTrustAnchor(GetTrustAnchorRequest getTrustAnchorRequest) throws ValidationException,
            ResourceNotFoundException, AccessDeniedException, AwsServiceException, SdkClientException, RolesAnywhereException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetTrustAnchorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetTrustAnchorResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getTrustAnchorRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTrustAnchorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTrustAnchor");

            return clientHandler.execute(new ClientExecutionParams<GetTrustAnchorRequest, GetTrustAnchorResponse>()
                    .withOperationName("GetTrustAnchor").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getTrustAnchorRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetTrustAnchorRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Imports the certificate revocation list (CRL). A CRL is a list of certificates that have been revoked by the
     * issuing certificate Authority (CA).In order to be properly imported, a CRL must be in PEM format. IAM Roles
     * Anywhere validates against the CRL before issuing credentials.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:ImportCrl</code>.
     * </p>
     *
     * @param importCrlRequest
     * @return Result of the ImportCrl operation returned by the service.
     * @throws ValidationException
     *         Validation exception error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RolesAnywhereException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RolesAnywhereClient.ImportCrl
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/ImportCrl" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ImportCrlResponse importCrl(ImportCrlRequest importCrlRequest) throws ValidationException, AccessDeniedException,
            AwsServiceException, SdkClientException, RolesAnywhereException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ImportCrlResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ImportCrlResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(importCrlRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, importCrlRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ImportCrl");

            return clientHandler.execute(new ClientExecutionParams<ImportCrlRequest, ImportCrlResponse>()
                    .withOperationName("ImportCrl").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(importCrlRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ImportCrlRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all certificate revocation lists (CRL) in the authenticated account and Amazon Web Services Region.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:ListCrls</code>.
     * </p>
     *
     * @param listCrlsRequest
     * @return Result of the ListCrls operation returned by the service.
     * @throws ValidationException
     *         Validation exception error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RolesAnywhereException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RolesAnywhereClient.ListCrls
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/ListCrls" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListCrlsResponse listCrls(ListCrlsRequest listCrlsRequest) throws ValidationException, AccessDeniedException,
            AwsServiceException, SdkClientException, RolesAnywhereException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListCrlsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListCrlsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listCrlsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listCrlsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListCrls");

            return clientHandler.execute(new ClientExecutionParams<ListCrlsRequest, ListCrlsResponse>()
                    .withOperationName("ListCrls").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listCrlsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListCrlsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all profiles in the authenticated account and Amazon Web Services Region.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:ListProfiles</code>.
     * </p>
     *
     * @param listProfilesRequest
     * @return Result of the ListProfiles operation returned by the service.
     * @throws ValidationException
     *         Validation exception error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RolesAnywhereException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RolesAnywhereClient.ListProfiles
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/ListProfiles" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListProfilesResponse listProfiles(ListProfilesRequest listProfilesRequest) throws ValidationException,
            AccessDeniedException, AwsServiceException, SdkClientException, RolesAnywhereException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListProfilesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListProfilesResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listProfilesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listProfilesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListProfiles");

            return clientHandler.execute(new ClientExecutionParams<ListProfilesRequest, ListProfilesResponse>()
                    .withOperationName("ListProfiles").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listProfilesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListProfilesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the subjects in the authenticated account and Amazon Web Services Region.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:ListSubjects</code>.
     * </p>
     *
     * @param listSubjectsRequest
     * @return Result of the ListSubjects operation returned by the service.
     * @throws ValidationException
     *         Validation exception error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RolesAnywhereException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RolesAnywhereClient.ListSubjects
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/ListSubjects" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListSubjectsResponse listSubjects(ListSubjectsRequest listSubjectsRequest) throws ValidationException,
            AccessDeniedException, AwsServiceException, SdkClientException, RolesAnywhereException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListSubjectsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListSubjectsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listSubjectsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSubjectsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSubjects");

            return clientHandler.execute(new ClientExecutionParams<ListSubjectsRequest, ListSubjectsResponse>()
                    .withOperationName("ListSubjects").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listSubjectsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListSubjectsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the tags attached to the resource.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:ListTagsForResource</code>.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ValidationException
     *         Validation exception error.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RolesAnywhereException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RolesAnywhereClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ValidationException, ResourceNotFoundException, AccessDeniedException, AwsServiceException,
            SdkClientException, RolesAnywhereException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the trust anchors in the authenticated account and Amazon Web Services Region.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:ListTrustAnchors</code>.
     * </p>
     *
     * @param listTrustAnchorsRequest
     * @return Result of the ListTrustAnchors operation returned by the service.
     * @throws ValidationException
     *         Validation exception error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RolesAnywhereException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RolesAnywhereClient.ListTrustAnchors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/ListTrustAnchors"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTrustAnchorsResponse listTrustAnchors(ListTrustAnchorsRequest listTrustAnchorsRequest) throws ValidationException,
            AccessDeniedException, AwsServiceException, SdkClientException, RolesAnywhereException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTrustAnchorsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListTrustAnchorsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTrustAnchorsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTrustAnchorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTrustAnchors");

            return clientHandler.execute(new ClientExecutionParams<ListTrustAnchorsRequest, ListTrustAnchorsResponse>()
                    .withOperationName("ListTrustAnchors").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTrustAnchorsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTrustAnchorsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Put an entry in the attribute mapping rules that will be enforced by a given profile. A mapping specifies a
     * certificate field and one or more specifiers that have contextual meanings.
     * </p>
     *
     * @param putAttributeMappingRequest
     * @return Result of the PutAttributeMapping operation returned by the service.
     * @throws ValidationException
     *         Validation exception error.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RolesAnywhereException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RolesAnywhereClient.PutAttributeMapping
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/PutAttributeMapping"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutAttributeMappingResponse putAttributeMapping(PutAttributeMappingRequest putAttributeMappingRequest)
            throws ValidationException, ResourceNotFoundException, AccessDeniedException, AwsServiceException,
            SdkClientException, RolesAnywhereException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutAttributeMappingResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutAttributeMappingResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putAttributeMappingRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putAttributeMappingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutAttributeMapping");

            return clientHandler.execute(new ClientExecutionParams<PutAttributeMappingRequest, PutAttributeMappingResponse>()
                    .withOperationName("PutAttributeMapping").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(putAttributeMappingRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutAttributeMappingRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Attaches a list of <i>notification settings</i> to a trust anchor.
     * </p>
     * <p>
     * A notification setting includes information such as event name, threshold, status of the notification setting,
     * and the channel to notify.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:PutNotificationSettings</code>.
     * </p>
     *
     * @param putNotificationSettingsRequest
     * @return Result of the PutNotificationSettings operation returned by the service.
     * @throws ValidationException
     *         Validation exception error.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RolesAnywhereException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RolesAnywhereClient.PutNotificationSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/PutNotificationSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutNotificationSettingsResponse putNotificationSettings(PutNotificationSettingsRequest putNotificationSettingsRequest)
            throws ValidationException, ResourceNotFoundException, AccessDeniedException, AwsServiceException,
            SdkClientException, RolesAnywhereException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutNotificationSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutNotificationSettingsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putNotificationSettingsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putNotificationSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutNotificationSettings");

            return clientHandler
                    .execute(new ClientExecutionParams<PutNotificationSettingsRequest, PutNotificationSettingsResponse>()
                            .withOperationName("PutNotificationSettings").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(putNotificationSettingsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new PutNotificationSettingsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Resets the <i>custom notification setting</i> to IAM Roles Anywhere default setting.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:ResetNotificationSettings</code>.
     * </p>
     *
     * @param resetNotificationSettingsRequest
     * @return Result of the ResetNotificationSettings operation returned by the service.
     * @throws ValidationException
     *         Validation exception error.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RolesAnywhereException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RolesAnywhereClient.ResetNotificationSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/ResetNotificationSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ResetNotificationSettingsResponse resetNotificationSettings(
            ResetNotificationSettingsRequest resetNotificationSettingsRequest) throws ValidationException,
            ResourceNotFoundException, AccessDeniedException, AwsServiceException, SdkClientException, RolesAnywhereException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ResetNotificationSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ResetNotificationSettingsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(resetNotificationSettingsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, resetNotificationSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ResetNotificationSettings");

            return clientHandler
                    .execute(new ClientExecutionParams<ResetNotificationSettingsRequest, ResetNotificationSettingsResponse>()
                            .withOperationName("ResetNotificationSettings").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(resetNotificationSettingsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ResetNotificationSettingsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Attaches tags to a resource.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:TagResource</code>.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ValidationException
     *         Validation exception error.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws TooManyTagsException
     *         Too many tags.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RolesAnywhereException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RolesAnywhereClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ValidationException,
            ResourceNotFoundException, AccessDeniedException, TooManyTagsException, AwsServiceException, SdkClientException,
            RolesAnywhereException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes tags from the resource.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:UntagResource</code>.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ValidationException
     *         Validation exception error.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RolesAnywhereException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RolesAnywhereClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ValidationException,
            ResourceNotFoundException, AccessDeniedException, AwsServiceException, SdkClientException, RolesAnywhereException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the certificate revocation list (CRL). A CRL is a list of certificates that have been revoked by the
     * issuing certificate authority (CA). IAM Roles Anywhere validates against the CRL before issuing credentials.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:UpdateCrl</code>.
     * </p>
     *
     * @param updateCrlRequest
     * @return Result of the UpdateCrl operation returned by the service.
     * @throws ValidationException
     *         Validation exception error.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RolesAnywhereException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RolesAnywhereClient.UpdateCrl
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/UpdateCrl" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateCrlResponse updateCrl(UpdateCrlRequest updateCrlRequest) throws ValidationException, ResourceNotFoundException,
            AccessDeniedException, AwsServiceException, SdkClientException, RolesAnywhereException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateCrlResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateCrlResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateCrlRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateCrlRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateCrl");

            return clientHandler.execute(new ClientExecutionParams<UpdateCrlRequest, UpdateCrlResponse>()
                    .withOperationName("UpdateCrl").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(updateCrlRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateCrlRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a <i>profile</i>, a list of the roles that IAM Roles Anywhere service is trusted to assume. You use
     * profiles to intersect permissions with IAM managed policies.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:UpdateProfile</code>.
     * </p>
     *
     * @param updateProfileRequest
     * @return Result of the UpdateProfile operation returned by the service.
     * @throws ValidationException
     *         Validation exception error.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RolesAnywhereException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RolesAnywhereClient.UpdateProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/UpdateProfile" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateProfileResponse updateProfile(UpdateProfileRequest updateProfileRequest) throws ValidationException,
            ResourceNotFoundException, AccessDeniedException, AwsServiceException, SdkClientException, RolesAnywhereException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateProfileResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateProfileResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateProfileRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateProfile");

            return clientHandler.execute(new ClientExecutionParams<UpdateProfileRequest, UpdateProfileResponse>()
                    .withOperationName("UpdateProfile").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateProfileRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateProfileRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a trust anchor. You establish trust between IAM Roles Anywhere and your certificate authority (CA) by
     * configuring a trust anchor. You can define a trust anchor as a reference to an Private Certificate Authority
     * (Private CA) or by uploading a CA certificate. Your Amazon Web Services workloads can authenticate with the trust
     * anchor using certificates issued by the CA in exchange for temporary Amazon Web Services credentials.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>rolesanywhere:UpdateTrustAnchor</code>.
     * </p>
     *
     * @param updateTrustAnchorRequest
     * @return Result of the UpdateTrustAnchor operation returned by the service.
     * @throws ValidationException
     *         Validation exception error.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RolesAnywhereException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RolesAnywhereClient.UpdateTrustAnchor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rolesanywhere-2018-05-10/UpdateTrustAnchor"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateTrustAnchorResponse updateTrustAnchor(UpdateTrustAnchorRequest updateTrustAnchorRequest)
            throws ValidationException, ResourceNotFoundException, AccessDeniedException, AwsServiceException,
            SdkClientException, RolesAnywhereException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateTrustAnchorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateTrustAnchorResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateTrustAnchorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateTrustAnchorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RolesAnywhere");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateTrustAnchor");

            return clientHandler.execute(new ClientExecutionParams<UpdateTrustAnchorRequest, UpdateTrustAnchorResponse>()
                    .withOperationName("UpdateTrustAnchor").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateTrustAnchorRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateTrustAnchorRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        RolesAnywhereServiceClientConfigurationBuilder serviceConfigBuilder = new RolesAnywhereServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(RolesAnywhereException::builder)
                .protocol(AwsJsonProtocol.REST_JSON).protocolVersion("1.1");
    }

    @Override
    public final RolesAnywhereServiceClientConfiguration serviceClientConfiguration() {
        return new RolesAnywhereServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
