/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.route53.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Optional;
import javax.annotation.Generated;
import software.amazon.awssdk.core.runtime.TypeConverter;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A complex type that contains information about the health check.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class HealthCheckConfig implements ToCopyableBuilder<HealthCheckConfig.Builder, HealthCheckConfig> {
    private final String ipAddress;

    private final Integer port;

    private final String type;

    private final String resourcePath;

    private final String fullyQualifiedDomainName;

    private final String searchString;

    private final Integer requestInterval;

    private final Integer failureThreshold;

    private final Boolean measureLatency;

    private final Boolean inverted;

    private final Integer healthThreshold;

    private final List<String> childHealthChecks;

    private final Boolean enableSNI;

    private final List<String> regions;

    private final AlarmIdentifier alarmIdentifier;

    private final String insufficientDataHealthStatus;

    private HealthCheckConfig(BuilderImpl builder) {
        this.ipAddress = builder.ipAddress;
        this.port = builder.port;
        this.type = builder.type;
        this.resourcePath = builder.resourcePath;
        this.fullyQualifiedDomainName = builder.fullyQualifiedDomainName;
        this.searchString = builder.searchString;
        this.requestInterval = builder.requestInterval;
        this.failureThreshold = builder.failureThreshold;
        this.measureLatency = builder.measureLatency;
        this.inverted = builder.inverted;
        this.healthThreshold = builder.healthThreshold;
        this.childHealthChecks = builder.childHealthChecks;
        this.enableSNI = builder.enableSNI;
        this.regions = builder.regions;
        this.alarmIdentifier = builder.alarmIdentifier;
        this.insufficientDataHealthStatus = builder.insufficientDataHealthStatus;
    }

    /**
     * <p>
     * The IPv4 or IPv6 IP address of the endpoint that you want Amazon Route 53 to perform health checks on. If you
     * don't specify a value for <code>IPAddress</code>, Amazon Route 53 sends a DNS request to resolve the domain name
     * that you specify in <code>FullyQualifiedDomainName</code> at the interval that you specify in
     * <code>RequestInterval</code>. Using an IP address returned by DNS, Amazon Route 53 then checks the health of the
     * endpoint.
     * </p>
     * <p>
     * Use one of the following formats for the value of <code>IPAddress</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>IPv4 address</b>: four values between 0 and 255, separated by periods (.), for example,
     * <code>192.0.2.44</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>IPv6 address</b>: eight groups of four hexadecimal values, separated by colons (:), for example,
     * <code>2001:0db8:85a3:0000:0000:abcd:0001:2345</code>. You can also shorten IPv6 addresses as described in RFC
     * 5952, for example, <code>2001:db8:85a3::abcd:1:2345</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the endpoint is an EC2 instance, we recommend that you create an Elastic IP address, associate it with your
     * EC2 instance, and specify the Elastic IP address for <code>IPAddress</code>. This ensures that the IP address of
     * your instance will never change.
     * </p>
     * <p>
     * For more information, see <a>HealthCheckConfig$FullyQualifiedDomainName</a>.
     * </p>
     * <p>
     * Constraints: Amazon Route 53 can't check the health of endpoints for which the IP address is in local, private,
     * non-routable, or multicast ranges. For more information about IP addresses for which you can't create health
     * checks, see the following documents:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://tools.ietf.org/html/rfc5735">RFC 5735, Special Use IPv4 Addresses</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://tools.ietf.org/html/rfc6598">RFC 6598, IANA-Reserved IPv4 Prefix for Shared Address Space</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://tools.ietf.org/html/rfc5156">RFC 5156, Special-Use IPv6 Addresses</a>
     * </p>
     * </li>
     * </ul>
     * <p>
     * When the value of <code>Type</code> is <code>CALCULATED</code> or <code>CLOUDWATCH_METRIC</code>, omit
     * <code>IPAddress</code>.
     * </p>
     * 
     * @return The IPv4 or IPv6 IP address of the endpoint that you want Amazon Route 53 to perform health checks on. If
     *         you don't specify a value for <code>IPAddress</code>, Amazon Route 53 sends a DNS request to resolve the
     *         domain name that you specify in <code>FullyQualifiedDomainName</code> at the interval that you specify in
     *         <code>RequestInterval</code>. Using an IP address returned by DNS, Amazon Route 53 then checks the health
     *         of the endpoint.</p>
     *         <p>
     *         Use one of the following formats for the value of <code>IPAddress</code>:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>IPv4 address</b>: four values between 0 and 255, separated by periods (.), for example,
     *         <code>192.0.2.44</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>IPv6 address</b>: eight groups of four hexadecimal values, separated by colons (:), for example,
     *         <code>2001:0db8:85a3:0000:0000:abcd:0001:2345</code>. You can also shorten IPv6 addresses as described in
     *         RFC 5952, for example, <code>2001:db8:85a3::abcd:1:2345</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If the endpoint is an EC2 instance, we recommend that you create an Elastic IP address, associate it with
     *         your EC2 instance, and specify the Elastic IP address for <code>IPAddress</code>. This ensures that the
     *         IP address of your instance will never change.
     *         </p>
     *         <p>
     *         For more information, see <a>HealthCheckConfig$FullyQualifiedDomainName</a>.
     *         </p>
     *         <p>
     *         Constraints: Amazon Route 53 can't check the health of endpoints for which the IP address is in local,
     *         private, non-routable, or multicast ranges. For more information about IP addresses for which you can't
     *         create health checks, see the following documents:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <a href="https://tools.ietf.org/html/rfc5735">RFC 5735, Special Use IPv4 Addresses</a>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <a href="https://tools.ietf.org/html/rfc6598">RFC 6598, IANA-Reserved IPv4 Prefix for Shared Address
     *         Space</a>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <a href="https://tools.ietf.org/html/rfc5156">RFC 5156, Special-Use IPv6 Addresses</a>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         When the value of <code>Type</code> is <code>CALCULATED</code> or <code>CLOUDWATCH_METRIC</code>, omit
     *         <code>IPAddress</code>.
     */
    public String ipAddress() {
        return ipAddress;
    }

    /**
     * <p>
     * The port on the endpoint on which you want Amazon Route 53 to perform health checks. Specify a value for
     * <code>Port</code> only when you specify a value for <code>IPAddress</code>.
     * </p>
     * 
     * @return The port on the endpoint on which you want Amazon Route 53 to perform health checks. Specify a value for
     *         <code>Port</code> only when you specify a value for <code>IPAddress</code>.
     */
    public Integer port() {
        return port;
    }

    /**
     * <p>
     * The type of health check that you want to create, which indicates how Amazon Route 53 determines whether an
     * endpoint is healthy.
     * </p>
     * <important>
     * <p>
     * You can't change the value of <code>Type</code> after you create a health check.
     * </p>
     * </important>
     * <p>
     * You can create the following types of health checks:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>HTTP</b>: Amazon Route 53 tries to establish a TCP connection. If successful, Amazon Route 53 submits an HTTP
     * request and waits for an HTTP status code of 200 or greater and less than 400.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>HTTPS</b>: Amazon Route 53 tries to establish a TCP connection. If successful, Amazon Route 53 submits an
     * HTTPS request and waits for an HTTP status code of 200 or greater and less than 400.
     * </p>
     * <important>
     * <p>
     * If you specify <code>HTTPS</code> for the value of <code>Type</code>, the endpoint must support TLS v1.0 or
     * later.
     * </p>
     * </important></li>
     * <li>
     * <p>
     * <b>HTTP_STR_MATCH</b>: Amazon Route 53 tries to establish a TCP connection. If successful, Amazon Route 53
     * submits an HTTP request and searches the first 5,120 bytes of the response body for the string that you specify
     * in <code>SearchString</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>HTTPS_STR_MATCH</b>: Amazon Route 53 tries to establish a TCP connection. If successful, Amazon Route 53
     * submits an <code>HTTPS</code> request and searches the first 5,120 bytes of the response body for the string that
     * you specify in <code>SearchString</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>TCP</b>: Amazon Route 53 tries to establish a TCP connection.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>CLOUDWATCH_METRIC</b>: The health check is associated with a CloudWatch alarm. If the state of the alarm is
     * <code>OK</code>, the health check is considered healthy. If the state is <code>ALARM</code>, the health check is
     * considered unhealthy. If CloudWatch doesn't have sufficient data to determine whether the state is
     * <code>OK</code> or <code>ALARM</code>, the health check status depends on the setting for
     * <code>InsufficientDataHealthStatus</code>: <code>Healthy</code>, <code>Unhealthy</code>, or
     * <code>LastKnownStatus</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>CALCULATED</b>: For health checks that monitor the status of other health checks, Amazon Route 53 adds up the
     * number of health checks that Amazon Route 53 health checkers consider to be healthy and compares that number with
     * the value of <code>HealthThreshold</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a href=
     * "http://docs.aws.amazon.com/Route53/latest/DeveloperGuide/dns-failover-determining-health-of-endpoints.html">How
     * Amazon Route 53 Determines Whether an Endpoint Is Healthy</a> in the <i>Amazon Route 53 Developer Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link HealthCheckType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeString}.
     * </p>
     * 
     * @return The type of health check that you want to create, which indicates how Amazon Route 53 determines whether
     *         an endpoint is healthy.</p> <important>
     *         <p>
     *         You can't change the value of <code>Type</code> after you create a health check.
     *         </p>
     *         </important>
     *         <p>
     *         You can create the following types of health checks:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>HTTP</b>: Amazon Route 53 tries to establish a TCP connection. If successful, Amazon Route 53 submits
     *         an HTTP request and waits for an HTTP status code of 200 or greater and less than 400.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>HTTPS</b>: Amazon Route 53 tries to establish a TCP connection. If successful, Amazon Route 53 submits
     *         an HTTPS request and waits for an HTTP status code of 200 or greater and less than 400.
     *         </p>
     *         <important>
     *         <p>
     *         If you specify <code>HTTPS</code> for the value of <code>Type</code>, the endpoint must support TLS v1.0
     *         or later.
     *         </p>
     *         </important></li>
     *         <li>
     *         <p>
     *         <b>HTTP_STR_MATCH</b>: Amazon Route 53 tries to establish a TCP connection. If successful, Amazon Route
     *         53 submits an HTTP request and searches the first 5,120 bytes of the response body for the string that
     *         you specify in <code>SearchString</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>HTTPS_STR_MATCH</b>: Amazon Route 53 tries to establish a TCP connection. If successful, Amazon Route
     *         53 submits an <code>HTTPS</code> request and searches the first 5,120 bytes of the response body for the
     *         string that you specify in <code>SearchString</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>TCP</b>: Amazon Route 53 tries to establish a TCP connection.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>CLOUDWATCH_METRIC</b>: The health check is associated with a CloudWatch alarm. If the state of the
     *         alarm is <code>OK</code>, the health check is considered healthy. If the state is <code>ALARM</code>, the
     *         health check is considered unhealthy. If CloudWatch doesn't have sufficient data to determine whether the
     *         state is <code>OK</code> or <code>ALARM</code>, the health check status depends on the setting for
     *         <code>InsufficientDataHealthStatus</code>: <code>Healthy</code>, <code>Unhealthy</code>, or
     *         <code>LastKnownStatus</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>CALCULATED</b>: For health checks that monitor the status of other health checks, Amazon Route 53 adds
     *         up the number of health checks that Amazon Route 53 health checkers consider to be healthy and compares
     *         that number with the value of <code>HealthThreshold</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For more information, see <a href=
     *         "http://docs.aws.amazon.com/Route53/latest/DeveloperGuide/dns-failover-determining-health-of-endpoints.html"
     *         >How Amazon Route 53 Determines Whether an Endpoint Is Healthy</a> in the <i>Amazon Route 53 Developer
     *         Guide</i>.
     * @see HealthCheckType
     */
    public HealthCheckType type() {
        return HealthCheckType.fromValue(type);
    }

    /**
     * <p>
     * The type of health check that you want to create, which indicates how Amazon Route 53 determines whether an
     * endpoint is healthy.
     * </p>
     * <important>
     * <p>
     * You can't change the value of <code>Type</code> after you create a health check.
     * </p>
     * </important>
     * <p>
     * You can create the following types of health checks:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>HTTP</b>: Amazon Route 53 tries to establish a TCP connection. If successful, Amazon Route 53 submits an HTTP
     * request and waits for an HTTP status code of 200 or greater and less than 400.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>HTTPS</b>: Amazon Route 53 tries to establish a TCP connection. If successful, Amazon Route 53 submits an
     * HTTPS request and waits for an HTTP status code of 200 or greater and less than 400.
     * </p>
     * <important>
     * <p>
     * If you specify <code>HTTPS</code> for the value of <code>Type</code>, the endpoint must support TLS v1.0 or
     * later.
     * </p>
     * </important></li>
     * <li>
     * <p>
     * <b>HTTP_STR_MATCH</b>: Amazon Route 53 tries to establish a TCP connection. If successful, Amazon Route 53
     * submits an HTTP request and searches the first 5,120 bytes of the response body for the string that you specify
     * in <code>SearchString</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>HTTPS_STR_MATCH</b>: Amazon Route 53 tries to establish a TCP connection. If successful, Amazon Route 53
     * submits an <code>HTTPS</code> request and searches the first 5,120 bytes of the response body for the string that
     * you specify in <code>SearchString</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>TCP</b>: Amazon Route 53 tries to establish a TCP connection.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>CLOUDWATCH_METRIC</b>: The health check is associated with a CloudWatch alarm. If the state of the alarm is
     * <code>OK</code>, the health check is considered healthy. If the state is <code>ALARM</code>, the health check is
     * considered unhealthy. If CloudWatch doesn't have sufficient data to determine whether the state is
     * <code>OK</code> or <code>ALARM</code>, the health check status depends on the setting for
     * <code>InsufficientDataHealthStatus</code>: <code>Healthy</code>, <code>Unhealthy</code>, or
     * <code>LastKnownStatus</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>CALCULATED</b>: For health checks that monitor the status of other health checks, Amazon Route 53 adds up the
     * number of health checks that Amazon Route 53 health checkers consider to be healthy and compares that number with
     * the value of <code>HealthThreshold</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a href=
     * "http://docs.aws.amazon.com/Route53/latest/DeveloperGuide/dns-failover-determining-health-of-endpoints.html">How
     * Amazon Route 53 Determines Whether an Endpoint Is Healthy</a> in the <i>Amazon Route 53 Developer Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link HealthCheckType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeString}.
     * </p>
     * 
     * @return The type of health check that you want to create, which indicates how Amazon Route 53 determines whether
     *         an endpoint is healthy.</p> <important>
     *         <p>
     *         You can't change the value of <code>Type</code> after you create a health check.
     *         </p>
     *         </important>
     *         <p>
     *         You can create the following types of health checks:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>HTTP</b>: Amazon Route 53 tries to establish a TCP connection. If successful, Amazon Route 53 submits
     *         an HTTP request and waits for an HTTP status code of 200 or greater and less than 400.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>HTTPS</b>: Amazon Route 53 tries to establish a TCP connection. If successful, Amazon Route 53 submits
     *         an HTTPS request and waits for an HTTP status code of 200 or greater and less than 400.
     *         </p>
     *         <important>
     *         <p>
     *         If you specify <code>HTTPS</code> for the value of <code>Type</code>, the endpoint must support TLS v1.0
     *         or later.
     *         </p>
     *         </important></li>
     *         <li>
     *         <p>
     *         <b>HTTP_STR_MATCH</b>: Amazon Route 53 tries to establish a TCP connection. If successful, Amazon Route
     *         53 submits an HTTP request and searches the first 5,120 bytes of the response body for the string that
     *         you specify in <code>SearchString</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>HTTPS_STR_MATCH</b>: Amazon Route 53 tries to establish a TCP connection. If successful, Amazon Route
     *         53 submits an <code>HTTPS</code> request and searches the first 5,120 bytes of the response body for the
     *         string that you specify in <code>SearchString</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>TCP</b>: Amazon Route 53 tries to establish a TCP connection.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>CLOUDWATCH_METRIC</b>: The health check is associated with a CloudWatch alarm. If the state of the
     *         alarm is <code>OK</code>, the health check is considered healthy. If the state is <code>ALARM</code>, the
     *         health check is considered unhealthy. If CloudWatch doesn't have sufficient data to determine whether the
     *         state is <code>OK</code> or <code>ALARM</code>, the health check status depends on the setting for
     *         <code>InsufficientDataHealthStatus</code>: <code>Healthy</code>, <code>Unhealthy</code>, or
     *         <code>LastKnownStatus</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>CALCULATED</b>: For health checks that monitor the status of other health checks, Amazon Route 53 adds
     *         up the number of health checks that Amazon Route 53 health checkers consider to be healthy and compares
     *         that number with the value of <code>HealthThreshold</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For more information, see <a href=
     *         "http://docs.aws.amazon.com/Route53/latest/DeveloperGuide/dns-failover-determining-health-of-endpoints.html"
     *         >How Amazon Route 53 Determines Whether an Endpoint Is Healthy</a> in the <i>Amazon Route 53 Developer
     *         Guide</i>.
     * @see HealthCheckType
     */
    public String typeString() {
        return type;
    }

    /**
     * <p>
     * The path, if any, that you want Amazon Route 53 to request when performing health checks. The path can be any
     * value for which your endpoint will return an HTTP status code of 2xx or 3xx when the endpoint is healthy, for
     * example, the file /docs/route53-health-check.html.
     * </p>
     * 
     * @return The path, if any, that you want Amazon Route 53 to request when performing health checks. The path can be
     *         any value for which your endpoint will return an HTTP status code of 2xx or 3xx when the endpoint is
     *         healthy, for example, the file /docs/route53-health-check.html.
     */
    public String resourcePath() {
        return resourcePath;
    }

    /**
     * <p>
     * Amazon Route 53 behavior depends on whether you specify a value for <code>IPAddress</code>.
     * </p>
     * <p>
     * <b>If you specify a value for</b> <code>IPAddress</code>:
     * </p>
     * <p>
     * Amazon Route 53 sends health check requests to the specified IPv4 or IPv6 address and passes the value of
     * <code>FullyQualifiedDomainName</code> in the <code>Host</code> header for all health checks except TCP health
     * checks. This is typically the fully qualified DNS name of the endpoint on which you want Amazon Route 53 to
     * perform health checks.
     * </p>
     * <p>
     * When Amazon Route 53 checks the health of an endpoint, here is how it constructs the <code>Host</code> header:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If you specify a value of <code>80</code> for <code>Port</code> and <code>HTTP</code> or
     * <code>HTTP_STR_MATCH</code> for <code>Type</code>, Amazon Route 53 passes the value of
     * <code>FullyQualifiedDomainName</code> to the endpoint in the Host header.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you specify a value of <code>443</code> for <code>Port</code> and <code>HTTPS</code> or
     * <code>HTTPS_STR_MATCH</code> for <code>Type</code>, Amazon Route 53 passes the value of
     * <code>FullyQualifiedDomainName</code> to the endpoint in the <code>Host</code> header.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you specify another value for <code>Port</code> and any value except <code>TCP</code> for <code>Type</code>,
     * Amazon Route 53 passes <code>FullyQualifiedDomainName:Port</code> to the endpoint in the <code>Host</code>
     * header.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If you don't specify a value for <code>FullyQualifiedDomainName</code>, Amazon Route 53 substitutes the value of
     * <code>IPAddress</code> in the <code>Host</code> header in each of the preceding cases.
     * </p>
     * <p>
     * <b>If you don't specify a value for <code>IPAddress</code> </b>:
     * </p>
     * <p>
     * Amazon Route 53 sends a DNS request to the domain that you specify for <code>FullyQualifiedDomainName</code> at
     * the interval that you specify for <code>RequestInterval</code>. Using an IPv4 address that DNS returns, Amazon
     * Route 53 then checks the health of the endpoint.
     * </p>
     * <note>
     * <p>
     * If you don't specify a value for <code>IPAddress</code>, Amazon Route 53 uses only IPv4 to send health checks to
     * the endpoint. If there's no resource record set with a type of A for the name that you specify for
     * <code>FullyQualifiedDomainName</code>, the health check fails with a "DNS resolution failed" error.
     * </p>
     * </note>
     * <p>
     * If you want to check the health of weighted, latency, or failover resource record sets and you choose to specify
     * the endpoint only by <code>FullyQualifiedDomainName</code>, we recommend that you create a separate health check
     * for each endpoint. For example, create a health check for each HTTP server that is serving content for
     * www.example.com. For the value of <code>FullyQualifiedDomainName</code>, specify the domain name of the server
     * (such as us-east-2-www.example.com), not the name of the resource record sets (www.example.com).
     * </p>
     * <important>
     * <p>
     * In this configuration, if you create a health check for which the value of <code>FullyQualifiedDomainName</code>
     * matches the name of the resource record sets and you then associate the health check with those resource record
     * sets, health check results will be unpredictable.
     * </p>
     * </important>
     * <p>
     * In addition, if the value that you specify for <code>Type</code> is <code>HTTP</code>, <code>HTTPS</code>,
     * <code>HTTP_STR_MATCH</code>, or <code>HTTPS_STR_MATCH</code>, Amazon Route 53 passes the value of
     * <code>FullyQualifiedDomainName</code> in the <code>Host</code> header, as it does when you specify a value for
     * <code>IPAddress</code>. If the value of <code>Type</code> is <code>TCP</code>, Amazon Route 53 doesn't pass a
     * <code>Host</code> header.
     * </p>
     * 
     * @return Amazon Route 53 behavior depends on whether you specify a value for <code>IPAddress</code>.</p>
     *         <p>
     *         <b>If you specify a value for</b> <code>IPAddress</code>:
     *         </p>
     *         <p>
     *         Amazon Route 53 sends health check requests to the specified IPv4 or IPv6 address and passes the value of
     *         <code>FullyQualifiedDomainName</code> in the <code>Host</code> header for all health checks except TCP
     *         health checks. This is typically the fully qualified DNS name of the endpoint on which you want Amazon
     *         Route 53 to perform health checks.
     *         </p>
     *         <p>
     *         When Amazon Route 53 checks the health of an endpoint, here is how it constructs the <code>Host</code>
     *         header:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If you specify a value of <code>80</code> for <code>Port</code> and <code>HTTP</code> or
     *         <code>HTTP_STR_MATCH</code> for <code>Type</code>, Amazon Route 53 passes the value of
     *         <code>FullyQualifiedDomainName</code> to the endpoint in the Host header.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If you specify a value of <code>443</code> for <code>Port</code> and <code>HTTPS</code> or
     *         <code>HTTPS_STR_MATCH</code> for <code>Type</code>, Amazon Route 53 passes the value of
     *         <code>FullyQualifiedDomainName</code> to the endpoint in the <code>Host</code> header.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If you specify another value for <code>Port</code> and any value except <code>TCP</code> for
     *         <code>Type</code>, Amazon Route 53 passes <code>FullyQualifiedDomainName:Port</code> to the endpoint in
     *         the <code>Host</code> header.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If you don't specify a value for <code>FullyQualifiedDomainName</code>, Amazon Route 53 substitutes the
     *         value of <code>IPAddress</code> in the <code>Host</code> header in each of the preceding cases.
     *         </p>
     *         <p>
     *         <b>If you don't specify a value for <code>IPAddress</code> </b>:
     *         </p>
     *         <p>
     *         Amazon Route 53 sends a DNS request to the domain that you specify for
     *         <code>FullyQualifiedDomainName</code> at the interval that you specify for <code>RequestInterval</code>.
     *         Using an IPv4 address that DNS returns, Amazon Route 53 then checks the health of the endpoint.
     *         </p>
     *         <note>
     *         <p>
     *         If you don't specify a value for <code>IPAddress</code>, Amazon Route 53 uses only IPv4 to send health
     *         checks to the endpoint. If there's no resource record set with a type of A for the name that you specify
     *         for <code>FullyQualifiedDomainName</code>, the health check fails with a "DNS resolution failed" error.
     *         </p>
     *         </note>
     *         <p>
     *         If you want to check the health of weighted, latency, or failover resource record sets and you choose to
     *         specify the endpoint only by <code>FullyQualifiedDomainName</code>, we recommend that you create a
     *         separate health check for each endpoint. For example, create a health check for each HTTP server that is
     *         serving content for www.example.com. For the value of <code>FullyQualifiedDomainName</code>, specify the
     *         domain name of the server (such as us-east-2-www.example.com), not the name of the resource record sets
     *         (www.example.com).
     *         </p>
     *         <important>
     *         <p>
     *         In this configuration, if you create a health check for which the value of
     *         <code>FullyQualifiedDomainName</code> matches the name of the resource record sets and you then associate
     *         the health check with those resource record sets, health check results will be unpredictable.
     *         </p>
     *         </important>
     *         <p>
     *         In addition, if the value that you specify for <code>Type</code> is <code>HTTP</code>, <code>HTTPS</code>, <code>HTTP_STR_MATCH</code>, or <code>HTTPS_STR_MATCH</code>, Amazon Route 53 passes the value of
     *         <code>FullyQualifiedDomainName</code> in the <code>Host</code> header, as it does when you specify a
     *         value for <code>IPAddress</code>. If the value of <code>Type</code> is <code>TCP</code>, Amazon Route 53
     *         doesn't pass a <code>Host</code> header.
     */
    public String fullyQualifiedDomainName() {
        return fullyQualifiedDomainName;
    }

    /**
     * <p>
     * If the value of Type is <code>HTTP_STR_MATCH</code> or <code>HTTP_STR_MATCH</code>, the string that you want
     * Amazon Route 53 to search for in the response body from the specified resource. If the string appears in the
     * response body, Amazon Route 53 considers the resource healthy.
     * </p>
     * <p>
     * Amazon Route 53 considers case when searching for <code>SearchString</code> in the response body.
     * </p>
     * 
     * @return If the value of Type is <code>HTTP_STR_MATCH</code> or <code>HTTP_STR_MATCH</code>, the string that you
     *         want Amazon Route 53 to search for in the response body from the specified resource. If the string
     *         appears in the response body, Amazon Route 53 considers the resource healthy.</p>
     *         <p>
     *         Amazon Route 53 considers case when searching for <code>SearchString</code> in the response body.
     */
    public String searchString() {
        return searchString;
    }

    /**
     * <p>
     * The number of seconds between the time that Amazon Route 53 gets a response from your endpoint and the time that
     * it sends the next health check request. Each Amazon Route 53 health checker makes requests at this interval.
     * </p>
     * <important>
     * <p>
     * You can't change the value of <code>RequestInterval</code> after you create a health check.
     * </p>
     * </important>
     * <p>
     * If you don't specify a value for <code>RequestInterval</code>, the default value is <code>30</code> seconds.
     * </p>
     * 
     * @return The number of seconds between the time that Amazon Route 53 gets a response from your endpoint and the
     *         time that it sends the next health check request. Each Amazon Route 53 health checker makes requests at
     *         this interval.</p> <important>
     *         <p>
     *         You can't change the value of <code>RequestInterval</code> after you create a health check.
     *         </p>
     *         </important>
     *         <p>
     *         If you don't specify a value for <code>RequestInterval</code>, the default value is <code>30</code>
     *         seconds.
     */
    public Integer requestInterval() {
        return requestInterval;
    }

    /**
     * <p>
     * The number of consecutive health checks that an endpoint must pass or fail for Amazon Route 53 to change the
     * current status of the endpoint from unhealthy to healthy or vice versa. For more information, see <a href=
     * "http://docs.aws.amazon.com/Route53/latest/DeveloperGuide/dns-failover-determining-health-of-endpoints.html">How
     * Amazon Route 53 Determines Whether an Endpoint Is Healthy</a> in the <i>Amazon Route 53 Developer Guide</i>.
     * </p>
     * <p>
     * If you don't specify a value for <code>FailureThreshold</code>, the default value is three health checks.
     * </p>
     * 
     * @return The number of consecutive health checks that an endpoint must pass or fail for Amazon Route 53 to change
     *         the current status of the endpoint from unhealthy to healthy or vice versa. For more information, see <a
     *         href=
     *         "http://docs.aws.amazon.com/Route53/latest/DeveloperGuide/dns-failover-determining-health-of-endpoints.html"
     *         >How Amazon Route 53 Determines Whether an Endpoint Is Healthy</a> in the <i>Amazon Route 53 Developer
     *         Guide</i>.</p>
     *         <p>
     *         If you don't specify a value for <code>FailureThreshold</code>, the default value is three health checks.
     */
    public Integer failureThreshold() {
        return failureThreshold;
    }

    /**
     * <p>
     * Specify whether you want Amazon Route 53 to measure the latency between health checkers in multiple AWS regions
     * and your endpoint, and to display CloudWatch latency graphs on the <b>Health Checks</b> page in the Amazon Route
     * 53 console.
     * </p>
     * <important>
     * <p>
     * You can't change the value of <code>MeasureLatency</code> after you create a health check.
     * </p>
     * </important>
     * 
     * @return Specify whether you want Amazon Route 53 to measure the latency between health checkers in multiple AWS
     *         regions and your endpoint, and to display CloudWatch latency graphs on the <b>Health Checks</b> page in
     *         the Amazon Route 53 console.</p> <important>
     *         <p>
     *         You can't change the value of <code>MeasureLatency</code> after you create a health check.
     *         </p>
     */
    public Boolean measureLatency() {
        return measureLatency;
    }

    /**
     * <p>
     * Specify whether you want Amazon Route 53 to invert the status of a health check, for example, to consider a
     * health check unhealthy when it otherwise would be considered healthy.
     * </p>
     * 
     * @return Specify whether you want Amazon Route 53 to invert the status of a health check, for example, to consider
     *         a health check unhealthy when it otherwise would be considered healthy.
     */
    public Boolean inverted() {
        return inverted;
    }

    /**
     * <p>
     * The number of child health checks that are associated with a <code>CALCULATED</code> health that Amazon Route 53
     * must consider healthy for the <code>CALCULATED</code> health check to be considered healthy. To specify the child
     * health checks that you want to associate with a <code>CALCULATED</code> health check, use the
     * <a>HealthCheckConfig$ChildHealthChecks</a> and <a>HealthCheckConfig$ChildHealthChecks</a> elements.
     * </p>
     * <p>
     * Note the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If you specify a number greater than the number of child health checks, Amazon Route 53 always considers this
     * health check to be unhealthy.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you specify <code>0</code>, Amazon Route 53 always considers this health check to be healthy.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The number of child health checks that are associated with a <code>CALCULATED</code> health that Amazon
     *         Route 53 must consider healthy for the <code>CALCULATED</code> health check to be considered healthy. To
     *         specify the child health checks that you want to associate with a <code>CALCULATED</code> health check,
     *         use the <a>HealthCheckConfig$ChildHealthChecks</a> and <a>HealthCheckConfig$ChildHealthChecks</a>
     *         elements.</p>
     *         <p>
     *         Note the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If you specify a number greater than the number of child health checks, Amazon Route 53 always considers
     *         this health check to be unhealthy.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If you specify <code>0</code>, Amazon Route 53 always considers this health check to be healthy.
     *         </p>
     *         </li>
     */
    public Integer healthThreshold() {
        return healthThreshold;
    }

    /**
     * <p>
     * (CALCULATED Health Checks Only) A complex type that contains one <code>ChildHealthCheck</code> element for each
     * health check that you want to associate with a <code>CALCULATED</code> health check.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return (CALCULATED Health Checks Only) A complex type that contains one <code>ChildHealthCheck</code> element
     *         for each health check that you want to associate with a <code>CALCULATED</code> health check.
     */
    public List<String> childHealthChecks() {
        return childHealthChecks;
    }

    /**
     * <p>
     * Specify whether you want Amazon Route 53 to send the value of <code>FullyQualifiedDomainName</code> to the
     * endpoint in the <code>client_hello</code> message during TLS negotiation. This allows the endpoint to respond to
     * <code>HTTPS</code> health check requests with the applicable SSL/TLS certificate.
     * </p>
     * <p>
     * Some endpoints require that <code>HTTPS</code> requests include the host name in the <code>client_hello</code>
     * message. If you don't enable SNI, the status of the health check will be <code>SSL alert handshake_failure</code>
     * . A health check can also have that status for other reasons. If SNI is enabled and you're still getting the
     * error, check the SSL/TLS configuration on your endpoint and confirm that your certificate is valid.
     * </p>
     * <p>
     * The SSL/TLS certificate on your endpoint includes a domain name in the <code>Common Name</code> field and
     * possibly several more in the <code>Subject Alternative Names</code> field. One of the domain names in the
     * certificate should match the value that you specify for <code>FullyQualifiedDomainName</code>. If the endpoint
     * responds to the <code>client_hello</code> message with a certificate that does not include the domain name that
     * you specified in <code>FullyQualifiedDomainName</code>, a health checker will retry the handshake. In the second
     * attempt, the health checker will omit <code>FullyQualifiedDomainName</code> from the <code>client_hello</code>
     * message.
     * </p>
     * 
     * @return Specify whether you want Amazon Route 53 to send the value of <code>FullyQualifiedDomainName</code> to
     *         the endpoint in the <code>client_hello</code> message during TLS negotiation. This allows the endpoint to
     *         respond to <code>HTTPS</code> health check requests with the applicable SSL/TLS certificate.</p>
     *         <p>
     *         Some endpoints require that <code>HTTPS</code> requests include the host name in the
     *         <code>client_hello</code> message. If you don't enable SNI, the status of the health check will be
     *         <code>SSL alert handshake_failure</code>. A health check can also have that status for other reasons. If
     *         SNI is enabled and you're still getting the error, check the SSL/TLS configuration on your endpoint and
     *         confirm that your certificate is valid.
     *         </p>
     *         <p>
     *         The SSL/TLS certificate on your endpoint includes a domain name in the <code>Common Name</code> field and
     *         possibly several more in the <code>Subject Alternative Names</code> field. One of the domain names in the
     *         certificate should match the value that you specify for <code>FullyQualifiedDomainName</code>. If the
     *         endpoint responds to the <code>client_hello</code> message with a certificate that does not include the
     *         domain name that you specified in <code>FullyQualifiedDomainName</code>, a health checker will retry the
     *         handshake. In the second attempt, the health checker will omit <code>FullyQualifiedDomainName</code> from
     *         the <code>client_hello</code> message.
     */
    public Boolean enableSNI() {
        return enableSNI;
    }

    /**
     * <p>
     * A complex type that contains one <code>Region</code> element for each region from which you want Amazon Route 53
     * health checkers to check the specified endpoint.
     * </p>
     * <p>
     * If you don't specify any regions, Amazon Route 53 health checkers automatically performs checks from all of the
     * regions that are listed under <b>Valid Values</b>.
     * </p>
     * <p>
     * If you update a health check to remove a region that has been performing health checks, Amazon Route 53 will
     * briefly continue to perform checks from that region to ensure that some health checkers are always checking the
     * endpoint (for example, if you replace three regions with four different regions).
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A complex type that contains one <code>Region</code> element for each region from which you want Amazon
     *         Route 53 health checkers to check the specified endpoint.</p>
     *         <p>
     *         If you don't specify any regions, Amazon Route 53 health checkers automatically performs checks from all
     *         of the regions that are listed under <b>Valid Values</b>.
     *         </p>
     *         <p>
     *         If you update a health check to remove a region that has been performing health checks, Amazon Route 53
     *         will briefly continue to perform checks from that region to ensure that some health checkers are always
     *         checking the endpoint (for example, if you replace three regions with four different regions).
     */
    public List<HealthCheckRegion> regions() {
        return TypeConverter.convert(regions, HealthCheckRegion::fromValue);
    }

    /**
     * <p>
     * A complex type that contains one <code>Region</code> element for each region from which you want Amazon Route 53
     * health checkers to check the specified endpoint.
     * </p>
     * <p>
     * If you don't specify any regions, Amazon Route 53 health checkers automatically performs checks from all of the
     * regions that are listed under <b>Valid Values</b>.
     * </p>
     * <p>
     * If you update a health check to remove a region that has been performing health checks, Amazon Route 53 will
     * briefly continue to perform checks from that region to ensure that some health checkers are always checking the
     * endpoint (for example, if you replace three regions with four different regions).
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A complex type that contains one <code>Region</code> element for each region from which you want Amazon
     *         Route 53 health checkers to check the specified endpoint.</p>
     *         <p>
     *         If you don't specify any regions, Amazon Route 53 health checkers automatically performs checks from all
     *         of the regions that are listed under <b>Valid Values</b>.
     *         </p>
     *         <p>
     *         If you update a health check to remove a region that has been performing health checks, Amazon Route 53
     *         will briefly continue to perform checks from that region to ensure that some health checkers are always
     *         checking the endpoint (for example, if you replace three regions with four different regions).
     */
    public List<String> regionsStrings() {
        return regions;
    }

    /**
     * <p>
     * A complex type that identifies the CloudWatch alarm that you want Amazon Route 53 health checkers to use to
     * determine whether this health check is healthy.
     * </p>
     * 
     * @return A complex type that identifies the CloudWatch alarm that you want Amazon Route 53 health checkers to use
     *         to determine whether this health check is healthy.
     */
    public AlarmIdentifier alarmIdentifier() {
        return alarmIdentifier;
    }

    /**
     * <p>
     * When CloudWatch has insufficient data about the metric to determine the alarm state, the status that you want
     * Amazon Route 53 to assign to the health check:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Healthy</code>: Amazon Route 53 considers the health check to be healthy.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Unhealthy</code>: Amazon Route 53 considers the health check to be unhealthy.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LastKnownStatus</code>: Amazon Route 53 uses the status of the health check from the last time that
     * CloudWatch had sufficient data to determine the alarm state. For new health checks that have no last known
     * status, the default status for the health check is healthy.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #insufficientDataHealthStatus} will return {@link InsufficientDataHealthStatus#UNKNOWN_TO_SDK_VERSION}.
     * The raw value returned by the service is available from {@link #insufficientDataHealthStatusString}.
     * </p>
     * 
     * @return When CloudWatch has insufficient data about the metric to determine the alarm state, the status that you
     *         want Amazon Route 53 to assign to the health check:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Healthy</code>: Amazon Route 53 considers the health check to be healthy.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Unhealthy</code>: Amazon Route 53 considers the health check to be unhealthy.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>LastKnownStatus</code>: Amazon Route 53 uses the status of the health check from the last time that
     *         CloudWatch had sufficient data to determine the alarm state. For new health checks that have no last
     *         known status, the default status for the health check is healthy.
     *         </p>
     *         </li>
     * @see InsufficientDataHealthStatus
     */
    public InsufficientDataHealthStatus insufficientDataHealthStatus() {
        return InsufficientDataHealthStatus.fromValue(insufficientDataHealthStatus);
    }

    /**
     * <p>
     * When CloudWatch has insufficient data about the metric to determine the alarm state, the status that you want
     * Amazon Route 53 to assign to the health check:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Healthy</code>: Amazon Route 53 considers the health check to be healthy.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Unhealthy</code>: Amazon Route 53 considers the health check to be unhealthy.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LastKnownStatus</code>: Amazon Route 53 uses the status of the health check from the last time that
     * CloudWatch had sufficient data to determine the alarm state. For new health checks that have no last known
     * status, the default status for the health check is healthy.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #insufficientDataHealthStatus} will return {@link InsufficientDataHealthStatus#UNKNOWN_TO_SDK_VERSION}.
     * The raw value returned by the service is available from {@link #insufficientDataHealthStatusString}.
     * </p>
     * 
     * @return When CloudWatch has insufficient data about the metric to determine the alarm state, the status that you
     *         want Amazon Route 53 to assign to the health check:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Healthy</code>: Amazon Route 53 considers the health check to be healthy.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Unhealthy</code>: Amazon Route 53 considers the health check to be unhealthy.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>LastKnownStatus</code>: Amazon Route 53 uses the status of the health check from the last time that
     *         CloudWatch had sufficient data to determine the alarm state. For new health checks that have no last
     *         known status, the default status for the health check is healthy.
     *         </p>
     *         </li>
     * @see InsufficientDataHealthStatus
     */
    public String insufficientDataHealthStatusString() {
        return insufficientDataHealthStatus;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + ((ipAddress() == null) ? 0 : ipAddress().hashCode());
        hashCode = 31 * hashCode + ((port() == null) ? 0 : port().hashCode());
        hashCode = 31 * hashCode + ((typeString() == null) ? 0 : typeString().hashCode());
        hashCode = 31 * hashCode + ((resourcePath() == null) ? 0 : resourcePath().hashCode());
        hashCode = 31 * hashCode + ((fullyQualifiedDomainName() == null) ? 0 : fullyQualifiedDomainName().hashCode());
        hashCode = 31 * hashCode + ((searchString() == null) ? 0 : searchString().hashCode());
        hashCode = 31 * hashCode + ((requestInterval() == null) ? 0 : requestInterval().hashCode());
        hashCode = 31 * hashCode + ((failureThreshold() == null) ? 0 : failureThreshold().hashCode());
        hashCode = 31 * hashCode + ((measureLatency() == null) ? 0 : measureLatency().hashCode());
        hashCode = 31 * hashCode + ((inverted() == null) ? 0 : inverted().hashCode());
        hashCode = 31 * hashCode + ((healthThreshold() == null) ? 0 : healthThreshold().hashCode());
        hashCode = 31 * hashCode + ((childHealthChecks() == null) ? 0 : childHealthChecks().hashCode());
        hashCode = 31 * hashCode + ((enableSNI() == null) ? 0 : enableSNI().hashCode());
        hashCode = 31 * hashCode + ((regionsStrings() == null) ? 0 : regionsStrings().hashCode());
        hashCode = 31 * hashCode + ((alarmIdentifier() == null) ? 0 : alarmIdentifier().hashCode());
        hashCode = 31 * hashCode
                + ((insufficientDataHealthStatusString() == null) ? 0 : insufficientDataHealthStatusString().hashCode());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof HealthCheckConfig)) {
            return false;
        }
        HealthCheckConfig other = (HealthCheckConfig) obj;
        if (other.ipAddress() == null ^ this.ipAddress() == null) {
            return false;
        }
        if (other.ipAddress() != null && !other.ipAddress().equals(this.ipAddress())) {
            return false;
        }
        if (other.port() == null ^ this.port() == null) {
            return false;
        }
        if (other.port() != null && !other.port().equals(this.port())) {
            return false;
        }
        if (other.typeString() == null ^ this.typeString() == null) {
            return false;
        }
        if (other.typeString() != null && !other.typeString().equals(this.typeString())) {
            return false;
        }
        if (other.resourcePath() == null ^ this.resourcePath() == null) {
            return false;
        }
        if (other.resourcePath() != null && !other.resourcePath().equals(this.resourcePath())) {
            return false;
        }
        if (other.fullyQualifiedDomainName() == null ^ this.fullyQualifiedDomainName() == null) {
            return false;
        }
        if (other.fullyQualifiedDomainName() != null && !other.fullyQualifiedDomainName().equals(this.fullyQualifiedDomainName())) {
            return false;
        }
        if (other.searchString() == null ^ this.searchString() == null) {
            return false;
        }
        if (other.searchString() != null && !other.searchString().equals(this.searchString())) {
            return false;
        }
        if (other.requestInterval() == null ^ this.requestInterval() == null) {
            return false;
        }
        if (other.requestInterval() != null && !other.requestInterval().equals(this.requestInterval())) {
            return false;
        }
        if (other.failureThreshold() == null ^ this.failureThreshold() == null) {
            return false;
        }
        if (other.failureThreshold() != null && !other.failureThreshold().equals(this.failureThreshold())) {
            return false;
        }
        if (other.measureLatency() == null ^ this.measureLatency() == null) {
            return false;
        }
        if (other.measureLatency() != null && !other.measureLatency().equals(this.measureLatency())) {
            return false;
        }
        if (other.inverted() == null ^ this.inverted() == null) {
            return false;
        }
        if (other.inverted() != null && !other.inverted().equals(this.inverted())) {
            return false;
        }
        if (other.healthThreshold() == null ^ this.healthThreshold() == null) {
            return false;
        }
        if (other.healthThreshold() != null && !other.healthThreshold().equals(this.healthThreshold())) {
            return false;
        }
        if (other.childHealthChecks() == null ^ this.childHealthChecks() == null) {
            return false;
        }
        if (other.childHealthChecks() != null && !other.childHealthChecks().equals(this.childHealthChecks())) {
            return false;
        }
        if (other.enableSNI() == null ^ this.enableSNI() == null) {
            return false;
        }
        if (other.enableSNI() != null && !other.enableSNI().equals(this.enableSNI())) {
            return false;
        }
        if (other.regionsStrings() == null ^ this.regionsStrings() == null) {
            return false;
        }
        if (other.regionsStrings() != null && !other.regionsStrings().equals(this.regionsStrings())) {
            return false;
        }
        if (other.alarmIdentifier() == null ^ this.alarmIdentifier() == null) {
            return false;
        }
        if (other.alarmIdentifier() != null && !other.alarmIdentifier().equals(this.alarmIdentifier())) {
            return false;
        }
        if (other.insufficientDataHealthStatusString() == null ^ this.insufficientDataHealthStatusString() == null) {
            return false;
        }
        if (other.insufficientDataHealthStatusString() != null
                && !other.insufficientDataHealthStatusString().equals(this.insufficientDataHealthStatusString())) {
            return false;
        }
        return true;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("{");
        if (ipAddress() != null) {
            sb.append("IPAddress: ").append(ipAddress()).append(",");
        }
        if (port() != null) {
            sb.append("Port: ").append(port()).append(",");
        }
        if (typeString() != null) {
            sb.append("Type: ").append(typeString()).append(",");
        }
        if (resourcePath() != null) {
            sb.append("ResourcePath: ").append(resourcePath()).append(",");
        }
        if (fullyQualifiedDomainName() != null) {
            sb.append("FullyQualifiedDomainName: ").append(fullyQualifiedDomainName()).append(",");
        }
        if (searchString() != null) {
            sb.append("SearchString: ").append(searchString()).append(",");
        }
        if (requestInterval() != null) {
            sb.append("RequestInterval: ").append(requestInterval()).append(",");
        }
        if (failureThreshold() != null) {
            sb.append("FailureThreshold: ").append(failureThreshold()).append(",");
        }
        if (measureLatency() != null) {
            sb.append("MeasureLatency: ").append(measureLatency()).append(",");
        }
        if (inverted() != null) {
            sb.append("Inverted: ").append(inverted()).append(",");
        }
        if (healthThreshold() != null) {
            sb.append("HealthThreshold: ").append(healthThreshold()).append(",");
        }
        if (childHealthChecks() != null) {
            sb.append("ChildHealthChecks: ").append(childHealthChecks()).append(",");
        }
        if (enableSNI() != null) {
            sb.append("EnableSNI: ").append(enableSNI()).append(",");
        }
        if (regionsStrings() != null) {
            sb.append("Regions: ").append(regionsStrings()).append(",");
        }
        if (alarmIdentifier() != null) {
            sb.append("AlarmIdentifier: ").append(alarmIdentifier()).append(",");
        }
        if (insufficientDataHealthStatusString() != null) {
            sb.append("InsufficientDataHealthStatus: ").append(insufficientDataHealthStatusString()).append(",");
        }
        if (sb.length() > 1) {
            sb.setLength(sb.length() - 1);
        }
        sb.append("}");
        return sb.toString();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "IPAddress":
            return Optional.of(clazz.cast(ipAddress()));
        case "Port":
            return Optional.of(clazz.cast(port()));
        case "Type":
            return Optional.of(clazz.cast(typeString()));
        case "ResourcePath":
            return Optional.of(clazz.cast(resourcePath()));
        case "FullyQualifiedDomainName":
            return Optional.of(clazz.cast(fullyQualifiedDomainName()));
        case "SearchString":
            return Optional.of(clazz.cast(searchString()));
        case "RequestInterval":
            return Optional.of(clazz.cast(requestInterval()));
        case "FailureThreshold":
            return Optional.of(clazz.cast(failureThreshold()));
        case "MeasureLatency":
            return Optional.of(clazz.cast(measureLatency()));
        case "Inverted":
            return Optional.of(clazz.cast(inverted()));
        case "HealthThreshold":
            return Optional.of(clazz.cast(healthThreshold()));
        case "ChildHealthChecks":
            return Optional.of(clazz.cast(childHealthChecks()));
        case "EnableSNI":
            return Optional.of(clazz.cast(enableSNI()));
        case "Regions":
            return Optional.of(clazz.cast(regionsStrings()));
        case "AlarmIdentifier":
            return Optional.of(clazz.cast(alarmIdentifier()));
        case "InsufficientDataHealthStatus":
            return Optional.of(clazz.cast(insufficientDataHealthStatusString()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends CopyableBuilder<Builder, HealthCheckConfig> {
        /**
         * <p>
         * The IPv4 or IPv6 IP address of the endpoint that you want Amazon Route 53 to perform health checks on. If you
         * don't specify a value for <code>IPAddress</code>, Amazon Route 53 sends a DNS request to resolve the domain
         * name that you specify in <code>FullyQualifiedDomainName</code> at the interval that you specify in
         * <code>RequestInterval</code>. Using an IP address returned by DNS, Amazon Route 53 then checks the health of
         * the endpoint.
         * </p>
         * <p>
         * Use one of the following formats for the value of <code>IPAddress</code>:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>IPv4 address</b>: four values between 0 and 255, separated by periods (.), for example,
         * <code>192.0.2.44</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>IPv6 address</b>: eight groups of four hexadecimal values, separated by colons (:), for example,
         * <code>2001:0db8:85a3:0000:0000:abcd:0001:2345</code>. You can also shorten IPv6 addresses as described in RFC
         * 5952, for example, <code>2001:db8:85a3::abcd:1:2345</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If the endpoint is an EC2 instance, we recommend that you create an Elastic IP address, associate it with
         * your EC2 instance, and specify the Elastic IP address for <code>IPAddress</code>. This ensures that the IP
         * address of your instance will never change.
         * </p>
         * <p>
         * For more information, see <a>HealthCheckConfig$FullyQualifiedDomainName</a>.
         * </p>
         * <p>
         * Constraints: Amazon Route 53 can't check the health of endpoints for which the IP address is in local,
         * private, non-routable, or multicast ranges. For more information about IP addresses for which you can't
         * create health checks, see the following documents:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <a href="https://tools.ietf.org/html/rfc5735">RFC 5735, Special Use IPv4 Addresses</a>
         * </p>
         * </li>
         * <li>
         * <p>
         * <a href="https://tools.ietf.org/html/rfc6598">RFC 6598, IANA-Reserved IPv4 Prefix for Shared Address
         * Space</a>
         * </p>
         * </li>
         * <li>
         * <p>
         * <a href="https://tools.ietf.org/html/rfc5156">RFC 5156, Special-Use IPv6 Addresses</a>
         * </p>
         * </li>
         * </ul>
         * <p>
         * When the value of <code>Type</code> is <code>CALCULATED</code> or <code>CLOUDWATCH_METRIC</code>, omit
         * <code>IPAddress</code>.
         * </p>
         * 
         * @param ipAddress
         *        The IPv4 or IPv6 IP address of the endpoint that you want Amazon Route 53 to perform health checks on.
         *        If you don't specify a value for <code>IPAddress</code>, Amazon Route 53 sends a DNS request to
         *        resolve the domain name that you specify in <code>FullyQualifiedDomainName</code> at the interval that
         *        you specify in <code>RequestInterval</code>. Using an IP address returned by DNS, Amazon Route 53 then
         *        checks the health of the endpoint.</p>
         *        <p>
         *        Use one of the following formats for the value of <code>IPAddress</code>:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>IPv4 address</b>: four values between 0 and 255, separated by periods (.), for example,
         *        <code>192.0.2.44</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>IPv6 address</b>: eight groups of four hexadecimal values, separated by colons (:), for example,
         *        <code>2001:0db8:85a3:0000:0000:abcd:0001:2345</code>. You can also shorten IPv6 addresses as described
         *        in RFC 5952, for example, <code>2001:db8:85a3::abcd:1:2345</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If the endpoint is an EC2 instance, we recommend that you create an Elastic IP address, associate it
         *        with your EC2 instance, and specify the Elastic IP address for <code>IPAddress</code>. This ensures
         *        that the IP address of your instance will never change.
         *        </p>
         *        <p>
         *        For more information, see <a>HealthCheckConfig$FullyQualifiedDomainName</a>.
         *        </p>
         *        <p>
         *        Constraints: Amazon Route 53 can't check the health of endpoints for which the IP address is in local,
         *        private, non-routable, or multicast ranges. For more information about IP addresses for which you
         *        can't create health checks, see the following documents:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <a href="https://tools.ietf.org/html/rfc5735">RFC 5735, Special Use IPv4 Addresses</a>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <a href="https://tools.ietf.org/html/rfc6598">RFC 6598, IANA-Reserved IPv4 Prefix for Shared Address
         *        Space</a>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <a href="https://tools.ietf.org/html/rfc5156">RFC 5156, Special-Use IPv6 Addresses</a>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        When the value of <code>Type</code> is <code>CALCULATED</code> or <code>CLOUDWATCH_METRIC</code>, omit
         *        <code>IPAddress</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipAddress(String ipAddress);

        /**
         * <p>
         * The port on the endpoint on which you want Amazon Route 53 to perform health checks. Specify a value for
         * <code>Port</code> only when you specify a value for <code>IPAddress</code>.
         * </p>
         * 
         * @param port
         *        The port on the endpoint on which you want Amazon Route 53 to perform health checks. Specify a value
         *        for <code>Port</code> only when you specify a value for <code>IPAddress</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder port(Integer port);

        /**
         * <p>
         * The type of health check that you want to create, which indicates how Amazon Route 53 determines whether an
         * endpoint is healthy.
         * </p>
         * <important>
         * <p>
         * You can't change the value of <code>Type</code> after you create a health check.
         * </p>
         * </important>
         * <p>
         * You can create the following types of health checks:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>HTTP</b>: Amazon Route 53 tries to establish a TCP connection. If successful, Amazon Route 53 submits an
         * HTTP request and waits for an HTTP status code of 200 or greater and less than 400.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>HTTPS</b>: Amazon Route 53 tries to establish a TCP connection. If successful, Amazon Route 53 submits an
         * HTTPS request and waits for an HTTP status code of 200 or greater and less than 400.
         * </p>
         * <important>
         * <p>
         * If you specify <code>HTTPS</code> for the value of <code>Type</code>, the endpoint must support TLS v1.0 or
         * later.
         * </p>
         * </important></li>
         * <li>
         * <p>
         * <b>HTTP_STR_MATCH</b>: Amazon Route 53 tries to establish a TCP connection. If successful, Amazon Route 53
         * submits an HTTP request and searches the first 5,120 bytes of the response body for the string that you
         * specify in <code>SearchString</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>HTTPS_STR_MATCH</b>: Amazon Route 53 tries to establish a TCP connection. If successful, Amazon Route 53
         * submits an <code>HTTPS</code> request and searches the first 5,120 bytes of the response body for the string
         * that you specify in <code>SearchString</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>TCP</b>: Amazon Route 53 tries to establish a TCP connection.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>CLOUDWATCH_METRIC</b>: The health check is associated with a CloudWatch alarm. If the state of the alarm
         * is <code>OK</code>, the health check is considered healthy. If the state is <code>ALARM</code>, the health
         * check is considered unhealthy. If CloudWatch doesn't have sufficient data to determine whether the state is
         * <code>OK</code> or <code>ALARM</code>, the health check status depends on the setting for
         * <code>InsufficientDataHealthStatus</code>: <code>Healthy</code>, <code>Unhealthy</code>, or
         * <code>LastKnownStatus</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>CALCULATED</b>: For health checks that monitor the status of other health checks, Amazon Route 53 adds up
         * the number of health checks that Amazon Route 53 health checkers consider to be healthy and compares that
         * number with the value of <code>HealthThreshold</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For more information, see <a href=
         * "http://docs.aws.amazon.com/Route53/latest/DeveloperGuide/dns-failover-determining-health-of-endpoints.html"
         * >How Amazon Route 53 Determines Whether an Endpoint Is Healthy</a> in the <i>Amazon Route 53 Developer
         * Guide</i>.
         * </p>
         * 
         * @param type
         *        The type of health check that you want to create, which indicates how Amazon Route 53 determines
         *        whether an endpoint is healthy.</p> <important>
         *        <p>
         *        You can't change the value of <code>Type</code> after you create a health check.
         *        </p>
         *        </important>
         *        <p>
         *        You can create the following types of health checks:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>HTTP</b>: Amazon Route 53 tries to establish a TCP connection. If successful, Amazon Route 53
         *        submits an HTTP request and waits for an HTTP status code of 200 or greater and less than 400.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>HTTPS</b>: Amazon Route 53 tries to establish a TCP connection. If successful, Amazon Route 53
         *        submits an HTTPS request and waits for an HTTP status code of 200 or greater and less than 400.
         *        </p>
         *        <important>
         *        <p>
         *        If you specify <code>HTTPS</code> for the value of <code>Type</code>, the endpoint must support TLS
         *        v1.0 or later.
         *        </p>
         *        </important></li>
         *        <li>
         *        <p>
         *        <b>HTTP_STR_MATCH</b>: Amazon Route 53 tries to establish a TCP connection. If successful, Amazon
         *        Route 53 submits an HTTP request and searches the first 5,120 bytes of the response body for the
         *        string that you specify in <code>SearchString</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>HTTPS_STR_MATCH</b>: Amazon Route 53 tries to establish a TCP connection. If successful, Amazon
         *        Route 53 submits an <code>HTTPS</code> request and searches the first 5,120 bytes of the response body
         *        for the string that you specify in <code>SearchString</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>TCP</b>: Amazon Route 53 tries to establish a TCP connection.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>CLOUDWATCH_METRIC</b>: The health check is associated with a CloudWatch alarm. If the state of the
         *        alarm is <code>OK</code>, the health check is considered healthy. If the state is <code>ALARM</code>,
         *        the health check is considered unhealthy. If CloudWatch doesn't have sufficient data to determine
         *        whether the state is <code>OK</code> or <code>ALARM</code>, the health check status depends on the
         *        setting for <code>InsufficientDataHealthStatus</code>: <code>Healthy</code>, <code>Unhealthy</code>,
         *        or <code>LastKnownStatus</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>CALCULATED</b>: For health checks that monitor the status of other health checks, Amazon Route 53
         *        adds up the number of health checks that Amazon Route 53 health checkers consider to be healthy and
         *        compares that number with the value of <code>HealthThreshold</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For more information, see <a href=
         *        "http://docs.aws.amazon.com/Route53/latest/DeveloperGuide/dns-failover-determining-health-of-endpoints.html"
         *        >How Amazon Route 53 Determines Whether an Endpoint Is Healthy</a> in the <i>Amazon Route 53 Developer
         *        Guide</i>.
         * @see HealthCheckType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HealthCheckType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of health check that you want to create, which indicates how Amazon Route 53 determines whether an
         * endpoint is healthy.
         * </p>
         * <important>
         * <p>
         * You can't change the value of <code>Type</code> after you create a health check.
         * </p>
         * </important>
         * <p>
         * You can create the following types of health checks:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>HTTP</b>: Amazon Route 53 tries to establish a TCP connection. If successful, Amazon Route 53 submits an
         * HTTP request and waits for an HTTP status code of 200 or greater and less than 400.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>HTTPS</b>: Amazon Route 53 tries to establish a TCP connection. If successful, Amazon Route 53 submits an
         * HTTPS request and waits for an HTTP status code of 200 or greater and less than 400.
         * </p>
         * <important>
         * <p>
         * If you specify <code>HTTPS</code> for the value of <code>Type</code>, the endpoint must support TLS v1.0 or
         * later.
         * </p>
         * </important></li>
         * <li>
         * <p>
         * <b>HTTP_STR_MATCH</b>: Amazon Route 53 tries to establish a TCP connection. If successful, Amazon Route 53
         * submits an HTTP request and searches the first 5,120 bytes of the response body for the string that you
         * specify in <code>SearchString</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>HTTPS_STR_MATCH</b>: Amazon Route 53 tries to establish a TCP connection. If successful, Amazon Route 53
         * submits an <code>HTTPS</code> request and searches the first 5,120 bytes of the response body for the string
         * that you specify in <code>SearchString</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>TCP</b>: Amazon Route 53 tries to establish a TCP connection.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>CLOUDWATCH_METRIC</b>: The health check is associated with a CloudWatch alarm. If the state of the alarm
         * is <code>OK</code>, the health check is considered healthy. If the state is <code>ALARM</code>, the health
         * check is considered unhealthy. If CloudWatch doesn't have sufficient data to determine whether the state is
         * <code>OK</code> or <code>ALARM</code>, the health check status depends on the setting for
         * <code>InsufficientDataHealthStatus</code>: <code>Healthy</code>, <code>Unhealthy</code>, or
         * <code>LastKnownStatus</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>CALCULATED</b>: For health checks that monitor the status of other health checks, Amazon Route 53 adds up
         * the number of health checks that Amazon Route 53 health checkers consider to be healthy and compares that
         * number with the value of <code>HealthThreshold</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For more information, see <a href=
         * "http://docs.aws.amazon.com/Route53/latest/DeveloperGuide/dns-failover-determining-health-of-endpoints.html"
         * >How Amazon Route 53 Determines Whether an Endpoint Is Healthy</a> in the <i>Amazon Route 53 Developer
         * Guide</i>.
         * </p>
         * 
         * @param type
         *        The type of health check that you want to create, which indicates how Amazon Route 53 determines
         *        whether an endpoint is healthy.</p> <important>
         *        <p>
         *        You can't change the value of <code>Type</code> after you create a health check.
         *        </p>
         *        </important>
         *        <p>
         *        You can create the following types of health checks:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>HTTP</b>: Amazon Route 53 tries to establish a TCP connection. If successful, Amazon Route 53
         *        submits an HTTP request and waits for an HTTP status code of 200 or greater and less than 400.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>HTTPS</b>: Amazon Route 53 tries to establish a TCP connection. If successful, Amazon Route 53
         *        submits an HTTPS request and waits for an HTTP status code of 200 or greater and less than 400.
         *        </p>
         *        <important>
         *        <p>
         *        If you specify <code>HTTPS</code> for the value of <code>Type</code>, the endpoint must support TLS
         *        v1.0 or later.
         *        </p>
         *        </important></li>
         *        <li>
         *        <p>
         *        <b>HTTP_STR_MATCH</b>: Amazon Route 53 tries to establish a TCP connection. If successful, Amazon
         *        Route 53 submits an HTTP request and searches the first 5,120 bytes of the response body for the
         *        string that you specify in <code>SearchString</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>HTTPS_STR_MATCH</b>: Amazon Route 53 tries to establish a TCP connection. If successful, Amazon
         *        Route 53 submits an <code>HTTPS</code> request and searches the first 5,120 bytes of the response body
         *        for the string that you specify in <code>SearchString</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>TCP</b>: Amazon Route 53 tries to establish a TCP connection.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>CLOUDWATCH_METRIC</b>: The health check is associated with a CloudWatch alarm. If the state of the
         *        alarm is <code>OK</code>, the health check is considered healthy. If the state is <code>ALARM</code>,
         *        the health check is considered unhealthy. If CloudWatch doesn't have sufficient data to determine
         *        whether the state is <code>OK</code> or <code>ALARM</code>, the health check status depends on the
         *        setting for <code>InsufficientDataHealthStatus</code>: <code>Healthy</code>, <code>Unhealthy</code>,
         *        or <code>LastKnownStatus</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>CALCULATED</b>: For health checks that monitor the status of other health checks, Amazon Route 53
         *        adds up the number of health checks that Amazon Route 53 health checkers consider to be healthy and
         *        compares that number with the value of <code>HealthThreshold</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For more information, see <a href=
         *        "http://docs.aws.amazon.com/Route53/latest/DeveloperGuide/dns-failover-determining-health-of-endpoints.html"
         *        >How Amazon Route 53 Determines Whether an Endpoint Is Healthy</a> in the <i>Amazon Route 53 Developer
         *        Guide</i>.
         * @see HealthCheckType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HealthCheckType
         */
        Builder type(HealthCheckType type);

        /**
         * <p>
         * The path, if any, that you want Amazon Route 53 to request when performing health checks. The path can be any
         * value for which your endpoint will return an HTTP status code of 2xx or 3xx when the endpoint is healthy, for
         * example, the file /docs/route53-health-check.html.
         * </p>
         * 
         * @param resourcePath
         *        The path, if any, that you want Amazon Route 53 to request when performing health checks. The path can
         *        be any value for which your endpoint will return an HTTP status code of 2xx or 3xx when the endpoint
         *        is healthy, for example, the file /docs/route53-health-check.html.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourcePath(String resourcePath);

        /**
         * <p>
         * Amazon Route 53 behavior depends on whether you specify a value for <code>IPAddress</code>.
         * </p>
         * <p>
         * <b>If you specify a value for</b> <code>IPAddress</code>:
         * </p>
         * <p>
         * Amazon Route 53 sends health check requests to the specified IPv4 or IPv6 address and passes the value of
         * <code>FullyQualifiedDomainName</code> in the <code>Host</code> header for all health checks except TCP health
         * checks. This is typically the fully qualified DNS name of the endpoint on which you want Amazon Route 53 to
         * perform health checks.
         * </p>
         * <p>
         * When Amazon Route 53 checks the health of an endpoint, here is how it constructs the <code>Host</code>
         * header:
         * </p>
         * <ul>
         * <li>
         * <p>
         * If you specify a value of <code>80</code> for <code>Port</code> and <code>HTTP</code> or
         * <code>HTTP_STR_MATCH</code> for <code>Type</code>, Amazon Route 53 passes the value of
         * <code>FullyQualifiedDomainName</code> to the endpoint in the Host header.
         * </p>
         * </li>
         * <li>
         * <p>
         * If you specify a value of <code>443</code> for <code>Port</code> and <code>HTTPS</code> or
         * <code>HTTPS_STR_MATCH</code> for <code>Type</code>, Amazon Route 53 passes the value of
         * <code>FullyQualifiedDomainName</code> to the endpoint in the <code>Host</code> header.
         * </p>
         * </li>
         * <li>
         * <p>
         * If you specify another value for <code>Port</code> and any value except <code>TCP</code> for
         * <code>Type</code>, Amazon Route 53 passes <code>FullyQualifiedDomainName:Port</code> to the endpoint in the
         * <code>Host</code> header.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If you don't specify a value for <code>FullyQualifiedDomainName</code>, Amazon Route 53 substitutes the value
         * of <code>IPAddress</code> in the <code>Host</code> header in each of the preceding cases.
         * </p>
         * <p>
         * <b>If you don't specify a value for <code>IPAddress</code> </b>:
         * </p>
         * <p>
         * Amazon Route 53 sends a DNS request to the domain that you specify for <code>FullyQualifiedDomainName</code>
         * at the interval that you specify for <code>RequestInterval</code>. Using an IPv4 address that DNS returns,
         * Amazon Route 53 then checks the health of the endpoint.
         * </p>
         * <note>
         * <p>
         * If you don't specify a value for <code>IPAddress</code>, Amazon Route 53 uses only IPv4 to send health checks
         * to the endpoint. If there's no resource record set with a type of A for the name that you specify for
         * <code>FullyQualifiedDomainName</code>, the health check fails with a "DNS resolution failed" error.
         * </p>
         * </note>
         * <p>
         * If you want to check the health of weighted, latency, or failover resource record sets and you choose to
         * specify the endpoint only by <code>FullyQualifiedDomainName</code>, we recommend that you create a separate
         * health check for each endpoint. For example, create a health check for each HTTP server that is serving
         * content for www.example.com. For the value of <code>FullyQualifiedDomainName</code>, specify the domain name
         * of the server (such as us-east-2-www.example.com), not the name of the resource record sets
         * (www.example.com).
         * </p>
         * <important>
         * <p>
         * In this configuration, if you create a health check for which the value of
         * <code>FullyQualifiedDomainName</code> matches the name of the resource record sets and you then associate the
         * health check with those resource record sets, health check results will be unpredictable.
         * </p>
         * </important>
         * <p>
         * In addition, if the value that you specify for <code>Type</code> is <code>HTTP</code>, <code>HTTPS</code>,
         * <code>HTTP_STR_MATCH</code>, or <code>HTTPS_STR_MATCH</code>, Amazon Route 53 passes the value of
         * <code>FullyQualifiedDomainName</code> in the <code>Host</code> header, as it does when you specify a value
         * for <code>IPAddress</code>. If the value of <code>Type</code> is <code>TCP</code>, Amazon Route 53 doesn't
         * pass a <code>Host</code> header.
         * </p>
         * 
         * @param fullyQualifiedDomainName
         *        Amazon Route 53 behavior depends on whether you specify a value for <code>IPAddress</code>.</p>
         *        <p>
         *        <b>If you specify a value for</b> <code>IPAddress</code>:
         *        </p>
         *        <p>
         *        Amazon Route 53 sends health check requests to the specified IPv4 or IPv6 address and passes the value
         *        of <code>FullyQualifiedDomainName</code> in the <code>Host</code> header for all health checks except
         *        TCP health checks. This is typically the fully qualified DNS name of the endpoint on which you want
         *        Amazon Route 53 to perform health checks.
         *        </p>
         *        <p>
         *        When Amazon Route 53 checks the health of an endpoint, here is how it constructs the <code>Host</code>
         *        header:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If you specify a value of <code>80</code> for <code>Port</code> and <code>HTTP</code> or
         *        <code>HTTP_STR_MATCH</code> for <code>Type</code>, Amazon Route 53 passes the value of
         *        <code>FullyQualifiedDomainName</code> to the endpoint in the Host header.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you specify a value of <code>443</code> for <code>Port</code> and <code>HTTPS</code> or
         *        <code>HTTPS_STR_MATCH</code> for <code>Type</code>, Amazon Route 53 passes the value of
         *        <code>FullyQualifiedDomainName</code> to the endpoint in the <code>Host</code> header.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you specify another value for <code>Port</code> and any value except <code>TCP</code> for
         *        <code>Type</code>, Amazon Route 53 passes <code>FullyQualifiedDomainName:Port</code> to the endpoint
         *        in the <code>Host</code> header.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If you don't specify a value for <code>FullyQualifiedDomainName</code>, Amazon Route 53 substitutes
         *        the value of <code>IPAddress</code> in the <code>Host</code> header in each of the preceding cases.
         *        </p>
         *        <p>
         *        <b>If you don't specify a value for <code>IPAddress</code> </b>:
         *        </p>
         *        <p>
         *        Amazon Route 53 sends a DNS request to the domain that you specify for
         *        <code>FullyQualifiedDomainName</code> at the interval that you specify for
         *        <code>RequestInterval</code>. Using an IPv4 address that DNS returns, Amazon Route 53 then checks the
         *        health of the endpoint.
         *        </p>
         *        <note>
         *        <p>
         *        If you don't specify a value for <code>IPAddress</code>, Amazon Route 53 uses only IPv4 to send health
         *        checks to the endpoint. If there's no resource record set with a type of A for the name that you
         *        specify for <code>FullyQualifiedDomainName</code>, the health check fails with a
         *        "DNS resolution failed" error.
         *        </p>
         *        </note>
         *        <p>
         *        If you want to check the health of weighted, latency, or failover resource record sets and you choose
         *        to specify the endpoint only by <code>FullyQualifiedDomainName</code>, we recommend that you create a
         *        separate health check for each endpoint. For example, create a health check for each HTTP server that
         *        is serving content for www.example.com. For the value of <code>FullyQualifiedDomainName</code>,
         *        specify the domain name of the server (such as us-east-2-www.example.com), not the name of the
         *        resource record sets (www.example.com).
         *        </p>
         *        <important>
         *        <p>
         *        In this configuration, if you create a health check for which the value of
         *        <code>FullyQualifiedDomainName</code> matches the name of the resource record sets and you then
         *        associate the health check with those resource record sets, health check results will be
         *        unpredictable.
         *        </p>
         *        </important>
         *        <p>
         *        In addition, if the value that you specify for <code>Type</code> is <code>HTTP</code>,
         *        <code>HTTPS</code>, <code>HTTP_STR_MATCH</code>, or <code>HTTPS_STR_MATCH</code>, Amazon Route 53
         *        passes the value of <code>FullyQualifiedDomainName</code> in the <code>Host</code> header, as it does
         *        when you specify a value for <code>IPAddress</code>. If the value of <code>Type</code> is
         *        <code>TCP</code>, Amazon Route 53 doesn't pass a <code>Host</code> header.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fullyQualifiedDomainName(String fullyQualifiedDomainName);

        /**
         * <p>
         * If the value of Type is <code>HTTP_STR_MATCH</code> or <code>HTTP_STR_MATCH</code>, the string that you want
         * Amazon Route 53 to search for in the response body from the specified resource. If the string appears in the
         * response body, Amazon Route 53 considers the resource healthy.
         * </p>
         * <p>
         * Amazon Route 53 considers case when searching for <code>SearchString</code> in the response body.
         * </p>
         * 
         * @param searchString
         *        If the value of Type is <code>HTTP_STR_MATCH</code> or <code>HTTP_STR_MATCH</code>, the string that
         *        you want Amazon Route 53 to search for in the response body from the specified resource. If the string
         *        appears in the response body, Amazon Route 53 considers the resource healthy.</p>
         *        <p>
         *        Amazon Route 53 considers case when searching for <code>SearchString</code> in the response body.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder searchString(String searchString);

        /**
         * <p>
         * The number of seconds between the time that Amazon Route 53 gets a response from your endpoint and the time
         * that it sends the next health check request. Each Amazon Route 53 health checker makes requests at this
         * interval.
         * </p>
         * <important>
         * <p>
         * You can't change the value of <code>RequestInterval</code> after you create a health check.
         * </p>
         * </important>
         * <p>
         * If you don't specify a value for <code>RequestInterval</code>, the default value is <code>30</code> seconds.
         * </p>
         * 
         * @param requestInterval
         *        The number of seconds between the time that Amazon Route 53 gets a response from your endpoint and the
         *        time that it sends the next health check request. Each Amazon Route 53 health checker makes requests
         *        at this interval.</p> <important>
         *        <p>
         *        You can't change the value of <code>RequestInterval</code> after you create a health check.
         *        </p>
         *        </important>
         *        <p>
         *        If you don't specify a value for <code>RequestInterval</code>, the default value is <code>30</code>
         *        seconds.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requestInterval(Integer requestInterval);

        /**
         * <p>
         * The number of consecutive health checks that an endpoint must pass or fail for Amazon Route 53 to change the
         * current status of the endpoint from unhealthy to healthy or vice versa. For more information, see <a href=
         * "http://docs.aws.amazon.com/Route53/latest/DeveloperGuide/dns-failover-determining-health-of-endpoints.html"
         * >How Amazon Route 53 Determines Whether an Endpoint Is Healthy</a> in the <i>Amazon Route 53 Developer
         * Guide</i>.
         * </p>
         * <p>
         * If you don't specify a value for <code>FailureThreshold</code>, the default value is three health checks.
         * </p>
         * 
         * @param failureThreshold
         *        The number of consecutive health checks that an endpoint must pass or fail for Amazon Route 53 to
         *        change the current status of the endpoint from unhealthy to healthy or vice versa. For more
         *        information, see <a href=
         *        "http://docs.aws.amazon.com/Route53/latest/DeveloperGuide/dns-failover-determining-health-of-endpoints.html"
         *        >How Amazon Route 53 Determines Whether an Endpoint Is Healthy</a> in the <i>Amazon Route 53 Developer
         *        Guide</i>.</p>
         *        <p>
         *        If you don't specify a value for <code>FailureThreshold</code>, the default value is three health
         *        checks.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureThreshold(Integer failureThreshold);

        /**
         * <p>
         * Specify whether you want Amazon Route 53 to measure the latency between health checkers in multiple AWS
         * regions and your endpoint, and to display CloudWatch latency graphs on the <b>Health Checks</b> page in the
         * Amazon Route 53 console.
         * </p>
         * <important>
         * <p>
         * You can't change the value of <code>MeasureLatency</code> after you create a health check.
         * </p>
         * </important>
         * 
         * @param measureLatency
         *        Specify whether you want Amazon Route 53 to measure the latency between health checkers in multiple
         *        AWS regions and your endpoint, and to display CloudWatch latency graphs on the <b>Health Checks</b>
         *        page in the Amazon Route 53 console.</p> <important>
         *        <p>
         *        You can't change the value of <code>MeasureLatency</code> after you create a health check.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder measureLatency(Boolean measureLatency);

        /**
         * <p>
         * Specify whether you want Amazon Route 53 to invert the status of a health check, for example, to consider a
         * health check unhealthy when it otherwise would be considered healthy.
         * </p>
         * 
         * @param inverted
         *        Specify whether you want Amazon Route 53 to invert the status of a health check, for example, to
         *        consider a health check unhealthy when it otherwise would be considered healthy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inverted(Boolean inverted);

        /**
         * <p>
         * The number of child health checks that are associated with a <code>CALCULATED</code> health that Amazon Route
         * 53 must consider healthy for the <code>CALCULATED</code> health check to be considered healthy. To specify
         * the child health checks that you want to associate with a <code>CALCULATED</code> health check, use the
         * <a>HealthCheckConfig$ChildHealthChecks</a> and <a>HealthCheckConfig$ChildHealthChecks</a> elements.
         * </p>
         * <p>
         * Note the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * If you specify a number greater than the number of child health checks, Amazon Route 53 always considers this
         * health check to be unhealthy.
         * </p>
         * </li>
         * <li>
         * <p>
         * If you specify <code>0</code>, Amazon Route 53 always considers this health check to be healthy.
         * </p>
         * </li>
         * </ul>
         * 
         * @param healthThreshold
         *        The number of child health checks that are associated with a <code>CALCULATED</code> health that
         *        Amazon Route 53 must consider healthy for the <code>CALCULATED</code> health check to be considered
         *        healthy. To specify the child health checks that you want to associate with a <code>CALCULATED</code>
         *        health check, use the <a>HealthCheckConfig$ChildHealthChecks</a> and
         *        <a>HealthCheckConfig$ChildHealthChecks</a> elements.</p>
         *        <p>
         *        Note the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If you specify a number greater than the number of child health checks, Amazon Route 53 always
         *        considers this health check to be unhealthy.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you specify <code>0</code>, Amazon Route 53 always considers this health check to be healthy.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder healthThreshold(Integer healthThreshold);

        /**
         * <p>
         * (CALCULATED Health Checks Only) A complex type that contains one <code>ChildHealthCheck</code> element for
         * each health check that you want to associate with a <code>CALCULATED</code> health check.
         * </p>
         * 
         * @param childHealthChecks
         *        (CALCULATED Health Checks Only) A complex type that contains one <code>ChildHealthCheck</code> element
         *        for each health check that you want to associate with a <code>CALCULATED</code> health check.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder childHealthChecks(Collection<String> childHealthChecks);

        /**
         * <p>
         * (CALCULATED Health Checks Only) A complex type that contains one <code>ChildHealthCheck</code> element for
         * each health check that you want to associate with a <code>CALCULATED</code> health check.
         * </p>
         * 
         * @param childHealthChecks
         *        (CALCULATED Health Checks Only) A complex type that contains one <code>ChildHealthCheck</code> element
         *        for each health check that you want to associate with a <code>CALCULATED</code> health check.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder childHealthChecks(String... childHealthChecks);

        /**
         * <p>
         * Specify whether you want Amazon Route 53 to send the value of <code>FullyQualifiedDomainName</code> to the
         * endpoint in the <code>client_hello</code> message during TLS negotiation. This allows the endpoint to respond
         * to <code>HTTPS</code> health check requests with the applicable SSL/TLS certificate.
         * </p>
         * <p>
         * Some endpoints require that <code>HTTPS</code> requests include the host name in the
         * <code>client_hello</code> message. If you don't enable SNI, the status of the health check will be
         * <code>SSL alert handshake_failure</code>. A health check can also have that status for other reasons. If SNI
         * is enabled and you're still getting the error, check the SSL/TLS configuration on your endpoint and confirm
         * that your certificate is valid.
         * </p>
         * <p>
         * The SSL/TLS certificate on your endpoint includes a domain name in the <code>Common Name</code> field and
         * possibly several more in the <code>Subject Alternative Names</code> field. One of the domain names in the
         * certificate should match the value that you specify for <code>FullyQualifiedDomainName</code>. If the
         * endpoint responds to the <code>client_hello</code> message with a certificate that does not include the
         * domain name that you specified in <code>FullyQualifiedDomainName</code>, a health checker will retry the
         * handshake. In the second attempt, the health checker will omit <code>FullyQualifiedDomainName</code> from the
         * <code>client_hello</code> message.
         * </p>
         * 
         * @param enableSNI
         *        Specify whether you want Amazon Route 53 to send the value of <code>FullyQualifiedDomainName</code> to
         *        the endpoint in the <code>client_hello</code> message during TLS negotiation. This allows the endpoint
         *        to respond to <code>HTTPS</code> health check requests with the applicable SSL/TLS certificate.</p>
         *        <p>
         *        Some endpoints require that <code>HTTPS</code> requests include the host name in the
         *        <code>client_hello</code> message. If you don't enable SNI, the status of the health check will be
         *        <code>SSL alert handshake_failure</code>. A health check can also have that status for other reasons.
         *        If SNI is enabled and you're still getting the error, check the SSL/TLS configuration on your endpoint
         *        and confirm that your certificate is valid.
         *        </p>
         *        <p>
         *        The SSL/TLS certificate on your endpoint includes a domain name in the <code>Common Name</code> field
         *        and possibly several more in the <code>Subject Alternative Names</code> field. One of the domain names
         *        in the certificate should match the value that you specify for <code>FullyQualifiedDomainName</code>.
         *        If the endpoint responds to the <code>client_hello</code> message with a certificate that does not
         *        include the domain name that you specified in <code>FullyQualifiedDomainName</code>, a health checker
         *        will retry the handshake. In the second attempt, the health checker will omit
         *        <code>FullyQualifiedDomainName</code> from the <code>client_hello</code> message.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enableSNI(Boolean enableSNI);

        /**
         * <p>
         * A complex type that contains one <code>Region</code> element for each region from which you want Amazon Route
         * 53 health checkers to check the specified endpoint.
         * </p>
         * <p>
         * If you don't specify any regions, Amazon Route 53 health checkers automatically performs checks from all of
         * the regions that are listed under <b>Valid Values</b>.
         * </p>
         * <p>
         * If you update a health check to remove a region that has been performing health checks, Amazon Route 53 will
         * briefly continue to perform checks from that region to ensure that some health checkers are always checking
         * the endpoint (for example, if you replace three regions with four different regions).
         * </p>
         * 
         * @param regions
         *        A complex type that contains one <code>Region</code> element for each region from which you want
         *        Amazon Route 53 health checkers to check the specified endpoint.</p>
         *        <p>
         *        If you don't specify any regions, Amazon Route 53 health checkers automatically performs checks from
         *        all of the regions that are listed under <b>Valid Values</b>.
         *        </p>
         *        <p>
         *        If you update a health check to remove a region that has been performing health checks, Amazon Route
         *        53 will briefly continue to perform checks from that region to ensure that some health checkers are
         *        always checking the endpoint (for example, if you replace three regions with four different regions).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder regions(Collection<String> regions);

        /**
         * <p>
         * A complex type that contains one <code>Region</code> element for each region from which you want Amazon Route
         * 53 health checkers to check the specified endpoint.
         * </p>
         * <p>
         * If you don't specify any regions, Amazon Route 53 health checkers automatically performs checks from all of
         * the regions that are listed under <b>Valid Values</b>.
         * </p>
         * <p>
         * If you update a health check to remove a region that has been performing health checks, Amazon Route 53 will
         * briefly continue to perform checks from that region to ensure that some health checkers are always checking
         * the endpoint (for example, if you replace three regions with four different regions).
         * </p>
         * 
         * @param regions
         *        A complex type that contains one <code>Region</code> element for each region from which you want
         *        Amazon Route 53 health checkers to check the specified endpoint.</p>
         *        <p>
         *        If you don't specify any regions, Amazon Route 53 health checkers automatically performs checks from
         *        all of the regions that are listed under <b>Valid Values</b>.
         *        </p>
         *        <p>
         *        If you update a health check to remove a region that has been performing health checks, Amazon Route
         *        53 will briefly continue to perform checks from that region to ensure that some health checkers are
         *        always checking the endpoint (for example, if you replace three regions with four different regions).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder regions(String... regions);

        /**
         * <p>
         * A complex type that identifies the CloudWatch alarm that you want Amazon Route 53 health checkers to use to
         * determine whether this health check is healthy.
         * </p>
         * 
         * @param alarmIdentifier
         *        A complex type that identifies the CloudWatch alarm that you want Amazon Route 53 health checkers to
         *        use to determine whether this health check is healthy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder alarmIdentifier(AlarmIdentifier alarmIdentifier);

        /**
         * <p>
         * When CloudWatch has insufficient data about the metric to determine the alarm state, the status that you want
         * Amazon Route 53 to assign to the health check:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Healthy</code>: Amazon Route 53 considers the health check to be healthy.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Unhealthy</code>: Amazon Route 53 considers the health check to be unhealthy.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>LastKnownStatus</code>: Amazon Route 53 uses the status of the health check from the last time that
         * CloudWatch had sufficient data to determine the alarm state. For new health checks that have no last known
         * status, the default status for the health check is healthy.
         * </p>
         * </li>
         * </ul>
         * 
         * @param insufficientDataHealthStatus
         *        When CloudWatch has insufficient data about the metric to determine the alarm state, the status that
         *        you want Amazon Route 53 to assign to the health check:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Healthy</code>: Amazon Route 53 considers the health check to be healthy.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Unhealthy</code>: Amazon Route 53 considers the health check to be unhealthy.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>LastKnownStatus</code>: Amazon Route 53 uses the status of the health check from the last time
         *        that CloudWatch had sufficient data to determine the alarm state. For new health checks that have no
         *        last known status, the default status for the health check is healthy.
         *        </p>
         *        </li>
         * @see InsufficientDataHealthStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InsufficientDataHealthStatus
         */
        Builder insufficientDataHealthStatus(String insufficientDataHealthStatus);

        /**
         * <p>
         * When CloudWatch has insufficient data about the metric to determine the alarm state, the status that you want
         * Amazon Route 53 to assign to the health check:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Healthy</code>: Amazon Route 53 considers the health check to be healthy.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Unhealthy</code>: Amazon Route 53 considers the health check to be unhealthy.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>LastKnownStatus</code>: Amazon Route 53 uses the status of the health check from the last time that
         * CloudWatch had sufficient data to determine the alarm state. For new health checks that have no last known
         * status, the default status for the health check is healthy.
         * </p>
         * </li>
         * </ul>
         * 
         * @param insufficientDataHealthStatus
         *        When CloudWatch has insufficient data about the metric to determine the alarm state, the status that
         *        you want Amazon Route 53 to assign to the health check:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Healthy</code>: Amazon Route 53 considers the health check to be healthy.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Unhealthy</code>: Amazon Route 53 considers the health check to be unhealthy.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>LastKnownStatus</code>: Amazon Route 53 uses the status of the health check from the last time
         *        that CloudWatch had sufficient data to determine the alarm state. For new health checks that have no
         *        last known status, the default status for the health check is healthy.
         *        </p>
         *        </li>
         * @see InsufficientDataHealthStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InsufficientDataHealthStatus
         */
        Builder insufficientDataHealthStatus(InsufficientDataHealthStatus insufficientDataHealthStatus);
    }

    static final class BuilderImpl implements Builder {
        private String ipAddress;

        private Integer port;

        private String type;

        private String resourcePath;

        private String fullyQualifiedDomainName;

        private String searchString;

        private Integer requestInterval;

        private Integer failureThreshold;

        private Boolean measureLatency;

        private Boolean inverted;

        private Integer healthThreshold;

        private List<String> childHealthChecks;

        private Boolean enableSNI;

        private List<String> regions;

        private AlarmIdentifier alarmIdentifier;

        private String insufficientDataHealthStatus;

        private BuilderImpl() {
        }

        private BuilderImpl(HealthCheckConfig model) {
            ipAddress(model.ipAddress);
            port(model.port);
            type(model.type);
            resourcePath(model.resourcePath);
            fullyQualifiedDomainName(model.fullyQualifiedDomainName);
            searchString(model.searchString);
            requestInterval(model.requestInterval);
            failureThreshold(model.failureThreshold);
            measureLatency(model.measureLatency);
            inverted(model.inverted);
            healthThreshold(model.healthThreshold);
            childHealthChecks(model.childHealthChecks);
            enableSNI(model.enableSNI);
            regions(model.regions);
            alarmIdentifier(model.alarmIdentifier);
            insufficientDataHealthStatus(model.insufficientDataHealthStatus);
        }

        public final String getIPAddress() {
            return ipAddress;
        }

        @Override
        public final Builder ipAddress(String ipAddress) {
            this.ipAddress = ipAddress;
            return this;
        }

        public final void setIPAddress(String ipAddress) {
            this.ipAddress = ipAddress;
        }

        public final Integer getPort() {
            return port;
        }

        @Override
        public final Builder port(Integer port) {
            this.port = port;
            return this;
        }

        public final void setPort(Integer port) {
            this.port = port;
        }

        public final String getType() {
            return type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(HealthCheckType type) {
            this.type(type.toString());
            return this;
        }

        public final void setType(String type) {
            this.type = type;
        }

        public final String getResourcePath() {
            return resourcePath;
        }

        @Override
        public final Builder resourcePath(String resourcePath) {
            this.resourcePath = resourcePath;
            return this;
        }

        public final void setResourcePath(String resourcePath) {
            this.resourcePath = resourcePath;
        }

        public final String getFullyQualifiedDomainName() {
            return fullyQualifiedDomainName;
        }

        @Override
        public final Builder fullyQualifiedDomainName(String fullyQualifiedDomainName) {
            this.fullyQualifiedDomainName = fullyQualifiedDomainName;
            return this;
        }

        public final void setFullyQualifiedDomainName(String fullyQualifiedDomainName) {
            this.fullyQualifiedDomainName = fullyQualifiedDomainName;
        }

        public final String getSearchString() {
            return searchString;
        }

        @Override
        public final Builder searchString(String searchString) {
            this.searchString = searchString;
            return this;
        }

        public final void setSearchString(String searchString) {
            this.searchString = searchString;
        }

        public final Integer getRequestInterval() {
            return requestInterval;
        }

        @Override
        public final Builder requestInterval(Integer requestInterval) {
            this.requestInterval = requestInterval;
            return this;
        }

        public final void setRequestInterval(Integer requestInterval) {
            this.requestInterval = requestInterval;
        }

        public final Integer getFailureThreshold() {
            return failureThreshold;
        }

        @Override
        public final Builder failureThreshold(Integer failureThreshold) {
            this.failureThreshold = failureThreshold;
            return this;
        }

        public final void setFailureThreshold(Integer failureThreshold) {
            this.failureThreshold = failureThreshold;
        }

        public final Boolean getMeasureLatency() {
            return measureLatency;
        }

        @Override
        public final Builder measureLatency(Boolean measureLatency) {
            this.measureLatency = measureLatency;
            return this;
        }

        public final void setMeasureLatency(Boolean measureLatency) {
            this.measureLatency = measureLatency;
        }

        public final Boolean getInverted() {
            return inverted;
        }

        @Override
        public final Builder inverted(Boolean inverted) {
            this.inverted = inverted;
            return this;
        }

        public final void setInverted(Boolean inverted) {
            this.inverted = inverted;
        }

        public final Integer getHealthThreshold() {
            return healthThreshold;
        }

        @Override
        public final Builder healthThreshold(Integer healthThreshold) {
            this.healthThreshold = healthThreshold;
            return this;
        }

        public final void setHealthThreshold(Integer healthThreshold) {
            this.healthThreshold = healthThreshold;
        }

        public final Collection<String> getChildHealthChecks() {
            return childHealthChecks;
        }

        @Override
        public final Builder childHealthChecks(Collection<String> childHealthChecks) {
            this.childHealthChecks = ChildHealthCheckListCopier.copy(childHealthChecks);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder childHealthChecks(String... childHealthChecks) {
            childHealthChecks(Arrays.asList(childHealthChecks));
            return this;
        }

        public final void setChildHealthChecks(Collection<String> childHealthChecks) {
            this.childHealthChecks = ChildHealthCheckListCopier.copy(childHealthChecks);
        }

        public final Boolean getEnableSNI() {
            return enableSNI;
        }

        @Override
        public final Builder enableSNI(Boolean enableSNI) {
            this.enableSNI = enableSNI;
            return this;
        }

        public final void setEnableSNI(Boolean enableSNI) {
            this.enableSNI = enableSNI;
        }

        public final Collection<String> getRegions() {
            return regions;
        }

        @Override
        public final Builder regions(Collection<String> regions) {
            this.regions = HealthCheckRegionListCopier.copy(regions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder regions(String... regions) {
            regions(Arrays.asList(regions));
            return this;
        }

        public final void setRegions(Collection<String> regions) {
            this.regions = HealthCheckRegionListCopier.copy(regions);
        }

        public final AlarmIdentifier.Builder getAlarmIdentifier() {
            return alarmIdentifier != null ? alarmIdentifier.toBuilder() : null;
        }

        @Override
        public final Builder alarmIdentifier(AlarmIdentifier alarmIdentifier) {
            this.alarmIdentifier = alarmIdentifier;
            return this;
        }

        public final void setAlarmIdentifier(AlarmIdentifier.BuilderImpl alarmIdentifier) {
            this.alarmIdentifier = alarmIdentifier != null ? alarmIdentifier.build() : null;
        }

        public final String getInsufficientDataHealthStatus() {
            return insufficientDataHealthStatus;
        }

        @Override
        public final Builder insufficientDataHealthStatus(String insufficientDataHealthStatus) {
            this.insufficientDataHealthStatus = insufficientDataHealthStatus;
            return this;
        }

        @Override
        public final Builder insufficientDataHealthStatus(InsufficientDataHealthStatus insufficientDataHealthStatus) {
            this.insufficientDataHealthStatus(insufficientDataHealthStatus.toString());
            return this;
        }

        public final void setInsufficientDataHealthStatus(String insufficientDataHealthStatus) {
            this.insufficientDataHealthStatus = insufficientDataHealthStatus;
        }

        @Override
        public HealthCheckConfig build() {
            return new HealthCheckConfig(this);
        }
    }
}
