/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.route53.paginators;

import java.util.Iterator;
import java.util.function.Function;
import javax.annotation.Generated;
import software.amazon.awssdk.core.pagination.NextPageFetcher;
import software.amazon.awssdk.core.pagination.PaginatedItemsIterable;
import software.amazon.awssdk.core.pagination.PaginatedResponsesIterator;
import software.amazon.awssdk.core.pagination.SdkIterable;
import software.amazon.awssdk.services.route53.Route53Client;
import software.amazon.awssdk.services.route53.model.HealthCheck;
import software.amazon.awssdk.services.route53.model.ListHealthChecksRequest;
import software.amazon.awssdk.services.route53.model.ListHealthChecksResponse;

/**
 * <p>
 * Represents the output for the
 * {@link software.amazon.awssdk.services.route53.Route53Client#listHealthChecksIterable(software.amazon.awssdk.services.route53.model.ListHealthChecksRequest)}
 * operation which is a paginated operation. This class is an iterable of
 * {@link software.amazon.awssdk.services.route53.model.ListHealthChecksResponse} that can be used to iterate through
 * all the response pages of the operation.
 * </p>
 * <p>
 * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet and
 * so there is no guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily
 * loading response pages by making service calls until there are no pages left or your iteration stops. If there are
 * errors in your request, you will see the failures only after you start iterating through the iterable.
 * </p>
 *
 * <p>
 * The following are few ways to iterate through the response pages:
 * </p>
 * 1) Using a Stream
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.route53.paginators.ListHealthChecksPaginator responses = client.listHealthChecksIterable(request);
 * responses.stream().forEach(....);
 * }
 * </pre>
 *
 * 2) Using For loop
 * 
 * <pre>
 * {
 *     &#064;code
 *     software.amazon.awssdk.services.route53.paginators.ListHealthChecksPaginator responses = client
 *             .listHealthChecksIterable(request);
 *     for (software.amazon.awssdk.services.route53.model.ListHealthChecksResponse response : responses) {
 *         // do something;
 *     }
 * }
 * </pre>
 *
 * 3) Use iterator directly
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.route53.paginators.ListHealthChecksPaginator responses = client.listHealthChecksIterable(request);
 * responses.iterator().forEachRemaining(....);
 * }
 * </pre>
 * <p>
 * <b>Note: If you prefer to have control on service calls, use the
 * {@link #listHealthChecks(software.amazon.awssdk.services.route53.model.ListHealthChecksRequest)} operation.</b>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ListHealthChecksPaginator implements SdkIterable<ListHealthChecksResponse> {
    private final Route53Client client;

    private final ListHealthChecksRequest firstRequest;

    private final NextPageFetcher nextPageFetcher;

    public ListHealthChecksPaginator(final Route53Client client, final ListHealthChecksRequest firstRequest) {
        this.client = client;
        this.firstRequest = firstRequest;
        this.nextPageFetcher = new ListHealthChecksResponseFetcher();
    }

    @Override
    public Iterator<ListHealthChecksResponse> iterator() {
        return new PaginatedResponsesIterator(nextPageFetcher);
    }

    /**
     * Returns an iterable to iterate through the paginated {@link ListHealthChecksResponse#healthChecks()} member. The
     * returned iterable is used to iterate through the results across all response pages and not a single page.
     *
     * This method is useful if you are interested in iterating over the paginated member in the response pages instead
     * of the top level pages. Similar to iteration over pages, this method internally makes service calls to get the
     * next list of results until the iteration stops or there are no more results.
     */
    public SdkIterable<HealthCheck> healthChecks() {
        Function<ListHealthChecksResponse, Iterator<HealthCheck>> getIterator = response -> response != null ? response
                .healthChecks().iterator() : null;

        return new PaginatedItemsIterable(this, getIterator);
    }

    private class ListHealthChecksResponseFetcher implements NextPageFetcher<ListHealthChecksResponse> {
        @Override
        public boolean hasNextPage(ListHealthChecksResponse previousPage) {
            return previousPage.isTruncated().booleanValue();
        }

        @Override
        public ListHealthChecksResponse nextPage(ListHealthChecksResponse previousPage) {
            if (previousPage == null) {
                return client.listHealthChecks(firstRequest);
            }
            return client.listHealthChecks(firstRequest.toBuilder().marker(previousPage.nextMarker()).build());
        }
    }
}
