/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.route53.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import javax.annotation.Generated;
import software.amazon.awssdk.core.AwsRequestOverrideConfig;
import software.amazon.awssdk.core.runtime.TypeConverter;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A complex type that contains information about a request to update a health check.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class UpdateHealthCheckRequest extends Route53Request implements
        ToCopyableBuilder<UpdateHealthCheckRequest.Builder, UpdateHealthCheckRequest> {
    private final String healthCheckId;

    private final Long healthCheckVersion;

    private final String ipAddress;

    private final Integer port;

    private final String resourcePath;

    private final String fullyQualifiedDomainName;

    private final String searchString;

    private final Integer failureThreshold;

    private final Boolean inverted;

    private final Integer healthThreshold;

    private final List<String> childHealthChecks;

    private final Boolean enableSNI;

    private final List<String> regions;

    private final AlarmIdentifier alarmIdentifier;

    private final String insufficientDataHealthStatus;

    private final List<String> resetElements;

    private UpdateHealthCheckRequest(BuilderImpl builder) {
        super(builder);
        this.healthCheckId = builder.healthCheckId;
        this.healthCheckVersion = builder.healthCheckVersion;
        this.ipAddress = builder.ipAddress;
        this.port = builder.port;
        this.resourcePath = builder.resourcePath;
        this.fullyQualifiedDomainName = builder.fullyQualifiedDomainName;
        this.searchString = builder.searchString;
        this.failureThreshold = builder.failureThreshold;
        this.inverted = builder.inverted;
        this.healthThreshold = builder.healthThreshold;
        this.childHealthChecks = builder.childHealthChecks;
        this.enableSNI = builder.enableSNI;
        this.regions = builder.regions;
        this.alarmIdentifier = builder.alarmIdentifier;
        this.insufficientDataHealthStatus = builder.insufficientDataHealthStatus;
        this.resetElements = builder.resetElements;
    }

    /**
     * <p>
     * The ID for the health check for which you want detailed information. When you created the health check,
     * <code>CreateHealthCheck</code> returned the ID in the response, in the <code>HealthCheckId</code> element.
     * </p>
     * 
     * @return The ID for the health check for which you want detailed information. When you created the health check,
     *         <code>CreateHealthCheck</code> returned the ID in the response, in the <code>HealthCheckId</code>
     *         element.
     */
    public String healthCheckId() {
        return healthCheckId;
    }

    /**
     * <p>
     * A sequential counter that Amazon Route 53 sets to <code>1</code> when you create a health check and increments by
     * 1 each time you update settings for the health check.
     * </p>
     * <p>
     * We recommend that you use <code>GetHealthCheck</code> or <code>ListHealthChecks</code> to get the current value
     * of <code>HealthCheckVersion</code> for the health check that you want to update, and that you include that value
     * in your <code>UpdateHealthCheck</code> request. This prevents Amazon Route 53 from overwriting an intervening
     * update:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If the value in the <code>UpdateHealthCheck</code> request matches the value of <code>HealthCheckVersion</code>
     * in the health check, Amazon Route 53 updates the health check with the new settings.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the value of <code>HealthCheckVersion</code> in the health check is greater, the health check was changed
     * after you got the version number. Amazon Route 53 does not update the health check, and it returns a
     * <code>HealthCheckVersionMismatch</code> error.
     * </p>
     * </li>
     * </ul>
     * 
     * @return A sequential counter that Amazon Route 53 sets to <code>1</code> when you create a health check and
     *         increments by 1 each time you update settings for the health check.</p>
     *         <p>
     *         We recommend that you use <code>GetHealthCheck</code> or <code>ListHealthChecks</code> to get the current
     *         value of <code>HealthCheckVersion</code> for the health check that you want to update, and that you
     *         include that value in your <code>UpdateHealthCheck</code> request. This prevents Amazon Route 53 from
     *         overwriting an intervening update:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If the value in the <code>UpdateHealthCheck</code> request matches the value of
     *         <code>HealthCheckVersion</code> in the health check, Amazon Route 53 updates the health check with the
     *         new settings.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If the value of <code>HealthCheckVersion</code> in the health check is greater, the health check was
     *         changed after you got the version number. Amazon Route 53 does not update the health check, and it
     *         returns a <code>HealthCheckVersionMismatch</code> error.
     *         </p>
     *         </li>
     */
    public Long healthCheckVersion() {
        return healthCheckVersion;
    }

    /**
     * <p>
     * The IPv4 or IPv6 IP address for the endpoint that you want Amazon Route 53 to perform health checks on. If you
     * don't specify a value for <code>IPAddress</code>, Amazon Route 53 sends a DNS request to resolve the domain name
     * that you specify in <code>FullyQualifiedDomainName</code> at the interval that you specify in
     * <code>RequestInterval</code>. Using an IP address that is returned by DNS, Amazon Route 53 then checks the health
     * of the endpoint.
     * </p>
     * <p>
     * Use one of the following formats for the value of <code>IPAddress</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>IPv4 address</b>: four values between 0 and 255, separated by periods (.), for example,
     * <code>192.0.2.44</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>IPv6 address</b>: eight groups of four hexadecimal values, separated by colons (:), for example,
     * <code>2001:0db8:85a3:0000:0000:abcd:0001:2345</code>. You can also shorten IPv6 addresses as described in RFC
     * 5952, for example, <code>2001:db8:85a3::abcd:1:2345</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the endpoint is an EC2 instance, we recommend that you create an Elastic IP address, associate it with your
     * EC2 instance, and specify the Elastic IP address for <code>IPAddress</code>. This ensures that the IP address of
     * your instance never changes. For more information, see the applicable documentation:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Linux: <a href="http://docs.aws.amazon.com/AWSEC2/latest/UserGuide/elastic-ip-addresses-eip.html">Elastic IP
     * Addresses (EIP)</a> in the <i>Amazon EC2 User Guide for Linux Instances</i>
     * </p>
     * </li>
     * <li>
     * <p>
     * Windows: <a href="http://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/elastic-ip-addresses-eip.html">Elastic IP
     * Addresses (EIP)</a> in the <i>Amazon EC2 User Guide for Windows Instances</i>
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * If a health check already has a value for <code>IPAddress</code>, you can change the value. However, you can't
     * update an existing health check to add or remove the value of <code>IPAddress</code>.
     * </p>
     * </note>
     * <p>
     * For more information, see <a>UpdateHealthCheckRequest$FullyQualifiedDomainName</a>.
     * </p>
     * <p>
     * Constraints: Amazon Route 53 can't check the health of endpoints for which the IP address is in local, private,
     * non-routable, or multicast ranges. For more information about IP addresses for which you can't create health
     * checks, see the following documents:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://tools.ietf.org/html/rfc5735">RFC 5735, Special Use IPv4 Addresses</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://tools.ietf.org/html/rfc6598">RFC 6598, IANA-Reserved IPv4 Prefix for Shared Address Space</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://tools.ietf.org/html/rfc5156">RFC 5156, Special-Use IPv6 Addresses</a>
     * </p>
     * </li>
     * </ul>
     * 
     * @return The IPv4 or IPv6 IP address for the endpoint that you want Amazon Route 53 to perform health checks on.
     *         If you don't specify a value for <code>IPAddress</code>, Amazon Route 53 sends a DNS request to resolve
     *         the domain name that you specify in <code>FullyQualifiedDomainName</code> at the interval that you
     *         specify in <code>RequestInterval</code>. Using an IP address that is returned by DNS, Amazon Route 53
     *         then checks the health of the endpoint.</p>
     *         <p>
     *         Use one of the following formats for the value of <code>IPAddress</code>:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>IPv4 address</b>: four values between 0 and 255, separated by periods (.), for example,
     *         <code>192.0.2.44</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>IPv6 address</b>: eight groups of four hexadecimal values, separated by colons (:), for example,
     *         <code>2001:0db8:85a3:0000:0000:abcd:0001:2345</code>. You can also shorten IPv6 addresses as described in
     *         RFC 5952, for example, <code>2001:db8:85a3::abcd:1:2345</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If the endpoint is an EC2 instance, we recommend that you create an Elastic IP address, associate it with
     *         your EC2 instance, and specify the Elastic IP address for <code>IPAddress</code>. This ensures that the
     *         IP address of your instance never changes. For more information, see the applicable documentation:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Linux: <a href="http://docs.aws.amazon.com/AWSEC2/latest/UserGuide/elastic-ip-addresses-eip.html">Elastic
     *         IP Addresses (EIP)</a> in the <i>Amazon EC2 User Guide for Linux Instances</i>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Windows: <a
     *         href="http://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/elastic-ip-addresses-eip.html">Elastic IP
     *         Addresses (EIP)</a> in the <i>Amazon EC2 User Guide for Windows Instances</i>
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         If a health check already has a value for <code>IPAddress</code>, you can change the value. However, you
     *         can't update an existing health check to add or remove the value of <code>IPAddress</code>.
     *         </p>
     *         </note>
     *         <p>
     *         For more information, see <a>UpdateHealthCheckRequest$FullyQualifiedDomainName</a>.
     *         </p>
     *         <p>
     *         Constraints: Amazon Route 53 can't check the health of endpoints for which the IP address is in local,
     *         private, non-routable, or multicast ranges. For more information about IP addresses for which you can't
     *         create health checks, see the following documents:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <a href="https://tools.ietf.org/html/rfc5735">RFC 5735, Special Use IPv4 Addresses</a>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <a href="https://tools.ietf.org/html/rfc6598">RFC 6598, IANA-Reserved IPv4 Prefix for Shared Address
     *         Space</a>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <a href="https://tools.ietf.org/html/rfc5156">RFC 5156, Special-Use IPv6 Addresses</a>
     *         </p>
     *         </li>
     */
    public String ipAddress() {
        return ipAddress;
    }

    /**
     * <p>
     * The port on the endpoint on which you want Amazon Route 53 to perform health checks.
     * </p>
     * 
     * @return The port on the endpoint on which you want Amazon Route 53 to perform health checks.
     */
    public Integer port() {
        return port;
    }

    /**
     * <p>
     * The path that you want Amazon Route 53 to request when performing health checks. The path can be any value for
     * which your endpoint will return an HTTP status code of 2xx or 3xx when the endpoint is healthy, for example the
     * file /docs/route53-health-check.html.
     * </p>
     * <p>
     * Specify this value only if you want to change it.
     * </p>
     * 
     * @return The path that you want Amazon Route 53 to request when performing health checks. The path can be any
     *         value for which your endpoint will return an HTTP status code of 2xx or 3xx when the endpoint is healthy,
     *         for example the file /docs/route53-health-check.html. </p>
     *         <p>
     *         Specify this value only if you want to change it.
     */
    public String resourcePath() {
        return resourcePath;
    }

    /**
     * <p>
     * Amazon Route 53 behavior depends on whether you specify a value for <code>IPAddress</code>.
     * </p>
     * <note>
     * <p>
     * If a health check already has a value for <code>IPAddress</code>, you can change the value. However, you can't
     * update an existing health check to add or remove the value of <code>IPAddress</code>.
     * </p>
     * </note>
     * <p>
     * <b>If you specify a value for</b> <code>IPAddress</code>:
     * </p>
     * <p>
     * Amazon Route 53 sends health check requests to the specified IPv4 or IPv6 address and passes the value of
     * <code>FullyQualifiedDomainName</code> in the <code>Host</code> header for all health checks except TCP health
     * checks. This is typically the fully qualified DNS name of the endpoint on which you want Amazon Route 53 to
     * perform health checks.
     * </p>
     * <p>
     * When Amazon Route 53 checks the health of an endpoint, here is how it constructs the <code>Host</code> header:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If you specify a value of <code>80</code> for <code>Port</code> and <code>HTTP</code> or
     * <code>HTTP_STR_MATCH</code> for <code>Type</code>, Amazon Route 53 passes the value of
     * <code>FullyQualifiedDomainName</code> to the endpoint in the <code>Host</code> header.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you specify a value of <code>443</code> for <code>Port</code> and <code>HTTPS</code> or
     * <code>HTTPS_STR_MATCH</code> for <code>Type</code>, Amazon Route 53 passes the value of
     * <code>FullyQualifiedDomainName</code> to the endpoint in the <code>Host</code> header.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you specify another value for <code>Port</code> and any value except <code>TCP</code> for <code>Type</code>,
     * Amazon Route 53 passes <i> <code>FullyQualifiedDomainName</code>:<code>Port</code> </i> to the endpoint in the
     * <code>Host</code> header.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If you don't specify a value for <code>FullyQualifiedDomainName</code>, Amazon Route 53 substitutes the value of
     * <code>IPAddress</code> in the <code>Host</code> header in each of the above cases.
     * </p>
     * <p>
     * <b>If you don't specify a value for</b> <code>IPAddress</code>:
     * </p>
     * <p>
     * If you don't specify a value for <code>IPAddress</code>, Amazon Route 53 sends a DNS request to the domain that
     * you specify in <code>FullyQualifiedDomainName</code> at the interval you specify in <code>RequestInterval</code>.
     * Using an IPv4 address that is returned by DNS, Amazon Route 53 then checks the health of the endpoint.
     * </p>
     * <note>
     * <p>
     * If you don't specify a value for <code>IPAddress</code>, Amazon Route 53 uses only IPv4 to send health checks to
     * the endpoint. If there's no resource record set with a type of A for the name that you specify for
     * <code>FullyQualifiedDomainName</code>, the health check fails with a "DNS resolution failed" error.
     * </p>
     * </note>
     * <p>
     * If you want to check the health of weighted, latency, or failover resource record sets and you choose to specify
     * the endpoint only by <code>FullyQualifiedDomainName</code>, we recommend that you create a separate health check
     * for each endpoint. For example, create a health check for each HTTP server that is serving content for
     * www.example.com. For the value of <code>FullyQualifiedDomainName</code>, specify the domain name of the server
     * (such as <code>us-east-2-www.example.com</code>), not the name of the resource record sets (www.example.com).
     * </p>
     * <important>
     * <p>
     * In this configuration, if the value of <code>FullyQualifiedDomainName</code> matches the name of the resource
     * record sets and you then associate the health check with those resource record sets, health check results will be
     * unpredictable.
     * </p>
     * </important>
     * <p>
     * In addition, if the value of <code>Type</code> is <code>HTTP</code>, <code>HTTPS</code>,
     * <code>HTTP_STR_MATCH</code>, or <code>HTTPS_STR_MATCH</code>, Amazon Route 53 passes the value of
     * <code>FullyQualifiedDomainName</code> in the <code>Host</code> header, as it does when you specify a value for
     * <code>IPAddress</code>. If the value of <code>Type</code> is <code>TCP</code>, Amazon Route 53 doesn't pass a
     * <code>Host</code> header.
     * </p>
     * 
     * @return Amazon Route 53 behavior depends on whether you specify a value for <code>IPAddress</code>.</p> <note>
     *         <p>
     *         If a health check already has a value for <code>IPAddress</code>, you can change the value. However, you
     *         can't update an existing health check to add or remove the value of <code>IPAddress</code>.
     *         </p>
     *         </note>
     *         <p>
     *         <b>If you specify a value for</b> <code>IPAddress</code>:
     *         </p>
     *         <p>
     *         Amazon Route 53 sends health check requests to the specified IPv4 or IPv6 address and passes the value of
     *         <code>FullyQualifiedDomainName</code> in the <code>Host</code> header for all health checks except TCP
     *         health checks. This is typically the fully qualified DNS name of the endpoint on which you want Amazon
     *         Route 53 to perform health checks.
     *         </p>
     *         <p>
     *         When Amazon Route 53 checks the health of an endpoint, here is how it constructs the <code>Host</code>
     *         header:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If you specify a value of <code>80</code> for <code>Port</code> and <code>HTTP</code> or
     *         <code>HTTP_STR_MATCH</code> for <code>Type</code>, Amazon Route 53 passes the value of
     *         <code>FullyQualifiedDomainName</code> to the endpoint in the <code>Host</code> header.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If you specify a value of <code>443</code> for <code>Port</code> and <code>HTTPS</code> or
     *         <code>HTTPS_STR_MATCH</code> for <code>Type</code>, Amazon Route 53 passes the value of
     *         <code>FullyQualifiedDomainName</code> to the endpoint in the <code>Host</code> header.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If you specify another value for <code>Port</code> and any value except <code>TCP</code> for
     *         <code>Type</code>, Amazon Route 53 passes <i> <code>FullyQualifiedDomainName</code>:<code>Port</code>
     *         </i> to the endpoint in the <code>Host</code> header.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If you don't specify a value for <code>FullyQualifiedDomainName</code>, Amazon Route 53 substitutes the
     *         value of <code>IPAddress</code> in the <code>Host</code> header in each of the above cases.
     *         </p>
     *         <p>
     *         <b>If you don't specify a value for</b> <code>IPAddress</code>:
     *         </p>
     *         <p>
     *         If you don't specify a value for <code>IPAddress</code>, Amazon Route 53 sends a DNS request to the
     *         domain that you specify in <code>FullyQualifiedDomainName</code> at the interval you specify in
     *         <code>RequestInterval</code>. Using an IPv4 address that is returned by DNS, Amazon Route 53 then checks
     *         the health of the endpoint.
     *         </p>
     *         <note>
     *         <p>
     *         If you don't specify a value for <code>IPAddress</code>, Amazon Route 53 uses only IPv4 to send health
     *         checks to the endpoint. If there's no resource record set with a type of A for the name that you specify
     *         for <code>FullyQualifiedDomainName</code>, the health check fails with a "DNS resolution failed" error.
     *         </p>
     *         </note>
     *         <p>
     *         If you want to check the health of weighted, latency, or failover resource record sets and you choose to
     *         specify the endpoint only by <code>FullyQualifiedDomainName</code>, we recommend that you create a
     *         separate health check for each endpoint. For example, create a health check for each HTTP server that is
     *         serving content for www.example.com. For the value of <code>FullyQualifiedDomainName</code>, specify the
     *         domain name of the server (such as <code>us-east-2-www.example.com</code>), not the name of the resource
     *         record sets (www.example.com).
     *         </p>
     *         <important>
     *         <p>
     *         In this configuration, if the value of <code>FullyQualifiedDomainName</code> matches the name of the
     *         resource record sets and you then associate the health check with those resource record sets, health
     *         check results will be unpredictable.
     *         </p>
     *         </important>
     *         <p>
     *         In addition, if the value of <code>Type</code> is <code>HTTP</code>, <code>HTTPS</code>,
     *         <code>HTTP_STR_MATCH</code>, or <code>HTTPS_STR_MATCH</code>, Amazon Route 53 passes the value of
     *         <code>FullyQualifiedDomainName</code> in the <code>Host</code> header, as it does when you specify a
     *         value for <code>IPAddress</code>. If the value of <code>Type</code> is <code>TCP</code>, Amazon Route 53
     *         doesn't pass a <code>Host</code> header.
     */
    public String fullyQualifiedDomainName() {
        return fullyQualifiedDomainName;
    }

    /**
     * <p>
     * If the value of <code>Type</code> is <code>HTTP_STR_MATCH</code> or <code>HTTP_STR_MATCH</code>, the string that
     * you want Amazon Route 53 to search for in the response body from the specified resource. If the string appears in
     * the response body, Amazon Route 53 considers the resource healthy. (You can't change the value of
     * <code>Type</code> when you update a health check.)
     * </p>
     * 
     * @return If the value of <code>Type</code> is <code>HTTP_STR_MATCH</code> or <code>HTTP_STR_MATCH</code>, the
     *         string that you want Amazon Route 53 to search for in the response body from the specified resource. If
     *         the string appears in the response body, Amazon Route 53 considers the resource healthy. (You can't
     *         change the value of <code>Type</code> when you update a health check.)
     */
    public String searchString() {
        return searchString;
    }

    /**
     * <p>
     * The number of consecutive health checks that an endpoint must pass or fail for Amazon Route 53 to change the
     * current status of the endpoint from unhealthy to healthy or vice versa. For more information, see <a href=
     * "http://docs.aws.amazon.com/Route53/latest/DeveloperGuide/dns-failover-determining-health-of-endpoints.html">How
     * Amazon Route 53 Determines Whether an Endpoint Is Healthy</a> in the <i>Amazon Route 53 Developer Guide</i>.
     * </p>
     * <p>
     * If you don't specify a value for <code>FailureThreshold</code>, the default value is three health checks.
     * </p>
     * 
     * @return The number of consecutive health checks that an endpoint must pass or fail for Amazon Route 53 to change
     *         the current status of the endpoint from unhealthy to healthy or vice versa. For more information, see <a
     *         href=
     *         "http://docs.aws.amazon.com/Route53/latest/DeveloperGuide/dns-failover-determining-health-of-endpoints.html"
     *         >How Amazon Route 53 Determines Whether an Endpoint Is Healthy</a> in the <i>Amazon Route 53 Developer
     *         Guide</i>.</p>
     *         <p>
     *         If you don't specify a value for <code>FailureThreshold</code>, the default value is three health checks.
     */
    public Integer failureThreshold() {
        return failureThreshold;
    }

    /**
     * <p>
     * Specify whether you want Amazon Route 53 to invert the status of a health check, for example, to consider a
     * health check unhealthy when it otherwise would be considered healthy.
     * </p>
     * 
     * @return Specify whether you want Amazon Route 53 to invert the status of a health check, for example, to consider
     *         a health check unhealthy when it otherwise would be considered healthy.
     */
    public Boolean inverted() {
        return inverted;
    }

    /**
     * <p>
     * The number of child health checks that are associated with a <code>CALCULATED</code> health that Amazon Route 53
     * must consider healthy for the <code>CALCULATED</code> health check to be considered healthy. To specify the child
     * health checks that you want to associate with a <code>CALCULATED</code> health check, use the
     * <code>ChildHealthChecks</code> and <code>ChildHealthCheck</code> elements.
     * </p>
     * <p>
     * Note the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If you specify a number greater than the number of child health checks, Amazon Route 53 always considers this
     * health check to be unhealthy.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you specify <code>0</code>, Amazon Route 53 always considers this health check to be healthy.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The number of child health checks that are associated with a <code>CALCULATED</code> health that Amazon
     *         Route 53 must consider healthy for the <code>CALCULATED</code> health check to be considered healthy. To
     *         specify the child health checks that you want to associate with a <code>CALCULATED</code> health check,
     *         use the <code>ChildHealthChecks</code> and <code>ChildHealthCheck</code> elements.</p>
     *         <p>
     *         Note the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If you specify a number greater than the number of child health checks, Amazon Route 53 always considers
     *         this health check to be unhealthy.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If you specify <code>0</code>, Amazon Route 53 always considers this health check to be healthy.
     *         </p>
     *         </li>
     */
    public Integer healthThreshold() {
        return healthThreshold;
    }

    /**
     * <p>
     * A complex type that contains one <code>ChildHealthCheck</code> element for each health check that you want to
     * associate with a <code>CALCULATED</code> health check.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A complex type that contains one <code>ChildHealthCheck</code> element for each health check that you
     *         want to associate with a <code>CALCULATED</code> health check.
     */
    public List<String> childHealthChecks() {
        return childHealthChecks;
    }

    /**
     * <p>
     * Specify whether you want Amazon Route 53 to send the value of <code>FullyQualifiedDomainName</code> to the
     * endpoint in the <code>client_hello</code> message during <code>TLS</code> negotiation. This allows the endpoint
     * to respond to <code>HTTPS</code> health check requests with the applicable SSL/TLS certificate.
     * </p>
     * <p>
     * Some endpoints require that HTTPS requests include the host name in the <code>client_hello</code> message. If you
     * don't enable SNI, the status of the health check will be SSL alert <code>handshake_failure</code>. A health check
     * can also have that status for other reasons. If SNI is enabled and you're still getting the error, check the
     * SSL/TLS configuration on your endpoint and confirm that your certificate is valid.
     * </p>
     * <p>
     * The SSL/TLS certificate on your endpoint includes a domain name in the <code>Common Name</code> field and
     * possibly several more in the <code>Subject Alternative Names</code> field. One of the domain names in the
     * certificate should match the value that you specify for <code>FullyQualifiedDomainName</code>. If the endpoint
     * responds to the <code>client_hello</code> message with a certificate that does not include the domain name that
     * you specified in <code>FullyQualifiedDomainName</code>, a health checker will retry the handshake. In the second
     * attempt, the health checker will omit <code>FullyQualifiedDomainName</code> from the <code>client_hello</code>
     * message.
     * </p>
     * 
     * @return Specify whether you want Amazon Route 53 to send the value of <code>FullyQualifiedDomainName</code> to
     *         the endpoint in the <code>client_hello</code> message during <code>TLS</code> negotiation. This allows
     *         the endpoint to respond to <code>HTTPS</code> health check requests with the applicable SSL/TLS
     *         certificate.</p>
     *         <p>
     *         Some endpoints require that HTTPS requests include the host name in the <code>client_hello</code>
     *         message. If you don't enable SNI, the status of the health check will be SSL alert
     *         <code>handshake_failure</code>. A health check can also have that status for other reasons. If SNI is
     *         enabled and you're still getting the error, check the SSL/TLS configuration on your endpoint and confirm
     *         that your certificate is valid.
     *         </p>
     *         <p>
     *         The SSL/TLS certificate on your endpoint includes a domain name in the <code>Common Name</code> field and
     *         possibly several more in the <code>Subject Alternative Names</code> field. One of the domain names in the
     *         certificate should match the value that you specify for <code>FullyQualifiedDomainName</code>. If the
     *         endpoint responds to the <code>client_hello</code> message with a certificate that does not include the
     *         domain name that you specified in <code>FullyQualifiedDomainName</code>, a health checker will retry the
     *         handshake. In the second attempt, the health checker will omit <code>FullyQualifiedDomainName</code> from
     *         the <code>client_hello</code> message.
     */
    public Boolean enableSNI() {
        return enableSNI;
    }

    /**
     * <p>
     * A complex type that contains one <code>Region</code> element for each region that you want Amazon Route 53 health
     * checkers to check the specified endpoint from.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A complex type that contains one <code>Region</code> element for each region that you want Amazon Route
     *         53 health checkers to check the specified endpoint from.
     */
    public List<HealthCheckRegion> regions() {
        return TypeConverter.convert(regions, HealthCheckRegion::fromValue);
    }

    /**
     * <p>
     * A complex type that contains one <code>Region</code> element for each region that you want Amazon Route 53 health
     * checkers to check the specified endpoint from.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A complex type that contains one <code>Region</code> element for each region that you want Amazon Route
     *         53 health checkers to check the specified endpoint from.
     */
    public List<String> regionsStrings() {
        return regions;
    }

    /**
     * Returns the value of the AlarmIdentifier property for this object.
     * 
     * @return The value of the AlarmIdentifier property for this object.
     */
    public AlarmIdentifier alarmIdentifier() {
        return alarmIdentifier;
    }

    /**
     * <p>
     * When CloudWatch has insufficient data about the metric to determine the alarm state, the status that you want
     * Amazon Route 53 to assign to the health check:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Healthy</code>: Amazon Route 53 considers the health check to be healthy.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Unhealthy</code>: Amazon Route 53 considers the health check to be unhealthy.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LastKnownStatus</code>: Amazon Route 53 uses the status of the health check from the last time CloudWatch
     * had sufficient data to determine the alarm state. For new health checks that have no last known status, the
     * default status for the health check is healthy.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #insufficientDataHealthStatus} will return {@link InsufficientDataHealthStatus#UNKNOWN_TO_SDK_VERSION}.
     * The raw value returned by the service is available from {@link #insufficientDataHealthStatusString}.
     * </p>
     * 
     * @return When CloudWatch has insufficient data about the metric to determine the alarm state, the status that you
     *         want Amazon Route 53 to assign to the health check:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Healthy</code>: Amazon Route 53 considers the health check to be healthy.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Unhealthy</code>: Amazon Route 53 considers the health check to be unhealthy.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>LastKnownStatus</code>: Amazon Route 53 uses the status of the health check from the last time
     *         CloudWatch had sufficient data to determine the alarm state. For new health checks that have no last
     *         known status, the default status for the health check is healthy.
     *         </p>
     *         </li>
     * @see InsufficientDataHealthStatus
     */
    public InsufficientDataHealthStatus insufficientDataHealthStatus() {
        return InsufficientDataHealthStatus.fromValue(insufficientDataHealthStatus);
    }

    /**
     * <p>
     * When CloudWatch has insufficient data about the metric to determine the alarm state, the status that you want
     * Amazon Route 53 to assign to the health check:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Healthy</code>: Amazon Route 53 considers the health check to be healthy.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Unhealthy</code>: Amazon Route 53 considers the health check to be unhealthy.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LastKnownStatus</code>: Amazon Route 53 uses the status of the health check from the last time CloudWatch
     * had sufficient data to determine the alarm state. For new health checks that have no last known status, the
     * default status for the health check is healthy.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #insufficientDataHealthStatus} will return {@link InsufficientDataHealthStatus#UNKNOWN_TO_SDK_VERSION}.
     * The raw value returned by the service is available from {@link #insufficientDataHealthStatusString}.
     * </p>
     * 
     * @return When CloudWatch has insufficient data about the metric to determine the alarm state, the status that you
     *         want Amazon Route 53 to assign to the health check:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Healthy</code>: Amazon Route 53 considers the health check to be healthy.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Unhealthy</code>: Amazon Route 53 considers the health check to be unhealthy.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>LastKnownStatus</code>: Amazon Route 53 uses the status of the health check from the last time
     *         CloudWatch had sufficient data to determine the alarm state. For new health checks that have no last
     *         known status, the default status for the health check is healthy.
     *         </p>
     *         </li>
     * @see InsufficientDataHealthStatus
     */
    public String insufficientDataHealthStatusString() {
        return insufficientDataHealthStatus;
    }

    /**
     * <p>
     * A complex type that contains one <code>ResettableElementName</code> element for each element that you want to
     * reset to the default value. Valid values for <code>ResettableElementName</code> include the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ChildHealthChecks</code>: Amazon Route 53 resets <a>HealthCheckConfig$ChildHealthChecks</a> to null.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FullyQualifiedDomainName</code>: Amazon Route 53 resets <a>HealthCheckConfig$FullyQualifiedDomainName</a>
     * to null.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Regions</code>: Amazon Route 53 resets the <a>HealthCheckConfig$Regions</a> list to the default set of
     * regions.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ResourcePath</code>: Amazon Route 53 resets <a>HealthCheckConfig$ResourcePath</a> to null.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A complex type that contains one <code>ResettableElementName</code> element for each element that you
     *         want to reset to the default value. Valid values for <code>ResettableElementName</code> include the
     *         following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ChildHealthChecks</code>: Amazon Route 53 resets <a>HealthCheckConfig$ChildHealthChecks</a> to
     *         null.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FullyQualifiedDomainName</code>: Amazon Route 53 resets
     *         <a>HealthCheckConfig$FullyQualifiedDomainName</a> to null.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Regions</code>: Amazon Route 53 resets the <a>HealthCheckConfig$Regions</a> list to the default set
     *         of regions.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ResourcePath</code>: Amazon Route 53 resets <a>HealthCheckConfig$ResourcePath</a> to null.
     *         </p>
     *         </li>
     */
    public List<ResettableElementName> resetElements() {
        return TypeConverter.convert(resetElements, ResettableElementName::fromValue);
    }

    /**
     * <p>
     * A complex type that contains one <code>ResettableElementName</code> element for each element that you want to
     * reset to the default value. Valid values for <code>ResettableElementName</code> include the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ChildHealthChecks</code>: Amazon Route 53 resets <a>HealthCheckConfig$ChildHealthChecks</a> to null.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FullyQualifiedDomainName</code>: Amazon Route 53 resets <a>HealthCheckConfig$FullyQualifiedDomainName</a>
     * to null.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Regions</code>: Amazon Route 53 resets the <a>HealthCheckConfig$Regions</a> list to the default set of
     * regions.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ResourcePath</code>: Amazon Route 53 resets <a>HealthCheckConfig$ResourcePath</a> to null.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A complex type that contains one <code>ResettableElementName</code> element for each element that you
     *         want to reset to the default value. Valid values for <code>ResettableElementName</code> include the
     *         following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ChildHealthChecks</code>: Amazon Route 53 resets <a>HealthCheckConfig$ChildHealthChecks</a> to
     *         null.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FullyQualifiedDomainName</code>: Amazon Route 53 resets
     *         <a>HealthCheckConfig$FullyQualifiedDomainName</a> to null.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Regions</code>: Amazon Route 53 resets the <a>HealthCheckConfig$Regions</a> list to the default set
     *         of regions.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ResourcePath</code>: Amazon Route 53 resets <a>HealthCheckConfig$ResourcePath</a> to null.
     *         </p>
     *         </li>
     */
    public List<String> resetElementsStrings() {
        return resetElements;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(healthCheckId());
        hashCode = 31 * hashCode + Objects.hashCode(healthCheckVersion());
        hashCode = 31 * hashCode + Objects.hashCode(ipAddress());
        hashCode = 31 * hashCode + Objects.hashCode(port());
        hashCode = 31 * hashCode + Objects.hashCode(resourcePath());
        hashCode = 31 * hashCode + Objects.hashCode(fullyQualifiedDomainName());
        hashCode = 31 * hashCode + Objects.hashCode(searchString());
        hashCode = 31 * hashCode + Objects.hashCode(failureThreshold());
        hashCode = 31 * hashCode + Objects.hashCode(inverted());
        hashCode = 31 * hashCode + Objects.hashCode(healthThreshold());
        hashCode = 31 * hashCode + Objects.hashCode(childHealthChecks());
        hashCode = 31 * hashCode + Objects.hashCode(enableSNI());
        hashCode = 31 * hashCode + Objects.hashCode(regionsStrings());
        hashCode = 31 * hashCode + Objects.hashCode(alarmIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(insufficientDataHealthStatusString());
        hashCode = 31 * hashCode + Objects.hashCode(resetElementsStrings());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof UpdateHealthCheckRequest)) {
            return false;
        }
        UpdateHealthCheckRequest other = (UpdateHealthCheckRequest) obj;
        return Objects.equals(healthCheckId(), other.healthCheckId())
                && Objects.equals(healthCheckVersion(), other.healthCheckVersion())
                && Objects.equals(ipAddress(), other.ipAddress()) && Objects.equals(port(), other.port())
                && Objects.equals(resourcePath(), other.resourcePath())
                && Objects.equals(fullyQualifiedDomainName(), other.fullyQualifiedDomainName())
                && Objects.equals(searchString(), other.searchString())
                && Objects.equals(failureThreshold(), other.failureThreshold()) && Objects.equals(inverted(), other.inverted())
                && Objects.equals(healthThreshold(), other.healthThreshold())
                && Objects.equals(childHealthChecks(), other.childHealthChecks())
                && Objects.equals(enableSNI(), other.enableSNI()) && Objects.equals(regionsStrings(), other.regionsStrings())
                && Objects.equals(alarmIdentifier(), other.alarmIdentifier())
                && Objects.equals(insufficientDataHealthStatusString(), other.insufficientDataHealthStatusString())
                && Objects.equals(resetElementsStrings(), other.resetElementsStrings());
    }

    @Override
    public String toString() {
        return ToString.builder("UpdateHealthCheckRequest").add("HealthCheckId", healthCheckId())
                .add("HealthCheckVersion", healthCheckVersion()).add("IPAddress", ipAddress()).add("Port", port())
                .add("ResourcePath", resourcePath()).add("FullyQualifiedDomainName", fullyQualifiedDomainName())
                .add("SearchString", searchString()).add("FailureThreshold", failureThreshold()).add("Inverted", inverted())
                .add("HealthThreshold", healthThreshold()).add("ChildHealthChecks", childHealthChecks())
                .add("EnableSNI", enableSNI()).add("Regions", regionsStrings()).add("AlarmIdentifier", alarmIdentifier())
                .add("InsufficientDataHealthStatus", insufficientDataHealthStatusString())
                .add("ResetElements", resetElementsStrings()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "HealthCheckId":
            return Optional.of(clazz.cast(healthCheckId()));
        case "HealthCheckVersion":
            return Optional.of(clazz.cast(healthCheckVersion()));
        case "IPAddress":
            return Optional.of(clazz.cast(ipAddress()));
        case "Port":
            return Optional.of(clazz.cast(port()));
        case "ResourcePath":
            return Optional.of(clazz.cast(resourcePath()));
        case "FullyQualifiedDomainName":
            return Optional.of(clazz.cast(fullyQualifiedDomainName()));
        case "SearchString":
            return Optional.of(clazz.cast(searchString()));
        case "FailureThreshold":
            return Optional.of(clazz.cast(failureThreshold()));
        case "Inverted":
            return Optional.of(clazz.cast(inverted()));
        case "HealthThreshold":
            return Optional.of(clazz.cast(healthThreshold()));
        case "ChildHealthChecks":
            return Optional.of(clazz.cast(childHealthChecks()));
        case "EnableSNI":
            return Optional.of(clazz.cast(enableSNI()));
        case "Regions":
            return Optional.of(clazz.cast(regionsStrings()));
        case "AlarmIdentifier":
            return Optional.of(clazz.cast(alarmIdentifier()));
        case "InsufficientDataHealthStatus":
            return Optional.of(clazz.cast(insufficientDataHealthStatusString()));
        case "ResetElements":
            return Optional.of(clazz.cast(resetElementsStrings()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends Route53Request.Builder, CopyableBuilder<Builder, UpdateHealthCheckRequest> {
        /**
         * <p>
         * The ID for the health check for which you want detailed information. When you created the health check,
         * <code>CreateHealthCheck</code> returned the ID in the response, in the <code>HealthCheckId</code> element.
         * </p>
         * 
         * @param healthCheckId
         *        The ID for the health check for which you want detailed information. When you created the health
         *        check, <code>CreateHealthCheck</code> returned the ID in the response, in the
         *        <code>HealthCheckId</code> element.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder healthCheckId(String healthCheckId);

        /**
         * <p>
         * A sequential counter that Amazon Route 53 sets to <code>1</code> when you create a health check and
         * increments by 1 each time you update settings for the health check.
         * </p>
         * <p>
         * We recommend that you use <code>GetHealthCheck</code> or <code>ListHealthChecks</code> to get the current
         * value of <code>HealthCheckVersion</code> for the health check that you want to update, and that you include
         * that value in your <code>UpdateHealthCheck</code> request. This prevents Amazon Route 53 from overwriting an
         * intervening update:
         * </p>
         * <ul>
         * <li>
         * <p>
         * If the value in the <code>UpdateHealthCheck</code> request matches the value of
         * <code>HealthCheckVersion</code> in the health check, Amazon Route 53 updates the health check with the new
         * settings.
         * </p>
         * </li>
         * <li>
         * <p>
         * If the value of <code>HealthCheckVersion</code> in the health check is greater, the health check was changed
         * after you got the version number. Amazon Route 53 does not update the health check, and it returns a
         * <code>HealthCheckVersionMismatch</code> error.
         * </p>
         * </li>
         * </ul>
         * 
         * @param healthCheckVersion
         *        A sequential counter that Amazon Route 53 sets to <code>1</code> when you create a health check and
         *        increments by 1 each time you update settings for the health check.</p>
         *        <p>
         *        We recommend that you use <code>GetHealthCheck</code> or <code>ListHealthChecks</code> to get the
         *        current value of <code>HealthCheckVersion</code> for the health check that you want to update, and
         *        that you include that value in your <code>UpdateHealthCheck</code> request. This prevents Amazon Route
         *        53 from overwriting an intervening update:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If the value in the <code>UpdateHealthCheck</code> request matches the value of
         *        <code>HealthCheckVersion</code> in the health check, Amazon Route 53 updates the health check with the
         *        new settings.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If the value of <code>HealthCheckVersion</code> in the health check is greater, the health check was
         *        changed after you got the version number. Amazon Route 53 does not update the health check, and it
         *        returns a <code>HealthCheckVersionMismatch</code> error.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder healthCheckVersion(Long healthCheckVersion);

        /**
         * <p>
         * The IPv4 or IPv6 IP address for the endpoint that you want Amazon Route 53 to perform health checks on. If
         * you don't specify a value for <code>IPAddress</code>, Amazon Route 53 sends a DNS request to resolve the
         * domain name that you specify in <code>FullyQualifiedDomainName</code> at the interval that you specify in
         * <code>RequestInterval</code>. Using an IP address that is returned by DNS, Amazon Route 53 then checks the
         * health of the endpoint.
         * </p>
         * <p>
         * Use one of the following formats for the value of <code>IPAddress</code>:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>IPv4 address</b>: four values between 0 and 255, separated by periods (.), for example,
         * <code>192.0.2.44</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>IPv6 address</b>: eight groups of four hexadecimal values, separated by colons (:), for example,
         * <code>2001:0db8:85a3:0000:0000:abcd:0001:2345</code>. You can also shorten IPv6 addresses as described in RFC
         * 5952, for example, <code>2001:db8:85a3::abcd:1:2345</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If the endpoint is an EC2 instance, we recommend that you create an Elastic IP address, associate it with
         * your EC2 instance, and specify the Elastic IP address for <code>IPAddress</code>. This ensures that the IP
         * address of your instance never changes. For more information, see the applicable documentation:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Linux: <a href="http://docs.aws.amazon.com/AWSEC2/latest/UserGuide/elastic-ip-addresses-eip.html">Elastic IP
         * Addresses (EIP)</a> in the <i>Amazon EC2 User Guide for Linux Instances</i>
         * </p>
         * </li>
         * <li>
         * <p>
         * Windows: <a
         * href="http://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/elastic-ip-addresses-eip.html">Elastic IP
         * Addresses (EIP)</a> in the <i>Amazon EC2 User Guide for Windows Instances</i>
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * If a health check already has a value for <code>IPAddress</code>, you can change the value. However, you
         * can't update an existing health check to add or remove the value of <code>IPAddress</code>.
         * </p>
         * </note>
         * <p>
         * For more information, see <a>UpdateHealthCheckRequest$FullyQualifiedDomainName</a>.
         * </p>
         * <p>
         * Constraints: Amazon Route 53 can't check the health of endpoints for which the IP address is in local,
         * private, non-routable, or multicast ranges. For more information about IP addresses for which you can't
         * create health checks, see the following documents:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <a href="https://tools.ietf.org/html/rfc5735">RFC 5735, Special Use IPv4 Addresses</a>
         * </p>
         * </li>
         * <li>
         * <p>
         * <a href="https://tools.ietf.org/html/rfc6598">RFC 6598, IANA-Reserved IPv4 Prefix for Shared Address
         * Space</a>
         * </p>
         * </li>
         * <li>
         * <p>
         * <a href="https://tools.ietf.org/html/rfc5156">RFC 5156, Special-Use IPv6 Addresses</a>
         * </p>
         * </li>
         * </ul>
         * 
         * @param ipAddress
         *        The IPv4 or IPv6 IP address for the endpoint that you want Amazon Route 53 to perform health checks
         *        on. If you don't specify a value for <code>IPAddress</code>, Amazon Route 53 sends a DNS request to
         *        resolve the domain name that you specify in <code>FullyQualifiedDomainName</code> at the interval that
         *        you specify in <code>RequestInterval</code>. Using an IP address that is returned by DNS, Amazon Route
         *        53 then checks the health of the endpoint.</p>
         *        <p>
         *        Use one of the following formats for the value of <code>IPAddress</code>:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>IPv4 address</b>: four values between 0 and 255, separated by periods (.), for example,
         *        <code>192.0.2.44</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>IPv6 address</b>: eight groups of four hexadecimal values, separated by colons (:), for example,
         *        <code>2001:0db8:85a3:0000:0000:abcd:0001:2345</code>. You can also shorten IPv6 addresses as described
         *        in RFC 5952, for example, <code>2001:db8:85a3::abcd:1:2345</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If the endpoint is an EC2 instance, we recommend that you create an Elastic IP address, associate it
         *        with your EC2 instance, and specify the Elastic IP address for <code>IPAddress</code>. This ensures
         *        that the IP address of your instance never changes. For more information, see the applicable
         *        documentation:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Linux: <a
         *        href="http://docs.aws.amazon.com/AWSEC2/latest/UserGuide/elastic-ip-addresses-eip.html">Elastic IP
         *        Addresses (EIP)</a> in the <i>Amazon EC2 User Guide for Linux Instances</i>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Windows: <a
         *        href="http://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/elastic-ip-addresses-eip.html">Elastic IP
         *        Addresses (EIP)</a> in the <i>Amazon EC2 User Guide for Windows Instances</i>
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        If a health check already has a value for <code>IPAddress</code>, you can change the value. However,
         *        you can't update an existing health check to add or remove the value of <code>IPAddress</code>.
         *        </p>
         *        </note>
         *        <p>
         *        For more information, see <a>UpdateHealthCheckRequest$FullyQualifiedDomainName</a>.
         *        </p>
         *        <p>
         *        Constraints: Amazon Route 53 can't check the health of endpoints for which the IP address is in local,
         *        private, non-routable, or multicast ranges. For more information about IP addresses for which you
         *        can't create health checks, see the following documents:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <a href="https://tools.ietf.org/html/rfc5735">RFC 5735, Special Use IPv4 Addresses</a>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <a href="https://tools.ietf.org/html/rfc6598">RFC 6598, IANA-Reserved IPv4 Prefix for Shared Address
         *        Space</a>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <a href="https://tools.ietf.org/html/rfc5156">RFC 5156, Special-Use IPv6 Addresses</a>
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipAddress(String ipAddress);

        /**
         * <p>
         * The port on the endpoint on which you want Amazon Route 53 to perform health checks.
         * </p>
         * 
         * @param port
         *        The port on the endpoint on which you want Amazon Route 53 to perform health checks.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder port(Integer port);

        /**
         * <p>
         * The path that you want Amazon Route 53 to request when performing health checks. The path can be any value
         * for which your endpoint will return an HTTP status code of 2xx or 3xx when the endpoint is healthy, for
         * example the file /docs/route53-health-check.html.
         * </p>
         * <p>
         * Specify this value only if you want to change it.
         * </p>
         * 
         * @param resourcePath
         *        The path that you want Amazon Route 53 to request when performing health checks. The path can be any
         *        value for which your endpoint will return an HTTP status code of 2xx or 3xx when the endpoint is
         *        healthy, for example the file /docs/route53-health-check.html. </p>
         *        <p>
         *        Specify this value only if you want to change it.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourcePath(String resourcePath);

        /**
         * <p>
         * Amazon Route 53 behavior depends on whether you specify a value for <code>IPAddress</code>.
         * </p>
         * <note>
         * <p>
         * If a health check already has a value for <code>IPAddress</code>, you can change the value. However, you
         * can't update an existing health check to add or remove the value of <code>IPAddress</code>.
         * </p>
         * </note>
         * <p>
         * <b>If you specify a value for</b> <code>IPAddress</code>:
         * </p>
         * <p>
         * Amazon Route 53 sends health check requests to the specified IPv4 or IPv6 address and passes the value of
         * <code>FullyQualifiedDomainName</code> in the <code>Host</code> header for all health checks except TCP health
         * checks. This is typically the fully qualified DNS name of the endpoint on which you want Amazon Route 53 to
         * perform health checks.
         * </p>
         * <p>
         * When Amazon Route 53 checks the health of an endpoint, here is how it constructs the <code>Host</code>
         * header:
         * </p>
         * <ul>
         * <li>
         * <p>
         * If you specify a value of <code>80</code> for <code>Port</code> and <code>HTTP</code> or
         * <code>HTTP_STR_MATCH</code> for <code>Type</code>, Amazon Route 53 passes the value of
         * <code>FullyQualifiedDomainName</code> to the endpoint in the <code>Host</code> header.
         * </p>
         * </li>
         * <li>
         * <p>
         * If you specify a value of <code>443</code> for <code>Port</code> and <code>HTTPS</code> or
         * <code>HTTPS_STR_MATCH</code> for <code>Type</code>, Amazon Route 53 passes the value of
         * <code>FullyQualifiedDomainName</code> to the endpoint in the <code>Host</code> header.
         * </p>
         * </li>
         * <li>
         * <p>
         * If you specify another value for <code>Port</code> and any value except <code>TCP</code> for
         * <code>Type</code>, Amazon Route 53 passes <i> <code>FullyQualifiedDomainName</code>:<code>Port</code> </i> to
         * the endpoint in the <code>Host</code> header.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If you don't specify a value for <code>FullyQualifiedDomainName</code>, Amazon Route 53 substitutes the value
         * of <code>IPAddress</code> in the <code>Host</code> header in each of the above cases.
         * </p>
         * <p>
         * <b>If you don't specify a value for</b> <code>IPAddress</code>:
         * </p>
         * <p>
         * If you don't specify a value for <code>IPAddress</code>, Amazon Route 53 sends a DNS request to the domain
         * that you specify in <code>FullyQualifiedDomainName</code> at the interval you specify in
         * <code>RequestInterval</code>. Using an IPv4 address that is returned by DNS, Amazon Route 53 then checks the
         * health of the endpoint.
         * </p>
         * <note>
         * <p>
         * If you don't specify a value for <code>IPAddress</code>, Amazon Route 53 uses only IPv4 to send health checks
         * to the endpoint. If there's no resource record set with a type of A for the name that you specify for
         * <code>FullyQualifiedDomainName</code>, the health check fails with a "DNS resolution failed" error.
         * </p>
         * </note>
         * <p>
         * If you want to check the health of weighted, latency, or failover resource record sets and you choose to
         * specify the endpoint only by <code>FullyQualifiedDomainName</code>, we recommend that you create a separate
         * health check for each endpoint. For example, create a health check for each HTTP server that is serving
         * content for www.example.com. For the value of <code>FullyQualifiedDomainName</code>, specify the domain name
         * of the server (such as <code>us-east-2-www.example.com</code>), not the name of the resource record sets
         * (www.example.com).
         * </p>
         * <important>
         * <p>
         * In this configuration, if the value of <code>FullyQualifiedDomainName</code> matches the name of the resource
         * record sets and you then associate the health check with those resource record sets, health check results
         * will be unpredictable.
         * </p>
         * </important>
         * <p>
         * In addition, if the value of <code>Type</code> is <code>HTTP</code>, <code>HTTPS</code>,
         * <code>HTTP_STR_MATCH</code>, or <code>HTTPS_STR_MATCH</code>, Amazon Route 53 passes the value of
         * <code>FullyQualifiedDomainName</code> in the <code>Host</code> header, as it does when you specify a value
         * for <code>IPAddress</code>. If the value of <code>Type</code> is <code>TCP</code>, Amazon Route 53 doesn't
         * pass a <code>Host</code> header.
         * </p>
         * 
         * @param fullyQualifiedDomainName
         *        Amazon Route 53 behavior depends on whether you specify a value for <code>IPAddress</code>.</p> <note>
         *        <p>
         *        If a health check already has a value for <code>IPAddress</code>, you can change the value. However,
         *        you can't update an existing health check to add or remove the value of <code>IPAddress</code>.
         *        </p>
         *        </note>
         *        <p>
         *        <b>If you specify a value for</b> <code>IPAddress</code>:
         *        </p>
         *        <p>
         *        Amazon Route 53 sends health check requests to the specified IPv4 or IPv6 address and passes the value
         *        of <code>FullyQualifiedDomainName</code> in the <code>Host</code> header for all health checks except
         *        TCP health checks. This is typically the fully qualified DNS name of the endpoint on which you want
         *        Amazon Route 53 to perform health checks.
         *        </p>
         *        <p>
         *        When Amazon Route 53 checks the health of an endpoint, here is how it constructs the <code>Host</code>
         *        header:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If you specify a value of <code>80</code> for <code>Port</code> and <code>HTTP</code> or
         *        <code>HTTP_STR_MATCH</code> for <code>Type</code>, Amazon Route 53 passes the value of
         *        <code>FullyQualifiedDomainName</code> to the endpoint in the <code>Host</code> header.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you specify a value of <code>443</code> for <code>Port</code> and <code>HTTPS</code> or
         *        <code>HTTPS_STR_MATCH</code> for <code>Type</code>, Amazon Route 53 passes the value of
         *        <code>FullyQualifiedDomainName</code> to the endpoint in the <code>Host</code> header.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you specify another value for <code>Port</code> and any value except <code>TCP</code> for
         *        <code>Type</code>, Amazon Route 53 passes <i> <code>FullyQualifiedDomainName</code>:<code>Port</code>
         *        </i> to the endpoint in the <code>Host</code> header.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If you don't specify a value for <code>FullyQualifiedDomainName</code>, Amazon Route 53 substitutes
         *        the value of <code>IPAddress</code> in the <code>Host</code> header in each of the above cases.
         *        </p>
         *        <p>
         *        <b>If you don't specify a value for</b> <code>IPAddress</code>:
         *        </p>
         *        <p>
         *        If you don't specify a value for <code>IPAddress</code>, Amazon Route 53 sends a DNS request to the
         *        domain that you specify in <code>FullyQualifiedDomainName</code> at the interval you specify in
         *        <code>RequestInterval</code>. Using an IPv4 address that is returned by DNS, Amazon Route 53 then
         *        checks the health of the endpoint.
         *        </p>
         *        <note>
         *        <p>
         *        If you don't specify a value for <code>IPAddress</code>, Amazon Route 53 uses only IPv4 to send health
         *        checks to the endpoint. If there's no resource record set with a type of A for the name that you
         *        specify for <code>FullyQualifiedDomainName</code>, the health check fails with a
         *        "DNS resolution failed" error.
         *        </p>
         *        </note>
         *        <p>
         *        If you want to check the health of weighted, latency, or failover resource record sets and you choose
         *        to specify the endpoint only by <code>FullyQualifiedDomainName</code>, we recommend that you create a
         *        separate health check for each endpoint. For example, create a health check for each HTTP server that
         *        is serving content for www.example.com. For the value of <code>FullyQualifiedDomainName</code>,
         *        specify the domain name of the server (such as <code>us-east-2-www.example.com</code>), not the name
         *        of the resource record sets (www.example.com).
         *        </p>
         *        <important>
         *        <p>
         *        In this configuration, if the value of <code>FullyQualifiedDomainName</code> matches the name of the
         *        resource record sets and you then associate the health check with those resource record sets, health
         *        check results will be unpredictable.
         *        </p>
         *        </important>
         *        <p>
         *        In addition, if the value of <code>Type</code> is <code>HTTP</code>, <code>HTTPS</code>,
         *        <code>HTTP_STR_MATCH</code>, or <code>HTTPS_STR_MATCH</code>, Amazon Route 53 passes the value of
         *        <code>FullyQualifiedDomainName</code> in the <code>Host</code> header, as it does when you specify a
         *        value for <code>IPAddress</code>. If the value of <code>Type</code> is <code>TCP</code>, Amazon Route
         *        53 doesn't pass a <code>Host</code> header.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fullyQualifiedDomainName(String fullyQualifiedDomainName);

        /**
         * <p>
         * If the value of <code>Type</code> is <code>HTTP_STR_MATCH</code> or <code>HTTP_STR_MATCH</code>, the string
         * that you want Amazon Route 53 to search for in the response body from the specified resource. If the string
         * appears in the response body, Amazon Route 53 considers the resource healthy. (You can't change the value of
         * <code>Type</code> when you update a health check.)
         * </p>
         * 
         * @param searchString
         *        If the value of <code>Type</code> is <code>HTTP_STR_MATCH</code> or <code>HTTP_STR_MATCH</code>, the
         *        string that you want Amazon Route 53 to search for in the response body from the specified resource.
         *        If the string appears in the response body, Amazon Route 53 considers the resource healthy. (You can't
         *        change the value of <code>Type</code> when you update a health check.)
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder searchString(String searchString);

        /**
         * <p>
         * The number of consecutive health checks that an endpoint must pass or fail for Amazon Route 53 to change the
         * current status of the endpoint from unhealthy to healthy or vice versa. For more information, see <a href=
         * "http://docs.aws.amazon.com/Route53/latest/DeveloperGuide/dns-failover-determining-health-of-endpoints.html"
         * >How Amazon Route 53 Determines Whether an Endpoint Is Healthy</a> in the <i>Amazon Route 53 Developer
         * Guide</i>.
         * </p>
         * <p>
         * If you don't specify a value for <code>FailureThreshold</code>, the default value is three health checks.
         * </p>
         * 
         * @param failureThreshold
         *        The number of consecutive health checks that an endpoint must pass or fail for Amazon Route 53 to
         *        change the current status of the endpoint from unhealthy to healthy or vice versa. For more
         *        information, see <a href=
         *        "http://docs.aws.amazon.com/Route53/latest/DeveloperGuide/dns-failover-determining-health-of-endpoints.html"
         *        >How Amazon Route 53 Determines Whether an Endpoint Is Healthy</a> in the <i>Amazon Route 53 Developer
         *        Guide</i>.</p>
         *        <p>
         *        If you don't specify a value for <code>FailureThreshold</code>, the default value is three health
         *        checks.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureThreshold(Integer failureThreshold);

        /**
         * <p>
         * Specify whether you want Amazon Route 53 to invert the status of a health check, for example, to consider a
         * health check unhealthy when it otherwise would be considered healthy.
         * </p>
         * 
         * @param inverted
         *        Specify whether you want Amazon Route 53 to invert the status of a health check, for example, to
         *        consider a health check unhealthy when it otherwise would be considered healthy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inverted(Boolean inverted);

        /**
         * <p>
         * The number of child health checks that are associated with a <code>CALCULATED</code> health that Amazon Route
         * 53 must consider healthy for the <code>CALCULATED</code> health check to be considered healthy. To specify
         * the child health checks that you want to associate with a <code>CALCULATED</code> health check, use the
         * <code>ChildHealthChecks</code> and <code>ChildHealthCheck</code> elements.
         * </p>
         * <p>
         * Note the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * If you specify a number greater than the number of child health checks, Amazon Route 53 always considers this
         * health check to be unhealthy.
         * </p>
         * </li>
         * <li>
         * <p>
         * If you specify <code>0</code>, Amazon Route 53 always considers this health check to be healthy.
         * </p>
         * </li>
         * </ul>
         * 
         * @param healthThreshold
         *        The number of child health checks that are associated with a <code>CALCULATED</code> health that
         *        Amazon Route 53 must consider healthy for the <code>CALCULATED</code> health check to be considered
         *        healthy. To specify the child health checks that you want to associate with a <code>CALCULATED</code>
         *        health check, use the <code>ChildHealthChecks</code> and <code>ChildHealthCheck</code> elements.</p>
         *        <p>
         *        Note the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If you specify a number greater than the number of child health checks, Amazon Route 53 always
         *        considers this health check to be unhealthy.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you specify <code>0</code>, Amazon Route 53 always considers this health check to be healthy.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder healthThreshold(Integer healthThreshold);

        /**
         * <p>
         * A complex type that contains one <code>ChildHealthCheck</code> element for each health check that you want to
         * associate with a <code>CALCULATED</code> health check.
         * </p>
         * 
         * @param childHealthChecks
         *        A complex type that contains one <code>ChildHealthCheck</code> element for each health check that you
         *        want to associate with a <code>CALCULATED</code> health check.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder childHealthChecks(Collection<String> childHealthChecks);

        /**
         * <p>
         * A complex type that contains one <code>ChildHealthCheck</code> element for each health check that you want to
         * associate with a <code>CALCULATED</code> health check.
         * </p>
         * 
         * @param childHealthChecks
         *        A complex type that contains one <code>ChildHealthCheck</code> element for each health check that you
         *        want to associate with a <code>CALCULATED</code> health check.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder childHealthChecks(String... childHealthChecks);

        /**
         * <p>
         * Specify whether you want Amazon Route 53 to send the value of <code>FullyQualifiedDomainName</code> to the
         * endpoint in the <code>client_hello</code> message during <code>TLS</code> negotiation. This allows the
         * endpoint to respond to <code>HTTPS</code> health check requests with the applicable SSL/TLS certificate.
         * </p>
         * <p>
         * Some endpoints require that HTTPS requests include the host name in the <code>client_hello</code> message. If
         * you don't enable SNI, the status of the health check will be SSL alert <code>handshake_failure</code>. A
         * health check can also have that status for other reasons. If SNI is enabled and you're still getting the
         * error, check the SSL/TLS configuration on your endpoint and confirm that your certificate is valid.
         * </p>
         * <p>
         * The SSL/TLS certificate on your endpoint includes a domain name in the <code>Common Name</code> field and
         * possibly several more in the <code>Subject Alternative Names</code> field. One of the domain names in the
         * certificate should match the value that you specify for <code>FullyQualifiedDomainName</code>. If the
         * endpoint responds to the <code>client_hello</code> message with a certificate that does not include the
         * domain name that you specified in <code>FullyQualifiedDomainName</code>, a health checker will retry the
         * handshake. In the second attempt, the health checker will omit <code>FullyQualifiedDomainName</code> from the
         * <code>client_hello</code> message.
         * </p>
         * 
         * @param enableSNI
         *        Specify whether you want Amazon Route 53 to send the value of <code>FullyQualifiedDomainName</code> to
         *        the endpoint in the <code>client_hello</code> message during <code>TLS</code> negotiation. This allows
         *        the endpoint to respond to <code>HTTPS</code> health check requests with the applicable SSL/TLS
         *        certificate.</p>
         *        <p>
         *        Some endpoints require that HTTPS requests include the host name in the <code>client_hello</code>
         *        message. If you don't enable SNI, the status of the health check will be SSL alert
         *        <code>handshake_failure</code>. A health check can also have that status for other reasons. If SNI is
         *        enabled and you're still getting the error, check the SSL/TLS configuration on your endpoint and
         *        confirm that your certificate is valid.
         *        </p>
         *        <p>
         *        The SSL/TLS certificate on your endpoint includes a domain name in the <code>Common Name</code> field
         *        and possibly several more in the <code>Subject Alternative Names</code> field. One of the domain names
         *        in the certificate should match the value that you specify for <code>FullyQualifiedDomainName</code>.
         *        If the endpoint responds to the <code>client_hello</code> message with a certificate that does not
         *        include the domain name that you specified in <code>FullyQualifiedDomainName</code>, a health checker
         *        will retry the handshake. In the second attempt, the health checker will omit
         *        <code>FullyQualifiedDomainName</code> from the <code>client_hello</code> message.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enableSNI(Boolean enableSNI);

        /**
         * <p>
         * A complex type that contains one <code>Region</code> element for each region that you want Amazon Route 53
         * health checkers to check the specified endpoint from.
         * </p>
         * 
         * @param regions
         *        A complex type that contains one <code>Region</code> element for each region that you want Amazon
         *        Route 53 health checkers to check the specified endpoint from.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder regions(Collection<String> regions);

        /**
         * <p>
         * A complex type that contains one <code>Region</code> element for each region that you want Amazon Route 53
         * health checkers to check the specified endpoint from.
         * </p>
         * 
         * @param regions
         *        A complex type that contains one <code>Region</code> element for each region that you want Amazon
         *        Route 53 health checkers to check the specified endpoint from.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder regions(String... regions);

        /**
         * Sets the value of the AlarmIdentifier property for this object.
         *
         * @param alarmIdentifier
         *        The new value for the AlarmIdentifier property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder alarmIdentifier(AlarmIdentifier alarmIdentifier);

        /**
         * Sets the value of the AlarmIdentifier property for this object.
         *
         * This is a convenience that creates an instance of the {@link AlarmIdentifier.Builder} avoiding the need to
         * create one manually via {@link AlarmIdentifier#builder()}.
         *
         * When the {@link Consumer} completes, {@link AlarmIdentifier.Builder#build()} is called immediately and its
         * result is passed to {@link #alarmIdentifier(AlarmIdentifier)}.
         * 
         * @param alarmIdentifier
         *        a consumer that will call methods on {@link AlarmIdentifier.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #alarmIdentifier(AlarmIdentifier)
         */
        default Builder alarmIdentifier(Consumer<AlarmIdentifier.Builder> alarmIdentifier) {
            return alarmIdentifier(AlarmIdentifier.builder().apply(alarmIdentifier).build());
        }

        /**
         * <p>
         * When CloudWatch has insufficient data about the metric to determine the alarm state, the status that you want
         * Amazon Route 53 to assign to the health check:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Healthy</code>: Amazon Route 53 considers the health check to be healthy.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Unhealthy</code>: Amazon Route 53 considers the health check to be unhealthy.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>LastKnownStatus</code>: Amazon Route 53 uses the status of the health check from the last time
         * CloudWatch had sufficient data to determine the alarm state. For new health checks that have no last known
         * status, the default status for the health check is healthy.
         * </p>
         * </li>
         * </ul>
         * 
         * @param insufficientDataHealthStatus
         *        When CloudWatch has insufficient data about the metric to determine the alarm state, the status that
         *        you want Amazon Route 53 to assign to the health check:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Healthy</code>: Amazon Route 53 considers the health check to be healthy.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Unhealthy</code>: Amazon Route 53 considers the health check to be unhealthy.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>LastKnownStatus</code>: Amazon Route 53 uses the status of the health check from the last time
         *        CloudWatch had sufficient data to determine the alarm state. For new health checks that have no last
         *        known status, the default status for the health check is healthy.
         *        </p>
         *        </li>
         * @see InsufficientDataHealthStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InsufficientDataHealthStatus
         */
        Builder insufficientDataHealthStatus(String insufficientDataHealthStatus);

        /**
         * <p>
         * When CloudWatch has insufficient data about the metric to determine the alarm state, the status that you want
         * Amazon Route 53 to assign to the health check:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Healthy</code>: Amazon Route 53 considers the health check to be healthy.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Unhealthy</code>: Amazon Route 53 considers the health check to be unhealthy.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>LastKnownStatus</code>: Amazon Route 53 uses the status of the health check from the last time
         * CloudWatch had sufficient data to determine the alarm state. For new health checks that have no last known
         * status, the default status for the health check is healthy.
         * </p>
         * </li>
         * </ul>
         * 
         * @param insufficientDataHealthStatus
         *        When CloudWatch has insufficient data about the metric to determine the alarm state, the status that
         *        you want Amazon Route 53 to assign to the health check:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Healthy</code>: Amazon Route 53 considers the health check to be healthy.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Unhealthy</code>: Amazon Route 53 considers the health check to be unhealthy.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>LastKnownStatus</code>: Amazon Route 53 uses the status of the health check from the last time
         *        CloudWatch had sufficient data to determine the alarm state. For new health checks that have no last
         *        known status, the default status for the health check is healthy.
         *        </p>
         *        </li>
         * @see InsufficientDataHealthStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InsufficientDataHealthStatus
         */
        Builder insufficientDataHealthStatus(InsufficientDataHealthStatus insufficientDataHealthStatus);

        /**
         * <p>
         * A complex type that contains one <code>ResettableElementName</code> element for each element that you want to
         * reset to the default value. Valid values for <code>ResettableElementName</code> include the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ChildHealthChecks</code>: Amazon Route 53 resets <a>HealthCheckConfig$ChildHealthChecks</a> to null.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FullyQualifiedDomainName</code>: Amazon Route 53 resets
         * <a>HealthCheckConfig$FullyQualifiedDomainName</a> to null.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Regions</code>: Amazon Route 53 resets the <a>HealthCheckConfig$Regions</a> list to the default set of
         * regions.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ResourcePath</code>: Amazon Route 53 resets <a>HealthCheckConfig$ResourcePath</a> to null.
         * </p>
         * </li>
         * </ul>
         * 
         * @param resetElements
         *        A complex type that contains one <code>ResettableElementName</code> element for each element that you
         *        want to reset to the default value. Valid values for <code>ResettableElementName</code> include the
         *        following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ChildHealthChecks</code>: Amazon Route 53 resets <a>HealthCheckConfig$ChildHealthChecks</a> to
         *        null.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FullyQualifiedDomainName</code>: Amazon Route 53 resets
         *        <a>HealthCheckConfig$FullyQualifiedDomainName</a> to null.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Regions</code>: Amazon Route 53 resets the <a>HealthCheckConfig$Regions</a> list to the default
         *        set of regions.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ResourcePath</code>: Amazon Route 53 resets <a>HealthCheckConfig$ResourcePath</a> to null.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resetElements(Collection<String> resetElements);

        /**
         * <p>
         * A complex type that contains one <code>ResettableElementName</code> element for each element that you want to
         * reset to the default value. Valid values for <code>ResettableElementName</code> include the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ChildHealthChecks</code>: Amazon Route 53 resets <a>HealthCheckConfig$ChildHealthChecks</a> to null.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FullyQualifiedDomainName</code>: Amazon Route 53 resets
         * <a>HealthCheckConfig$FullyQualifiedDomainName</a> to null.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Regions</code>: Amazon Route 53 resets the <a>HealthCheckConfig$Regions</a> list to the default set of
         * regions.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ResourcePath</code>: Amazon Route 53 resets <a>HealthCheckConfig$ResourcePath</a> to null.
         * </p>
         * </li>
         * </ul>
         * 
         * @param resetElements
         *        A complex type that contains one <code>ResettableElementName</code> element for each element that you
         *        want to reset to the default value. Valid values for <code>ResettableElementName</code> include the
         *        following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ChildHealthChecks</code>: Amazon Route 53 resets <a>HealthCheckConfig$ChildHealthChecks</a> to
         *        null.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FullyQualifiedDomainName</code>: Amazon Route 53 resets
         *        <a>HealthCheckConfig$FullyQualifiedDomainName</a> to null.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Regions</code>: Amazon Route 53 resets the <a>HealthCheckConfig$Regions</a> list to the default
         *        set of regions.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ResourcePath</code>: Amazon Route 53 resets <a>HealthCheckConfig$ResourcePath</a> to null.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resetElements(String... resetElements);

        @Override
        Builder requestOverrideConfig(AwsRequestOverrideConfig awsRequestOverrideConfig);
    }

    static final class BuilderImpl extends Route53Request.BuilderImpl implements Builder {
        private String healthCheckId;

        private Long healthCheckVersion;

        private String ipAddress;

        private Integer port;

        private String resourcePath;

        private String fullyQualifiedDomainName;

        private String searchString;

        private Integer failureThreshold;

        private Boolean inverted;

        private Integer healthThreshold;

        private List<String> childHealthChecks;

        private Boolean enableSNI;

        private List<String> regions;

        private AlarmIdentifier alarmIdentifier;

        private String insufficientDataHealthStatus;

        private List<String> resetElements;

        private BuilderImpl() {
        }

        private BuilderImpl(UpdateHealthCheckRequest model) {
            healthCheckId(model.healthCheckId);
            healthCheckVersion(model.healthCheckVersion);
            ipAddress(model.ipAddress);
            port(model.port);
            resourcePath(model.resourcePath);
            fullyQualifiedDomainName(model.fullyQualifiedDomainName);
            searchString(model.searchString);
            failureThreshold(model.failureThreshold);
            inverted(model.inverted);
            healthThreshold(model.healthThreshold);
            childHealthChecks(model.childHealthChecks);
            enableSNI(model.enableSNI);
            regions(model.regions);
            alarmIdentifier(model.alarmIdentifier);
            insufficientDataHealthStatus(model.insufficientDataHealthStatus);
            resetElements(model.resetElements);
        }

        public final String getHealthCheckId() {
            return healthCheckId;
        }

        @Override
        public final Builder healthCheckId(String healthCheckId) {
            this.healthCheckId = healthCheckId;
            return this;
        }

        public final void setHealthCheckId(String healthCheckId) {
            this.healthCheckId = healthCheckId;
        }

        public final Long getHealthCheckVersion() {
            return healthCheckVersion;
        }

        @Override
        public final Builder healthCheckVersion(Long healthCheckVersion) {
            this.healthCheckVersion = healthCheckVersion;
            return this;
        }

        public final void setHealthCheckVersion(Long healthCheckVersion) {
            this.healthCheckVersion = healthCheckVersion;
        }

        public final String getIPAddress() {
            return ipAddress;
        }

        @Override
        public final Builder ipAddress(String ipAddress) {
            this.ipAddress = ipAddress;
            return this;
        }

        public final void setIPAddress(String ipAddress) {
            this.ipAddress = ipAddress;
        }

        public final Integer getPort() {
            return port;
        }

        @Override
        public final Builder port(Integer port) {
            this.port = port;
            return this;
        }

        public final void setPort(Integer port) {
            this.port = port;
        }

        public final String getResourcePath() {
            return resourcePath;
        }

        @Override
        public final Builder resourcePath(String resourcePath) {
            this.resourcePath = resourcePath;
            return this;
        }

        public final void setResourcePath(String resourcePath) {
            this.resourcePath = resourcePath;
        }

        public final String getFullyQualifiedDomainName() {
            return fullyQualifiedDomainName;
        }

        @Override
        public final Builder fullyQualifiedDomainName(String fullyQualifiedDomainName) {
            this.fullyQualifiedDomainName = fullyQualifiedDomainName;
            return this;
        }

        public final void setFullyQualifiedDomainName(String fullyQualifiedDomainName) {
            this.fullyQualifiedDomainName = fullyQualifiedDomainName;
        }

        public final String getSearchString() {
            return searchString;
        }

        @Override
        public final Builder searchString(String searchString) {
            this.searchString = searchString;
            return this;
        }

        public final void setSearchString(String searchString) {
            this.searchString = searchString;
        }

        public final Integer getFailureThreshold() {
            return failureThreshold;
        }

        @Override
        public final Builder failureThreshold(Integer failureThreshold) {
            this.failureThreshold = failureThreshold;
            return this;
        }

        public final void setFailureThreshold(Integer failureThreshold) {
            this.failureThreshold = failureThreshold;
        }

        public final Boolean getInverted() {
            return inverted;
        }

        @Override
        public final Builder inverted(Boolean inverted) {
            this.inverted = inverted;
            return this;
        }

        public final void setInverted(Boolean inverted) {
            this.inverted = inverted;
        }

        public final Integer getHealthThreshold() {
            return healthThreshold;
        }

        @Override
        public final Builder healthThreshold(Integer healthThreshold) {
            this.healthThreshold = healthThreshold;
            return this;
        }

        public final void setHealthThreshold(Integer healthThreshold) {
            this.healthThreshold = healthThreshold;
        }

        public final Collection<String> getChildHealthChecks() {
            return childHealthChecks;
        }

        @Override
        public final Builder childHealthChecks(Collection<String> childHealthChecks) {
            this.childHealthChecks = ChildHealthCheckListCopier.copy(childHealthChecks);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder childHealthChecks(String... childHealthChecks) {
            childHealthChecks(Arrays.asList(childHealthChecks));
            return this;
        }

        public final void setChildHealthChecks(Collection<String> childHealthChecks) {
            this.childHealthChecks = ChildHealthCheckListCopier.copy(childHealthChecks);
        }

        public final Boolean getEnableSNI() {
            return enableSNI;
        }

        @Override
        public final Builder enableSNI(Boolean enableSNI) {
            this.enableSNI = enableSNI;
            return this;
        }

        public final void setEnableSNI(Boolean enableSNI) {
            this.enableSNI = enableSNI;
        }

        public final Collection<String> getRegions() {
            return regions;
        }

        @Override
        public final Builder regions(Collection<String> regions) {
            this.regions = HealthCheckRegionListCopier.copy(regions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder regions(String... regions) {
            regions(Arrays.asList(regions));
            return this;
        }

        public final void setRegions(Collection<String> regions) {
            this.regions = HealthCheckRegionListCopier.copy(regions);
        }

        public final AlarmIdentifier.Builder getAlarmIdentifier() {
            return alarmIdentifier != null ? alarmIdentifier.toBuilder() : null;
        }

        @Override
        public final Builder alarmIdentifier(AlarmIdentifier alarmIdentifier) {
            this.alarmIdentifier = alarmIdentifier;
            return this;
        }

        public final void setAlarmIdentifier(AlarmIdentifier.BuilderImpl alarmIdentifier) {
            this.alarmIdentifier = alarmIdentifier != null ? alarmIdentifier.build() : null;
        }

        public final String getInsufficientDataHealthStatus() {
            return insufficientDataHealthStatus;
        }

        @Override
        public final Builder insufficientDataHealthStatus(String insufficientDataHealthStatus) {
            this.insufficientDataHealthStatus = insufficientDataHealthStatus;
            return this;
        }

        @Override
        public final Builder insufficientDataHealthStatus(InsufficientDataHealthStatus insufficientDataHealthStatus) {
            this.insufficientDataHealthStatus(insufficientDataHealthStatus.toString());
            return this;
        }

        public final void setInsufficientDataHealthStatus(String insufficientDataHealthStatus) {
            this.insufficientDataHealthStatus = insufficientDataHealthStatus;
        }

        public final Collection<String> getResetElements() {
            return resetElements;
        }

        @Override
        public final Builder resetElements(Collection<String> resetElements) {
            this.resetElements = ResettableElementNameListCopier.copy(resetElements);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resetElements(String... resetElements) {
            resetElements(Arrays.asList(resetElements));
            return this;
        }

        public final void setResetElements(Collection<String> resetElements) {
            this.resetElements = ResettableElementNameListCopier.copy(resetElements);
        }

        @Override
        public Builder requestOverrideConfig(AwsRequestOverrideConfig awsRequestOverrideConfig) {
            super.requestOverrideConfig(awsRequestOverrideConfig);
            return this;
        }

        @Override
        public Builder requestOverrideConfig(Consumer<AwsRequestOverrideConfig.Builder> builderConsumer) {
            super.requestOverrideConfig(builderConsumer);
            return this;
        }

        @Override
        public UpdateHealthCheckRequest build() {
            return new UpdateHealthCheckRequest(this);
        }
    }
}
