/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.route53.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A complex type that lists the name servers in a delegation set, as well as the <code>CallerReference</code> and the
 * <code>ID</code> for the delegation set.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DelegationSet implements SdkPojo, Serializable, ToCopyableBuilder<DelegationSet.Builder, DelegationSet> {
    private static final SdkField<String> ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(DelegationSet::id))
            .setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").unmarshallLocationName("Id")
                    .build()).build();

    private static final SdkField<String> CALLER_REFERENCE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(DelegationSet::callerReference))
            .setter(setter(Builder::callerReference))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CallerReference")
                    .unmarshallLocationName("CallerReference").build()).build();

    private static final SdkField<List<String>> NAME_SERVERS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(DelegationSet::nameServers))
            .setter(setter(Builder::nameServers))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NameServers")
                    .unmarshallLocationName("NameServers").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("NameServer")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("NameServer").unmarshallLocationName("NameServer").build())
                                            .build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD,
            CALLER_REFERENCE_FIELD, NAME_SERVERS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String id;

    private final String callerReference;

    private final List<String> nameServers;

    private DelegationSet(BuilderImpl builder) {
        this.id = builder.id;
        this.callerReference = builder.callerReference;
        this.nameServers = builder.nameServers;
    }

    /**
     * <p>
     * The ID that Amazon Route 53 assigns to a reusable delegation set.
     * </p>
     * 
     * @return The ID that Amazon Route 53 assigns to a reusable delegation set.
     */
    public String id() {
        return id;
    }

    /**
     * <p>
     * The value that you specified for <code>CallerReference</code> when you created the reusable delegation set.
     * </p>
     * 
     * @return The value that you specified for <code>CallerReference</code> when you created the reusable delegation
     *         set.
     */
    public String callerReference() {
        return callerReference;
    }

    /**
     * <p>
     * A complex type that contains a list of the authoritative name servers for a hosted zone or for a reusable
     * delegation set.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A complex type that contains a list of the authoritative name servers for a hosted zone or for a reusable
     *         delegation set.
     */
    public List<String> nameServers() {
        return nameServers;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(callerReference());
        hashCode = 31 * hashCode + Objects.hashCode(nameServers());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DelegationSet)) {
            return false;
        }
        DelegationSet other = (DelegationSet) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(callerReference(), other.callerReference())
                && Objects.equals(nameServers(), other.nameServers());
    }

    @Override
    public String toString() {
        return ToString.builder("DelegationSet").add("Id", id()).add("CallerReference", callerReference())
                .add("NameServers", nameServers()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "CallerReference":
            return Optional.ofNullable(clazz.cast(callerReference()));
        case "NameServers":
            return Optional.ofNullable(clazz.cast(nameServers()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DelegationSet, T> g) {
        return obj -> g.apply((DelegationSet) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DelegationSet> {
        /**
         * <p>
         * The ID that Amazon Route 53 assigns to a reusable delegation set.
         * </p>
         * 
         * @param id
         *        The ID that Amazon Route 53 assigns to a reusable delegation set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The value that you specified for <code>CallerReference</code> when you created the reusable delegation set.
         * </p>
         * 
         * @param callerReference
         *        The value that you specified for <code>CallerReference</code> when you created the reusable delegation
         *        set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder callerReference(String callerReference);

        /**
         * <p>
         * A complex type that contains a list of the authoritative name servers for a hosted zone or for a reusable
         * delegation set.
         * </p>
         * 
         * @param nameServers
         *        A complex type that contains a list of the authoritative name servers for a hosted zone or for a
         *        reusable delegation set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nameServers(Collection<String> nameServers);

        /**
         * <p>
         * A complex type that contains a list of the authoritative name servers for a hosted zone or for a reusable
         * delegation set.
         * </p>
         * 
         * @param nameServers
         *        A complex type that contains a list of the authoritative name servers for a hosted zone or for a
         *        reusable delegation set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nameServers(String... nameServers);
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private String callerReference;

        private List<String> nameServers = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(DelegationSet model) {
            id(model.id);
            callerReference(model.callerReference);
            nameServers(model.nameServers);
        }

        public final String getId() {
            return id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final void setId(String id) {
            this.id = id;
        }

        public final String getCallerReference() {
            return callerReference;
        }

        @Override
        public final Builder callerReference(String callerReference) {
            this.callerReference = callerReference;
            return this;
        }

        public final void setCallerReference(String callerReference) {
            this.callerReference = callerReference;
        }

        public final Collection<String> getNameServers() {
            return nameServers;
        }

        @Override
        public final Builder nameServers(Collection<String> nameServers) {
            this.nameServers = DelegationSetNameServersCopier.copy(nameServers);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder nameServers(String... nameServers) {
            nameServers(Arrays.asList(nameServers));
            return this;
        }

        public final void setNameServers(Collection<String> nameServers) {
            this.nameServers = DelegationSetNameServersCopier.copy(nameServers);
        }

        @Override
        public DelegationSet build() {
            return new DelegationSet(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
