/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.route53.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A complex type that contains information about the CloudWatch alarm that Amazon Route 53 is monitoring for this
 * health check.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CloudWatchAlarmConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<CloudWatchAlarmConfiguration.Builder, CloudWatchAlarmConfiguration> {
    private static final SdkField<Integer> EVALUATION_PERIODS_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("EvaluationPeriods")
            .getter(getter(CloudWatchAlarmConfiguration::evaluationPeriods))
            .setter(setter(Builder::evaluationPeriods))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EvaluationPeriods")
                    .unmarshallLocationName("EvaluationPeriods").build()).build();

    private static final SdkField<Double> THRESHOLD_FIELD = SdkField
            .<Double> builder(MarshallingType.DOUBLE)
            .memberName("Threshold")
            .getter(getter(CloudWatchAlarmConfiguration::threshold))
            .setter(setter(Builder::threshold))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Threshold")
                    .unmarshallLocationName("Threshold").build()).build();

    private static final SdkField<String> COMPARISON_OPERATOR_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ComparisonOperator")
            .getter(getter(CloudWatchAlarmConfiguration::comparisonOperatorAsString))
            .setter(setter(Builder::comparisonOperator))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ComparisonOperator")
                    .unmarshallLocationName("ComparisonOperator").build()).build();

    private static final SdkField<Integer> PERIOD_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("Period")
            .getter(getter(CloudWatchAlarmConfiguration::period))
            .setter(setter(Builder::period))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Period")
                    .unmarshallLocationName("Period").build()).build();

    private static final SdkField<String> METRIC_NAME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("MetricName")
            .getter(getter(CloudWatchAlarmConfiguration::metricName))
            .setter(setter(Builder::metricName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MetricName")
                    .unmarshallLocationName("MetricName").build()).build();

    private static final SdkField<String> NAMESPACE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Namespace")
            .getter(getter(CloudWatchAlarmConfiguration::namespace))
            .setter(setter(Builder::namespace))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Namespace")
                    .unmarshallLocationName("Namespace").build()).build();

    private static final SdkField<String> STATISTIC_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Statistic")
            .getter(getter(CloudWatchAlarmConfiguration::statisticAsString))
            .setter(setter(Builder::statistic))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Statistic")
                    .unmarshallLocationName("Statistic").build()).build();

    private static final SdkField<List<Dimension>> DIMENSIONS_FIELD = SdkField
            .<List<Dimension>> builder(MarshallingType.LIST)
            .memberName("Dimensions")
            .getter(getter(CloudWatchAlarmConfiguration::dimensions))
            .setter(setter(Builder::dimensions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Dimensions")
                    .unmarshallLocationName("Dimensions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("Dimension")
                            .memberFieldInfo(
                                    SdkField.<Dimension> builder(MarshallingType.SDK_POJO)
                                            .constructor(Dimension::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Dimension").unmarshallLocationName("Dimension").build())
                                            .build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(EVALUATION_PERIODS_FIELD,
            THRESHOLD_FIELD, COMPARISON_OPERATOR_FIELD, PERIOD_FIELD, METRIC_NAME_FIELD, NAMESPACE_FIELD, STATISTIC_FIELD,
            DIMENSIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final Integer evaluationPeriods;

    private final Double threshold;

    private final String comparisonOperator;

    private final Integer period;

    private final String metricName;

    private final String namespace;

    private final String statistic;

    private final List<Dimension> dimensions;

    private CloudWatchAlarmConfiguration(BuilderImpl builder) {
        this.evaluationPeriods = builder.evaluationPeriods;
        this.threshold = builder.threshold;
        this.comparisonOperator = builder.comparisonOperator;
        this.period = builder.period;
        this.metricName = builder.metricName;
        this.namespace = builder.namespace;
        this.statistic = builder.statistic;
        this.dimensions = builder.dimensions;
    }

    /**
     * <p>
     * For the metric that the CloudWatch alarm is associated with, the number of periods that the metric is compared to
     * the threshold.
     * </p>
     * 
     * @return For the metric that the CloudWatch alarm is associated with, the number of periods that the metric is
     *         compared to the threshold.
     */
    public Integer evaluationPeriods() {
        return evaluationPeriods;
    }

    /**
     * <p>
     * For the metric that the CloudWatch alarm is associated with, the value the metric is compared with.
     * </p>
     * 
     * @return For the metric that the CloudWatch alarm is associated with, the value the metric is compared with.
     */
    public Double threshold() {
        return threshold;
    }

    /**
     * <p>
     * For the metric that the CloudWatch alarm is associated with, the arithmetic operation that is used for the
     * comparison.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #comparisonOperator} will return {@link ComparisonOperator#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #comparisonOperatorAsString}.
     * </p>
     * 
     * @return For the metric that the CloudWatch alarm is associated with, the arithmetic operation that is used for
     *         the comparison.
     * @see ComparisonOperator
     */
    public ComparisonOperator comparisonOperator() {
        return ComparisonOperator.fromValue(comparisonOperator);
    }

    /**
     * <p>
     * For the metric that the CloudWatch alarm is associated with, the arithmetic operation that is used for the
     * comparison.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #comparisonOperator} will return {@link ComparisonOperator#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #comparisonOperatorAsString}.
     * </p>
     * 
     * @return For the metric that the CloudWatch alarm is associated with, the arithmetic operation that is used for
     *         the comparison.
     * @see ComparisonOperator
     */
    public String comparisonOperatorAsString() {
        return comparisonOperator;
    }

    /**
     * <p>
     * For the metric that the CloudWatch alarm is associated with, the duration of one evaluation period in seconds.
     * </p>
     * 
     * @return For the metric that the CloudWatch alarm is associated with, the duration of one evaluation period in
     *         seconds.
     */
    public Integer period() {
        return period;
    }

    /**
     * <p>
     * The name of the CloudWatch metric that the alarm is associated with.
     * </p>
     * 
     * @return The name of the CloudWatch metric that the alarm is associated with.
     */
    public String metricName() {
        return metricName;
    }

    /**
     * <p>
     * The namespace of the metric that the alarm is associated with. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/DeveloperGuide/CW_Support_For_AWS.html">Amazon
     * CloudWatch Namespaces, Dimensions, and Metrics Reference</a> in the <i>Amazon CloudWatch User Guide</i>.
     * </p>
     * 
     * @return The namespace of the metric that the alarm is associated with. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/DeveloperGuide/CW_Support_For_AWS.html">Amazon
     *         CloudWatch Namespaces, Dimensions, and Metrics Reference</a> in the <i>Amazon CloudWatch User Guide</i>.
     */
    public String namespace() {
        return namespace;
    }

    /**
     * <p>
     * For the metric that the CloudWatch alarm is associated with, the statistic that is applied to the metric.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #statistic} will
     * return {@link Statistic#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statisticAsString}.
     * </p>
     * 
     * @return For the metric that the CloudWatch alarm is associated with, the statistic that is applied to the metric.
     * @see Statistic
     */
    public Statistic statistic() {
        return Statistic.fromValue(statistic);
    }

    /**
     * <p>
     * For the metric that the CloudWatch alarm is associated with, the statistic that is applied to the metric.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #statistic} will
     * return {@link Statistic#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statisticAsString}.
     * </p>
     * 
     * @return For the metric that the CloudWatch alarm is associated with, the statistic that is applied to the metric.
     * @see Statistic
     */
    public String statisticAsString() {
        return statistic;
    }

    /**
     * Returns true if the Dimensions property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasDimensions() {
        return dimensions != null && !(dimensions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * For the metric that the CloudWatch alarm is associated with, a complex type that contains information about the
     * dimensions for the metric. For information, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/DeveloperGuide/CW_Support_For_AWS.html">Amazon
     * CloudWatch Namespaces, Dimensions, and Metrics Reference</a> in the <i>Amazon CloudWatch User Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasDimensions()} to see if a value was sent in this field.
     * </p>
     * 
     * @return For the metric that the CloudWatch alarm is associated with, a complex type that contains information
     *         about the dimensions for the metric. For information, see <a
     *         href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/DeveloperGuide/CW_Support_For_AWS.html">Amazon
     *         CloudWatch Namespaces, Dimensions, and Metrics Reference</a> in the <i>Amazon CloudWatch User Guide</i>.
     */
    public List<Dimension> dimensions() {
        return dimensions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(evaluationPeriods());
        hashCode = 31 * hashCode + Objects.hashCode(threshold());
        hashCode = 31 * hashCode + Objects.hashCode(comparisonOperatorAsString());
        hashCode = 31 * hashCode + Objects.hashCode(period());
        hashCode = 31 * hashCode + Objects.hashCode(metricName());
        hashCode = 31 * hashCode + Objects.hashCode(namespace());
        hashCode = 31 * hashCode + Objects.hashCode(statisticAsString());
        hashCode = 31 * hashCode + Objects.hashCode(dimensions());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CloudWatchAlarmConfiguration)) {
            return false;
        }
        CloudWatchAlarmConfiguration other = (CloudWatchAlarmConfiguration) obj;
        return Objects.equals(evaluationPeriods(), other.evaluationPeriods()) && Objects.equals(threshold(), other.threshold())
                && Objects.equals(comparisonOperatorAsString(), other.comparisonOperatorAsString())
                && Objects.equals(period(), other.period()) && Objects.equals(metricName(), other.metricName())
                && Objects.equals(namespace(), other.namespace())
                && Objects.equals(statisticAsString(), other.statisticAsString())
                && Objects.equals(dimensions(), other.dimensions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("CloudWatchAlarmConfiguration").add("EvaluationPeriods", evaluationPeriods())
                .add("Threshold", threshold()).add("ComparisonOperator", comparisonOperatorAsString()).add("Period", period())
                .add("MetricName", metricName()).add("Namespace", namespace()).add("Statistic", statisticAsString())
                .add("Dimensions", dimensions()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "EvaluationPeriods":
            return Optional.ofNullable(clazz.cast(evaluationPeriods()));
        case "Threshold":
            return Optional.ofNullable(clazz.cast(threshold()));
        case "ComparisonOperator":
            return Optional.ofNullable(clazz.cast(comparisonOperatorAsString()));
        case "Period":
            return Optional.ofNullable(clazz.cast(period()));
        case "MetricName":
            return Optional.ofNullable(clazz.cast(metricName()));
        case "Namespace":
            return Optional.ofNullable(clazz.cast(namespace()));
        case "Statistic":
            return Optional.ofNullable(clazz.cast(statisticAsString()));
        case "Dimensions":
            return Optional.ofNullable(clazz.cast(dimensions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CloudWatchAlarmConfiguration, T> g) {
        return obj -> g.apply((CloudWatchAlarmConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CloudWatchAlarmConfiguration> {
        /**
         * <p>
         * For the metric that the CloudWatch alarm is associated with, the number of periods that the metric is
         * compared to the threshold.
         * </p>
         * 
         * @param evaluationPeriods
         *        For the metric that the CloudWatch alarm is associated with, the number of periods that the metric is
         *        compared to the threshold.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder evaluationPeriods(Integer evaluationPeriods);

        /**
         * <p>
         * For the metric that the CloudWatch alarm is associated with, the value the metric is compared with.
         * </p>
         * 
         * @param threshold
         *        For the metric that the CloudWatch alarm is associated with, the value the metric is compared with.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder threshold(Double threshold);

        /**
         * <p>
         * For the metric that the CloudWatch alarm is associated with, the arithmetic operation that is used for the
         * comparison.
         * </p>
         * 
         * @param comparisonOperator
         *        For the metric that the CloudWatch alarm is associated with, the arithmetic operation that is used for
         *        the comparison.
         * @see ComparisonOperator
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComparisonOperator
         */
        Builder comparisonOperator(String comparisonOperator);

        /**
         * <p>
         * For the metric that the CloudWatch alarm is associated with, the arithmetic operation that is used for the
         * comparison.
         * </p>
         * 
         * @param comparisonOperator
         *        For the metric that the CloudWatch alarm is associated with, the arithmetic operation that is used for
         *        the comparison.
         * @see ComparisonOperator
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComparisonOperator
         */
        Builder comparisonOperator(ComparisonOperator comparisonOperator);

        /**
         * <p>
         * For the metric that the CloudWatch alarm is associated with, the duration of one evaluation period in
         * seconds.
         * </p>
         * 
         * @param period
         *        For the metric that the CloudWatch alarm is associated with, the duration of one evaluation period in
         *        seconds.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder period(Integer period);

        /**
         * <p>
         * The name of the CloudWatch metric that the alarm is associated with.
         * </p>
         * 
         * @param metricName
         *        The name of the CloudWatch metric that the alarm is associated with.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricName(String metricName);

        /**
         * <p>
         * The namespace of the metric that the alarm is associated with. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/DeveloperGuide/CW_Support_For_AWS.html">Amazon
         * CloudWatch Namespaces, Dimensions, and Metrics Reference</a> in the <i>Amazon CloudWatch User Guide</i>.
         * </p>
         * 
         * @param namespace
         *        The namespace of the metric that the alarm is associated with. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/DeveloperGuide/CW_Support_For_AWS.html"
         *        >Amazon CloudWatch Namespaces, Dimensions, and Metrics Reference</a> in the <i>Amazon CloudWatch User
         *        Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder namespace(String namespace);

        /**
         * <p>
         * For the metric that the CloudWatch alarm is associated with, the statistic that is applied to the metric.
         * </p>
         * 
         * @param statistic
         *        For the metric that the CloudWatch alarm is associated with, the statistic that is applied to the
         *        metric.
         * @see Statistic
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Statistic
         */
        Builder statistic(String statistic);

        /**
         * <p>
         * For the metric that the CloudWatch alarm is associated with, the statistic that is applied to the metric.
         * </p>
         * 
         * @param statistic
         *        For the metric that the CloudWatch alarm is associated with, the statistic that is applied to the
         *        metric.
         * @see Statistic
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Statistic
         */
        Builder statistic(Statistic statistic);

        /**
         * <p>
         * For the metric that the CloudWatch alarm is associated with, a complex type that contains information about
         * the dimensions for the metric. For information, see <a
         * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/DeveloperGuide/CW_Support_For_AWS.html">Amazon
         * CloudWatch Namespaces, Dimensions, and Metrics Reference</a> in the <i>Amazon CloudWatch User Guide</i>.
         * </p>
         * 
         * @param dimensions
         *        For the metric that the CloudWatch alarm is associated with, a complex type that contains information
         *        about the dimensions for the metric. For information, see <a
         *        href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/DeveloperGuide/CW_Support_For_AWS.html"
         *        >Amazon CloudWatch Namespaces, Dimensions, and Metrics Reference</a> in the <i>Amazon CloudWatch User
         *        Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dimensions(Collection<Dimension> dimensions);

        /**
         * <p>
         * For the metric that the CloudWatch alarm is associated with, a complex type that contains information about
         * the dimensions for the metric. For information, see <a
         * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/DeveloperGuide/CW_Support_For_AWS.html">Amazon
         * CloudWatch Namespaces, Dimensions, and Metrics Reference</a> in the <i>Amazon CloudWatch User Guide</i>.
         * </p>
         * 
         * @param dimensions
         *        For the metric that the CloudWatch alarm is associated with, a complex type that contains information
         *        about the dimensions for the metric. For information, see <a
         *        href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/DeveloperGuide/CW_Support_For_AWS.html"
         *        >Amazon CloudWatch Namespaces, Dimensions, and Metrics Reference</a> in the <i>Amazon CloudWatch User
         *        Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dimensions(Dimension... dimensions);

        /**
         * <p>
         * For the metric that the CloudWatch alarm is associated with, a complex type that contains information about
         * the dimensions for the metric. For information, see <a
         * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/DeveloperGuide/CW_Support_For_AWS.html">Amazon
         * CloudWatch Namespaces, Dimensions, and Metrics Reference</a> in the <i>Amazon CloudWatch User Guide</i>.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Dimension>.Builder} avoiding the need to
         * create one manually via {@link List<Dimension>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Dimension>.Builder#build()} is called immediately and its
         * result is passed to {@link #dimensions(List<Dimension>)}.
         * 
         * @param dimensions
         *        a consumer that will call methods on {@link List<Dimension>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dimensions(List<Dimension>)
         */
        Builder dimensions(Consumer<Dimension.Builder>... dimensions);
    }

    static final class BuilderImpl implements Builder {
        private Integer evaluationPeriods;

        private Double threshold;

        private String comparisonOperator;

        private Integer period;

        private String metricName;

        private String namespace;

        private String statistic;

        private List<Dimension> dimensions = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CloudWatchAlarmConfiguration model) {
            evaluationPeriods(model.evaluationPeriods);
            threshold(model.threshold);
            comparisonOperator(model.comparisonOperator);
            period(model.period);
            metricName(model.metricName);
            namespace(model.namespace);
            statistic(model.statistic);
            dimensions(model.dimensions);
        }

        public final Integer getEvaluationPeriods() {
            return evaluationPeriods;
        }

        @Override
        public final Builder evaluationPeriods(Integer evaluationPeriods) {
            this.evaluationPeriods = evaluationPeriods;
            return this;
        }

        public final void setEvaluationPeriods(Integer evaluationPeriods) {
            this.evaluationPeriods = evaluationPeriods;
        }

        public final Double getThreshold() {
            return threshold;
        }

        @Override
        public final Builder threshold(Double threshold) {
            this.threshold = threshold;
            return this;
        }

        public final void setThreshold(Double threshold) {
            this.threshold = threshold;
        }

        public final String getComparisonOperator() {
            return comparisonOperator;
        }

        @Override
        public final Builder comparisonOperator(String comparisonOperator) {
            this.comparisonOperator = comparisonOperator;
            return this;
        }

        @Override
        public final Builder comparisonOperator(ComparisonOperator comparisonOperator) {
            this.comparisonOperator(comparisonOperator == null ? null : comparisonOperator.toString());
            return this;
        }

        public final void setComparisonOperator(String comparisonOperator) {
            this.comparisonOperator = comparisonOperator;
        }

        public final Integer getPeriod() {
            return period;
        }

        @Override
        public final Builder period(Integer period) {
            this.period = period;
            return this;
        }

        public final void setPeriod(Integer period) {
            this.period = period;
        }

        public final String getMetricName() {
            return metricName;
        }

        @Override
        public final Builder metricName(String metricName) {
            this.metricName = metricName;
            return this;
        }

        public final void setMetricName(String metricName) {
            this.metricName = metricName;
        }

        public final String getNamespace() {
            return namespace;
        }

        @Override
        public final Builder namespace(String namespace) {
            this.namespace = namespace;
            return this;
        }

        public final void setNamespace(String namespace) {
            this.namespace = namespace;
        }

        public final String getStatistic() {
            return statistic;
        }

        @Override
        public final Builder statistic(String statistic) {
            this.statistic = statistic;
            return this;
        }

        @Override
        public final Builder statistic(Statistic statistic) {
            this.statistic(statistic == null ? null : statistic.toString());
            return this;
        }

        public final void setStatistic(String statistic) {
            this.statistic = statistic;
        }

        public final Collection<Dimension.Builder> getDimensions() {
            return dimensions != null ? dimensions.stream().map(Dimension::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder dimensions(Collection<Dimension> dimensions) {
            this.dimensions = DimensionListCopier.copy(dimensions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dimensions(Dimension... dimensions) {
            dimensions(Arrays.asList(dimensions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dimensions(Consumer<Dimension.Builder>... dimensions) {
            dimensions(Stream.of(dimensions).map(c -> Dimension.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setDimensions(Collection<Dimension.BuilderImpl> dimensions) {
            this.dimensions = DimensionListCopier.copyFromBuilder(dimensions);
        }

        @Override
        public CloudWatchAlarmConfiguration build() {
            return new CloudWatchAlarmConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
