/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.route53.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A complex type that identifies the CloudWatch alarm that you want Amazon Route 53 health checkers to use to determine
 * whether the specified health check is healthy.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AlarmIdentifier implements SdkPojo, Serializable, ToCopyableBuilder<AlarmIdentifier.Builder, AlarmIdentifier> {
    private static final SdkField<String> REGION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Region")
            .getter(getter(AlarmIdentifier::regionAsString))
            .setter(setter(Builder::region))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Region")
                    .unmarshallLocationName("Region").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Name")
            .getter(getter(AlarmIdentifier::name))
            .setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name")
                    .unmarshallLocationName("Name").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(REGION_FIELD, NAME_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("Region", REGION_FIELD);
                    put("Name", NAME_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final String region;

    private final String name;

    private AlarmIdentifier(BuilderImpl builder) {
        this.region = builder.region;
        this.name = builder.name;
    }

    /**
     * <p>
     * For the CloudWatch alarm that you want Route 53 health checkers to use to determine whether this health check is
     * healthy, the region that the alarm was created in.
     * </p>
     * <p>
     * For the current list of CloudWatch regions, see <a
     * href="https://docs.aws.amazon.com/general/latest/gr/cw_region.html">Amazon CloudWatch endpoints and quotas</a> in
     * the <i>Amazon Web Services General Reference</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #region} will
     * return {@link CloudWatchRegion#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #regionAsString}.
     * </p>
     * 
     * @return For the CloudWatch alarm that you want Route 53 health checkers to use to determine whether this health
     *         check is healthy, the region that the alarm was created in.</p>
     *         <p>
     *         For the current list of CloudWatch regions, see <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/cw_region.html">Amazon CloudWatch endpoints and
     *         quotas</a> in the <i>Amazon Web Services General Reference</i>.
     * @see CloudWatchRegion
     */
    public final CloudWatchRegion region() {
        return CloudWatchRegion.fromValue(region);
    }

    /**
     * <p>
     * For the CloudWatch alarm that you want Route 53 health checkers to use to determine whether this health check is
     * healthy, the region that the alarm was created in.
     * </p>
     * <p>
     * For the current list of CloudWatch regions, see <a
     * href="https://docs.aws.amazon.com/general/latest/gr/cw_region.html">Amazon CloudWatch endpoints and quotas</a> in
     * the <i>Amazon Web Services General Reference</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #region} will
     * return {@link CloudWatchRegion#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #regionAsString}.
     * </p>
     * 
     * @return For the CloudWatch alarm that you want Route 53 health checkers to use to determine whether this health
     *         check is healthy, the region that the alarm was created in.</p>
     *         <p>
     *         For the current list of CloudWatch regions, see <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/cw_region.html">Amazon CloudWatch endpoints and
     *         quotas</a> in the <i>Amazon Web Services General Reference</i>.
     * @see CloudWatchRegion
     */
    public final String regionAsString() {
        return region;
    }

    /**
     * <p>
     * The name of the CloudWatch alarm that you want Amazon Route 53 health checkers to use to determine whether this
     * health check is healthy.
     * </p>
     * <note>
     * <p>
     * Route 53 supports CloudWatch alarms with the following features:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Standard-resolution metrics. High-resolution metrics aren't supported. For more information, see <a href=
     * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/DeveloperGuide/publishingMetrics.html#high-resolution-metrics"
     * >High-Resolution Metrics</a> in the <i>Amazon CloudWatch User Guide</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Statistics: Average, Minimum, Maximum, Sum, and SampleCount. Extended statistics aren't supported.
     * </p>
     * </li>
     * </ul>
     * </note>
     * 
     * @return The name of the CloudWatch alarm that you want Amazon Route 53 health checkers to use to determine
     *         whether this health check is healthy.</p> <note>
     *         <p>
     *         Route 53 supports CloudWatch alarms with the following features:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Standard-resolution metrics. High-resolution metrics aren't supported. For more information, see <a href=
     *         "https://docs.aws.amazon.com/AmazonCloudWatch/latest/DeveloperGuide/publishingMetrics.html#high-resolution-metrics"
     *         >High-Resolution Metrics</a> in the <i>Amazon CloudWatch User Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Statistics: Average, Minimum, Maximum, Sum, and SampleCount. Extended statistics aren't supported.
     *         </p>
     *         </li>
     *         </ul>
     */
    public final String name() {
        return name;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(regionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AlarmIdentifier)) {
            return false;
        }
        AlarmIdentifier other = (AlarmIdentifier) obj;
        return Objects.equals(regionAsString(), other.regionAsString()) && Objects.equals(name(), other.name());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AlarmIdentifier").add("Region", regionAsString()).add("Name", name()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Region":
            return Optional.ofNullable(clazz.cast(regionAsString()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<AlarmIdentifier, T> g) {
        return obj -> g.apply((AlarmIdentifier) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AlarmIdentifier> {
        /**
         * <p>
         * For the CloudWatch alarm that you want Route 53 health checkers to use to determine whether this health check
         * is healthy, the region that the alarm was created in.
         * </p>
         * <p>
         * For the current list of CloudWatch regions, see <a
         * href="https://docs.aws.amazon.com/general/latest/gr/cw_region.html">Amazon CloudWatch endpoints and
         * quotas</a> in the <i>Amazon Web Services General Reference</i>.
         * </p>
         * 
         * @param region
         *        For the CloudWatch alarm that you want Route 53 health checkers to use to determine whether this
         *        health check is healthy, the region that the alarm was created in.</p>
         *        <p>
         *        For the current list of CloudWatch regions, see <a
         *        href="https://docs.aws.amazon.com/general/latest/gr/cw_region.html">Amazon CloudWatch endpoints and
         *        quotas</a> in the <i>Amazon Web Services General Reference</i>.
         * @see CloudWatchRegion
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CloudWatchRegion
         */
        Builder region(String region);

        /**
         * <p>
         * For the CloudWatch alarm that you want Route 53 health checkers to use to determine whether this health check
         * is healthy, the region that the alarm was created in.
         * </p>
         * <p>
         * For the current list of CloudWatch regions, see <a
         * href="https://docs.aws.amazon.com/general/latest/gr/cw_region.html">Amazon CloudWatch endpoints and
         * quotas</a> in the <i>Amazon Web Services General Reference</i>.
         * </p>
         * 
         * @param region
         *        For the CloudWatch alarm that you want Route 53 health checkers to use to determine whether this
         *        health check is healthy, the region that the alarm was created in.</p>
         *        <p>
         *        For the current list of CloudWatch regions, see <a
         *        href="https://docs.aws.amazon.com/general/latest/gr/cw_region.html">Amazon CloudWatch endpoints and
         *        quotas</a> in the <i>Amazon Web Services General Reference</i>.
         * @see CloudWatchRegion
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CloudWatchRegion
         */
        Builder region(CloudWatchRegion region);

        /**
         * <p>
         * The name of the CloudWatch alarm that you want Amazon Route 53 health checkers to use to determine whether
         * this health check is healthy.
         * </p>
         * <note>
         * <p>
         * Route 53 supports CloudWatch alarms with the following features:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Standard-resolution metrics. High-resolution metrics aren't supported. For more information, see <a href=
         * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/DeveloperGuide/publishingMetrics.html#high-resolution-metrics"
         * >High-Resolution Metrics</a> in the <i>Amazon CloudWatch User Guide</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Statistics: Average, Minimum, Maximum, Sum, and SampleCount. Extended statistics aren't supported.
         * </p>
         * </li>
         * </ul>
         * </note>
         * 
         * @param name
         *        The name of the CloudWatch alarm that you want Amazon Route 53 health checkers to use to determine
         *        whether this health check is healthy.</p> <note>
         *        <p>
         *        Route 53 supports CloudWatch alarms with the following features:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Standard-resolution metrics. High-resolution metrics aren't supported. For more information, see <a
         *        href=
         *        "https://docs.aws.amazon.com/AmazonCloudWatch/latest/DeveloperGuide/publishingMetrics.html#high-resolution-metrics"
         *        >High-Resolution Metrics</a> in the <i>Amazon CloudWatch User Guide</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Statistics: Average, Minimum, Maximum, Sum, and SampleCount. Extended statistics aren't supported.
         *        </p>
         *        </li>
         *        </ul>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);
    }

    static final class BuilderImpl implements Builder {
        private String region;

        private String name;

        private BuilderImpl() {
        }

        private BuilderImpl(AlarmIdentifier model) {
            region(model.region);
            name(model.name);
        }

        public final String getRegion() {
            return region;
        }

        public final void setRegion(String region) {
            this.region = region;
        }

        @Override
        public final Builder region(String region) {
            this.region = region;
            return this;
        }

        @Override
        public final Builder region(CloudWatchRegion region) {
            this.region(region == null ? null : region.toString());
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        @Override
        public AlarmIdentifier build() {
            return new AlarmIdentifier(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
