/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.route53.endpoints.internal;

import java.net.URI;
import java.util.Arrays;
import java.util.concurrent.CompletableFuture;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.endpoints.AwsEndpointAttribute;
import software.amazon.awssdk.awscore.endpoints.authscheme.SigV4AuthScheme;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.endpoints.Endpoint;
import software.amazon.awssdk.regions.Region;
import software.amazon.awssdk.services.route53.endpoints.Route53EndpointParams;
import software.amazon.awssdk.services.route53.endpoints.Route53EndpointProvider;
import software.amazon.awssdk.utils.CompletableFutureUtils;
import software.amazon.awssdk.utils.Validate;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
public final class DefaultRoute53EndpointProvider implements Route53EndpointProvider {
    @Override
    public CompletableFuture<Endpoint> resolveEndpoint(Route53EndpointParams params) {
        Validate.notNull(params.useDualStack(), "Parameter 'UseDualStack' must not be null");
        Validate.notNull(params.useFips(), "Parameter 'UseFIPS' must not be null");
        try {
            Region region = params.region();
            String regionId = region == null ? null : region.id();
            RuleResult result = endpointRule0(params, regionId);
            if (result.canContinue()) {
                throw SdkClientException.create("Rule engine did not reach an error or endpoint result");
            }
            if (result.isError()) {
                String errorMsg = result.error();
                if (errorMsg.contains("Invalid ARN") && errorMsg.contains(":s3:::")) {
                    errorMsg += ". Use the bucket name instead of simple bucket ARNs in GetBucketLocationRequest.";
                }
                throw SdkClientException.create(errorMsg);
            }
            return CompletableFuture.completedFuture(result.endpoint());
        } catch (Exception error) {
            return CompletableFutureUtils.failedFuture(error);
        }
    }

    private static RuleResult endpointRule0(Route53EndpointParams params, String region) {
        RuleResult result = endpointRule1(params);
        if (result.isResolved()) {
            return result;
        }
        result = endpointRule5(params, region);
        if (result.isResolved()) {
            return result;
        }
        return RuleResult.error("Invalid Configuration: Missing Region");
    }

    private static RuleResult endpointRule1(Route53EndpointParams params) {
        if (params.endpoint() != null) {
            if (params.useFips()) {
                return RuleResult.error("Invalid Configuration: FIPS and custom endpoint are not supported");
            }
            if (params.useDualStack()) {
                return RuleResult.error("Invalid Configuration: Dualstack and custom endpoint are not supported");
            }
            return RuleResult.endpoint(Endpoint.builder().url(URI.create(params.endpoint())).build());
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule5(Route53EndpointParams params, String region) {
        if (region != null) {
            return endpointRule6(params, region);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule6(Route53EndpointParams params, String region) {
        RulePartition partitionResult = RulesFunctions.awsPartition(region);
        if (partitionResult != null) {
            if ("aws".equals(partitionResult.name()) && !params.useFips() && !params.useDualStack()) {
                return RuleResult
                        .endpoint(Endpoint
                                .builder()
                                .url(URI.create("https://route53.amazonaws.com"))
                                .putAttribute(
                                        AwsEndpointAttribute.AUTH_SCHEMES,
                                        Arrays.asList(SigV4AuthScheme.builder().signingName("route53").signingRegion("us-east-1")
                                                .build())).build());
            }
            if ("aws".equals(partitionResult.name()) && params.useFips() && !params.useDualStack()) {
                return RuleResult
                        .endpoint(Endpoint
                                .builder()
                                .url(URI.create("https://route53-fips.amazonaws.com"))
                                .putAttribute(
                                        AwsEndpointAttribute.AUTH_SCHEMES,
                                        Arrays.asList(SigV4AuthScheme.builder().signingName("route53").signingRegion("us-east-1")
                                                .build())).build());
            }
            if ("aws-cn".equals(partitionResult.name()) && !params.useFips() && !params.useDualStack()) {
                return RuleResult.endpoint(Endpoint
                        .builder()
                        .url(URI.create("https://route53.amazonaws.com.cn"))
                        .putAttribute(
                                AwsEndpointAttribute.AUTH_SCHEMES,
                                Arrays.asList(SigV4AuthScheme.builder().signingName("route53").signingRegion("cn-northwest-1")
                                        .build())).build());
            }
            if ("aws-us-gov".equals(partitionResult.name()) && !params.useFips() && !params.useDualStack()) {
                return RuleResult.endpoint(Endpoint
                        .builder()
                        .url(URI.create("https://route53.us-gov.amazonaws.com"))
                        .putAttribute(
                                AwsEndpointAttribute.AUTH_SCHEMES,
                                Arrays.asList(SigV4AuthScheme.builder().signingName("route53").signingRegion("us-gov-west-1")
                                        .build())).build());
            }
            if ("aws-us-gov".equals(partitionResult.name()) && params.useFips() && !params.useDualStack()) {
                return RuleResult.endpoint(Endpoint
                        .builder()
                        .url(URI.create("https://route53.us-gov.amazonaws.com"))
                        .putAttribute(
                                AwsEndpointAttribute.AUTH_SCHEMES,
                                Arrays.asList(SigV4AuthScheme.builder().signingName("route53").signingRegion("us-gov-west-1")
                                        .build())).build());
            }
            if ("aws-iso".equals(partitionResult.name()) && !params.useFips() && !params.useDualStack()) {
                return RuleResult.endpoint(Endpoint
                        .builder()
                        .url(URI.create("https://route53.c2s.ic.gov"))
                        .putAttribute(
                                AwsEndpointAttribute.AUTH_SCHEMES,
                                Arrays.asList(SigV4AuthScheme.builder().signingName("route53").signingRegion("us-iso-east-1")
                                        .build())).build());
            }
            if ("aws-iso-b".equals(partitionResult.name()) && !params.useFips() && !params.useDualStack()) {
                return RuleResult.endpoint(Endpoint
                        .builder()
                        .url(URI.create("https://route53.sc2s.sgov.gov"))
                        .putAttribute(
                                AwsEndpointAttribute.AUTH_SCHEMES,
                                Arrays.asList(SigV4AuthScheme.builder().signingName("route53").signingRegion("us-isob-east-1")
                                        .build())).build());
            }
            if ("aws-iso-e".equals(partitionResult.name()) && !params.useFips() && !params.useDualStack()) {
                return RuleResult.endpoint(Endpoint
                        .builder()
                        .url(URI.create("https://route53.cloud.adc-e.uk"))
                        .putAttribute(
                                AwsEndpointAttribute.AUTH_SCHEMES,
                                Arrays.asList(SigV4AuthScheme.builder().signingName("route53").signingRegion("eu-isoe-west-1")
                                        .build())).build());
            }
            if ("aws-iso-f".equals(partitionResult.name()) && !params.useFips() && !params.useDualStack()) {
                return RuleResult.endpoint(Endpoint
                        .builder()
                        .url(URI.create("https://route53.csp.hci.ic.gov"))
                        .putAttribute(
                                AwsEndpointAttribute.AUTH_SCHEMES,
                                Arrays.asList(SigV4AuthScheme.builder().signingName("route53").signingRegion("us-isof-south-1")
                                        .build())).build());
            }
            RuleResult result = endpointRule16(params, partitionResult, region);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule20(params, partitionResult, region);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule24(params, partitionResult, region);
            if (result.isResolved()) {
                return result;
            }
            return RuleResult.endpoint(Endpoint.builder()
                    .url(URI.create("https://route53." + region + "." + partitionResult.dnsSuffix())).build());
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule16(Route53EndpointParams params, RulePartition partitionResult, String region) {
        if (params.useFips() && params.useDualStack()) {
            RuleResult result = endpointRule17(params, partitionResult, region);
            if (result.isResolved()) {
                return result;
            }
            return RuleResult.error("FIPS and DualStack are enabled, but this partition does not support one or both");
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule17(Route53EndpointParams params, RulePartition partitionResult, String region) {
        if (partitionResult.supportsFIPS() && partitionResult.supportsDualStack()) {
            return RuleResult.endpoint(Endpoint.builder()
                    .url(URI.create("https://route53-fips." + region + "." + partitionResult.dualStackDnsSuffix())).build());
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule20(Route53EndpointParams params, RulePartition partitionResult, String region) {
        if (params.useFips()) {
            RuleResult result = endpointRule21(params, partitionResult, region);
            if (result.isResolved()) {
                return result;
            }
            return RuleResult.error("FIPS is enabled but this partition does not support FIPS");
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule21(Route53EndpointParams params, RulePartition partitionResult, String region) {
        if (partitionResult.supportsFIPS()) {
            return RuleResult.endpoint(Endpoint.builder()
                    .url(URI.create("https://route53-fips." + region + "." + partitionResult.dnsSuffix())).build());
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule24(Route53EndpointParams params, RulePartition partitionResult, String region) {
        if (params.useDualStack()) {
            RuleResult result = endpointRule25(params, partitionResult, region);
            if (result.isResolved()) {
                return result;
            }
            return RuleResult.error("DualStack is enabled but this partition does not support DualStack");
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule25(Route53EndpointParams params, RulePartition partitionResult, String region) {
        if (partitionResult.supportsDualStack()) {
            return RuleResult.endpoint(Endpoint.builder()
                    .url(URI.create("https://route53." + region + "." + partitionResult.dualStackDnsSuffix())).build());
        }
        return RuleResult.carryOn();
    }

    @Override
    public boolean equals(Object rhs) {
        return rhs != null && getClass().equals(rhs.getClass());
    }

    @Override
    public int hashCode() {
        return getClass().hashCode();
    }
}
