/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.route53profiles;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.route53profiles.internal.Route53ProfilesServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.route53profiles.model.AccessDeniedException;
import software.amazon.awssdk.services.route53profiles.model.AssociateProfileRequest;
import software.amazon.awssdk.services.route53profiles.model.AssociateProfileResponse;
import software.amazon.awssdk.services.route53profiles.model.AssociateResourceToProfileRequest;
import software.amazon.awssdk.services.route53profiles.model.AssociateResourceToProfileResponse;
import software.amazon.awssdk.services.route53profiles.model.ConflictException;
import software.amazon.awssdk.services.route53profiles.model.CreateProfileRequest;
import software.amazon.awssdk.services.route53profiles.model.CreateProfileResponse;
import software.amazon.awssdk.services.route53profiles.model.DeleteProfileRequest;
import software.amazon.awssdk.services.route53profiles.model.DeleteProfileResponse;
import software.amazon.awssdk.services.route53profiles.model.DisassociateProfileRequest;
import software.amazon.awssdk.services.route53profiles.model.DisassociateProfileResponse;
import software.amazon.awssdk.services.route53profiles.model.DisassociateResourceFromProfileRequest;
import software.amazon.awssdk.services.route53profiles.model.DisassociateResourceFromProfileResponse;
import software.amazon.awssdk.services.route53profiles.model.GetProfileAssociationRequest;
import software.amazon.awssdk.services.route53profiles.model.GetProfileAssociationResponse;
import software.amazon.awssdk.services.route53profiles.model.GetProfileRequest;
import software.amazon.awssdk.services.route53profiles.model.GetProfileResourceAssociationRequest;
import software.amazon.awssdk.services.route53profiles.model.GetProfileResourceAssociationResponse;
import software.amazon.awssdk.services.route53profiles.model.GetProfileResponse;
import software.amazon.awssdk.services.route53profiles.model.InternalServiceErrorException;
import software.amazon.awssdk.services.route53profiles.model.InvalidNextTokenException;
import software.amazon.awssdk.services.route53profiles.model.InvalidParameterException;
import software.amazon.awssdk.services.route53profiles.model.LimitExceededException;
import software.amazon.awssdk.services.route53profiles.model.ListProfileAssociationsRequest;
import software.amazon.awssdk.services.route53profiles.model.ListProfileAssociationsResponse;
import software.amazon.awssdk.services.route53profiles.model.ListProfileResourceAssociationsRequest;
import software.amazon.awssdk.services.route53profiles.model.ListProfileResourceAssociationsResponse;
import software.amazon.awssdk.services.route53profiles.model.ListProfilesRequest;
import software.amazon.awssdk.services.route53profiles.model.ListProfilesResponse;
import software.amazon.awssdk.services.route53profiles.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.route53profiles.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.route53profiles.model.ResourceExistsException;
import software.amazon.awssdk.services.route53profiles.model.ResourceNotFoundException;
import software.amazon.awssdk.services.route53profiles.model.Route53ProfilesException;
import software.amazon.awssdk.services.route53profiles.model.TagResourceRequest;
import software.amazon.awssdk.services.route53profiles.model.TagResourceResponse;
import software.amazon.awssdk.services.route53profiles.model.ThrottlingException;
import software.amazon.awssdk.services.route53profiles.model.UntagResourceRequest;
import software.amazon.awssdk.services.route53profiles.model.UntagResourceResponse;
import software.amazon.awssdk.services.route53profiles.model.UpdateProfileResourceAssociationRequest;
import software.amazon.awssdk.services.route53profiles.model.UpdateProfileResourceAssociationResponse;
import software.amazon.awssdk.services.route53profiles.model.ValidationException;
import software.amazon.awssdk.services.route53profiles.transform.AssociateProfileRequestMarshaller;
import software.amazon.awssdk.services.route53profiles.transform.AssociateResourceToProfileRequestMarshaller;
import software.amazon.awssdk.services.route53profiles.transform.CreateProfileRequestMarshaller;
import software.amazon.awssdk.services.route53profiles.transform.DeleteProfileRequestMarshaller;
import software.amazon.awssdk.services.route53profiles.transform.DisassociateProfileRequestMarshaller;
import software.amazon.awssdk.services.route53profiles.transform.DisassociateResourceFromProfileRequestMarshaller;
import software.amazon.awssdk.services.route53profiles.transform.GetProfileAssociationRequestMarshaller;
import software.amazon.awssdk.services.route53profiles.transform.GetProfileRequestMarshaller;
import software.amazon.awssdk.services.route53profiles.transform.GetProfileResourceAssociationRequestMarshaller;
import software.amazon.awssdk.services.route53profiles.transform.ListProfileAssociationsRequestMarshaller;
import software.amazon.awssdk.services.route53profiles.transform.ListProfileResourceAssociationsRequestMarshaller;
import software.amazon.awssdk.services.route53profiles.transform.ListProfilesRequestMarshaller;
import software.amazon.awssdk.services.route53profiles.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.route53profiles.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.route53profiles.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.route53profiles.transform.UpdateProfileResourceAssociationRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link Route53ProfilesAsyncClient}.
 *
 * @see Route53ProfilesAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultRoute53ProfilesAsyncClient implements Route53ProfilesAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultRoute53ProfilesAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultRoute53ProfilesAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Associates a Route 53 Profiles profile with a VPC. A VPC can have only one Profile associated with it, but a
     * Profile can be associated with 1000 of VPCs (and you can request a higher quota). For more information, see <a
     * href="https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/DNSLimitations.html#limits-api-entities">https://
     * docs.aws.amazon.com/Route53/latest/DeveloperGuide/DNSLimitations.html#limits-api-entities</a>.
     * </p>
     *
     * @param associateProfileRequest
     * @return A Java Future containing the result of the AssociateProfile operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource you are associating is not found.</li>
     *         <li>ResourceExistsException The resource you are trying to associate, has already been associated.</li>
     *         <li>LimitExceededException The request caused one or more limits to be exceeded.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>ValidationException You have provided an invalid command.</li>
     *         <li>AccessDeniedException The current account doesn't have the IAM permissions required to perform the
     *         specified operation.</li>
     *         <li>ConflictException The request you submitted conflicts with an existing request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ProfilesException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ProfilesAsyncClient.AssociateProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53profiles-2018-05-10/AssociateProfile"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateProfileResponse> associateProfile(AssociateProfileRequest associateProfileRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(associateProfileRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateProfile");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AssociateProfileResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AssociateProfileResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateProfileRequest, AssociateProfileResponse>()
                            .withOperationName("AssociateProfile").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new AssociateProfileRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(associateProfileRequest));
            CompletableFuture<AssociateProfileResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Associates a DNS reource configuration to a Route 53 Profile.
     * </p>
     *
     * @param associateResourceToProfileRequest
     * @return A Java Future containing the result of the AssociateResourceToProfile operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource you are associating is not found.</li>
     *         <li>LimitExceededException The request caused one or more limits to be exceeded.</li>
     *         <li>InternalServiceErrorException An internal server error occured. Retry your request.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>ValidationException You have provided an invalid command.</li>
     *         <li>AccessDeniedException The current account doesn't have the IAM permissions required to perform the
     *         specified operation.</li>
     *         <li>ConflictException The request you submitted conflicts with an existing request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ProfilesException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ProfilesAsyncClient.AssociateResourceToProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53profiles-2018-05-10/AssociateResourceToProfile"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateResourceToProfileResponse> associateResourceToProfile(
            AssociateResourceToProfileRequest associateResourceToProfileRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(associateResourceToProfileRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateResourceToProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateResourceToProfile");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateResourceToProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AssociateResourceToProfileResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AssociateResourceToProfileResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateResourceToProfileRequest, AssociateResourceToProfileResponse>()
                            .withOperationName("AssociateResourceToProfile").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new AssociateResourceToProfileRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(associateResourceToProfileRequest));
            CompletableFuture<AssociateResourceToProfileResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an empty Route 53 Profile.
     * </p>
     *
     * @param createProfileRequest
     * @return A Java Future containing the result of the CreateProfile operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>LimitExceededException The request caused one or more limits to be exceeded.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>ValidationException You have provided an invalid command.</li>
     *         <li>AccessDeniedException The current account doesn't have the IAM permissions required to perform the
     *         specified operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ProfilesException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ProfilesAsyncClient.CreateProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53profiles-2018-05-10/CreateProfile" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateProfileResponse> createProfile(CreateProfileRequest createProfileRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createProfileRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateProfile");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateProfileResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateProfileResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateProfileResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateProfileRequest, CreateProfileResponse>()
                            .withOperationName("CreateProfile").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateProfileRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createProfileRequest));
            CompletableFuture<CreateProfileResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified Route 53 Profile. Before you can delete a profile, you must first disassociate it from all
     * VPCs.
     * </p>
     *
     * @param deleteProfileRequest
     * @return A Java Future containing the result of the DeleteProfile operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource you are associating is not found.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>ValidationException You have provided an invalid command.</li>
     *         <li>AccessDeniedException The current account doesn't have the IAM permissions required to perform the
     *         specified operation.</li>
     *         <li>ConflictException The request you submitted conflicts with an existing request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ProfilesException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ProfilesAsyncClient.DeleteProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53profiles-2018-05-10/DeleteProfile" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteProfileResponse> deleteProfile(DeleteProfileRequest deleteProfileRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteProfileRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteProfile");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteProfileResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteProfileResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteProfileResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteProfileRequest, DeleteProfileResponse>()
                            .withOperationName("DeleteProfile").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteProfileRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteProfileRequest));
            CompletableFuture<DeleteProfileResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Dissociates a specified Route 53 Profile from the specified VPC.
     * </p>
     *
     * @param disassociateProfileRequest
     * @return A Java Future containing the result of the DisassociateProfile operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource you are associating is not found.</li>
     *         <li>LimitExceededException The request caused one or more limits to be exceeded.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>ValidationException You have provided an invalid command.</li>
     *         <li>AccessDeniedException The current account doesn't have the IAM permissions required to perform the
     *         specified operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ProfilesException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ProfilesAsyncClient.DisassociateProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53profiles-2018-05-10/DisassociateProfile"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateProfileResponse> disassociateProfile(
            DisassociateProfileRequest disassociateProfileRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disassociateProfileRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disassociateProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateProfile");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisassociateProfileResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateProfileResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateProfileRequest, DisassociateProfileResponse>()
                            .withOperationName("DisassociateProfile").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DisassociateProfileRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(disassociateProfileRequest));
            CompletableFuture<DisassociateProfileResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Dissoaciated a specified resource, from the Route 53 Profile.
     * </p>
     *
     * @param disassociateResourceFromProfileRequest
     * @return A Java Future containing the result of the DisassociateResourceFromProfile operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource you are associating is not found.</li>
     *         <li>LimitExceededException The request caused one or more limits to be exceeded.</li>
     *         <li>InternalServiceErrorException An internal server error occured. Retry your request.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>ValidationException You have provided an invalid command.</li>
     *         <li>AccessDeniedException The current account doesn't have the IAM permissions required to perform the
     *         specified operation.</li>
     *         <li>ConflictException The request you submitted conflicts with an existing request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ProfilesException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ProfilesAsyncClient.DisassociateResourceFromProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53profiles-2018-05-10/DisassociateResourceFromProfile"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateResourceFromProfileResponse> disassociateResourceFromProfile(
            DisassociateResourceFromProfileRequest disassociateResourceFromProfileRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disassociateResourceFromProfileRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                disassociateResourceFromProfileRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateResourceFromProfile");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateResourceFromProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisassociateResourceFromProfileResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateResourceFromProfileResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateResourceFromProfileRequest, DisassociateResourceFromProfileResponse>()
                            .withOperationName("DisassociateResourceFromProfile").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DisassociateResourceFromProfileRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(disassociateResourceFromProfileRequest));
            CompletableFuture<DisassociateResourceFromProfileResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about a specified Route 53 Profile, such as whether whether the Profile is shared, and the
     * current status of the Profile.
     * </p>
     *
     * @param getProfileRequest
     * @return A Java Future containing the result of the GetProfile operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource you are associating is not found.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>ValidationException You have provided an invalid command.</li>
     *         <li>AccessDeniedException The current account doesn't have the IAM permissions required to perform the
     *         specified operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ProfilesException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ProfilesAsyncClient.GetProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53profiles-2018-05-10/GetProfile" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetProfileResponse> getProfile(GetProfileRequest getProfileRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getProfileRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetProfile");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetProfileResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetProfileResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetProfileResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetProfileRequest, GetProfileResponse>().withOperationName("GetProfile")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetProfileRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getProfileRequest));
            CompletableFuture<GetProfileResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a Route 53 Profile association for a VPC. A VPC can have only one Profile association, but a Profile
     * can be associated with up to 5000 VPCs.
     * </p>
     *
     * @param getProfileAssociationRequest
     * @return A Java Future containing the result of the GetProfileAssociation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource you are associating is not found.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>ValidationException You have provided an invalid command.</li>
     *         <li>AccessDeniedException The current account doesn't have the IAM permissions required to perform the
     *         specified operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ProfilesException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ProfilesAsyncClient.GetProfileAssociation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53profiles-2018-05-10/GetProfileAssociation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetProfileAssociationResponse> getProfileAssociation(
            GetProfileAssociationRequest getProfileAssociationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getProfileAssociationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getProfileAssociationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetProfileAssociation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetProfileAssociationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetProfileAssociationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetProfileAssociationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetProfileAssociationRequest, GetProfileAssociationResponse>()
                            .withOperationName("GetProfileAssociation").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetProfileAssociationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getProfileAssociationRequest));
            CompletableFuture<GetProfileAssociationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about a specified Route 53 Profile resource association.
     * </p>
     *
     * @param getProfileResourceAssociationRequest
     * @return A Java Future containing the result of the GetProfileResourceAssociation operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource you are associating is not found.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>ValidationException You have provided an invalid command.</li>
     *         <li>AccessDeniedException The current account doesn't have the IAM permissions required to perform the
     *         specified operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ProfilesException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ProfilesAsyncClient.GetProfileResourceAssociation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53profiles-2018-05-10/GetProfileResourceAssociation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetProfileResourceAssociationResponse> getProfileResourceAssociation(
            GetProfileResourceAssociationRequest getProfileResourceAssociationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getProfileResourceAssociationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getProfileResourceAssociationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetProfileResourceAssociation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetProfileResourceAssociationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetProfileResourceAssociationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetProfileResourceAssociationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetProfileResourceAssociationRequest, GetProfileResourceAssociationResponse>()
                            .withOperationName("GetProfileResourceAssociation").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetProfileResourceAssociationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getProfileResourceAssociationRequest));
            CompletableFuture<GetProfileResourceAssociationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all the VPCs that the specified Route 53 Profile is associated with.
     * </p>
     *
     * @param listProfileAssociationsRequest
     * @return A Java Future containing the result of the ListProfileAssociations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>ValidationException You have provided an invalid command.</li>
     *         <li>AccessDeniedException The current account doesn't have the IAM permissions required to perform the
     *         specified operation.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> you provided isn;t valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ProfilesException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ProfilesAsyncClient.ListProfileAssociations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53profiles-2018-05-10/ListProfileAssociations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListProfileAssociationsResponse> listProfileAssociations(
            ListProfileAssociationsRequest listProfileAssociationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listProfileAssociationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listProfileAssociationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListProfileAssociations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListProfileAssociationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListProfileAssociationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListProfileAssociationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListProfileAssociationsRequest, ListProfileAssociationsResponse>()
                            .withOperationName("ListProfileAssociations").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListProfileAssociationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listProfileAssociationsRequest));
            CompletableFuture<ListProfileAssociationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all the resource associations for the specified Route 53 Profile.
     * </p>
     *
     * @param listProfileResourceAssociationsRequest
     * @return A Java Future containing the result of the ListProfileResourceAssociations operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource you are associating is not found.</li>
     *         <li>InternalServiceErrorException An internal server error occured. Retry your request.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>ValidationException You have provided an invalid command.</li>
     *         <li>AccessDeniedException The current account doesn't have the IAM permissions required to perform the
     *         specified operation.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> you provided isn;t valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ProfilesException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ProfilesAsyncClient.ListProfileResourceAssociations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53profiles-2018-05-10/ListProfileResourceAssociations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListProfileResourceAssociationsResponse> listProfileResourceAssociations(
            ListProfileResourceAssociationsRequest listProfileResourceAssociationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listProfileResourceAssociationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listProfileResourceAssociationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListProfileResourceAssociations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListProfileResourceAssociationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListProfileResourceAssociationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListProfileResourceAssociationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListProfileResourceAssociationsRequest, ListProfileResourceAssociationsResponse>()
                            .withOperationName("ListProfileResourceAssociations").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListProfileResourceAssociationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listProfileResourceAssociationsRequest));
            CompletableFuture<ListProfileResourceAssociationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all the Route 53 Profiles associated with your Amazon Web Services account.
     * </p>
     *
     * @param listProfilesRequest
     * @return A Java Future containing the result of the ListProfiles operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>ValidationException You have provided an invalid command.</li>
     *         <li>AccessDeniedException The current account doesn't have the IAM permissions required to perform the
     *         specified operation.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> you provided isn;t valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ProfilesException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ProfilesAsyncClient.ListProfiles
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53profiles-2018-05-10/ListProfiles" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListProfilesResponse> listProfiles(ListProfilesRequest listProfilesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listProfilesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listProfilesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListProfiles");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListProfilesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListProfilesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListProfilesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListProfilesRequest, ListProfilesResponse>()
                            .withOperationName("ListProfiles").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListProfilesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listProfilesRequest));
            CompletableFuture<ListProfilesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the tags that you associated with the specified resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource you are associating is not found.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>ValidationException You have provided an invalid command.</li>
     *         <li>AccessDeniedException The current account doesn't have the IAM permissions required to perform the
     *         specified operation.</li>
     *         <li>ConflictException The request you submitted conflicts with an existing request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ProfilesException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ProfilesAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53profiles-2018-05-10/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds one or more tags to a specified resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource you are associating is not found.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>ValidationException You have provided an invalid command.</li>
     *         <li>AccessDeniedException The current account doesn't have the IAM permissions required to perform the
     *         specified operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ProfilesException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ProfilesAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53profiles-2018-05-10/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes one or more tags from a specified resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource you are associating is not found.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>ValidationException You have provided an invalid command.</li>
     *         <li>AccessDeniedException The current account doesn't have the IAM permissions required to perform the
     *         specified operation.</li>
     *         <li>ConflictException The request you submitted conflicts with an existing request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ProfilesException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ProfilesAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53profiles-2018-05-10/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the specified Route 53 Profile resourse association.
     * </p>
     *
     * @param updateProfileResourceAssociationRequest
     * @return A Java Future containing the result of the UpdateProfileResourceAssociation operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource you are associating is not found.</li>
     *         <li>LimitExceededException The request caused one or more limits to be exceeded.</li>
     *         <li>InternalServiceErrorException An internal server error occured. Retry your request.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>ValidationException You have provided an invalid command.</li>
     *         <li>AccessDeniedException The current account doesn't have the IAM permissions required to perform the
     *         specified operation.</li>
     *         <li>ConflictException The request you submitted conflicts with an existing request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ProfilesException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ProfilesAsyncClient.UpdateProfileResourceAssociation
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53profiles-2018-05-10/UpdateProfileResourceAssociation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateProfileResourceAssociationResponse> updateProfileResourceAssociation(
            UpdateProfileResourceAssociationRequest updateProfileResourceAssociationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateProfileResourceAssociationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateProfileResourceAssociationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateProfileResourceAssociation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateProfileResourceAssociationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, UpdateProfileResourceAssociationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateProfileResourceAssociationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateProfileResourceAssociationRequest, UpdateProfileResourceAssociationResponse>()
                            .withOperationName("UpdateProfileResourceAssociation").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateProfileResourceAssociationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateProfileResourceAssociationRequest));
            CompletableFuture<UpdateProfileResourceAssociationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final Route53ProfilesServiceClientConfiguration serviceClientConfiguration() {
        return new Route53ProfilesServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(Route53ProfilesException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterException")
                                .exceptionBuilderSupplier(InvalidParameterException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServiceErrorException")
                                .exceptionBuilderSupplier(InternalServiceErrorException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceExistsException")
                                .exceptionBuilderSupplier(ResourceExistsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNextTokenException")
                                .exceptionBuilderSupplier(InvalidNextTokenException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        Route53ProfilesServiceClientConfigurationBuilder serviceConfigBuilder = new Route53ProfilesServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
