/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.route53profiles;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.route53profiles.internal.Route53ProfilesServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.route53profiles.model.AccessDeniedException;
import software.amazon.awssdk.services.route53profiles.model.AssociateProfileRequest;
import software.amazon.awssdk.services.route53profiles.model.AssociateProfileResponse;
import software.amazon.awssdk.services.route53profiles.model.AssociateResourceToProfileRequest;
import software.amazon.awssdk.services.route53profiles.model.AssociateResourceToProfileResponse;
import software.amazon.awssdk.services.route53profiles.model.ConflictException;
import software.amazon.awssdk.services.route53profiles.model.CreateProfileRequest;
import software.amazon.awssdk.services.route53profiles.model.CreateProfileResponse;
import software.amazon.awssdk.services.route53profiles.model.DeleteProfileRequest;
import software.amazon.awssdk.services.route53profiles.model.DeleteProfileResponse;
import software.amazon.awssdk.services.route53profiles.model.DisassociateProfileRequest;
import software.amazon.awssdk.services.route53profiles.model.DisassociateProfileResponse;
import software.amazon.awssdk.services.route53profiles.model.DisassociateResourceFromProfileRequest;
import software.amazon.awssdk.services.route53profiles.model.DisassociateResourceFromProfileResponse;
import software.amazon.awssdk.services.route53profiles.model.GetProfileAssociationRequest;
import software.amazon.awssdk.services.route53profiles.model.GetProfileAssociationResponse;
import software.amazon.awssdk.services.route53profiles.model.GetProfileRequest;
import software.amazon.awssdk.services.route53profiles.model.GetProfileResourceAssociationRequest;
import software.amazon.awssdk.services.route53profiles.model.GetProfileResourceAssociationResponse;
import software.amazon.awssdk.services.route53profiles.model.GetProfileResponse;
import software.amazon.awssdk.services.route53profiles.model.InternalServiceErrorException;
import software.amazon.awssdk.services.route53profiles.model.InvalidNextTokenException;
import software.amazon.awssdk.services.route53profiles.model.InvalidParameterException;
import software.amazon.awssdk.services.route53profiles.model.LimitExceededException;
import software.amazon.awssdk.services.route53profiles.model.ListProfileAssociationsRequest;
import software.amazon.awssdk.services.route53profiles.model.ListProfileAssociationsResponse;
import software.amazon.awssdk.services.route53profiles.model.ListProfileResourceAssociationsRequest;
import software.amazon.awssdk.services.route53profiles.model.ListProfileResourceAssociationsResponse;
import software.amazon.awssdk.services.route53profiles.model.ListProfilesRequest;
import software.amazon.awssdk.services.route53profiles.model.ListProfilesResponse;
import software.amazon.awssdk.services.route53profiles.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.route53profiles.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.route53profiles.model.ResourceExistsException;
import software.amazon.awssdk.services.route53profiles.model.ResourceNotFoundException;
import software.amazon.awssdk.services.route53profiles.model.Route53ProfilesException;
import software.amazon.awssdk.services.route53profiles.model.TagResourceRequest;
import software.amazon.awssdk.services.route53profiles.model.TagResourceResponse;
import software.amazon.awssdk.services.route53profiles.model.ThrottlingException;
import software.amazon.awssdk.services.route53profiles.model.UntagResourceRequest;
import software.amazon.awssdk.services.route53profiles.model.UntagResourceResponse;
import software.amazon.awssdk.services.route53profiles.model.UpdateProfileResourceAssociationRequest;
import software.amazon.awssdk.services.route53profiles.model.UpdateProfileResourceAssociationResponse;
import software.amazon.awssdk.services.route53profiles.model.ValidationException;
import software.amazon.awssdk.services.route53profiles.transform.AssociateProfileRequestMarshaller;
import software.amazon.awssdk.services.route53profiles.transform.AssociateResourceToProfileRequestMarshaller;
import software.amazon.awssdk.services.route53profiles.transform.CreateProfileRequestMarshaller;
import software.amazon.awssdk.services.route53profiles.transform.DeleteProfileRequestMarshaller;
import software.amazon.awssdk.services.route53profiles.transform.DisassociateProfileRequestMarshaller;
import software.amazon.awssdk.services.route53profiles.transform.DisassociateResourceFromProfileRequestMarshaller;
import software.amazon.awssdk.services.route53profiles.transform.GetProfileAssociationRequestMarshaller;
import software.amazon.awssdk.services.route53profiles.transform.GetProfileRequestMarshaller;
import software.amazon.awssdk.services.route53profiles.transform.GetProfileResourceAssociationRequestMarshaller;
import software.amazon.awssdk.services.route53profiles.transform.ListProfileAssociationsRequestMarshaller;
import software.amazon.awssdk.services.route53profiles.transform.ListProfileResourceAssociationsRequestMarshaller;
import software.amazon.awssdk.services.route53profiles.transform.ListProfilesRequestMarshaller;
import software.amazon.awssdk.services.route53profiles.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.route53profiles.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.route53profiles.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.route53profiles.transform.UpdateProfileResourceAssociationRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link Route53ProfilesClient}.
 *
 * @see Route53ProfilesClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultRoute53ProfilesClient implements Route53ProfilesClient {
    private static final Logger log = Logger.loggerFor(DefaultRoute53ProfilesClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultRoute53ProfilesClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Associates a Route 53 Profiles profile with a VPC. A VPC can have only one Profile associated with it, but a
     * Profile can be associated with 1000 of VPCs (and you can request a higher quota). For more information, see <a
     * href="https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/DNSLimitations.html#limits-api-entities">https://
     * docs.aws.amazon.com/Route53/latest/DeveloperGuide/DNSLimitations.html#limits-api-entities</a>.
     * </p>
     *
     * @param associateProfileRequest
     * @return Result of the AssociateProfile operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource you are associating is not found.
     * @throws ResourceExistsException
     *         The resource you are trying to associate, has already been associated.
     * @throws LimitExceededException
     *         The request caused one or more limits to be exceeded.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws ValidationException
     *         You have provided an invalid command.
     * @throws AccessDeniedException
     *         The current account doesn't have the IAM permissions required to perform the specified operation.
     * @throws ConflictException
     *         The request you submitted conflicts with an existing request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ProfilesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ProfilesClient.AssociateProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53profiles-2018-05-10/AssociateProfile"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AssociateProfileResponse associateProfile(AssociateProfileRequest associateProfileRequest)
            throws ResourceNotFoundException, ResourceExistsException, LimitExceededException, InvalidParameterException,
            ThrottlingException, ValidationException, AccessDeniedException, ConflictException, AwsServiceException,
            SdkClientException, Route53ProfilesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AssociateProfileResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                AssociateProfileResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(associateProfileRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateProfile");

            return clientHandler.execute(new ClientExecutionParams<AssociateProfileRequest, AssociateProfileResponse>()
                    .withOperationName("AssociateProfile").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(associateProfileRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new AssociateProfileRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Associates a DNS reource configuration to a Route 53 Profile.
     * </p>
     *
     * @param associateResourceToProfileRequest
     * @return Result of the AssociateResourceToProfile operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource you are associating is not found.
     * @throws LimitExceededException
     *         The request caused one or more limits to be exceeded.
     * @throws InternalServiceErrorException
     *         An internal server error occured. Retry your request.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws ValidationException
     *         You have provided an invalid command.
     * @throws AccessDeniedException
     *         The current account doesn't have the IAM permissions required to perform the specified operation.
     * @throws ConflictException
     *         The request you submitted conflicts with an existing request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ProfilesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ProfilesClient.AssociateResourceToProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53profiles-2018-05-10/AssociateResourceToProfile"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AssociateResourceToProfileResponse associateResourceToProfile(
            AssociateResourceToProfileRequest associateResourceToProfileRequest) throws ResourceNotFoundException,
            LimitExceededException, InternalServiceErrorException, InvalidParameterException, ThrottlingException,
            ValidationException, AccessDeniedException, ConflictException, AwsServiceException, SdkClientException,
            Route53ProfilesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AssociateResourceToProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, AssociateResourceToProfileResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(associateResourceToProfileRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateResourceToProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateResourceToProfile");

            return clientHandler
                    .execute(new ClientExecutionParams<AssociateResourceToProfileRequest, AssociateResourceToProfileResponse>()
                            .withOperationName("AssociateResourceToProfile").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(associateResourceToProfileRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new AssociateResourceToProfileRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates an empty Route 53 Profile.
     * </p>
     *
     * @param createProfileRequest
     * @return Result of the CreateProfile operation returned by the service.
     * @throws LimitExceededException
     *         The request caused one or more limits to be exceeded.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws ValidationException
     *         You have provided an invalid command.
     * @throws AccessDeniedException
     *         The current account doesn't have the IAM permissions required to perform the specified operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ProfilesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ProfilesClient.CreateProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53profiles-2018-05-10/CreateProfile" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateProfileResponse createProfile(CreateProfileRequest createProfileRequest) throws LimitExceededException,
            InvalidParameterException, ThrottlingException, ValidationException, AccessDeniedException, AwsServiceException,
            SdkClientException, Route53ProfilesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateProfileResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateProfileResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createProfileRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateProfile");

            return clientHandler.execute(new ClientExecutionParams<CreateProfileRequest, CreateProfileResponse>()
                    .withOperationName("CreateProfile").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createProfileRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateProfileRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified Route 53 Profile. Before you can delete a profile, you must first disassociate it from all
     * VPCs.
     * </p>
     *
     * @param deleteProfileRequest
     * @return Result of the DeleteProfile operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource you are associating is not found.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws ValidationException
     *         You have provided an invalid command.
     * @throws AccessDeniedException
     *         The current account doesn't have the IAM permissions required to perform the specified operation.
     * @throws ConflictException
     *         The request you submitted conflicts with an existing request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ProfilesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ProfilesClient.DeleteProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53profiles-2018-05-10/DeleteProfile" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteProfileResponse deleteProfile(DeleteProfileRequest deleteProfileRequest) throws ResourceNotFoundException,
            ThrottlingException, ValidationException, AccessDeniedException, ConflictException, AwsServiceException,
            SdkClientException, Route53ProfilesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteProfileResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteProfileResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteProfileRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteProfile");

            return clientHandler.execute(new ClientExecutionParams<DeleteProfileRequest, DeleteProfileResponse>()
                    .withOperationName("DeleteProfile").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteProfileRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteProfileRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Dissociates a specified Route 53 Profile from the specified VPC.
     * </p>
     *
     * @param disassociateProfileRequest
     * @return Result of the DisassociateProfile operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource you are associating is not found.
     * @throws LimitExceededException
     *         The request caused one or more limits to be exceeded.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws ValidationException
     *         You have provided an invalid command.
     * @throws AccessDeniedException
     *         The current account doesn't have the IAM permissions required to perform the specified operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ProfilesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ProfilesClient.DisassociateProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53profiles-2018-05-10/DisassociateProfile"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisassociateProfileResponse disassociateProfile(DisassociateProfileRequest disassociateProfileRequest)
            throws ResourceNotFoundException, LimitExceededException, InvalidParameterException, ThrottlingException,
            ValidationException, AccessDeniedException, AwsServiceException, SdkClientException, Route53ProfilesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisassociateProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisassociateProfileResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disassociateProfileRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disassociateProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateProfile");

            return clientHandler.execute(new ClientExecutionParams<DisassociateProfileRequest, DisassociateProfileResponse>()
                    .withOperationName("DisassociateProfile").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(disassociateProfileRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DisassociateProfileRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Dissoaciated a specified resource, from the Route 53 Profile.
     * </p>
     *
     * @param disassociateResourceFromProfileRequest
     * @return Result of the DisassociateResourceFromProfile operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource you are associating is not found.
     * @throws LimitExceededException
     *         The request caused one or more limits to be exceeded.
     * @throws InternalServiceErrorException
     *         An internal server error occured. Retry your request.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws ValidationException
     *         You have provided an invalid command.
     * @throws AccessDeniedException
     *         The current account doesn't have the IAM permissions required to perform the specified operation.
     * @throws ConflictException
     *         The request you submitted conflicts with an existing request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ProfilesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ProfilesClient.DisassociateResourceFromProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53profiles-2018-05-10/DisassociateResourceFromProfile"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisassociateResourceFromProfileResponse disassociateResourceFromProfile(
            DisassociateResourceFromProfileRequest disassociateResourceFromProfileRequest) throws ResourceNotFoundException,
            LimitExceededException, InternalServiceErrorException, InvalidParameterException, ThrottlingException,
            ValidationException, AccessDeniedException, ConflictException, AwsServiceException, SdkClientException,
            Route53ProfilesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisassociateResourceFromProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisassociateResourceFromProfileResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disassociateResourceFromProfileRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                disassociateResourceFromProfileRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateResourceFromProfile");

            return clientHandler
                    .execute(new ClientExecutionParams<DisassociateResourceFromProfileRequest, DisassociateResourceFromProfileResponse>()
                            .withOperationName("DisassociateResourceFromProfile").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(disassociateResourceFromProfileRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DisassociateResourceFromProfileRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about a specified Route 53 Profile, such as whether whether the Profile is shared, and the
     * current status of the Profile.
     * </p>
     *
     * @param getProfileRequest
     * @return Result of the GetProfile operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource you are associating is not found.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws ValidationException
     *         You have provided an invalid command.
     * @throws AccessDeniedException
     *         The current account doesn't have the IAM permissions required to perform the specified operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ProfilesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ProfilesClient.GetProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53profiles-2018-05-10/GetProfile" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetProfileResponse getProfile(GetProfileRequest getProfileRequest) throws ResourceNotFoundException,
            ThrottlingException, ValidationException, AccessDeniedException, AwsServiceException, SdkClientException,
            Route53ProfilesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetProfileResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetProfileResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getProfileRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetProfile");

            return clientHandler.execute(new ClientExecutionParams<GetProfileRequest, GetProfileResponse>()
                    .withOperationName("GetProfile").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getProfileRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetProfileRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves a Route 53 Profile association for a VPC. A VPC can have only one Profile association, but a Profile
     * can be associated with up to 5000 VPCs.
     * </p>
     *
     * @param getProfileAssociationRequest
     * @return Result of the GetProfileAssociation operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource you are associating is not found.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws ValidationException
     *         You have provided an invalid command.
     * @throws AccessDeniedException
     *         The current account doesn't have the IAM permissions required to perform the specified operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ProfilesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ProfilesClient.GetProfileAssociation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53profiles-2018-05-10/GetProfileAssociation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetProfileAssociationResponse getProfileAssociation(GetProfileAssociationRequest getProfileAssociationRequest)
            throws ResourceNotFoundException, ThrottlingException, ValidationException, AccessDeniedException,
            AwsServiceException, SdkClientException, Route53ProfilesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetProfileAssociationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetProfileAssociationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getProfileAssociationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getProfileAssociationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetProfileAssociation");

            return clientHandler.execute(new ClientExecutionParams<GetProfileAssociationRequest, GetProfileAssociationResponse>()
                    .withOperationName("GetProfileAssociation").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getProfileAssociationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetProfileAssociationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about a specified Route 53 Profile resource association.
     * </p>
     *
     * @param getProfileResourceAssociationRequest
     * @return Result of the GetProfileResourceAssociation operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource you are associating is not found.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws ValidationException
     *         You have provided an invalid command.
     * @throws AccessDeniedException
     *         The current account doesn't have the IAM permissions required to perform the specified operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ProfilesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ProfilesClient.GetProfileResourceAssociation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53profiles-2018-05-10/GetProfileResourceAssociation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetProfileResourceAssociationResponse getProfileResourceAssociation(
            GetProfileResourceAssociationRequest getProfileResourceAssociationRequest) throws ResourceNotFoundException,
            InvalidParameterException, ThrottlingException, ValidationException, AccessDeniedException, AwsServiceException,
            SdkClientException, Route53ProfilesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetProfileResourceAssociationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetProfileResourceAssociationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getProfileResourceAssociationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getProfileResourceAssociationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetProfileResourceAssociation");

            return clientHandler
                    .execute(new ClientExecutionParams<GetProfileResourceAssociationRequest, GetProfileResourceAssociationResponse>()
                            .withOperationName("GetProfileResourceAssociation").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getProfileResourceAssociationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetProfileResourceAssociationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all the VPCs that the specified Route 53 Profile is associated with.
     * </p>
     *
     * @param listProfileAssociationsRequest
     * @return Result of the ListProfileAssociations operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws ValidationException
     *         You have provided an invalid command.
     * @throws AccessDeniedException
     *         The current account doesn't have the IAM permissions required to perform the specified operation.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> you provided isn;t valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ProfilesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ProfilesClient.ListProfileAssociations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53profiles-2018-05-10/ListProfileAssociations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListProfileAssociationsResponse listProfileAssociations(ListProfileAssociationsRequest listProfileAssociationsRequest)
            throws InvalidParameterException, ThrottlingException, ValidationException, AccessDeniedException,
            InvalidNextTokenException, AwsServiceException, SdkClientException, Route53ProfilesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListProfileAssociationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListProfileAssociationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listProfileAssociationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listProfileAssociationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListProfileAssociations");

            return clientHandler
                    .execute(new ClientExecutionParams<ListProfileAssociationsRequest, ListProfileAssociationsResponse>()
                            .withOperationName("ListProfileAssociations").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listProfileAssociationsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListProfileAssociationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all the resource associations for the specified Route 53 Profile.
     * </p>
     *
     * @param listProfileResourceAssociationsRequest
     * @return Result of the ListProfileResourceAssociations operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource you are associating is not found.
     * @throws InternalServiceErrorException
     *         An internal server error occured. Retry your request.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws ValidationException
     *         You have provided an invalid command.
     * @throws AccessDeniedException
     *         The current account doesn't have the IAM permissions required to perform the specified operation.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> you provided isn;t valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ProfilesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ProfilesClient.ListProfileResourceAssociations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53profiles-2018-05-10/ListProfileResourceAssociations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListProfileResourceAssociationsResponse listProfileResourceAssociations(
            ListProfileResourceAssociationsRequest listProfileResourceAssociationsRequest) throws ResourceNotFoundException,
            InternalServiceErrorException, InvalidParameterException, ThrottlingException, ValidationException,
            AccessDeniedException, InvalidNextTokenException, AwsServiceException, SdkClientException, Route53ProfilesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListProfileResourceAssociationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListProfileResourceAssociationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listProfileResourceAssociationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listProfileResourceAssociationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListProfileResourceAssociations");

            return clientHandler
                    .execute(new ClientExecutionParams<ListProfileResourceAssociationsRequest, ListProfileResourceAssociationsResponse>()
                            .withOperationName("ListProfileResourceAssociations").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listProfileResourceAssociationsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListProfileResourceAssociationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all the Route 53 Profiles associated with your Amazon Web Services account.
     * </p>
     *
     * @param listProfilesRequest
     * @return Result of the ListProfiles operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws ValidationException
     *         You have provided an invalid command.
     * @throws AccessDeniedException
     *         The current account doesn't have the IAM permissions required to perform the specified operation.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> you provided isn;t valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ProfilesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ProfilesClient.ListProfiles
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53profiles-2018-05-10/ListProfiles" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListProfilesResponse listProfiles(ListProfilesRequest listProfilesRequest) throws InvalidParameterException,
            ThrottlingException, ValidationException, AccessDeniedException, InvalidNextTokenException, AwsServiceException,
            SdkClientException, Route53ProfilesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListProfilesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListProfilesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listProfilesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listProfilesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListProfiles");

            return clientHandler.execute(new ClientExecutionParams<ListProfilesRequest, ListProfilesResponse>()
                    .withOperationName("ListProfiles").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listProfilesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListProfilesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the tags that you associated with the specified resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource you are associating is not found.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws ValidationException
     *         You have provided an invalid command.
     * @throws AccessDeniedException
     *         The current account doesn't have the IAM permissions required to perform the specified operation.
     * @throws ConflictException
     *         The request you submitted conflicts with an existing request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ProfilesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ProfilesClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53profiles-2018-05-10/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, ThrottlingException, ValidationException, AccessDeniedException, ConflictException,
            AwsServiceException, SdkClientException, Route53ProfilesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds one or more tags to a specified resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource you are associating is not found.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws ValidationException
     *         You have provided an invalid command.
     * @throws AccessDeniedException
     *         The current account doesn't have the IAM permissions required to perform the specified operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ProfilesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ProfilesClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53profiles-2018-05-10/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ResourceNotFoundException,
            ThrottlingException, ValidationException, AccessDeniedException, AwsServiceException, SdkClientException,
            Route53ProfilesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes one or more tags from a specified resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource you are associating is not found.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws ValidationException
     *         You have provided an invalid command.
     * @throws AccessDeniedException
     *         The current account doesn't have the IAM permissions required to perform the specified operation.
     * @throws ConflictException
     *         The request you submitted conflicts with an existing request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ProfilesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ProfilesClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53profiles-2018-05-10/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            ThrottlingException, ValidationException, AccessDeniedException, ConflictException, AwsServiceException,
            SdkClientException, Route53ProfilesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the specified Route 53 Profile resourse association.
     * </p>
     *
     * @param updateProfileResourceAssociationRequest
     * @return Result of the UpdateProfileResourceAssociation operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource you are associating is not found.
     * @throws LimitExceededException
     *         The request caused one or more limits to be exceeded.
     * @throws InternalServiceErrorException
     *         An internal server error occured. Retry your request.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws ValidationException
     *         You have provided an invalid command.
     * @throws AccessDeniedException
     *         The current account doesn't have the IAM permissions required to perform the specified operation.
     * @throws ConflictException
     *         The request you submitted conflicts with an existing request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ProfilesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ProfilesClient.UpdateProfileResourceAssociation
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53profiles-2018-05-10/UpdateProfileResourceAssociation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateProfileResourceAssociationResponse updateProfileResourceAssociation(
            UpdateProfileResourceAssociationRequest updateProfileResourceAssociationRequest) throws ResourceNotFoundException,
            LimitExceededException, InternalServiceErrorException, InvalidParameterException, ThrottlingException,
            ValidationException, AccessDeniedException, ConflictException, AwsServiceException, SdkClientException,
            Route53ProfilesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateProfileResourceAssociationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateProfileResourceAssociationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateProfileResourceAssociationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateProfileResourceAssociationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateProfileResourceAssociation");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateProfileResourceAssociationRequest, UpdateProfileResourceAssociationResponse>()
                            .withOperationName("UpdateProfileResourceAssociation").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(updateProfileResourceAssociationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateProfileResourceAssociationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        Route53ProfilesServiceClientConfigurationBuilder serviceConfigBuilder = new Route53ProfilesServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(Route53ProfilesException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterException")
                                .exceptionBuilderSupplier(InvalidParameterException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServiceErrorException")
                                .exceptionBuilderSupplier(InternalServiceErrorException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceExistsException")
                                .exceptionBuilderSupplier(ResourceExistsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNextTokenException")
                                .exceptionBuilderSupplier(InvalidNextTokenException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).build());
    }

    @Override
    public final Route53ProfilesServiceClientConfiguration serviceClientConfiguration() {
        return new Route53ProfilesServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
