/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.route53recoverycluster;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.SdkClient;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.regions.ServiceMetadata;
import software.amazon.awssdk.services.route53recoverycluster.model.AccessDeniedException;
import software.amazon.awssdk.services.route53recoverycluster.model.ConflictException;
import software.amazon.awssdk.services.route53recoverycluster.model.EndpointTemporarilyUnavailableException;
import software.amazon.awssdk.services.route53recoverycluster.model.GetRoutingControlStateRequest;
import software.amazon.awssdk.services.route53recoverycluster.model.GetRoutingControlStateResponse;
import software.amazon.awssdk.services.route53recoverycluster.model.InternalServerException;
import software.amazon.awssdk.services.route53recoverycluster.model.ListRoutingControlsRequest;
import software.amazon.awssdk.services.route53recoverycluster.model.ListRoutingControlsResponse;
import software.amazon.awssdk.services.route53recoverycluster.model.ResourceNotFoundException;
import software.amazon.awssdk.services.route53recoverycluster.model.Route53RecoveryClusterException;
import software.amazon.awssdk.services.route53recoverycluster.model.ServiceLimitExceededException;
import software.amazon.awssdk.services.route53recoverycluster.model.ThrottlingException;
import software.amazon.awssdk.services.route53recoverycluster.model.UpdateRoutingControlStateRequest;
import software.amazon.awssdk.services.route53recoverycluster.model.UpdateRoutingControlStateResponse;
import software.amazon.awssdk.services.route53recoverycluster.model.UpdateRoutingControlStatesRequest;
import software.amazon.awssdk.services.route53recoverycluster.model.UpdateRoutingControlStatesResponse;
import software.amazon.awssdk.services.route53recoverycluster.model.ValidationException;
import software.amazon.awssdk.services.route53recoverycluster.paginators.ListRoutingControlsIterable;

/**
 * Service client for accessing Route53 Recovery Cluster. This can be created using the static {@link #builder()}
 * method.
 *
 * <p>
 * Welcome to the Routing Control (Recovery Cluster) API Reference Guide for Amazon Route 53 Application Recovery
 * Controller.
 * </p>
 * <p>
 * With Route 53 ARC, you can use routing control with extreme reliability to recover applications by rerouting traffic
 * across Availability Zones or Amazon Web Services Regions. Routing controls are simple on/off switches hosted on a
 * highly available cluster in Route 53 ARC. A cluster provides a set of five redundant Regional endpoints against which
 * you can run API calls to get or update the state of routing controls. To implement failover, you set one routing
 * control On and another one Off, to reroute traffic from one Availability Zone or Amazon Web Services Region to
 * another.
 * </p>
 * <p>
 * <i>Be aware that you must specify a Regional endpoint for a cluster when you work with API cluster operations to get
 * or update routing control states in Route 53 ARC.</i> In addition, you must specify the US West (Oregon) Region for
 * Route 53 ARC API calls. For example, use the parameter <code>--region us-west-2</code> with AWS CLI commands. For
 * more information, see <a href="https://docs.aws.amazon.com/r53recovery/latest/dg/routing-control.update.api.html">
 * Get and update routing control states using the API</a> in the Amazon Route 53 Application Recovery Controller
 * Developer Guide.
 * </p>
 * <p>
 * This API guide includes information about the API operations for how to get and update routing control states in
 * Route 53 ARC. To work with routing control in Route 53 ARC, you must first create the required components (clusters,
 * control panels, and routing controls) using the recovery cluster configuration API.
 * </p>
 * <p>
 * For more information about working with routing control in Route 53 ARC, see the following:
 * </p>
 * <ul>
 * <li>
 * <p>
 * Create clusters, control panels, and routing controls by using API operations. For more information, see the <a
 * href="https://docs.aws.amazon.com/recovery-cluster/latest/api/">Recovery Control Configuration API Reference Guide
 * for Amazon Route 53 Application Recovery Controller</a>.
 * </p>
 * </li>
 * <li>
 * <p>
 * Learn about the components in recovery control, including clusters, routing controls, and control panels, and how to
 * work with Route 53 ARC in the Amazon Web Services console. For more information, see <a href=
 * "https://docs.aws.amazon.com/r53recovery/latest/dg/introduction-components.html#introduction-components-routing">
 * Recovery control components</a> in the Amazon Route 53 Application Recovery Controller Developer Guide.
 * </p>
 * </li>
 * <li>
 * <p>
 * Route 53 ARC also provides readiness checks that continually audit resources to help make sure that your applications
 * are scaled and ready to handle failover traffic. For more information about the related API operations, see the <a
 * href="https://docs.aws.amazon.com/recovery-readiness/latest/api/">Recovery Readiness API Reference Guide for Amazon
 * Route 53 Application Recovery Controller</a>.
 * </p>
 * </li>
 * <li>
 * <p>
 * For more information about creating resilient applications and preparing for recovery readiness with Route 53 ARC,
 * see the <a href="https://docs.aws.amazon.com/r53recovery/latest/dg/">Amazon Route 53 Application Recovery Controller
 * Developer Guide</a>.
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface Route53RecoveryClusterClient extends SdkClient {
    String SERVICE_NAME = "route53-recovery-cluster";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "route53-recovery-cluster";

    /**
     * <p>
     * Get the state for a routing control. A routing control is a simple on/off switch that you can use to route
     * traffic to cells. When a routing control state is On, traffic flows to a cell. When the state is Off, traffic
     * does not flow.
     * </p>
     * <p>
     * Before you can create a routing control, you must first create a cluster, and then host the control in a control
     * panel on the cluster. For more information, see <a
     * href="https://docs.aws.amazon.com/r53recovery/latest/dg/routing-control.create.html"> Create routing control
     * structures</a> in the Amazon Route 53 Application Recovery Controller Developer Guide. You access one of the
     * endpoints for the cluster to get or update the routing control state to redirect traffic for your application.
     * </p>
     * <p>
     * <i>You must specify Regional endpoints when you work with API cluster operations to get or update routing control
     * states in Route 53 ARC.</i>
     * </p>
     * <p>
     * To see a code example for getting a routing control state, including accessing Regional cluster endpoints in
     * sequence, see <a href="https://docs.aws.amazon.com/r53recovery/latest/dg/service_code_examples_actions.html">API
     * examples</a> in the Amazon Route 53 Application Recovery Controller Developer Guide.
     * </p>
     * <p>
     * Learn more about working with routing controls in the following topics in the Amazon Route 53 Application
     * Recovery Controller Developer Guide:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/r53recovery/latest/dg/routing-control.update.html"> Viewing and updating
     * routing control states</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/r53recovery/latest/dg/routing-control.html">Working with routing controls in
     * Route 53 ARC</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param getRoutingControlStateRequest
     * @return Result of the GetRoutingControlState operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws InternalServerException
     *         There was an unexpected error during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a routing control or control panel that was not found.
     * @throws ValidationException
     *         There was a validation error on the request.
     * @throws ThrottlingException
     *         The request was denied because of request throttling.
     * @throws EndpointTemporarilyUnavailableException
     *         The cluster endpoint isn't available. Try another cluster endpoint.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryClusterException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryClusterClient.GetRoutingControlState
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-cluster-2019-12-02/GetRoutingControlState"
     *      target="_top">AWS API Documentation</a>
     */
    default GetRoutingControlStateResponse getRoutingControlState(GetRoutingControlStateRequest getRoutingControlStateRequest)
            throws AccessDeniedException, InternalServerException, ResourceNotFoundException, ValidationException,
            ThrottlingException, EndpointTemporarilyUnavailableException, AwsServiceException, SdkClientException,
            Route53RecoveryClusterException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Get the state for a routing control. A routing control is a simple on/off switch that you can use to route
     * traffic to cells. When a routing control state is On, traffic flows to a cell. When the state is Off, traffic
     * does not flow.
     * </p>
     * <p>
     * Before you can create a routing control, you must first create a cluster, and then host the control in a control
     * panel on the cluster. For more information, see <a
     * href="https://docs.aws.amazon.com/r53recovery/latest/dg/routing-control.create.html"> Create routing control
     * structures</a> in the Amazon Route 53 Application Recovery Controller Developer Guide. You access one of the
     * endpoints for the cluster to get or update the routing control state to redirect traffic for your application.
     * </p>
     * <p>
     * <i>You must specify Regional endpoints when you work with API cluster operations to get or update routing control
     * states in Route 53 ARC.</i>
     * </p>
     * <p>
     * To see a code example for getting a routing control state, including accessing Regional cluster endpoints in
     * sequence, see <a href="https://docs.aws.amazon.com/r53recovery/latest/dg/service_code_examples_actions.html">API
     * examples</a> in the Amazon Route 53 Application Recovery Controller Developer Guide.
     * </p>
     * <p>
     * Learn more about working with routing controls in the following topics in the Amazon Route 53 Application
     * Recovery Controller Developer Guide:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/r53recovery/latest/dg/routing-control.update.html"> Viewing and updating
     * routing control states</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/r53recovery/latest/dg/routing-control.html">Working with routing controls in
     * Route 53 ARC</a>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetRoutingControlStateRequest.Builder} avoiding the
     * need to create one manually via {@link GetRoutingControlStateRequest#builder()}
     * </p>
     *
     * @param getRoutingControlStateRequest
     *        A {@link Consumer} that will call methods on {@link GetRoutingControlStateRequest.Builder} to create a
     *        request.
     * @return Result of the GetRoutingControlState operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws InternalServerException
     *         There was an unexpected error during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a routing control or control panel that was not found.
     * @throws ValidationException
     *         There was a validation error on the request.
     * @throws ThrottlingException
     *         The request was denied because of request throttling.
     * @throws EndpointTemporarilyUnavailableException
     *         The cluster endpoint isn't available. Try another cluster endpoint.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryClusterException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryClusterClient.GetRoutingControlState
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-cluster-2019-12-02/GetRoutingControlState"
     *      target="_top">AWS API Documentation</a>
     */
    default GetRoutingControlStateResponse getRoutingControlState(
            Consumer<GetRoutingControlStateRequest.Builder> getRoutingControlStateRequest) throws AccessDeniedException,
            InternalServerException, ResourceNotFoundException, ValidationException, ThrottlingException,
            EndpointTemporarilyUnavailableException, AwsServiceException, SdkClientException, Route53RecoveryClusterException {
        return getRoutingControlState(GetRoutingControlStateRequest.builder().applyMutation(getRoutingControlStateRequest)
                .build());
    }

    /**
     * <p>
     * List routing control names and Amazon Resource Names (ARNs), as well as the routing control state for each
     * routing control, along with the control panel name and control panel ARN for the routing controls. If you specify
     * a control panel ARN, this call lists the routing controls in the control panel. Otherwise, it lists all the
     * routing controls in the cluster.
     * </p>
     * <p>
     * A routing control is a simple on/off switch in Route 53 ARC that you can use to route traffic to cells. When a
     * routing control state is On, traffic flows to a cell. When the state is Off, traffic does not flow.
     * </p>
     * <p>
     * Before you can create a routing control, you must first create a cluster, and then host the control in a control
     * panel on the cluster. For more information, see <a
     * href="https://docs.aws.amazon.com/r53recovery/latest/dg/routing-control.create.html"> Create routing control
     * structures</a> in the Amazon Route 53 Application Recovery Controller Developer Guide. You access one of the
     * endpoints for the cluster to get or update the routing control state to redirect traffic for your application.
     * </p>
     * <p>
     * <i>You must specify Regional endpoints when you work with API cluster operations to use this API operation to
     * list routing controls in Route 53 ARC.</i>
     * </p>
     * <p>
     * Learn more about working with routing controls in the following topics in the Amazon Route 53 Application
     * Recovery Controller Developer Guide:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/r53recovery/latest/dg/routing-control.update.html"> Viewing and updating
     * routing control states</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/r53recovery/latest/dg/routing-control.html">Working with routing controls in
     * Route 53 ARC</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param listRoutingControlsRequest
     * @return Result of the ListRoutingControls operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws InternalServerException
     *         There was an unexpected error during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a routing control or control panel that was not found.
     * @throws ValidationException
     *         There was a validation error on the request.
     * @throws ThrottlingException
     *         The request was denied because of request throttling.
     * @throws EndpointTemporarilyUnavailableException
     *         The cluster endpoint isn't available. Try another cluster endpoint.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryClusterException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryClusterClient.ListRoutingControls
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-cluster-2019-12-02/ListRoutingControls"
     *      target="_top">AWS API Documentation</a>
     */
    default ListRoutingControlsResponse listRoutingControls(ListRoutingControlsRequest listRoutingControlsRequest)
            throws AccessDeniedException, InternalServerException, ResourceNotFoundException, ValidationException,
            ThrottlingException, EndpointTemporarilyUnavailableException, AwsServiceException, SdkClientException,
            Route53RecoveryClusterException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * List routing control names and Amazon Resource Names (ARNs), as well as the routing control state for each
     * routing control, along with the control panel name and control panel ARN for the routing controls. If you specify
     * a control panel ARN, this call lists the routing controls in the control panel. Otherwise, it lists all the
     * routing controls in the cluster.
     * </p>
     * <p>
     * A routing control is a simple on/off switch in Route 53 ARC that you can use to route traffic to cells. When a
     * routing control state is On, traffic flows to a cell. When the state is Off, traffic does not flow.
     * </p>
     * <p>
     * Before you can create a routing control, you must first create a cluster, and then host the control in a control
     * panel on the cluster. For more information, see <a
     * href="https://docs.aws.amazon.com/r53recovery/latest/dg/routing-control.create.html"> Create routing control
     * structures</a> in the Amazon Route 53 Application Recovery Controller Developer Guide. You access one of the
     * endpoints for the cluster to get or update the routing control state to redirect traffic for your application.
     * </p>
     * <p>
     * <i>You must specify Regional endpoints when you work with API cluster operations to use this API operation to
     * list routing controls in Route 53 ARC.</i>
     * </p>
     * <p>
     * Learn more about working with routing controls in the following topics in the Amazon Route 53 Application
     * Recovery Controller Developer Guide:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/r53recovery/latest/dg/routing-control.update.html"> Viewing and updating
     * routing control states</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/r53recovery/latest/dg/routing-control.html">Working with routing controls in
     * Route 53 ARC</a>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListRoutingControlsRequest.Builder} avoiding the
     * need to create one manually via {@link ListRoutingControlsRequest#builder()}
     * </p>
     *
     * @param listRoutingControlsRequest
     *        A {@link Consumer} that will call methods on {@link ListRoutingControlsRequest.Builder} to create a
     *        request.
     * @return Result of the ListRoutingControls operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws InternalServerException
     *         There was an unexpected error during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a routing control or control panel that was not found.
     * @throws ValidationException
     *         There was a validation error on the request.
     * @throws ThrottlingException
     *         The request was denied because of request throttling.
     * @throws EndpointTemporarilyUnavailableException
     *         The cluster endpoint isn't available. Try another cluster endpoint.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryClusterException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryClusterClient.ListRoutingControls
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-cluster-2019-12-02/ListRoutingControls"
     *      target="_top">AWS API Documentation</a>
     */
    default ListRoutingControlsResponse listRoutingControls(
            Consumer<ListRoutingControlsRequest.Builder> listRoutingControlsRequest) throws AccessDeniedException,
            InternalServerException, ResourceNotFoundException, ValidationException, ThrottlingException,
            EndpointTemporarilyUnavailableException, AwsServiceException, SdkClientException, Route53RecoveryClusterException {
        return listRoutingControls(ListRoutingControlsRequest.builder().applyMutation(listRoutingControlsRequest).build());
    }

    /**
     * <p>
     * List routing control names and Amazon Resource Names (ARNs), as well as the routing control state for each
     * routing control, along with the control panel name and control panel ARN for the routing controls. If you specify
     * a control panel ARN, this call lists the routing controls in the control panel. Otherwise, it lists all the
     * routing controls in the cluster.
     * </p>
     * <p>
     * A routing control is a simple on/off switch in Route 53 ARC that you can use to route traffic to cells. When a
     * routing control state is On, traffic flows to a cell. When the state is Off, traffic does not flow.
     * </p>
     * <p>
     * Before you can create a routing control, you must first create a cluster, and then host the control in a control
     * panel on the cluster. For more information, see <a
     * href="https://docs.aws.amazon.com/r53recovery/latest/dg/routing-control.create.html"> Create routing control
     * structures</a> in the Amazon Route 53 Application Recovery Controller Developer Guide. You access one of the
     * endpoints for the cluster to get or update the routing control state to redirect traffic for your application.
     * </p>
     * <p>
     * <i>You must specify Regional endpoints when you work with API cluster operations to use this API operation to
     * list routing controls in Route 53 ARC.</i>
     * </p>
     * <p>
     * Learn more about working with routing controls in the following topics in the Amazon Route 53 Application
     * Recovery Controller Developer Guide:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/r53recovery/latest/dg/routing-control.update.html"> Viewing and updating
     * routing control states</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/r53recovery/latest/dg/routing-control.html">Working with routing controls in
     * Route 53 ARC</a>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listRoutingControls(software.amazon.awssdk.services.route53recoverycluster.model.ListRoutingControlsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53recoverycluster.paginators.ListRoutingControlsIterable responses = client.listRoutingControlsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.route53recoverycluster.paginators.ListRoutingControlsIterable responses = client
     *             .listRoutingControlsPaginator(request);
     *     for (software.amazon.awssdk.services.route53recoverycluster.model.ListRoutingControlsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53recoverycluster.paginators.ListRoutingControlsIterable responses = client.listRoutingControlsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listRoutingControls(software.amazon.awssdk.services.route53recoverycluster.model.ListRoutingControlsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listRoutingControlsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws InternalServerException
     *         There was an unexpected error during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a routing control or control panel that was not found.
     * @throws ValidationException
     *         There was a validation error on the request.
     * @throws ThrottlingException
     *         The request was denied because of request throttling.
     * @throws EndpointTemporarilyUnavailableException
     *         The cluster endpoint isn't available. Try another cluster endpoint.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryClusterException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryClusterClient.ListRoutingControls
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-cluster-2019-12-02/ListRoutingControls"
     *      target="_top">AWS API Documentation</a>
     */
    default ListRoutingControlsIterable listRoutingControlsPaginator(ListRoutingControlsRequest listRoutingControlsRequest)
            throws AccessDeniedException, InternalServerException, ResourceNotFoundException, ValidationException,
            ThrottlingException, EndpointTemporarilyUnavailableException, AwsServiceException, SdkClientException,
            Route53RecoveryClusterException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * List routing control names and Amazon Resource Names (ARNs), as well as the routing control state for each
     * routing control, along with the control panel name and control panel ARN for the routing controls. If you specify
     * a control panel ARN, this call lists the routing controls in the control panel. Otherwise, it lists all the
     * routing controls in the cluster.
     * </p>
     * <p>
     * A routing control is a simple on/off switch in Route 53 ARC that you can use to route traffic to cells. When a
     * routing control state is On, traffic flows to a cell. When the state is Off, traffic does not flow.
     * </p>
     * <p>
     * Before you can create a routing control, you must first create a cluster, and then host the control in a control
     * panel on the cluster. For more information, see <a
     * href="https://docs.aws.amazon.com/r53recovery/latest/dg/routing-control.create.html"> Create routing control
     * structures</a> in the Amazon Route 53 Application Recovery Controller Developer Guide. You access one of the
     * endpoints for the cluster to get or update the routing control state to redirect traffic for your application.
     * </p>
     * <p>
     * <i>You must specify Regional endpoints when you work with API cluster operations to use this API operation to
     * list routing controls in Route 53 ARC.</i>
     * </p>
     * <p>
     * Learn more about working with routing controls in the following topics in the Amazon Route 53 Application
     * Recovery Controller Developer Guide:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/r53recovery/latest/dg/routing-control.update.html"> Viewing and updating
     * routing control states</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/r53recovery/latest/dg/routing-control.html">Working with routing controls in
     * Route 53 ARC</a>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listRoutingControls(software.amazon.awssdk.services.route53recoverycluster.model.ListRoutingControlsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53recoverycluster.paginators.ListRoutingControlsIterable responses = client.listRoutingControlsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.route53recoverycluster.paginators.ListRoutingControlsIterable responses = client
     *             .listRoutingControlsPaginator(request);
     *     for (software.amazon.awssdk.services.route53recoverycluster.model.ListRoutingControlsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53recoverycluster.paginators.ListRoutingControlsIterable responses = client.listRoutingControlsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listRoutingControls(software.amazon.awssdk.services.route53recoverycluster.model.ListRoutingControlsRequest)}
     * operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListRoutingControlsRequest.Builder} avoiding the
     * need to create one manually via {@link ListRoutingControlsRequest#builder()}
     * </p>
     *
     * @param listRoutingControlsRequest
     *        A {@link Consumer} that will call methods on {@link ListRoutingControlsRequest.Builder} to create a
     *        request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws InternalServerException
     *         There was an unexpected error during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a routing control or control panel that was not found.
     * @throws ValidationException
     *         There was a validation error on the request.
     * @throws ThrottlingException
     *         The request was denied because of request throttling.
     * @throws EndpointTemporarilyUnavailableException
     *         The cluster endpoint isn't available. Try another cluster endpoint.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryClusterException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryClusterClient.ListRoutingControls
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-cluster-2019-12-02/ListRoutingControls"
     *      target="_top">AWS API Documentation</a>
     */
    default ListRoutingControlsIterable listRoutingControlsPaginator(
            Consumer<ListRoutingControlsRequest.Builder> listRoutingControlsRequest) throws AccessDeniedException,
            InternalServerException, ResourceNotFoundException, ValidationException, ThrottlingException,
            EndpointTemporarilyUnavailableException, AwsServiceException, SdkClientException, Route53RecoveryClusterException {
        return listRoutingControlsPaginator(ListRoutingControlsRequest.builder().applyMutation(listRoutingControlsRequest)
                .build());
    }

    /**
     * <p>
     * Set the state of the routing control to reroute traffic. You can set the value to be On or Off. When the state is
     * On, traffic flows to a cell. When the state is Off, traffic does not flow.
     * </p>
     * <p>
     * With Route 53 ARC, you can add safety rules for routing controls, which are safeguards for routing control state
     * updates that help prevent unexpected outcomes, like fail open traffic routing. However, there are scenarios when
     * you might want to bypass the routing control safeguards that are enforced with safety rules that you've
     * configured. For example, you might want to fail over quickly for disaster recovery, and one or more safety rules
     * might be unexpectedly preventing you from updating a routing control state to reroute traffic. In a "break glass"
     * scenario like this, you can override one or more safety rules to change a routing control state and fail over
     * your application.
     * </p>
     * <p>
     * The <code>SafetyRulesToOverride</code> property enables you override one or more safety rules and update routing
     * control states. For more information, see <a
     * href="https://docs.aws.amazon.com/r53recovery/latest/dg/routing-control.override-safety-rule.html"> Override
     * safety rules to reroute traffic</a> in the Amazon Route 53 Application Recovery Controller Developer Guide.
     * </p>
     * <p>
     * <i>You must specify Regional endpoints when you work with API cluster operations to get or update routing control
     * states in Route 53 ARC.</i>
     * </p>
     * <p>
     * To see a code example for getting a routing control state, including accessing Regional cluster endpoints in
     * sequence, see <a href="https://docs.aws.amazon.com/r53recovery/latest/dg/service_code_examples_actions.html">API
     * examples</a> in the Amazon Route 53 Application Recovery Controller Developer Guide.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/r53recovery/latest/dg/routing-control.update.html"> Viewing and updating
     * routing control states</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/r53recovery/latest/dg/routing-control.html">Working with routing controls
     * overall</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param updateRoutingControlStateRequest
     * @return Result of the UpdateRoutingControlState operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws InternalServerException
     *         There was an unexpected error during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a routing control or control panel that was not found.
     * @throws ValidationException
     *         There was a validation error on the request.
     * @throws ThrottlingException
     *         The request was denied because of request throttling.
     * @throws EndpointTemporarilyUnavailableException
     *         The cluster endpoint isn't available. Try another cluster endpoint.
     * @throws ConflictException
     *         There was a conflict with this request. Try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryClusterException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryClusterClient.UpdateRoutingControlState
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-cluster-2019-12-02/UpdateRoutingControlState"
     *      target="_top">AWS API Documentation</a>
     */
    default UpdateRoutingControlStateResponse updateRoutingControlState(
            UpdateRoutingControlStateRequest updateRoutingControlStateRequest) throws AccessDeniedException,
            InternalServerException, ResourceNotFoundException, ValidationException, ThrottlingException,
            EndpointTemporarilyUnavailableException, ConflictException, AwsServiceException, SdkClientException,
            Route53RecoveryClusterException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Set the state of the routing control to reroute traffic. You can set the value to be On or Off. When the state is
     * On, traffic flows to a cell. When the state is Off, traffic does not flow.
     * </p>
     * <p>
     * With Route 53 ARC, you can add safety rules for routing controls, which are safeguards for routing control state
     * updates that help prevent unexpected outcomes, like fail open traffic routing. However, there are scenarios when
     * you might want to bypass the routing control safeguards that are enforced with safety rules that you've
     * configured. For example, you might want to fail over quickly for disaster recovery, and one or more safety rules
     * might be unexpectedly preventing you from updating a routing control state to reroute traffic. In a "break glass"
     * scenario like this, you can override one or more safety rules to change a routing control state and fail over
     * your application.
     * </p>
     * <p>
     * The <code>SafetyRulesToOverride</code> property enables you override one or more safety rules and update routing
     * control states. For more information, see <a
     * href="https://docs.aws.amazon.com/r53recovery/latest/dg/routing-control.override-safety-rule.html"> Override
     * safety rules to reroute traffic</a> in the Amazon Route 53 Application Recovery Controller Developer Guide.
     * </p>
     * <p>
     * <i>You must specify Regional endpoints when you work with API cluster operations to get or update routing control
     * states in Route 53 ARC.</i>
     * </p>
     * <p>
     * To see a code example for getting a routing control state, including accessing Regional cluster endpoints in
     * sequence, see <a href="https://docs.aws.amazon.com/r53recovery/latest/dg/service_code_examples_actions.html">API
     * examples</a> in the Amazon Route 53 Application Recovery Controller Developer Guide.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/r53recovery/latest/dg/routing-control.update.html"> Viewing and updating
     * routing control states</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/r53recovery/latest/dg/routing-control.html">Working with routing controls
     * overall</a>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateRoutingControlStateRequest.Builder} avoiding
     * the need to create one manually via {@link UpdateRoutingControlStateRequest#builder()}
     * </p>
     *
     * @param updateRoutingControlStateRequest
     *        A {@link Consumer} that will call methods on {@link UpdateRoutingControlStateRequest.Builder} to create a
     *        request.
     * @return Result of the UpdateRoutingControlState operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws InternalServerException
     *         There was an unexpected error during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a routing control or control panel that was not found.
     * @throws ValidationException
     *         There was a validation error on the request.
     * @throws ThrottlingException
     *         The request was denied because of request throttling.
     * @throws EndpointTemporarilyUnavailableException
     *         The cluster endpoint isn't available. Try another cluster endpoint.
     * @throws ConflictException
     *         There was a conflict with this request. Try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryClusterException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryClusterClient.UpdateRoutingControlState
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-cluster-2019-12-02/UpdateRoutingControlState"
     *      target="_top">AWS API Documentation</a>
     */
    default UpdateRoutingControlStateResponse updateRoutingControlState(
            Consumer<UpdateRoutingControlStateRequest.Builder> updateRoutingControlStateRequest) throws AccessDeniedException,
            InternalServerException, ResourceNotFoundException, ValidationException, ThrottlingException,
            EndpointTemporarilyUnavailableException, ConflictException, AwsServiceException, SdkClientException,
            Route53RecoveryClusterException {
        return updateRoutingControlState(UpdateRoutingControlStateRequest.builder()
                .applyMutation(updateRoutingControlStateRequest).build());
    }

    /**
     * <p>
     * Set multiple routing control states. You can set the value for each state to be On or Off. When the state is On,
     * traffic flows to a cell. When it's Off, traffic does not flow.
     * </p>
     * <p>
     * With Route 53 ARC, you can add safety rules for routing controls, which are safeguards for routing control state
     * updates that help prevent unexpected outcomes, like fail open traffic routing. However, there are scenarios when
     * you might want to bypass the routing control safeguards that are enforced with safety rules that you've
     * configured. For example, you might want to fail over quickly for disaster recovery, and one or more safety rules
     * might be unexpectedly preventing you from updating a routing control state to reroute traffic. In a "break glass"
     * scenario like this, you can override one or more safety rules to change a routing control state and fail over
     * your application.
     * </p>
     * <p>
     * The <code>SafetyRulesToOverride</code> property enables you override one or more safety rules and update routing
     * control states. For more information, see <a
     * href="https://docs.aws.amazon.com/r53recovery/latest/dg/routing-control.override-safety-rule.html"> Override
     * safety rules to reroute traffic</a> in the Amazon Route 53 Application Recovery Controller Developer Guide.
     * </p>
     * <p>
     * <i>You must specify Regional endpoints when you work with API cluster operations to get or update routing control
     * states in Route 53 ARC.</i>
     * </p>
     * <p>
     * To see a code example for getting a routing control state, including accessing Regional cluster endpoints in
     * sequence, see <a href="https://docs.aws.amazon.com/r53recovery/latest/dg/service_code_examples_actions.html">API
     * examples</a> in the Amazon Route 53 Application Recovery Controller Developer Guide.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/r53recovery/latest/dg/routing-control.update.html"> Viewing and updating
     * routing control states</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/r53recovery/latest/dg/routing-control.html">Working with routing controls
     * overall</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param updateRoutingControlStatesRequest
     * @return Result of the UpdateRoutingControlStates operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws InternalServerException
     *         There was an unexpected error during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a routing control or control panel that was not found.
     * @throws ValidationException
     *         There was a validation error on the request.
     * @throws ThrottlingException
     *         The request was denied because of request throttling.
     * @throws EndpointTemporarilyUnavailableException
     *         The cluster endpoint isn't available. Try another cluster endpoint.
     * @throws ConflictException
     *         There was a conflict with this request. Try again.
     * @throws ServiceLimitExceededException
     *         The request can't update that many routing control states at the same time. Try again with fewer routing
     *         control states.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryClusterException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryClusterClient.UpdateRoutingControlStates
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-cluster-2019-12-02/UpdateRoutingControlStates"
     *      target="_top">AWS API Documentation</a>
     */
    default UpdateRoutingControlStatesResponse updateRoutingControlStates(
            UpdateRoutingControlStatesRequest updateRoutingControlStatesRequest) throws AccessDeniedException,
            InternalServerException, ResourceNotFoundException, ValidationException, ThrottlingException,
            EndpointTemporarilyUnavailableException, ConflictException, ServiceLimitExceededException, AwsServiceException,
            SdkClientException, Route53RecoveryClusterException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Set multiple routing control states. You can set the value for each state to be On or Off. When the state is On,
     * traffic flows to a cell. When it's Off, traffic does not flow.
     * </p>
     * <p>
     * With Route 53 ARC, you can add safety rules for routing controls, which are safeguards for routing control state
     * updates that help prevent unexpected outcomes, like fail open traffic routing. However, there are scenarios when
     * you might want to bypass the routing control safeguards that are enforced with safety rules that you've
     * configured. For example, you might want to fail over quickly for disaster recovery, and one or more safety rules
     * might be unexpectedly preventing you from updating a routing control state to reroute traffic. In a "break glass"
     * scenario like this, you can override one or more safety rules to change a routing control state and fail over
     * your application.
     * </p>
     * <p>
     * The <code>SafetyRulesToOverride</code> property enables you override one or more safety rules and update routing
     * control states. For more information, see <a
     * href="https://docs.aws.amazon.com/r53recovery/latest/dg/routing-control.override-safety-rule.html"> Override
     * safety rules to reroute traffic</a> in the Amazon Route 53 Application Recovery Controller Developer Guide.
     * </p>
     * <p>
     * <i>You must specify Regional endpoints when you work with API cluster operations to get or update routing control
     * states in Route 53 ARC.</i>
     * </p>
     * <p>
     * To see a code example for getting a routing control state, including accessing Regional cluster endpoints in
     * sequence, see <a href="https://docs.aws.amazon.com/r53recovery/latest/dg/service_code_examples_actions.html">API
     * examples</a> in the Amazon Route 53 Application Recovery Controller Developer Guide.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/r53recovery/latest/dg/routing-control.update.html"> Viewing and updating
     * routing control states</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/r53recovery/latest/dg/routing-control.html">Working with routing controls
     * overall</a>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateRoutingControlStatesRequest.Builder} avoiding
     * the need to create one manually via {@link UpdateRoutingControlStatesRequest#builder()}
     * </p>
     *
     * @param updateRoutingControlStatesRequest
     *        A {@link Consumer} that will call methods on {@link UpdateRoutingControlStatesRequest.Builder} to create a
     *        request.
     * @return Result of the UpdateRoutingControlStates operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws InternalServerException
     *         There was an unexpected error during processing of the request.
     * @throws ResourceNotFoundException
     *         The request references a routing control or control panel that was not found.
     * @throws ValidationException
     *         There was a validation error on the request.
     * @throws ThrottlingException
     *         The request was denied because of request throttling.
     * @throws EndpointTemporarilyUnavailableException
     *         The cluster endpoint isn't available. Try another cluster endpoint.
     * @throws ConflictException
     *         There was a conflict with this request. Try again.
     * @throws ServiceLimitExceededException
     *         The request can't update that many routing control states at the same time. Try again with fewer routing
     *         control states.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryClusterException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryClusterClient.UpdateRoutingControlStates
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-cluster-2019-12-02/UpdateRoutingControlStates"
     *      target="_top">AWS API Documentation</a>
     */
    default UpdateRoutingControlStatesResponse updateRoutingControlStates(
            Consumer<UpdateRoutingControlStatesRequest.Builder> updateRoutingControlStatesRequest) throws AccessDeniedException,
            InternalServerException, ResourceNotFoundException, ValidationException, ThrottlingException,
            EndpointTemporarilyUnavailableException, ConflictException, ServiceLimitExceededException, AwsServiceException,
            SdkClientException, Route53RecoveryClusterException {
        return updateRoutingControlStates(UpdateRoutingControlStatesRequest.builder()
                .applyMutation(updateRoutingControlStatesRequest).build());
    }

    /**
     * Create a {@link Route53RecoveryClusterClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static Route53RecoveryClusterClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link Route53RecoveryClusterClient}.
     */
    static Route53RecoveryClusterClientBuilder builder() {
        return new DefaultRoute53RecoveryClusterClientBuilder();
    }

    static ServiceMetadata serviceMetadata() {
        return ServiceMetadata.of(SERVICE_METADATA_ID);
    }
}
