/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.route53recoverycontrolconfig.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An assertion rule enforces that, when you change a routing control state, that the criteria that you set in the rule
 * configuration is met. Otherwise, the change to the routing control is not accepted. For example, the criteria might
 * be that at least one routing control state is On after the transation so that traffic continues to flow to at least
 * one cell for the application. This ensures that you avoid a fail-open scenario.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AssertionRule implements SdkPojo, Serializable, ToCopyableBuilder<AssertionRule.Builder, AssertionRule> {
    private static final SdkField<List<String>> ASSERTED_CONTROLS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("AssertedControls")
            .getter(getter(AssertionRule::assertedControls))
            .setter(setter(Builder::assertedControls))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AssertedControls").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> CONTROL_PANEL_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ControlPanelArn").getter(getter(AssertionRule::controlPanelArn))
            .setter(setter(Builder::controlPanelArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ControlPanelArn").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(AssertionRule::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<RuleConfig> RULE_CONFIG_FIELD = SdkField.<RuleConfig> builder(MarshallingType.SDK_POJO)
            .memberName("RuleConfig").getter(getter(AssertionRule::ruleConfig)).setter(setter(Builder::ruleConfig))
            .constructor(RuleConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RuleConfig").build()).build();

    private static final SdkField<String> SAFETY_RULE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SafetyRuleArn").getter(getter(AssertionRule::safetyRuleArn)).setter(setter(Builder::safetyRuleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SafetyRuleArn").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(AssertionRule::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<Integer> WAIT_PERIOD_MS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("WaitPeriodMs").getter(getter(AssertionRule::waitPeriodMs)).setter(setter(Builder::waitPeriodMs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WaitPeriodMs").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ASSERTED_CONTROLS_FIELD,
            CONTROL_PANEL_ARN_FIELD, NAME_FIELD, RULE_CONFIG_FIELD, SAFETY_RULE_ARN_FIELD, STATUS_FIELD, WAIT_PERIOD_MS_FIELD));

    private static final long serialVersionUID = 1L;

    private final List<String> assertedControls;

    private final String controlPanelArn;

    private final String name;

    private final RuleConfig ruleConfig;

    private final String safetyRuleArn;

    private final String status;

    private final Integer waitPeriodMs;

    private AssertionRule(BuilderImpl builder) {
        this.assertedControls = builder.assertedControls;
        this.controlPanelArn = builder.controlPanelArn;
        this.name = builder.name;
        this.ruleConfig = builder.ruleConfig;
        this.safetyRuleArn = builder.safetyRuleArn;
        this.status = builder.status;
        this.waitPeriodMs = builder.waitPeriodMs;
    }

    /**
     * For responses, this returns true if the service returned a value for the AssertedControls property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAssertedControls() {
        return assertedControls != null && !(assertedControls instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The routing controls that are part of transactions that are evaluated to determine if a request to change a
     * routing control state is allowed. For example, you might include three routing controls, one for each of three
     * Amazon Web Services Regions.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAssertedControls} method.
     * </p>
     * 
     * @return The routing controls that are part of transactions that are evaluated to determine if a request to change
     *         a routing control state is allowed. For example, you might include three routing controls, one for each
     *         of three Amazon Web Services Regions.
     */
    public final List<String> assertedControls() {
        return assertedControls;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the control panel.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the control panel.
     */
    public final String controlPanelArn() {
        return controlPanelArn;
    }

    /**
     * <p>
     * Name of the assertion rule. You can use any non-white space character in the name.
     * </p>
     * 
     * @return Name of the assertion rule. You can use any non-white space character in the name.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The criteria that you set for specific assertion routing controls (AssertedControls) that designate how many
     * routing control states must be ON as the result of a transaction. For example, if you have three assertion
     * routing controls, you might specify atleast 2 for your rule configuration. This means that at least two assertion
     * routing control states must be ON, so that at least two Amazon Web Services Regions have traffic flowing to them.
     * </p>
     * 
     * @return The criteria that you set for specific assertion routing controls (AssertedControls) that designate how
     *         many routing control states must be ON as the result of a transaction. For example, if you have three
     *         assertion routing controls, you might specify atleast 2 for your rule configuration. This means that at
     *         least two assertion routing control states must be ON, so that at least two Amazon Web Services Regions
     *         have traffic flowing to them.
     */
    public final RuleConfig ruleConfig() {
        return ruleConfig;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the assertion rule.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the assertion rule.
     */
    public final String safetyRuleArn() {
        return safetyRuleArn;
    }

    /**
     * <p>
     * The deployment status of an assertion rule. Status can be one of the following: PENDING, DEPLOYED,
     * PENDING_DELETION.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link Status#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The deployment status of an assertion rule. Status can be one of the following: PENDING, DEPLOYED,
     *         PENDING_DELETION.
     * @see Status
     */
    public final Status status() {
        return Status.fromValue(status);
    }

    /**
     * <p>
     * The deployment status of an assertion rule. Status can be one of the following: PENDING, DEPLOYED,
     * PENDING_DELETION.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link Status#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The deployment status of an assertion rule. Status can be one of the following: PENDING, DEPLOYED,
     *         PENDING_DELETION.
     * @see Status
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * An evaluation period, in milliseconds (ms), during which any request against the target routing controls will
     * fail. This helps prevent "flapping" of state. The wait period is 5000 ms by default, but you can choose a custom
     * value.
     * </p>
     * 
     * @return An evaluation period, in milliseconds (ms), during which any request against the target routing controls
     *         will fail. This helps prevent "flapping" of state. The wait period is 5000 ms by default, but you can
     *         choose a custom value.
     */
    public final Integer waitPeriodMs() {
        return waitPeriodMs;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasAssertedControls() ? assertedControls() : null);
        hashCode = 31 * hashCode + Objects.hashCode(controlPanelArn());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(ruleConfig());
        hashCode = 31 * hashCode + Objects.hashCode(safetyRuleArn());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(waitPeriodMs());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AssertionRule)) {
            return false;
        }
        AssertionRule other = (AssertionRule) obj;
        return hasAssertedControls() == other.hasAssertedControls()
                && Objects.equals(assertedControls(), other.assertedControls())
                && Objects.equals(controlPanelArn(), other.controlPanelArn()) && Objects.equals(name(), other.name())
                && Objects.equals(ruleConfig(), other.ruleConfig()) && Objects.equals(safetyRuleArn(), other.safetyRuleArn())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(waitPeriodMs(), other.waitPeriodMs());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AssertionRule").add("AssertedControls", hasAssertedControls() ? assertedControls() : null)
                .add("ControlPanelArn", controlPanelArn()).add("Name", name()).add("RuleConfig", ruleConfig())
                .add("SafetyRuleArn", safetyRuleArn()).add("Status", statusAsString()).add("WaitPeriodMs", waitPeriodMs())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AssertedControls":
            return Optional.ofNullable(clazz.cast(assertedControls()));
        case "ControlPanelArn":
            return Optional.ofNullable(clazz.cast(controlPanelArn()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "RuleConfig":
            return Optional.ofNullable(clazz.cast(ruleConfig()));
        case "SafetyRuleArn":
            return Optional.ofNullable(clazz.cast(safetyRuleArn()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "WaitPeriodMs":
            return Optional.ofNullable(clazz.cast(waitPeriodMs()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AssertionRule, T> g) {
        return obj -> g.apply((AssertionRule) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AssertionRule> {
        /**
         * <p>
         * The routing controls that are part of transactions that are evaluated to determine if a request to change a
         * routing control state is allowed. For example, you might include three routing controls, one for each of
         * three Amazon Web Services Regions.
         * </p>
         * 
         * @param assertedControls
         *        The routing controls that are part of transactions that are evaluated to determine if a request to
         *        change a routing control state is allowed. For example, you might include three routing controls, one
         *        for each of three Amazon Web Services Regions.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder assertedControls(Collection<String> assertedControls);

        /**
         * <p>
         * The routing controls that are part of transactions that are evaluated to determine if a request to change a
         * routing control state is allowed. For example, you might include three routing controls, one for each of
         * three Amazon Web Services Regions.
         * </p>
         * 
         * @param assertedControls
         *        The routing controls that are part of transactions that are evaluated to determine if a request to
         *        change a routing control state is allowed. For example, you might include three routing controls, one
         *        for each of three Amazon Web Services Regions.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder assertedControls(String... assertedControls);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the control panel.
         * </p>
         * 
         * @param controlPanelArn
         *        The Amazon Resource Name (ARN) of the control panel.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder controlPanelArn(String controlPanelArn);

        /**
         * <p>
         * Name of the assertion rule. You can use any non-white space character in the name.
         * </p>
         * 
         * @param name
         *        Name of the assertion rule. You can use any non-white space character in the name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The criteria that you set for specific assertion routing controls (AssertedControls) that designate how many
         * routing control states must be ON as the result of a transaction. For example, if you have three assertion
         * routing controls, you might specify atleast 2 for your rule configuration. This means that at least two
         * assertion routing control states must be ON, so that at least two Amazon Web Services Regions have traffic
         * flowing to them.
         * </p>
         * 
         * @param ruleConfig
         *        The criteria that you set for specific assertion routing controls (AssertedControls) that designate
         *        how many routing control states must be ON as the result of a transaction. For example, if you have
         *        three assertion routing controls, you might specify atleast 2 for your rule configuration. This means
         *        that at least two assertion routing control states must be ON, so that at least two Amazon Web
         *        Services Regions have traffic flowing to them.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ruleConfig(RuleConfig ruleConfig);

        /**
         * <p>
         * The criteria that you set for specific assertion routing controls (AssertedControls) that designate how many
         * routing control states must be ON as the result of a transaction. For example, if you have three assertion
         * routing controls, you might specify atleast 2 for your rule configuration. This means that at least two
         * assertion routing control states must be ON, so that at least two Amazon Web Services Regions have traffic
         * flowing to them.
         * </p>
         * This is a convenience method that creates an instance of the {@link RuleConfig.Builder} avoiding the need to
         * create one manually via {@link RuleConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link RuleConfig.Builder#build()} is called immediately and its result
         * is passed to {@link #ruleConfig(RuleConfig)}.
         * 
         * @param ruleConfig
         *        a consumer that will call methods on {@link RuleConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #ruleConfig(RuleConfig)
         */
        default Builder ruleConfig(Consumer<RuleConfig.Builder> ruleConfig) {
            return ruleConfig(RuleConfig.builder().applyMutation(ruleConfig).build());
        }

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the assertion rule.
         * </p>
         * 
         * @param safetyRuleArn
         *        The Amazon Resource Name (ARN) of the assertion rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder safetyRuleArn(String safetyRuleArn);

        /**
         * <p>
         * The deployment status of an assertion rule. Status can be one of the following: PENDING, DEPLOYED,
         * PENDING_DELETION.
         * </p>
         * 
         * @param status
         *        The deployment status of an assertion rule. Status can be one of the following: PENDING, DEPLOYED,
         *        PENDING_DELETION.
         * @see Status
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Status
         */
        Builder status(String status);

        /**
         * <p>
         * The deployment status of an assertion rule. Status can be one of the following: PENDING, DEPLOYED,
         * PENDING_DELETION.
         * </p>
         * 
         * @param status
         *        The deployment status of an assertion rule. Status can be one of the following: PENDING, DEPLOYED,
         *        PENDING_DELETION.
         * @see Status
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Status
         */
        Builder status(Status status);

        /**
         * <p>
         * An evaluation period, in milliseconds (ms), during which any request against the target routing controls will
         * fail. This helps prevent "flapping" of state. The wait period is 5000 ms by default, but you can choose a
         * custom value.
         * </p>
         * 
         * @param waitPeriodMs
         *        An evaluation period, in milliseconds (ms), during which any request against the target routing
         *        controls will fail. This helps prevent "flapping" of state. The wait period is 5000 ms by default, but
         *        you can choose a custom value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder waitPeriodMs(Integer waitPeriodMs);
    }

    static final class BuilderImpl implements Builder {
        private List<String> assertedControls = DefaultSdkAutoConstructList.getInstance();

        private String controlPanelArn;

        private String name;

        private RuleConfig ruleConfig;

        private String safetyRuleArn;

        private String status;

        private Integer waitPeriodMs;

        private BuilderImpl() {
        }

        private BuilderImpl(AssertionRule model) {
            assertedControls(model.assertedControls);
            controlPanelArn(model.controlPanelArn);
            name(model.name);
            ruleConfig(model.ruleConfig);
            safetyRuleArn(model.safetyRuleArn);
            status(model.status);
            waitPeriodMs(model.waitPeriodMs);
        }

        public final Collection<String> getAssertedControls() {
            if (assertedControls instanceof SdkAutoConstructList) {
                return null;
            }
            return assertedControls;
        }

        public final void setAssertedControls(Collection<String> assertedControls) {
            this.assertedControls = ___listOf__stringMin1Max256PatternAZaZ09Copier.copy(assertedControls);
        }

        @Override
        public final Builder assertedControls(Collection<String> assertedControls) {
            this.assertedControls = ___listOf__stringMin1Max256PatternAZaZ09Copier.copy(assertedControls);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder assertedControls(String... assertedControls) {
            assertedControls(Arrays.asList(assertedControls));
            return this;
        }

        public final String getControlPanelArn() {
            return controlPanelArn;
        }

        public final void setControlPanelArn(String controlPanelArn) {
            this.controlPanelArn = controlPanelArn;
        }

        @Override
        public final Builder controlPanelArn(String controlPanelArn) {
            this.controlPanelArn = controlPanelArn;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final RuleConfig.Builder getRuleConfig() {
            return ruleConfig != null ? ruleConfig.toBuilder() : null;
        }

        public final void setRuleConfig(RuleConfig.BuilderImpl ruleConfig) {
            this.ruleConfig = ruleConfig != null ? ruleConfig.build() : null;
        }

        @Override
        public final Builder ruleConfig(RuleConfig ruleConfig) {
            this.ruleConfig = ruleConfig;
            return this;
        }

        public final String getSafetyRuleArn() {
            return safetyRuleArn;
        }

        public final void setSafetyRuleArn(String safetyRuleArn) {
            this.safetyRuleArn = safetyRuleArn;
        }

        @Override
        public final Builder safetyRuleArn(String safetyRuleArn) {
            this.safetyRuleArn = safetyRuleArn;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(Status status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final Integer getWaitPeriodMs() {
            return waitPeriodMs;
        }

        public final void setWaitPeriodMs(Integer waitPeriodMs) {
            this.waitPeriodMs = waitPeriodMs;
        }

        @Override
        public final Builder waitPeriodMs(Integer waitPeriodMs) {
            this.waitPeriodMs = waitPeriodMs;
            return this;
        }

        @Override
        public AssertionRule build() {
            return new AssertionRule(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
