/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.route53recoverycontrolconfig;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.AccessDeniedException;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.ConflictException;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.CreateClusterRequest;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.CreateClusterResponse;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.CreateControlPanelRequest;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.CreateControlPanelResponse;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.CreateRoutingControlRequest;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.CreateRoutingControlResponse;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.CreateSafetyRuleRequest;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.CreateSafetyRuleResponse;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.DeleteClusterRequest;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.DeleteClusterResponse;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.DeleteControlPanelRequest;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.DeleteControlPanelResponse;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.DeleteRoutingControlRequest;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.DeleteRoutingControlResponse;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.DeleteSafetyRuleRequest;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.DeleteSafetyRuleResponse;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.DescribeClusterRequest;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.DescribeClusterResponse;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.DescribeControlPanelRequest;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.DescribeControlPanelResponse;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.DescribeRoutingControlRequest;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.DescribeRoutingControlResponse;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.DescribeSafetyRuleRequest;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.DescribeSafetyRuleResponse;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.InternalServerException;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.ListAssociatedRoute53HealthChecksRequest;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.ListAssociatedRoute53HealthChecksResponse;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.ListClustersRequest;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.ListClustersResponse;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.ListControlPanelsRequest;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.ListControlPanelsResponse;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.ListRoutingControlsRequest;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.ListRoutingControlsResponse;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.ListSafetyRulesRequest;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.ListSafetyRulesResponse;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.ResourceNotFoundException;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.Route53RecoveryControlConfigException;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.Route53RecoveryControlConfigRequest;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.ThrottlingException;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.UpdateControlPanelRequest;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.UpdateControlPanelResponse;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.UpdateRoutingControlRequest;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.UpdateRoutingControlResponse;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.UpdateSafetyRuleRequest;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.UpdateSafetyRuleResponse;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.ValidationException;
import software.amazon.awssdk.services.route53recoverycontrolconfig.paginators.ListAssociatedRoute53HealthChecksIterable;
import software.amazon.awssdk.services.route53recoverycontrolconfig.paginators.ListClustersIterable;
import software.amazon.awssdk.services.route53recoverycontrolconfig.paginators.ListControlPanelsIterable;
import software.amazon.awssdk.services.route53recoverycontrolconfig.paginators.ListRoutingControlsIterable;
import software.amazon.awssdk.services.route53recoverycontrolconfig.paginators.ListSafetyRulesIterable;
import software.amazon.awssdk.services.route53recoverycontrolconfig.transform.CreateClusterRequestMarshaller;
import software.amazon.awssdk.services.route53recoverycontrolconfig.transform.CreateControlPanelRequestMarshaller;
import software.amazon.awssdk.services.route53recoverycontrolconfig.transform.CreateRoutingControlRequestMarshaller;
import software.amazon.awssdk.services.route53recoverycontrolconfig.transform.CreateSafetyRuleRequestMarshaller;
import software.amazon.awssdk.services.route53recoverycontrolconfig.transform.DeleteClusterRequestMarshaller;
import software.amazon.awssdk.services.route53recoverycontrolconfig.transform.DeleteControlPanelRequestMarshaller;
import software.amazon.awssdk.services.route53recoverycontrolconfig.transform.DeleteRoutingControlRequestMarshaller;
import software.amazon.awssdk.services.route53recoverycontrolconfig.transform.DeleteSafetyRuleRequestMarshaller;
import software.amazon.awssdk.services.route53recoverycontrolconfig.transform.DescribeClusterRequestMarshaller;
import software.amazon.awssdk.services.route53recoverycontrolconfig.transform.DescribeControlPanelRequestMarshaller;
import software.amazon.awssdk.services.route53recoverycontrolconfig.transform.DescribeRoutingControlRequestMarshaller;
import software.amazon.awssdk.services.route53recoverycontrolconfig.transform.DescribeSafetyRuleRequestMarshaller;
import software.amazon.awssdk.services.route53recoverycontrolconfig.transform.ListAssociatedRoute53HealthChecksRequestMarshaller;
import software.amazon.awssdk.services.route53recoverycontrolconfig.transform.ListClustersRequestMarshaller;
import software.amazon.awssdk.services.route53recoverycontrolconfig.transform.ListControlPanelsRequestMarshaller;
import software.amazon.awssdk.services.route53recoverycontrolconfig.transform.ListRoutingControlsRequestMarshaller;
import software.amazon.awssdk.services.route53recoverycontrolconfig.transform.ListSafetyRulesRequestMarshaller;
import software.amazon.awssdk.services.route53recoverycontrolconfig.transform.UpdateControlPanelRequestMarshaller;
import software.amazon.awssdk.services.route53recoverycontrolconfig.transform.UpdateRoutingControlRequestMarshaller;
import software.amazon.awssdk.services.route53recoverycontrolconfig.transform.UpdateSafetyRuleRequestMarshaller;
import software.amazon.awssdk.services.route53recoverycontrolconfig.waiters.Route53RecoveryControlConfigWaiter;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link Route53RecoveryControlConfigClient}.
 *
 * @see Route53RecoveryControlConfigClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultRoute53RecoveryControlConfigClient implements Route53RecoveryControlConfigClient {
    private static final Logger log = Logger.loggerFor(DefaultRoute53RecoveryControlConfigClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultRoute53RecoveryControlConfigClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Create a new cluster. A cluster is a set of redundant Regional endpoints against which you can run API calls to
     * update or get the state of one or more routing controls. Each cluster has a name, status, Amazon Resource Name
     * (ARN), and an array of the five cluster endpoints (one for each supported Amazon Web Services Region) that you
     * can use with API calls to the Amazon Route 53 Application Recovery Controller cluster data plane.
     * </p>
     *
     * @param createClusterRequest
     *        Creates a cluster.
     * @return Result of the CreateCluster operation returned by the service.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string and input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - InternalServiceError. Temporary service error. Retry the request.
     * @throws ServiceQuotaExceededException
     *         402 response
     * @throws AccessDeniedException
     *         403 response - AccessDeniedException. You do not hace sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         404 response - MalformedQueryString. The query string contains a syntax error or resource not found.
     * @throws ThrottlingException
     *         429 response - LimitExceededException or TooManyRequestsException.
     * @throws ConflictException
     *         409 response - ConflictException. You might be using a predefined variable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryControlConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryControlConfigClient.CreateCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-control-config-2020-11-02/CreateCluster"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateClusterResponse createCluster(CreateClusterRequest createClusterRequest) throws ValidationException,
            InternalServerException, ServiceQuotaExceededException, AccessDeniedException, ResourceNotFoundException,
            ThrottlingException, ConflictException, AwsServiceException, SdkClientException,
            Route53RecoveryControlConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateClusterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateClusterResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Control Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateCluster");

            return clientHandler.execute(new ClientExecutionParams<CreateClusterRequest, CreateClusterResponse>()
                    .withOperationName("CreateCluster").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createClusterRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateClusterRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new control panel. A control panel represents a group of routing controls that can be changed together
     * in a single transaction. You can use a control panel to centrally view the operational status of applications
     * across your organization, and trigger multi-app failovers in a single transaction, for example, to fail over an
     * Availability Zone or AWS Region.
     * </p>
     *
     * @param createControlPanelRequest
     *        The details of the control panel that you're creating.
     * @return Result of the CreateControlPanel operation returned by the service.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string and input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - InternalServiceError. Temporary service error. Retry the request.
     * @throws ServiceQuotaExceededException
     *         402 response
     * @throws AccessDeniedException
     *         403 response - AccessDeniedException. You do not hace sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         404 response - MalformedQueryString. The query string contains a syntax error or resource not found.
     * @throws ThrottlingException
     *         429 response - LimitExceededException or TooManyRequestsException.
     * @throws ConflictException
     *         409 response - ConflictException. You might be using a predefined variable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryControlConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryControlConfigClient.CreateControlPanel
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-control-config-2020-11-02/CreateControlPanel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateControlPanelResponse createControlPanel(CreateControlPanelRequest createControlPanelRequest)
            throws ValidationException, InternalServerException, ServiceQuotaExceededException, AccessDeniedException,
            ResourceNotFoundException, ThrottlingException, ConflictException, AwsServiceException, SdkClientException,
            Route53RecoveryControlConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateControlPanelResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateControlPanelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createControlPanelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Control Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateControlPanel");

            return clientHandler.execute(new ClientExecutionParams<CreateControlPanelRequest, CreateControlPanelResponse>()
                    .withOperationName("CreateControlPanel").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createControlPanelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateControlPanelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new routing control.
     * </p>
     * <p>
     * A routing control has one of two states: ON and OFF. You can map the routing control state to the state of an
     * Amazon Route 53 health check, which can be used to control traffic routing.
     * </p>
     * <p>
     * To get or update the routing control state, see the Recovery Cluster (data plane) API actions for Amazon Route 53
     * Application Recovery Controller.
     * </p>
     *
     * @param createRoutingControlRequest
     *        The details of the routing control that you're creating.
     * @return Result of the CreateRoutingControl operation returned by the service.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string and input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - InternalServiceError. Temporary service error. Retry the request.
     * @throws ServiceQuotaExceededException
     *         402 response
     * @throws AccessDeniedException
     *         403 response - AccessDeniedException. You do not hace sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         404 response - MalformedQueryString. The query string contains a syntax error or resource not found.
     * @throws ThrottlingException
     *         429 response - LimitExceededException or TooManyRequestsException.
     * @throws ConflictException
     *         409 response - ConflictException. You might be using a predefined variable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryControlConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryControlConfigClient.CreateRoutingControl
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-control-config-2020-11-02/CreateRoutingControl"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateRoutingControlResponse createRoutingControl(CreateRoutingControlRequest createRoutingControlRequest)
            throws ValidationException, InternalServerException, ServiceQuotaExceededException, AccessDeniedException,
            ResourceNotFoundException, ThrottlingException, ConflictException, AwsServiceException, SdkClientException,
            Route53RecoveryControlConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateRoutingControlResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateRoutingControlResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createRoutingControlRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Control Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateRoutingControl");

            return clientHandler.execute(new ClientExecutionParams<CreateRoutingControlRequest, CreateRoutingControlResponse>()
                    .withOperationName("CreateRoutingControl").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createRoutingControlRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateRoutingControlRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a safety rule in a control panel. Safety rules let you add safeguards around enabling and disabling
     * routing controls, to help prevent unexpected outcomes.
     * </p>
     * <p>
     * There are two types of safety rules: assertion rules and gating rules.
     * </p>
     * <p>
     * Assertion rule: An assertion rule enforces that, when a routing control state is changed, the criteria set by the
     * rule configuration is met. Otherwise, the change to the routing control is not accepted.
     * </p>
     * <p>
     * Gating rule: A gating rule verifies that a set of gating controls evaluates as true, based on a rule
     * configuration that you specify. If the gating rule evaluates to true, Amazon Route 53 Application Recovery
     * Controller allows a set of routing control state changes to run and complete against the set of target controls.
     * </p>
     *
     * @param createSafetyRuleRequest
     *        The request body that you include when you create a safety rule.
     * @return Result of the CreateSafetyRule operation returned by the service.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string and input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - InternalServiceError. Temporary service error. Retry the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryControlConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryControlConfigClient.CreateSafetyRule
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-control-config-2020-11-02/CreateSafetyRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateSafetyRuleResponse createSafetyRule(CreateSafetyRuleRequest createSafetyRuleRequest) throws ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, Route53RecoveryControlConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateSafetyRuleResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateSafetyRuleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSafetyRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Control Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSafetyRule");

            return clientHandler.execute(new ClientExecutionParams<CreateSafetyRuleRequest, CreateSafetyRuleResponse>()
                    .withOperationName("CreateSafetyRule").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createSafetyRuleRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateSafetyRuleRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Delete a cluster.
     * </p>
     *
     * @param deleteClusterRequest
     * @return Result of the DeleteCluster operation returned by the service.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string and input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - InternalServiceError. Temporary service error. Retry the request.
     * @throws AccessDeniedException
     *         403 response - AccessDeniedException. You do not hace sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         404 response - MalformedQueryString. The query string contains a syntax error or resource not found.
     * @throws ThrottlingException
     *         429 response - LimitExceededException or TooManyRequestsException.
     * @throws ConflictException
     *         409 response - ConflictException. You might be using a predefined variable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryControlConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryControlConfigClient.DeleteCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-control-config-2020-11-02/DeleteCluster"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteClusterResponse deleteCluster(DeleteClusterRequest deleteClusterRequest) throws ValidationException,
            InternalServerException, AccessDeniedException, ResourceNotFoundException, ThrottlingException, ConflictException,
            AwsServiceException, SdkClientException, Route53RecoveryControlConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteClusterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteClusterResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Control Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteCluster");

            return clientHandler.execute(new ClientExecutionParams<DeleteClusterRequest, DeleteClusterResponse>()
                    .withOperationName("DeleteCluster").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteClusterRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteClusterRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a control panel.
     * </p>
     *
     * @param deleteControlPanelRequest
     * @return Result of the DeleteControlPanel operation returned by the service.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string and input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - InternalServiceError. Temporary service error. Retry the request.
     * @throws AccessDeniedException
     *         403 response - AccessDeniedException. You do not hace sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         404 response - MalformedQueryString. The query string contains a syntax error or resource not found.
     * @throws ThrottlingException
     *         429 response - LimitExceededException or TooManyRequestsException.
     * @throws ConflictException
     *         409 response - ConflictException. You might be using a predefined variable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryControlConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryControlConfigClient.DeleteControlPanel
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-control-config-2020-11-02/DeleteControlPanel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteControlPanelResponse deleteControlPanel(DeleteControlPanelRequest deleteControlPanelRequest)
            throws ValidationException, InternalServerException, AccessDeniedException, ResourceNotFoundException,
            ThrottlingException, ConflictException, AwsServiceException, SdkClientException,
            Route53RecoveryControlConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteControlPanelResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteControlPanelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteControlPanelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Control Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteControlPanel");

            return clientHandler.execute(new ClientExecutionParams<DeleteControlPanelRequest, DeleteControlPanelResponse>()
                    .withOperationName("DeleteControlPanel").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteControlPanelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteControlPanelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a routing control.
     * </p>
     *
     * @param deleteRoutingControlRequest
     * @return Result of the DeleteRoutingControl operation returned by the service.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string and input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - InternalServiceError. Temporary service error. Retry the request.
     * @throws AccessDeniedException
     *         403 response - AccessDeniedException. You do not hace sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         404 response - MalformedQueryString. The query string contains a syntax error or resource not found.
     * @throws ThrottlingException
     *         429 response - LimitExceededException or TooManyRequestsException.
     * @throws ConflictException
     *         409 response - ConflictException. You might be using a predefined variable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryControlConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryControlConfigClient.DeleteRoutingControl
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-control-config-2020-11-02/DeleteRoutingControl"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteRoutingControlResponse deleteRoutingControl(DeleteRoutingControlRequest deleteRoutingControlRequest)
            throws ValidationException, InternalServerException, AccessDeniedException, ResourceNotFoundException,
            ThrottlingException, ConflictException, AwsServiceException, SdkClientException,
            Route53RecoveryControlConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteRoutingControlResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteRoutingControlResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteRoutingControlRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Control Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRoutingControl");

            return clientHandler.execute(new ClientExecutionParams<DeleteRoutingControlRequest, DeleteRoutingControlResponse>()
                    .withOperationName("DeleteRoutingControl").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteRoutingControlRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteRoutingControlRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a safety rule.
     * </p>
     * /&gt;
     *
     * @param deleteSafetyRuleRequest
     * @return Result of the DeleteSafetyRule operation returned by the service.
     * @throws ResourceNotFoundException
     *         404 response - MalformedQueryString. The query string contains a syntax error or resource not found.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string and input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - InternalServiceError. Temporary service error. Retry the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryControlConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryControlConfigClient.DeleteSafetyRule
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-control-config-2020-11-02/DeleteSafetyRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteSafetyRuleResponse deleteSafetyRule(DeleteSafetyRuleRequest deleteSafetyRuleRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, Route53RecoveryControlConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteSafetyRuleResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteSafetyRuleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSafetyRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Control Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSafetyRule");

            return clientHandler.execute(new ClientExecutionParams<DeleteSafetyRuleRequest, DeleteSafetyRuleResponse>()
                    .withOperationName("DeleteSafetyRule").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteSafetyRuleRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteSafetyRuleRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Display the details about a cluster. The response includes the cluster name, endpoints, status, and Amazon
     * Resource Name (ARN).
     * </p>
     *
     * @param describeClusterRequest
     * @return Result of the DescribeCluster operation returned by the service.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string and input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - InternalServiceError. Temporary service error. Retry the request.
     * @throws AccessDeniedException
     *         403 response - AccessDeniedException. You do not hace sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         404 response - MalformedQueryString. The query string contains a syntax error or resource not found.
     * @throws ThrottlingException
     *         429 response - LimitExceededException or TooManyRequestsException.
     * @throws ConflictException
     *         409 response - ConflictException. You might be using a predefined variable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryControlConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryControlConfigClient.DescribeCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-control-config-2020-11-02/DescribeCluster"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeClusterResponse describeCluster(DescribeClusterRequest describeClusterRequest) throws ValidationException,
            InternalServerException, AccessDeniedException, ResourceNotFoundException, ThrottlingException, ConflictException,
            AwsServiceException, SdkClientException, Route53RecoveryControlConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeClusterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeClusterResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Control Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeCluster");

            return clientHandler.execute(new ClientExecutionParams<DescribeClusterRequest, DescribeClusterResponse>()
                    .withOperationName("DescribeCluster").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeClusterRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeClusterRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Displays details about a control panel.
     * </p>
     *
     * @param describeControlPanelRequest
     * @return Result of the DescribeControlPanel operation returned by the service.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string and input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - InternalServiceError. Temporary service error. Retry the request.
     * @throws AccessDeniedException
     *         403 response - AccessDeniedException. You do not hace sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         404 response - MalformedQueryString. The query string contains a syntax error or resource not found.
     * @throws ThrottlingException
     *         429 response - LimitExceededException or TooManyRequestsException.
     * @throws ConflictException
     *         409 response - ConflictException. You might be using a predefined variable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryControlConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryControlConfigClient.DescribeControlPanel
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-control-config-2020-11-02/DescribeControlPanel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeControlPanelResponse describeControlPanel(DescribeControlPanelRequest describeControlPanelRequest)
            throws ValidationException, InternalServerException, AccessDeniedException, ResourceNotFoundException,
            ThrottlingException, ConflictException, AwsServiceException, SdkClientException,
            Route53RecoveryControlConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeControlPanelResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeControlPanelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeControlPanelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Control Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeControlPanel");

            return clientHandler.execute(new ClientExecutionParams<DescribeControlPanelRequest, DescribeControlPanelResponse>()
                    .withOperationName("DescribeControlPanel").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeControlPanelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeControlPanelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Displays details about a routing control. A routing control has one of two states: ON and OFF. You can map the
     * routing control state to the state of an Amazon Route 53 health check, which can be used to control routing.
     * </p>
     * <p>
     * To get or update the routing control state, see the Recovery Cluster (data plane) API actions for Amazon Route 53
     * Application Recovery Controller.
     * </p>
     *
     * @param describeRoutingControlRequest
     * @return Result of the DescribeRoutingControl operation returned by the service.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string and input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - InternalServiceError. Temporary service error. Retry the request.
     * @throws AccessDeniedException
     *         403 response - AccessDeniedException. You do not hace sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         404 response - MalformedQueryString. The query string contains a syntax error or resource not found.
     * @throws ThrottlingException
     *         429 response - LimitExceededException or TooManyRequestsException.
     * @throws ConflictException
     *         409 response - ConflictException. You might be using a predefined variable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryControlConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryControlConfigClient.DescribeRoutingControl
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-control-config-2020-11-02/DescribeRoutingControl"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeRoutingControlResponse describeRoutingControl(DescribeRoutingControlRequest describeRoutingControlRequest)
            throws ValidationException, InternalServerException, AccessDeniedException, ResourceNotFoundException,
            ThrottlingException, ConflictException, AwsServiceException, SdkClientException,
            Route53RecoveryControlConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeRoutingControlResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeRoutingControlResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeRoutingControlRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Control Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeRoutingControl");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeRoutingControlRequest, DescribeRoutingControlResponse>()
                            .withOperationName("DescribeRoutingControl").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeRoutingControlRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeRoutingControlRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes the safety rules (that is, the assertion rules and gating rules) for the routing controls in a control
     * panel.
     * </p>
     *
     * @param describeSafetyRuleRequest
     * @return Result of the DescribeSafetyRule operation returned by the service.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string and input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws ResourceNotFoundException
     *         404 response - MalformedQueryString. The query string contains a syntax error or resource not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryControlConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryControlConfigClient.DescribeSafetyRule
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-control-config-2020-11-02/DescribeSafetyRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeSafetyRuleResponse describeSafetyRule(DescribeSafetyRuleRequest describeSafetyRuleRequest)
            throws ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            Route53RecoveryControlConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeSafetyRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeSafetyRuleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeSafetyRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Control Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeSafetyRule");

            return clientHandler.execute(new ClientExecutionParams<DescribeSafetyRuleRequest, DescribeSafetyRuleResponse>()
                    .withOperationName("DescribeSafetyRule").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeSafetyRuleRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeSafetyRuleRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns an array of all Amazon Route 53 health checks associated with a specific routing control.
     * </p>
     *
     * @param listAssociatedRoute53HealthChecksRequest
     * @return Result of the ListAssociatedRoute53HealthChecks operation returned by the service.
     * @throws ResourceNotFoundException
     *         404 response - MalformedQueryString. The query string contains a syntax error or resource not found.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string and input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - InternalServiceError. Temporary service error. Retry the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryControlConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryControlConfigClient.ListAssociatedRoute53HealthChecks
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-control-config-2020-11-02/ListAssociatedRoute53HealthChecks"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListAssociatedRoute53HealthChecksResponse listAssociatedRoute53HealthChecks(
            ListAssociatedRoute53HealthChecksRequest listAssociatedRoute53HealthChecksRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, AwsServiceException, SdkClientException,
            Route53RecoveryControlConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAssociatedRoute53HealthChecksResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListAssociatedRoute53HealthChecksResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listAssociatedRoute53HealthChecksRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Control Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAssociatedRoute53HealthChecks");

            return clientHandler
                    .execute(new ClientExecutionParams<ListAssociatedRoute53HealthChecksRequest, ListAssociatedRoute53HealthChecksResponse>()
                            .withOperationName("ListAssociatedRoute53HealthChecks").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listAssociatedRoute53HealthChecksRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListAssociatedRoute53HealthChecksRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns an array of all Amazon Route 53 health checks associated with a specific routing control.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listAssociatedRoute53HealthChecks(software.amazon.awssdk.services.route53recoverycontrolconfig.model.ListAssociatedRoute53HealthChecksRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53recoverycontrolconfig.paginators.ListAssociatedRoute53HealthChecksIterable responses = client.listAssociatedRoute53HealthChecksPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.route53recoverycontrolconfig.paginators.ListAssociatedRoute53HealthChecksIterable responses = client
     *             .listAssociatedRoute53HealthChecksPaginator(request);
     *     for (software.amazon.awssdk.services.route53recoverycontrolconfig.model.ListAssociatedRoute53HealthChecksResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53recoverycontrolconfig.paginators.ListAssociatedRoute53HealthChecksIterable responses = client.listAssociatedRoute53HealthChecksPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAssociatedRoute53HealthChecks(software.amazon.awssdk.services.route53recoverycontrolconfig.model.ListAssociatedRoute53HealthChecksRequest)}
     * operation.</b>
     * </p>
     *
     * @param listAssociatedRoute53HealthChecksRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         404 response - MalformedQueryString. The query string contains a syntax error or resource not found.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string and input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - InternalServiceError. Temporary service error. Retry the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryControlConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryControlConfigClient.ListAssociatedRoute53HealthChecks
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-control-config-2020-11-02/ListAssociatedRoute53HealthChecks"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListAssociatedRoute53HealthChecksIterable listAssociatedRoute53HealthChecksPaginator(
            ListAssociatedRoute53HealthChecksRequest listAssociatedRoute53HealthChecksRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, AwsServiceException, SdkClientException,
            Route53RecoveryControlConfigException {
        return new ListAssociatedRoute53HealthChecksIterable(this,
                applyPaginatorUserAgent(listAssociatedRoute53HealthChecksRequest));
    }

    /**
     * <p>
     * Returns an array of all the clusters in an account.
     * </p>
     *
     * @param listClustersRequest
     * @return Result of the ListClusters operation returned by the service.
     * @throws ResourceNotFoundException
     *         404 response - MalformedQueryString. The query string contains a syntax error or resource not found.
     * @throws ThrottlingException
     *         429 response - LimitExceededException or TooManyRequestsException.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string and input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - InternalServiceError. Temporary service error. Retry the request.
     * @throws AccessDeniedException
     *         403 response - AccessDeniedException. You do not hace sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryControlConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryControlConfigClient.ListClusters
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-control-config-2020-11-02/ListClusters"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListClustersResponse listClusters(ListClustersRequest listClustersRequest) throws ResourceNotFoundException,
            ThrottlingException, ValidationException, InternalServerException, AccessDeniedException, AwsServiceException,
            SdkClientException, Route53RecoveryControlConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListClustersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListClustersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listClustersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Control Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListClusters");

            return clientHandler.execute(new ClientExecutionParams<ListClustersRequest, ListClustersResponse>()
                    .withOperationName("ListClusters").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listClustersRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListClustersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns an array of all the clusters in an account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listClusters(software.amazon.awssdk.services.route53recoverycontrolconfig.model.ListClustersRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53recoverycontrolconfig.paginators.ListClustersIterable responses = client.listClustersPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.route53recoverycontrolconfig.paginators.ListClustersIterable responses = client
     *             .listClustersPaginator(request);
     *     for (software.amazon.awssdk.services.route53recoverycontrolconfig.model.ListClustersResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53recoverycontrolconfig.paginators.ListClustersIterable responses = client.listClustersPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listClusters(software.amazon.awssdk.services.route53recoverycontrolconfig.model.ListClustersRequest)}
     * operation.</b>
     * </p>
     *
     * @param listClustersRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         404 response - MalformedQueryString. The query string contains a syntax error or resource not found.
     * @throws ThrottlingException
     *         429 response - LimitExceededException or TooManyRequestsException.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string and input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - InternalServiceError. Temporary service error. Retry the request.
     * @throws AccessDeniedException
     *         403 response - AccessDeniedException. You do not hace sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryControlConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryControlConfigClient.ListClusters
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-control-config-2020-11-02/ListClusters"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListClustersIterable listClustersPaginator(ListClustersRequest listClustersRequest) throws ResourceNotFoundException,
            ThrottlingException, ValidationException, InternalServerException, AccessDeniedException, AwsServiceException,
            SdkClientException, Route53RecoveryControlConfigException {
        return new ListClustersIterable(this, applyPaginatorUserAgent(listClustersRequest));
    }

    /**
     * <p>
     * Returns an array of control panels for a cluster.
     * </p>
     *
     * @param listControlPanelsRequest
     * @return Result of the ListControlPanels operation returned by the service.
     * @throws ResourceNotFoundException
     *         404 response - MalformedQueryString. The query string contains a syntax error or resource not found.
     * @throws ThrottlingException
     *         429 response - LimitExceededException or TooManyRequestsException.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string and input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - InternalServiceError. Temporary service error. Retry the request.
     * @throws AccessDeniedException
     *         403 response - AccessDeniedException. You do not hace sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryControlConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryControlConfigClient.ListControlPanels
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-control-config-2020-11-02/ListControlPanels"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListControlPanelsResponse listControlPanels(ListControlPanelsRequest listControlPanelsRequest)
            throws ResourceNotFoundException, ThrottlingException, ValidationException, InternalServerException,
            AccessDeniedException, AwsServiceException, SdkClientException, Route53RecoveryControlConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListControlPanelsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListControlPanelsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listControlPanelsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Control Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListControlPanels");

            return clientHandler.execute(new ClientExecutionParams<ListControlPanelsRequest, ListControlPanelsResponse>()
                    .withOperationName("ListControlPanels").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listControlPanelsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListControlPanelsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns an array of control panels for a cluster.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listControlPanels(software.amazon.awssdk.services.route53recoverycontrolconfig.model.ListControlPanelsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53recoverycontrolconfig.paginators.ListControlPanelsIterable responses = client.listControlPanelsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.route53recoverycontrolconfig.paginators.ListControlPanelsIterable responses = client
     *             .listControlPanelsPaginator(request);
     *     for (software.amazon.awssdk.services.route53recoverycontrolconfig.model.ListControlPanelsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53recoverycontrolconfig.paginators.ListControlPanelsIterable responses = client.listControlPanelsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listControlPanels(software.amazon.awssdk.services.route53recoverycontrolconfig.model.ListControlPanelsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listControlPanelsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         404 response - MalformedQueryString. The query string contains a syntax error or resource not found.
     * @throws ThrottlingException
     *         429 response - LimitExceededException or TooManyRequestsException.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string and input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - InternalServiceError. Temporary service error. Retry the request.
     * @throws AccessDeniedException
     *         403 response - AccessDeniedException. You do not hace sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryControlConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryControlConfigClient.ListControlPanels
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-control-config-2020-11-02/ListControlPanels"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListControlPanelsIterable listControlPanelsPaginator(ListControlPanelsRequest listControlPanelsRequest)
            throws ResourceNotFoundException, ThrottlingException, ValidationException, InternalServerException,
            AccessDeniedException, AwsServiceException, SdkClientException, Route53RecoveryControlConfigException {
        return new ListControlPanelsIterable(this, applyPaginatorUserAgent(listControlPanelsRequest));
    }

    /**
     * <p>
     * Returns an array of routing controls for a control panel. A routing control is an Amazon Route 53 Application
     * Recovery Controller construct that has one of two states: ON and OFF. You can map the routing control state to
     * the state of an Amazon Route 53 health check, which can be used to control routing.
     * </p>
     *
     * @param listRoutingControlsRequest
     * @return Result of the ListRoutingControls operation returned by the service.
     * @throws ResourceNotFoundException
     *         404 response - MalformedQueryString. The query string contains a syntax error or resource not found.
     * @throws ThrottlingException
     *         429 response - LimitExceededException or TooManyRequestsException.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string and input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - InternalServiceError. Temporary service error. Retry the request.
     * @throws AccessDeniedException
     *         403 response - AccessDeniedException. You do not hace sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryControlConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryControlConfigClient.ListRoutingControls
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-control-config-2020-11-02/ListRoutingControls"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListRoutingControlsResponse listRoutingControls(ListRoutingControlsRequest listRoutingControlsRequest)
            throws ResourceNotFoundException, ThrottlingException, ValidationException, InternalServerException,
            AccessDeniedException, AwsServiceException, SdkClientException, Route53RecoveryControlConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListRoutingControlsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListRoutingControlsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRoutingControlsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Control Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRoutingControls");

            return clientHandler.execute(new ClientExecutionParams<ListRoutingControlsRequest, ListRoutingControlsResponse>()
                    .withOperationName("ListRoutingControls").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listRoutingControlsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListRoutingControlsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns an array of routing controls for a control panel. A routing control is an Amazon Route 53 Application
     * Recovery Controller construct that has one of two states: ON and OFF. You can map the routing control state to
     * the state of an Amazon Route 53 health check, which can be used to control routing.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listRoutingControls(software.amazon.awssdk.services.route53recoverycontrolconfig.model.ListRoutingControlsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53recoverycontrolconfig.paginators.ListRoutingControlsIterable responses = client.listRoutingControlsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.route53recoverycontrolconfig.paginators.ListRoutingControlsIterable responses = client
     *             .listRoutingControlsPaginator(request);
     *     for (software.amazon.awssdk.services.route53recoverycontrolconfig.model.ListRoutingControlsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53recoverycontrolconfig.paginators.ListRoutingControlsIterable responses = client.listRoutingControlsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listRoutingControls(software.amazon.awssdk.services.route53recoverycontrolconfig.model.ListRoutingControlsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listRoutingControlsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         404 response - MalformedQueryString. The query string contains a syntax error or resource not found.
     * @throws ThrottlingException
     *         429 response - LimitExceededException or TooManyRequestsException.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string and input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - InternalServiceError. Temporary service error. Retry the request.
     * @throws AccessDeniedException
     *         403 response - AccessDeniedException. You do not hace sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryControlConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryControlConfigClient.ListRoutingControls
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-control-config-2020-11-02/ListRoutingControls"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListRoutingControlsIterable listRoutingControlsPaginator(ListRoutingControlsRequest listRoutingControlsRequest)
            throws ResourceNotFoundException, ThrottlingException, ValidationException, InternalServerException,
            AccessDeniedException, AwsServiceException, SdkClientException, Route53RecoveryControlConfigException {
        return new ListRoutingControlsIterable(this, applyPaginatorUserAgent(listRoutingControlsRequest));
    }

    /**
     * <p>
     * List the safety rules (the assertion rules and gating rules) that you've defined for the routing controls in a
     * control panel.
     * </p>
     *
     * @param listSafetyRulesRequest
     * @return Result of the ListSafetyRules operation returned by the service.
     * @throws ResourceNotFoundException
     *         404 response - MalformedQueryString. The query string contains a syntax error or resource not found.
     * @throws ThrottlingException
     *         429 response - LimitExceededException or TooManyRequestsException.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string and input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - InternalServiceError. Temporary service error. Retry the request.
     * @throws AccessDeniedException
     *         403 response - AccessDeniedException. You do not hace sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryControlConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryControlConfigClient.ListSafetyRules
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-control-config-2020-11-02/ListSafetyRules"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListSafetyRulesResponse listSafetyRules(ListSafetyRulesRequest listSafetyRulesRequest)
            throws ResourceNotFoundException, ThrottlingException, ValidationException, InternalServerException,
            AccessDeniedException, AwsServiceException, SdkClientException, Route53RecoveryControlConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListSafetyRulesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListSafetyRulesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSafetyRulesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Control Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSafetyRules");

            return clientHandler.execute(new ClientExecutionParams<ListSafetyRulesRequest, ListSafetyRulesResponse>()
                    .withOperationName("ListSafetyRules").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listSafetyRulesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListSafetyRulesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * List the safety rules (the assertion rules and gating rules) that you've defined for the routing controls in a
     * control panel.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listSafetyRules(software.amazon.awssdk.services.route53recoverycontrolconfig.model.ListSafetyRulesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53recoverycontrolconfig.paginators.ListSafetyRulesIterable responses = client.listSafetyRulesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.route53recoverycontrolconfig.paginators.ListSafetyRulesIterable responses = client
     *             .listSafetyRulesPaginator(request);
     *     for (software.amazon.awssdk.services.route53recoverycontrolconfig.model.ListSafetyRulesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53recoverycontrolconfig.paginators.ListSafetyRulesIterable responses = client.listSafetyRulesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSafetyRules(software.amazon.awssdk.services.route53recoverycontrolconfig.model.ListSafetyRulesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listSafetyRulesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         404 response - MalformedQueryString. The query string contains a syntax error or resource not found.
     * @throws ThrottlingException
     *         429 response - LimitExceededException or TooManyRequestsException.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string and input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - InternalServiceError. Temporary service error. Retry the request.
     * @throws AccessDeniedException
     *         403 response - AccessDeniedException. You do not hace sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryControlConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryControlConfigClient.ListSafetyRules
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-control-config-2020-11-02/ListSafetyRules"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListSafetyRulesIterable listSafetyRulesPaginator(ListSafetyRulesRequest listSafetyRulesRequest)
            throws ResourceNotFoundException, ThrottlingException, ValidationException, InternalServerException,
            AccessDeniedException, AwsServiceException, SdkClientException, Route53RecoveryControlConfigException {
        return new ListSafetyRulesIterable(this, applyPaginatorUserAgent(listSafetyRulesRequest));
    }

    /**
     * <p>
     * Updates a control panel. The only update you can make to a control panel is to change the name of the control
     * panel.
     * </p>
     *
     * @param updateControlPanelRequest
     *        The details of the control panel that you're updating.
     * @return Result of the UpdateControlPanel operation returned by the service.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string and input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - InternalServiceError. Temporary service error. Retry the request.
     * @throws AccessDeniedException
     *         403 response - AccessDeniedException. You do not hace sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         404 response - MalformedQueryString. The query string contains a syntax error or resource not found.
     * @throws ThrottlingException
     *         429 response - LimitExceededException or TooManyRequestsException.
     * @throws ConflictException
     *         409 response - ConflictException. You might be using a predefined variable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryControlConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryControlConfigClient.UpdateControlPanel
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-control-config-2020-11-02/UpdateControlPanel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateControlPanelResponse updateControlPanel(UpdateControlPanelRequest updateControlPanelRequest)
            throws ValidationException, InternalServerException, AccessDeniedException, ResourceNotFoundException,
            ThrottlingException, ConflictException, AwsServiceException, SdkClientException,
            Route53RecoveryControlConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateControlPanelResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateControlPanelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateControlPanelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Control Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateControlPanel");

            return clientHandler.execute(new ClientExecutionParams<UpdateControlPanelRequest, UpdateControlPanelResponse>()
                    .withOperationName("UpdateControlPanel").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateControlPanelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateControlPanelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a routing control. You can only update the name of the routing control. To get or update the routing
     * control state, see the Recovery Cluster (data plane) API actions for Amazon Route 53 Application Recovery
     * Controller.
     * </p>
     *
     * @param updateRoutingControlRequest
     *        The details of the routing control that you're updating.
     * @return Result of the UpdateRoutingControl operation returned by the service.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string and input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - InternalServiceError. Temporary service error. Retry the request.
     * @throws AccessDeniedException
     *         403 response - AccessDeniedException. You do not hace sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         404 response - MalformedQueryString. The query string contains a syntax error or resource not found.
     * @throws ThrottlingException
     *         429 response - LimitExceededException or TooManyRequestsException.
     * @throws ConflictException
     *         409 response - ConflictException. You might be using a predefined variable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryControlConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryControlConfigClient.UpdateRoutingControl
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-control-config-2020-11-02/UpdateRoutingControl"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateRoutingControlResponse updateRoutingControl(UpdateRoutingControlRequest updateRoutingControlRequest)
            throws ValidationException, InternalServerException, AccessDeniedException, ResourceNotFoundException,
            ThrottlingException, ConflictException, AwsServiceException, SdkClientException,
            Route53RecoveryControlConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateRoutingControlResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateRoutingControlResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateRoutingControlRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Control Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateRoutingControl");

            return clientHandler.execute(new ClientExecutionParams<UpdateRoutingControlRequest, UpdateRoutingControlResponse>()
                    .withOperationName("UpdateRoutingControl").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateRoutingControlRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateRoutingControlRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Update a safety rule (an assertion rule or gating rule) for the routing controls in a control panel. You can only
     * update the name and the waiting period for a safety rule. To make other updates, delete the safety rule and
     * create a new safety rule.
     * </p>
     *
     * @param updateSafetyRuleRequest
     * @return Result of the UpdateSafetyRule operation returned by the service.
     * @throws ResourceNotFoundException
     *         404 response - MalformedQueryString. The query string contains a syntax error or resource not found.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string and input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - InternalServiceError. Temporary service error. Retry the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryControlConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryControlConfigClient.UpdateSafetyRule
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-control-config-2020-11-02/UpdateSafetyRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateSafetyRuleResponse updateSafetyRule(UpdateSafetyRuleRequest updateSafetyRuleRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, Route53RecoveryControlConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateSafetyRuleResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateSafetyRuleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateSafetyRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Control Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSafetyRule");

            return clientHandler.execute(new ClientExecutionParams<UpdateSafetyRuleRequest, UpdateSafetyRuleResponse>()
                    .withOperationName("UpdateSafetyRule").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateSafetyRuleRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateSafetyRuleRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(Route53RecoveryControlConfigException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends Route53RecoveryControlConfigRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    @Override
    public Route53RecoveryControlConfigWaiter waiter() {
        return Route53RecoveryControlConfigWaiter.builder().client(this).build();
    }
}
