/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.route53recoverycontrolconfig;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.route53recoverycontrolconfig.internal.Route53RecoveryControlConfigServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.AccessDeniedException;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.ConflictException;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.CreateClusterRequest;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.CreateClusterResponse;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.CreateControlPanelRequest;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.CreateControlPanelResponse;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.CreateRoutingControlRequest;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.CreateRoutingControlResponse;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.CreateSafetyRuleRequest;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.CreateSafetyRuleResponse;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.DeleteClusterRequest;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.DeleteClusterResponse;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.DeleteControlPanelRequest;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.DeleteControlPanelResponse;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.DeleteRoutingControlRequest;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.DeleteRoutingControlResponse;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.DeleteSafetyRuleRequest;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.DeleteSafetyRuleResponse;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.DescribeClusterRequest;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.DescribeClusterResponse;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.DescribeControlPanelRequest;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.DescribeControlPanelResponse;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.DescribeRoutingControlRequest;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.DescribeRoutingControlResponse;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.DescribeSafetyRuleRequest;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.DescribeSafetyRuleResponse;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.GetResourcePolicyRequest;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.GetResourcePolicyResponse;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.InternalServerException;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.ListAssociatedRoute53HealthChecksRequest;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.ListAssociatedRoute53HealthChecksResponse;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.ListClustersRequest;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.ListClustersResponse;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.ListControlPanelsRequest;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.ListControlPanelsResponse;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.ListRoutingControlsRequest;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.ListRoutingControlsResponse;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.ListSafetyRulesRequest;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.ListSafetyRulesResponse;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.ResourceNotFoundException;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.Route53RecoveryControlConfigException;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.TagResourceRequest;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.TagResourceResponse;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.ThrottlingException;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.UntagResourceRequest;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.UntagResourceResponse;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.UpdateControlPanelRequest;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.UpdateControlPanelResponse;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.UpdateRoutingControlRequest;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.UpdateRoutingControlResponse;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.UpdateSafetyRuleRequest;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.UpdateSafetyRuleResponse;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.ValidationException;
import software.amazon.awssdk.services.route53recoverycontrolconfig.transform.CreateClusterRequestMarshaller;
import software.amazon.awssdk.services.route53recoverycontrolconfig.transform.CreateControlPanelRequestMarshaller;
import software.amazon.awssdk.services.route53recoverycontrolconfig.transform.CreateRoutingControlRequestMarshaller;
import software.amazon.awssdk.services.route53recoverycontrolconfig.transform.CreateSafetyRuleRequestMarshaller;
import software.amazon.awssdk.services.route53recoverycontrolconfig.transform.DeleteClusterRequestMarshaller;
import software.amazon.awssdk.services.route53recoverycontrolconfig.transform.DeleteControlPanelRequestMarshaller;
import software.amazon.awssdk.services.route53recoverycontrolconfig.transform.DeleteRoutingControlRequestMarshaller;
import software.amazon.awssdk.services.route53recoverycontrolconfig.transform.DeleteSafetyRuleRequestMarshaller;
import software.amazon.awssdk.services.route53recoverycontrolconfig.transform.DescribeClusterRequestMarshaller;
import software.amazon.awssdk.services.route53recoverycontrolconfig.transform.DescribeControlPanelRequestMarshaller;
import software.amazon.awssdk.services.route53recoverycontrolconfig.transform.DescribeRoutingControlRequestMarshaller;
import software.amazon.awssdk.services.route53recoverycontrolconfig.transform.DescribeSafetyRuleRequestMarshaller;
import software.amazon.awssdk.services.route53recoverycontrolconfig.transform.GetResourcePolicyRequestMarshaller;
import software.amazon.awssdk.services.route53recoverycontrolconfig.transform.ListAssociatedRoute53HealthChecksRequestMarshaller;
import software.amazon.awssdk.services.route53recoverycontrolconfig.transform.ListClustersRequestMarshaller;
import software.amazon.awssdk.services.route53recoverycontrolconfig.transform.ListControlPanelsRequestMarshaller;
import software.amazon.awssdk.services.route53recoverycontrolconfig.transform.ListRoutingControlsRequestMarshaller;
import software.amazon.awssdk.services.route53recoverycontrolconfig.transform.ListSafetyRulesRequestMarshaller;
import software.amazon.awssdk.services.route53recoverycontrolconfig.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.route53recoverycontrolconfig.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.route53recoverycontrolconfig.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.route53recoverycontrolconfig.transform.UpdateControlPanelRequestMarshaller;
import software.amazon.awssdk.services.route53recoverycontrolconfig.transform.UpdateRoutingControlRequestMarshaller;
import software.amazon.awssdk.services.route53recoverycontrolconfig.transform.UpdateSafetyRuleRequestMarshaller;
import software.amazon.awssdk.services.route53recoverycontrolconfig.waiters.Route53RecoveryControlConfigAsyncWaiter;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link Route53RecoveryControlConfigAsyncClient}.
 *
 * @see Route53RecoveryControlConfigAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultRoute53RecoveryControlConfigAsyncClient implements Route53RecoveryControlConfigAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultRoute53RecoveryControlConfigAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final Route53RecoveryControlConfigServiceClientConfiguration serviceClientConfiguration;

    private final ScheduledExecutorService executorService;

    protected DefaultRoute53RecoveryControlConfigAsyncClient(
            Route53RecoveryControlConfigServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
        this.executorService = clientConfiguration.option(SdkClientOption.SCHEDULED_EXECUTOR_SERVICE);
    }

    /**
     * <p>
     * Create a new cluster. A cluster is a set of redundant Regional endpoints against which you can run API calls to
     * update or get the state of one or more routing controls. Each cluster has a name, status, Amazon Resource Name
     * (ARN), and an array of the five cluster endpoints (one for each supported Amazon Web Services Region) that you
     * can use with API calls to the cluster data plane.
     * </p>
     *
     * @param createClusterRequest
     *        Creates a cluster.
     * @return A Java Future containing the result of the CreateCluster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException 400 response - Multiple causes. For example, you might have a malformed query
     *         string and input parameter might be out of range, or you used parameters together incorrectly.</li>
     *         <li>InternalServerException 500 response - InternalServiceError. Temporary service error. Retry the
     *         request.</li>
     *         <li>ServiceQuotaExceededException 402 response</li>
     *         <li>AccessDeniedException 403 response - AccessDeniedException. You do not have sufficient access to
     *         perform this action.</li>
     *         <li>ResourceNotFoundException 404 response - MalformedQueryString. The query string contains a syntax
     *         error or resource not found.</li>
     *         <li>ThrottlingException 429 response - LimitExceededException or TooManyRequestsException.</li>
     *         <li>ConflictException 409 response - ConflictException. You might be using a predefined variable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryControlConfigException Base class for all service exceptions. Unknown exceptions will
     *         be thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryControlConfigAsyncClient.CreateCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-control-config-2020-11-02/CreateCluster"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateClusterResponse> createCluster(CreateClusterRequest createClusterRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createClusterRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Control Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateCluster");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateClusterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateClusterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateClusterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateClusterRequest, CreateClusterResponse>()
                            .withOperationName("CreateCluster").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateClusterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createClusterRequest));
            CompletableFuture<CreateClusterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new control panel. A control panel represents a group of routing controls that can be changed together
     * in a single transaction. You can use a control panel to centrally view the operational status of applications
     * across your organization, and trigger multi-app failovers in a single transaction, for example, to fail over an
     * Availability Zone or Amazon Web Services Region.
     * </p>
     *
     * @param createControlPanelRequest
     *        The details of the control panel that you're creating.
     * @return A Java Future containing the result of the CreateControlPanel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException 400 response - Multiple causes. For example, you might have a malformed query
     *         string and input parameter might be out of range, or you used parameters together incorrectly.</li>
     *         <li>InternalServerException 500 response - InternalServiceError. Temporary service error. Retry the
     *         request.</li>
     *         <li>ServiceQuotaExceededException 402 response</li>
     *         <li>AccessDeniedException 403 response - AccessDeniedException. You do not have sufficient access to
     *         perform this action.</li>
     *         <li>ResourceNotFoundException 404 response - MalformedQueryString. The query string contains a syntax
     *         error or resource not found.</li>
     *         <li>ThrottlingException 429 response - LimitExceededException or TooManyRequestsException.</li>
     *         <li>ConflictException 409 response - ConflictException. You might be using a predefined variable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryControlConfigException Base class for all service exceptions. Unknown exceptions will
     *         be thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryControlConfigAsyncClient.CreateControlPanel
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-control-config-2020-11-02/CreateControlPanel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateControlPanelResponse> createControlPanel(CreateControlPanelRequest createControlPanelRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createControlPanelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createControlPanelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Control Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateControlPanel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateControlPanelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateControlPanelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateControlPanelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateControlPanelRequest, CreateControlPanelResponse>()
                            .withOperationName("CreateControlPanel").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateControlPanelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createControlPanelRequest));
            CompletableFuture<CreateControlPanelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new routing control.
     * </p>
     * <p>
     * A routing control has one of two states: ON and OFF. You can map the routing control state to the state of an
     * Amazon Route 53 health check, which can be used to control traffic routing.
     * </p>
     * <p>
     * To get or update the routing control state, see the Recovery Cluster (data plane) API actions for Amazon Route 53
     * Application Recovery Controller.
     * </p>
     *
     * @param createRoutingControlRequest
     *        The details of the routing control that you're creating.
     * @return A Java Future containing the result of the CreateRoutingControl operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException 400 response - Multiple causes. For example, you might have a malformed query
     *         string and input parameter might be out of range, or you used parameters together incorrectly.</li>
     *         <li>InternalServerException 500 response - InternalServiceError. Temporary service error. Retry the
     *         request.</li>
     *         <li>ServiceQuotaExceededException 402 response</li>
     *         <li>AccessDeniedException 403 response - AccessDeniedException. You do not have sufficient access to
     *         perform this action.</li>
     *         <li>ResourceNotFoundException 404 response - MalformedQueryString. The query string contains a syntax
     *         error or resource not found.</li>
     *         <li>ThrottlingException 429 response - LimitExceededException or TooManyRequestsException.</li>
     *         <li>ConflictException 409 response - ConflictException. You might be using a predefined variable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryControlConfigException Base class for all service exceptions. Unknown exceptions will
     *         be thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryControlConfigAsyncClient.CreateRoutingControl
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-control-config-2020-11-02/CreateRoutingControl"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateRoutingControlResponse> createRoutingControl(
            CreateRoutingControlRequest createRoutingControlRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createRoutingControlRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createRoutingControlRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Control Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateRoutingControl");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateRoutingControlResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateRoutingControlResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateRoutingControlResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateRoutingControlRequest, CreateRoutingControlResponse>()
                            .withOperationName("CreateRoutingControl").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateRoutingControlRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createRoutingControlRequest));
            CompletableFuture<CreateRoutingControlResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a safety rule in a control panel. Safety rules let you add safeguards around changing routing control
     * states, and for enabling and disabling routing controls, to help prevent unexpected outcomes.
     * </p>
     * <p>
     * There are two types of safety rules: assertion rules and gating rules.
     * </p>
     * <p>
     * Assertion rule: An assertion rule enforces that, when you change a routing control state, that a certain criteria
     * is met. For example, the criteria might be that at least one routing control state is On after the transaction so
     * that traffic continues to flow to at least one cell for the application. This ensures that you avoid a fail-open
     * scenario.
     * </p>
     * <p>
     * Gating rule: A gating rule lets you configure a gating routing control as an overall "on/off" switch for a group
     * of routing controls. Or, you can configure more complex gating scenarios, for example by configuring multiple
     * gating routing controls.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/r53recovery/latest/dg/routing-control.safety-rules.html">Safety rules</a> in
     * the Amazon Route 53 Application Recovery Controller Developer Guide.
     * </p>
     *
     * @param createSafetyRuleRequest
     *        The request body that you include when you create a safety rule.
     * @return A Java Future containing the result of the CreateSafetyRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException 400 response - Multiple causes. For example, you might have a malformed query
     *         string and input parameter might be out of range, or you used parameters together incorrectly.</li>
     *         <li>InternalServerException 500 response - InternalServiceError. Temporary service error. Retry the
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryControlConfigException Base class for all service exceptions. Unknown exceptions will
     *         be thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryControlConfigAsyncClient.CreateSafetyRule
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-control-config-2020-11-02/CreateSafetyRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateSafetyRuleResponse> createSafetyRule(CreateSafetyRuleRequest createSafetyRuleRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createSafetyRuleRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSafetyRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Control Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSafetyRule");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateSafetyRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateSafetyRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateSafetyRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateSafetyRuleRequest, CreateSafetyRuleResponse>()
                            .withOperationName("CreateSafetyRule").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateSafetyRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createSafetyRuleRequest));
            CompletableFuture<CreateSafetyRuleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Delete a cluster.
     * </p>
     *
     * @param deleteClusterRequest
     * @return A Java Future containing the result of the DeleteCluster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException 400 response - Multiple causes. For example, you might have a malformed query
     *         string and input parameter might be out of range, or you used parameters together incorrectly.</li>
     *         <li>InternalServerException 500 response - InternalServiceError. Temporary service error. Retry the
     *         request.</li>
     *         <li>AccessDeniedException 403 response - AccessDeniedException. You do not have sufficient access to
     *         perform this action.</li>
     *         <li>ResourceNotFoundException 404 response - MalformedQueryString. The query string contains a syntax
     *         error or resource not found.</li>
     *         <li>ThrottlingException 429 response - LimitExceededException or TooManyRequestsException.</li>
     *         <li>ConflictException 409 response - ConflictException. You might be using a predefined variable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryControlConfigException Base class for all service exceptions. Unknown exceptions will
     *         be thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryControlConfigAsyncClient.DeleteCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-control-config-2020-11-02/DeleteCluster"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteClusterResponse> deleteCluster(DeleteClusterRequest deleteClusterRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteClusterRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Control Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteCluster");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteClusterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteClusterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteClusterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteClusterRequest, DeleteClusterResponse>()
                            .withOperationName("DeleteCluster").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteClusterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteClusterRequest));
            CompletableFuture<DeleteClusterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a control panel.
     * </p>
     *
     * @param deleteControlPanelRequest
     * @return A Java Future containing the result of the DeleteControlPanel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException 400 response - Multiple causes. For example, you might have a malformed query
     *         string and input parameter might be out of range, or you used parameters together incorrectly.</li>
     *         <li>InternalServerException 500 response - InternalServiceError. Temporary service error. Retry the
     *         request.</li>
     *         <li>AccessDeniedException 403 response - AccessDeniedException. You do not have sufficient access to
     *         perform this action.</li>
     *         <li>ResourceNotFoundException 404 response - MalformedQueryString. The query string contains a syntax
     *         error or resource not found.</li>
     *         <li>ThrottlingException 429 response - LimitExceededException or TooManyRequestsException.</li>
     *         <li>ConflictException 409 response - ConflictException. You might be using a predefined variable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryControlConfigException Base class for all service exceptions. Unknown exceptions will
     *         be thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryControlConfigAsyncClient.DeleteControlPanel
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-control-config-2020-11-02/DeleteControlPanel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteControlPanelResponse> deleteControlPanel(DeleteControlPanelRequest deleteControlPanelRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteControlPanelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteControlPanelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Control Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteControlPanel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteControlPanelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteControlPanelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteControlPanelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteControlPanelRequest, DeleteControlPanelResponse>()
                            .withOperationName("DeleteControlPanel").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteControlPanelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteControlPanelRequest));
            CompletableFuture<DeleteControlPanelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a routing control.
     * </p>
     *
     * @param deleteRoutingControlRequest
     * @return A Java Future containing the result of the DeleteRoutingControl operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException 400 response - Multiple causes. For example, you might have a malformed query
     *         string and input parameter might be out of range, or you used parameters together incorrectly.</li>
     *         <li>InternalServerException 500 response - InternalServiceError. Temporary service error. Retry the
     *         request.</li>
     *         <li>AccessDeniedException 403 response - AccessDeniedException. You do not have sufficient access to
     *         perform this action.</li>
     *         <li>ResourceNotFoundException 404 response - MalformedQueryString. The query string contains a syntax
     *         error or resource not found.</li>
     *         <li>ThrottlingException 429 response - LimitExceededException or TooManyRequestsException.</li>
     *         <li>ConflictException 409 response - ConflictException. You might be using a predefined variable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryControlConfigException Base class for all service exceptions. Unknown exceptions will
     *         be thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryControlConfigAsyncClient.DeleteRoutingControl
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-control-config-2020-11-02/DeleteRoutingControl"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteRoutingControlResponse> deleteRoutingControl(
            DeleteRoutingControlRequest deleteRoutingControlRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteRoutingControlRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteRoutingControlRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Control Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRoutingControl");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteRoutingControlResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteRoutingControlResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteRoutingControlResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteRoutingControlRequest, DeleteRoutingControlResponse>()
                            .withOperationName("DeleteRoutingControl").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteRoutingControlRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteRoutingControlRequest));
            CompletableFuture<DeleteRoutingControlResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a safety rule.
     * </p>
     * /&gt;
     *
     * @param deleteSafetyRuleRequest
     * @return A Java Future containing the result of the DeleteSafetyRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException 404 response - MalformedQueryString. The query string contains a syntax
     *         error or resource not found.</li>
     *         <li>ValidationException 400 response - Multiple causes. For example, you might have a malformed query
     *         string and input parameter might be out of range, or you used parameters together incorrectly.</li>
     *         <li>InternalServerException 500 response - InternalServiceError. Temporary service error. Retry the
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryControlConfigException Base class for all service exceptions. Unknown exceptions will
     *         be thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryControlConfigAsyncClient.DeleteSafetyRule
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-control-config-2020-11-02/DeleteSafetyRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteSafetyRuleResponse> deleteSafetyRule(DeleteSafetyRuleRequest deleteSafetyRuleRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteSafetyRuleRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSafetyRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Control Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSafetyRule");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteSafetyRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteSafetyRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteSafetyRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteSafetyRuleRequest, DeleteSafetyRuleResponse>()
                            .withOperationName("DeleteSafetyRule").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteSafetyRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteSafetyRuleRequest));
            CompletableFuture<DeleteSafetyRuleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Display the details about a cluster. The response includes the cluster name, endpoints, status, and Amazon
     * Resource Name (ARN).
     * </p>
     *
     * @param describeClusterRequest
     * @return A Java Future containing the result of the DescribeCluster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException 400 response - Multiple causes. For example, you might have a malformed query
     *         string and input parameter might be out of range, or you used parameters together incorrectly.</li>
     *         <li>InternalServerException 500 response - InternalServiceError. Temporary service error. Retry the
     *         request.</li>
     *         <li>AccessDeniedException 403 response - AccessDeniedException. You do not have sufficient access to
     *         perform this action.</li>
     *         <li>ResourceNotFoundException 404 response - MalformedQueryString. The query string contains a syntax
     *         error or resource not found.</li>
     *         <li>ThrottlingException 429 response - LimitExceededException or TooManyRequestsException.</li>
     *         <li>ConflictException 409 response - ConflictException. You might be using a predefined variable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryControlConfigException Base class for all service exceptions. Unknown exceptions will
     *         be thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryControlConfigAsyncClient.DescribeCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-control-config-2020-11-02/DescribeCluster"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeClusterResponse> describeCluster(DescribeClusterRequest describeClusterRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeClusterRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Control Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeCluster");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeClusterResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeClusterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeClusterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeClusterRequest, DescribeClusterResponse>()
                            .withOperationName("DescribeCluster").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeClusterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeClusterRequest));
            CompletableFuture<DescribeClusterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Displays details about a control panel.
     * </p>
     *
     * @param describeControlPanelRequest
     * @return A Java Future containing the result of the DescribeControlPanel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException 400 response - Multiple causes. For example, you might have a malformed query
     *         string and input parameter might be out of range, or you used parameters together incorrectly.</li>
     *         <li>InternalServerException 500 response - InternalServiceError. Temporary service error. Retry the
     *         request.</li>
     *         <li>AccessDeniedException 403 response - AccessDeniedException. You do not have sufficient access to
     *         perform this action.</li>
     *         <li>ResourceNotFoundException 404 response - MalformedQueryString. The query string contains a syntax
     *         error or resource not found.</li>
     *         <li>ThrottlingException 429 response - LimitExceededException or TooManyRequestsException.</li>
     *         <li>ConflictException 409 response - ConflictException. You might be using a predefined variable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryControlConfigException Base class for all service exceptions. Unknown exceptions will
     *         be thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryControlConfigAsyncClient.DescribeControlPanel
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-control-config-2020-11-02/DescribeControlPanel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeControlPanelResponse> describeControlPanel(
            DescribeControlPanelRequest describeControlPanelRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeControlPanelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeControlPanelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Control Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeControlPanel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeControlPanelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeControlPanelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeControlPanelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeControlPanelRequest, DescribeControlPanelResponse>()
                            .withOperationName("DescribeControlPanel").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeControlPanelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeControlPanelRequest));
            CompletableFuture<DescribeControlPanelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Displays details about a routing control. A routing control has one of two states: ON and OFF. You can map the
     * routing control state to the state of an Amazon Route 53 health check, which can be used to control routing.
     * </p>
     * <p>
     * To get or update the routing control state, see the Recovery Cluster (data plane) API actions for Amazon Route 53
     * Application Recovery Controller.
     * </p>
     *
     * @param describeRoutingControlRequest
     * @return A Java Future containing the result of the DescribeRoutingControl operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException 400 response - Multiple causes. For example, you might have a malformed query
     *         string and input parameter might be out of range, or you used parameters together incorrectly.</li>
     *         <li>InternalServerException 500 response - InternalServiceError. Temporary service error. Retry the
     *         request.</li>
     *         <li>AccessDeniedException 403 response - AccessDeniedException. You do not have sufficient access to
     *         perform this action.</li>
     *         <li>ResourceNotFoundException 404 response - MalformedQueryString. The query string contains a syntax
     *         error or resource not found.</li>
     *         <li>ThrottlingException 429 response - LimitExceededException or TooManyRequestsException.</li>
     *         <li>ConflictException 409 response - ConflictException. You might be using a predefined variable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryControlConfigException Base class for all service exceptions. Unknown exceptions will
     *         be thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryControlConfigAsyncClient.DescribeRoutingControl
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-control-config-2020-11-02/DescribeRoutingControl"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeRoutingControlResponse> describeRoutingControl(
            DescribeRoutingControlRequest describeRoutingControlRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeRoutingControlRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeRoutingControlRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Control Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeRoutingControl");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeRoutingControlResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeRoutingControlResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeRoutingControlResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeRoutingControlRequest, DescribeRoutingControlResponse>()
                            .withOperationName("DescribeRoutingControl").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeRoutingControlRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeRoutingControlRequest));
            CompletableFuture<DescribeRoutingControlResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about a safety rule.
     * </p>
     *
     * @param describeSafetyRuleRequest
     * @return A Java Future containing the result of the DescribeSafetyRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException 400 response - Multiple causes. For example, you might have a malformed query
     *         string and input parameter might be out of range, or you used parameters together incorrectly.</li>
     *         <li>ResourceNotFoundException 404 response - MalformedQueryString. The query string contains a syntax
     *         error or resource not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryControlConfigException Base class for all service exceptions. Unknown exceptions will
     *         be thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryControlConfigAsyncClient.DescribeSafetyRule
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-control-config-2020-11-02/DescribeSafetyRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeSafetyRuleResponse> describeSafetyRule(DescribeSafetyRuleRequest describeSafetyRuleRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeSafetyRuleRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeSafetyRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Control Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeSafetyRule");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeSafetyRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeSafetyRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeSafetyRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeSafetyRuleRequest, DescribeSafetyRuleResponse>()
                            .withOperationName("DescribeSafetyRule").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeSafetyRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeSafetyRuleRequest));
            CompletableFuture<DescribeSafetyRuleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get information about the resource policy for a cluster.
     * </p>
     *
     * @param getResourcePolicyRequest
     * @return A Java Future containing the result of the GetResourcePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException 404 response - MalformedQueryString. The query string contains a syntax
     *         error or resource not found.</li>
     *         <li>InternalServerException 500 response - InternalServiceError. Temporary service error. Retry the
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryControlConfigException Base class for all service exceptions. Unknown exceptions will
     *         be thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryControlConfigAsyncClient.GetResourcePolicy
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-control-config-2020-11-02/GetResourcePolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetResourcePolicyResponse> getResourcePolicy(GetResourcePolicyRequest getResourcePolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getResourcePolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourcePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Control Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResourcePolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetResourcePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetResourcePolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetResourcePolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetResourcePolicyRequest, GetResourcePolicyResponse>()
                            .withOperationName("GetResourcePolicy").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetResourcePolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getResourcePolicyRequest));
            CompletableFuture<GetResourcePolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns an array of all Amazon Route 53 health checks associated with a specific routing control.
     * </p>
     *
     * @param listAssociatedRoute53HealthChecksRequest
     * @return A Java Future containing the result of the ListAssociatedRoute53HealthChecks operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException 404 response - MalformedQueryString. The query string contains a syntax
     *         error or resource not found.</li>
     *         <li>ValidationException 400 response - Multiple causes. For example, you might have a malformed query
     *         string and input parameter might be out of range, or you used parameters together incorrectly.</li>
     *         <li>InternalServerException 500 response - InternalServiceError. Temporary service error. Retry the
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryControlConfigException Base class for all service exceptions. Unknown exceptions will
     *         be thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryControlConfigAsyncClient.ListAssociatedRoute53HealthChecks
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-control-config-2020-11-02/ListAssociatedRoute53HealthChecks"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListAssociatedRoute53HealthChecksResponse> listAssociatedRoute53HealthChecks(
            ListAssociatedRoute53HealthChecksRequest listAssociatedRoute53HealthChecksRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAssociatedRoute53HealthChecksRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listAssociatedRoute53HealthChecksRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Control Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAssociatedRoute53HealthChecks");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAssociatedRoute53HealthChecksResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListAssociatedRoute53HealthChecksResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAssociatedRoute53HealthChecksResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAssociatedRoute53HealthChecksRequest, ListAssociatedRoute53HealthChecksResponse>()
                            .withOperationName("ListAssociatedRoute53HealthChecks").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListAssociatedRoute53HealthChecksRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listAssociatedRoute53HealthChecksRequest));
            CompletableFuture<ListAssociatedRoute53HealthChecksResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns an array of all the clusters in an account.
     * </p>
     *
     * @param listClustersRequest
     * @return A Java Future containing the result of the ListClusters operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException 404 response - MalformedQueryString. The query string contains a syntax
     *         error or resource not found.</li>
     *         <li>ThrottlingException 429 response - LimitExceededException or TooManyRequestsException.</li>
     *         <li>ValidationException 400 response - Multiple causes. For example, you might have a malformed query
     *         string and input parameter might be out of range, or you used parameters together incorrectly.</li>
     *         <li>InternalServerException 500 response - InternalServiceError. Temporary service error. Retry the
     *         request.</li>
     *         <li>AccessDeniedException 403 response - AccessDeniedException. You do not have sufficient access to
     *         perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryControlConfigException Base class for all service exceptions. Unknown exceptions will
     *         be thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryControlConfigAsyncClient.ListClusters
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-control-config-2020-11-02/ListClusters"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListClustersResponse> listClusters(ListClustersRequest listClustersRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listClustersRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listClustersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Control Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListClusters");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListClustersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListClustersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListClustersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListClustersRequest, ListClustersResponse>()
                            .withOperationName("ListClusters").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListClustersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listClustersRequest));
            CompletableFuture<ListClustersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns an array of control panels in an account or in a cluster.
     * </p>
     *
     * @param listControlPanelsRequest
     * @return A Java Future containing the result of the ListControlPanels operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException 404 response - MalformedQueryString. The query string contains a syntax
     *         error or resource not found.</li>
     *         <li>ThrottlingException 429 response - LimitExceededException or TooManyRequestsException.</li>
     *         <li>ValidationException 400 response - Multiple causes. For example, you might have a malformed query
     *         string and input parameter might be out of range, or you used parameters together incorrectly.</li>
     *         <li>InternalServerException 500 response - InternalServiceError. Temporary service error. Retry the
     *         request.</li>
     *         <li>AccessDeniedException 403 response - AccessDeniedException. You do not have sufficient access to
     *         perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryControlConfigException Base class for all service exceptions. Unknown exceptions will
     *         be thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryControlConfigAsyncClient.ListControlPanels
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-control-config-2020-11-02/ListControlPanels"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListControlPanelsResponse> listControlPanels(ListControlPanelsRequest listControlPanelsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listControlPanelsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listControlPanelsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Control Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListControlPanels");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListControlPanelsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListControlPanelsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListControlPanelsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListControlPanelsRequest, ListControlPanelsResponse>()
                            .withOperationName("ListControlPanels").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListControlPanelsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listControlPanelsRequest));
            CompletableFuture<ListControlPanelsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns an array of routing controls for a control panel. A routing control is an Amazon Route 53 Application
     * Recovery Controller construct that has one of two states: ON and OFF. You can map the routing control state to
     * the state of an Amazon Route 53 health check, which can be used to control routing.
     * </p>
     *
     * @param listRoutingControlsRequest
     * @return A Java Future containing the result of the ListRoutingControls operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException 404 response - MalformedQueryString. The query string contains a syntax
     *         error or resource not found.</li>
     *         <li>ThrottlingException 429 response - LimitExceededException or TooManyRequestsException.</li>
     *         <li>ValidationException 400 response - Multiple causes. For example, you might have a malformed query
     *         string and input parameter might be out of range, or you used parameters together incorrectly.</li>
     *         <li>InternalServerException 500 response - InternalServiceError. Temporary service error. Retry the
     *         request.</li>
     *         <li>AccessDeniedException 403 response - AccessDeniedException. You do not have sufficient access to
     *         perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryControlConfigException Base class for all service exceptions. Unknown exceptions will
     *         be thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryControlConfigAsyncClient.ListRoutingControls
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-control-config-2020-11-02/ListRoutingControls"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListRoutingControlsResponse> listRoutingControls(
            ListRoutingControlsRequest listRoutingControlsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listRoutingControlsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRoutingControlsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Control Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRoutingControls");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListRoutingControlsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListRoutingControlsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListRoutingControlsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListRoutingControlsRequest, ListRoutingControlsResponse>()
                            .withOperationName("ListRoutingControls").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListRoutingControlsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listRoutingControlsRequest));
            CompletableFuture<ListRoutingControlsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List the safety rules (the assertion rules and gating rules) that you've defined for the routing controls in a
     * control panel.
     * </p>
     *
     * @param listSafetyRulesRequest
     * @return A Java Future containing the result of the ListSafetyRules operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException 404 response - MalformedQueryString. The query string contains a syntax
     *         error or resource not found.</li>
     *         <li>ThrottlingException 429 response - LimitExceededException or TooManyRequestsException.</li>
     *         <li>ValidationException 400 response - Multiple causes. For example, you might have a malformed query
     *         string and input parameter might be out of range, or you used parameters together incorrectly.</li>
     *         <li>InternalServerException 500 response - InternalServiceError. Temporary service error. Retry the
     *         request.</li>
     *         <li>AccessDeniedException 403 response - AccessDeniedException. You do not have sufficient access to
     *         perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryControlConfigException Base class for all service exceptions. Unknown exceptions will
     *         be thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryControlConfigAsyncClient.ListSafetyRules
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-control-config-2020-11-02/ListSafetyRules"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListSafetyRulesResponse> listSafetyRules(ListSafetyRulesRequest listSafetyRulesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listSafetyRulesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSafetyRulesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Control Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSafetyRules");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSafetyRulesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListSafetyRulesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListSafetyRulesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSafetyRulesRequest, ListSafetyRulesResponse>()
                            .withOperationName("ListSafetyRules").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListSafetyRulesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listSafetyRulesRequest));
            CompletableFuture<ListSafetyRulesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the tags for a resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException 404 response - MalformedQueryString. The query string contains a syntax
     *         error or resource not found.</li>
     *         <li>ValidationException 400 response - Multiple causes. For example, you might have a malformed query
     *         string and input parameter might be out of range, or you used parameters together incorrectly.</li>
     *         <li>InternalServerException 500 response - InternalServiceError. Temporary service error. Retry the
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryControlConfigException Base class for all service exceptions. Unknown exceptions will
     *         be thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryControlConfigAsyncClient.ListTagsForResource
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-control-config-2020-11-02/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Control Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds a tag to a resource.
     * </p>
     *
     * @param tagResourceRequest
     *        Request of adding tag to the resource
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException 404 response - MalformedQueryString. The query string contains a syntax
     *         error or resource not found.</li>
     *         <li>ValidationException 400 response - Multiple causes. For example, you might have a malformed query
     *         string and input parameter might be out of range, or you used parameters together incorrectly.</li>
     *         <li>InternalServerException 500 response - InternalServiceError. Temporary service error. Retry the
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryControlConfigException Base class for all service exceptions. Unknown exceptions will
     *         be thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryControlConfigAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-control-config-2020-11-02/TagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Control Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes a tag from a resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException 404 response - MalformedQueryString. The query string contains a syntax
     *         error or resource not found.</li>
     *         <li>ValidationException 400 response - Multiple causes. For example, you might have a malformed query
     *         string and input parameter might be out of range, or you used parameters together incorrectly.</li>
     *         <li>InternalServerException 500 response - InternalServiceError. Temporary service error. Retry the
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryControlConfigException Base class for all service exceptions. Unknown exceptions will
     *         be thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryControlConfigAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-control-config-2020-11-02/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Control Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a control panel. The only update you can make to a control panel is to change the name of the control
     * panel.
     * </p>
     *
     * @param updateControlPanelRequest
     *        The details of the control panel that you're updating.
     * @return A Java Future containing the result of the UpdateControlPanel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException 400 response - Multiple causes. For example, you might have a malformed query
     *         string and input parameter might be out of range, or you used parameters together incorrectly.</li>
     *         <li>InternalServerException 500 response - InternalServiceError. Temporary service error. Retry the
     *         request.</li>
     *         <li>AccessDeniedException 403 response - AccessDeniedException. You do not have sufficient access to
     *         perform this action.</li>
     *         <li>ResourceNotFoundException 404 response - MalformedQueryString. The query string contains a syntax
     *         error or resource not found.</li>
     *         <li>ThrottlingException 429 response - LimitExceededException or TooManyRequestsException.</li>
     *         <li>ConflictException 409 response - ConflictException. You might be using a predefined variable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryControlConfigException Base class for all service exceptions. Unknown exceptions will
     *         be thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryControlConfigAsyncClient.UpdateControlPanel
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-control-config-2020-11-02/UpdateControlPanel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateControlPanelResponse> updateControlPanel(UpdateControlPanelRequest updateControlPanelRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateControlPanelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateControlPanelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Control Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateControlPanel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateControlPanelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateControlPanelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateControlPanelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateControlPanelRequest, UpdateControlPanelResponse>()
                            .withOperationName("UpdateControlPanel").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateControlPanelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateControlPanelRequest));
            CompletableFuture<UpdateControlPanelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a routing control. You can only update the name of the routing control. To get or update the routing
     * control state, see the Recovery Cluster (data plane) API actions for Amazon Route 53 Application Recovery
     * Controller.
     * </p>
     *
     * @param updateRoutingControlRequest
     *        The details of the routing control that you're updating.
     * @return A Java Future containing the result of the UpdateRoutingControl operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException 400 response - Multiple causes. For example, you might have a malformed query
     *         string and input parameter might be out of range, or you used parameters together incorrectly.</li>
     *         <li>InternalServerException 500 response - InternalServiceError. Temporary service error. Retry the
     *         request.</li>
     *         <li>AccessDeniedException 403 response - AccessDeniedException. You do not have sufficient access to
     *         perform this action.</li>
     *         <li>ResourceNotFoundException 404 response - MalformedQueryString. The query string contains a syntax
     *         error or resource not found.</li>
     *         <li>ThrottlingException 429 response - LimitExceededException or TooManyRequestsException.</li>
     *         <li>ConflictException 409 response - ConflictException. You might be using a predefined variable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryControlConfigException Base class for all service exceptions. Unknown exceptions will
     *         be thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryControlConfigAsyncClient.UpdateRoutingControl
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-control-config-2020-11-02/UpdateRoutingControl"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateRoutingControlResponse> updateRoutingControl(
            UpdateRoutingControlRequest updateRoutingControlRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateRoutingControlRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateRoutingControlRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Control Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateRoutingControl");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateRoutingControlResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateRoutingControlResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateRoutingControlResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateRoutingControlRequest, UpdateRoutingControlResponse>()
                            .withOperationName("UpdateRoutingControl").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateRoutingControlRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateRoutingControlRequest));
            CompletableFuture<UpdateRoutingControlResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Update a safety rule (an assertion rule or gating rule). You can only update the name and the waiting period for
     * a safety rule. To make other updates, delete the safety rule and create a new one.
     * </p>
     *
     * @param updateSafetyRuleRequest
     *        A rule that you add to Application Recovery Controller to ensure that recovery actions don't accidentally
     *        impair your application's availability.
     * @return A Java Future containing the result of the UpdateSafetyRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException 404 response - MalformedQueryString. The query string contains a syntax
     *         error or resource not found.</li>
     *         <li>ValidationException 400 response - Multiple causes. For example, you might have a malformed query
     *         string and input parameter might be out of range, or you used parameters together incorrectly.</li>
     *         <li>InternalServerException 500 response - InternalServiceError. Temporary service error. Retry the
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryControlConfigException Base class for all service exceptions. Unknown exceptions will
     *         be thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryControlConfigAsyncClient.UpdateSafetyRule
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-control-config-2020-11-02/UpdateSafetyRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateSafetyRuleResponse> updateSafetyRule(UpdateSafetyRuleRequest updateSafetyRuleRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateSafetyRuleRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateSafetyRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Control Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSafetyRule");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateSafetyRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateSafetyRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateSafetyRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateSafetyRuleRequest, UpdateSafetyRuleResponse>()
                            .withOperationName("UpdateSafetyRule").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateSafetyRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateSafetyRuleRequest));
            CompletableFuture<UpdateSafetyRuleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public Route53RecoveryControlConfigAsyncWaiter waiter() {
        return Route53RecoveryControlConfigAsyncWaiter.builder().client(this).scheduledExecutorService(executorService).build();
    }

    @Override
    public final Route53RecoveryControlConfigServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(Route53RecoveryControlConfigException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        Route53RecoveryControlConfigServiceClientConfigurationBuilder.BuilderInternal serviceConfigBuilder = Route53RecoveryControlConfigServiceClientConfigurationBuilder
                .builder(clientConfiguration.toBuilder());
        serviceConfigBuilder.overrideConfiguration(serviceClientConfiguration.overrideConfiguration());
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return serviceConfigBuilder.buildSdkClientConfiguration();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
