/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.route53recoverycontrolconfig.waiters;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Immutable;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.services.route53recoverycontrolconfig.Route53RecoveryControlConfigAsyncClient;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.DescribeClusterRequest;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.DescribeClusterResponse;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.DescribeControlPanelRequest;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.DescribeControlPanelResponse;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.DescribeRoutingControlRequest;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.DescribeRoutingControlResponse;
import software.amazon.awssdk.utils.SdkAutoCloseable;

/**
 * Waiter utility class that polls a resource until a desired state is reached or until it is determined that the
 * resource will never enter into the desired state. This can be created using the static {@link #builder()} method
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
@Immutable
public interface Route53RecoveryControlConfigAsyncWaiter extends SdkAutoCloseable {
    /**
     * Polls {@link Route53RecoveryControlConfigAsyncClient#describeCluster} API until the desired condition
     * {@code ClusterCreated} is met, or until it is determined that the resource will never enter into the desired
     * state
     *
     * @param describeClusterRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeClusterResponse>> waitUntilClusterCreated(
            DescribeClusterRequest describeClusterRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link Route53RecoveryControlConfigAsyncClient#describeCluster} API until the desired condition
     * {@code ClusterCreated} is met, or until it is determined that the resource will never enter into the desired
     * state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeClusterRequest#builder()}
     *
     * @param describeClusterRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeClusterResponse>> waitUntilClusterCreated(
            Consumer<DescribeClusterRequest.Builder> describeClusterRequest) {
        return waitUntilClusterCreated(DescribeClusterRequest.builder().applyMutation(describeClusterRequest).build());
    }

    /**
     * Polls {@link Route53RecoveryControlConfigAsyncClient#describeCluster} API until the desired condition
     * {@code ClusterCreated} is met, or until it is determined that the resource will never enter into the desired
     * state
     *
     * @param describeClusterRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeClusterResponse>> waitUntilClusterCreated(
            DescribeClusterRequest describeClusterRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link Route53RecoveryControlConfigAsyncClient#describeCluster} API until the desired condition
     * {@code ClusterCreated} is met, or until it is determined that the resource will never enter into the desired
     * state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeClusterRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeClusterResponse>> waitUntilClusterCreated(
            Consumer<DescribeClusterRequest.Builder> describeClusterRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilClusterCreated(DescribeClusterRequest.builder().applyMutation(describeClusterRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link Route53RecoveryControlConfigAsyncClient#describeCluster} API until the desired condition
     * {@code ClusterDeleted} is met, or until it is determined that the resource will never enter into the desired
     * state
     *
     * @param describeClusterRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeClusterResponse>> waitUntilClusterDeleted(
            DescribeClusterRequest describeClusterRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link Route53RecoveryControlConfigAsyncClient#describeCluster} API until the desired condition
     * {@code ClusterDeleted} is met, or until it is determined that the resource will never enter into the desired
     * state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeClusterRequest#builder()}
     *
     * @param describeClusterRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeClusterResponse>> waitUntilClusterDeleted(
            Consumer<DescribeClusterRequest.Builder> describeClusterRequest) {
        return waitUntilClusterDeleted(DescribeClusterRequest.builder().applyMutation(describeClusterRequest).build());
    }

    /**
     * Polls {@link Route53RecoveryControlConfigAsyncClient#describeCluster} API until the desired condition
     * {@code ClusterDeleted} is met, or until it is determined that the resource will never enter into the desired
     * state
     *
     * @param describeClusterRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeClusterResponse>> waitUntilClusterDeleted(
            DescribeClusterRequest describeClusterRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link Route53RecoveryControlConfigAsyncClient#describeCluster} API until the desired condition
     * {@code ClusterDeleted} is met, or until it is determined that the resource will never enter into the desired
     * state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeClusterRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeClusterResponse>> waitUntilClusterDeleted(
            Consumer<DescribeClusterRequest.Builder> describeClusterRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilClusterDeleted(DescribeClusterRequest.builder().applyMutation(describeClusterRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link Route53RecoveryControlConfigAsyncClient#describeControlPanel} API until the desired condition
     * {@code ControlPanelCreated} is met, or until it is determined that the resource will never enter into the desired
     * state
     *
     * @param describeControlPanelRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeControlPanelResponse>> waitUntilControlPanelCreated(
            DescribeControlPanelRequest describeControlPanelRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link Route53RecoveryControlConfigAsyncClient#describeControlPanel} API until the desired condition
     * {@code ControlPanelCreated} is met, or until it is determined that the resource will never enter into the desired
     * state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeControlPanelRequest#builder()}
     *
     * @param describeControlPanelRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeControlPanelResponse>> waitUntilControlPanelCreated(
            Consumer<DescribeControlPanelRequest.Builder> describeControlPanelRequest) {
        return waitUntilControlPanelCreated(DescribeControlPanelRequest.builder().applyMutation(describeControlPanelRequest)
                .build());
    }

    /**
     * Polls {@link Route53RecoveryControlConfigAsyncClient#describeControlPanel} API until the desired condition
     * {@code ControlPanelCreated} is met, or until it is determined that the resource will never enter into the desired
     * state
     *
     * @param describeControlPanelRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeControlPanelResponse>> waitUntilControlPanelCreated(
            DescribeControlPanelRequest describeControlPanelRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link Route53RecoveryControlConfigAsyncClient#describeControlPanel} API until the desired condition
     * {@code ControlPanelCreated} is met, or until it is determined that the resource will never enter into the desired
     * state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeControlPanelRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeControlPanelResponse>> waitUntilControlPanelCreated(
            Consumer<DescribeControlPanelRequest.Builder> describeControlPanelRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilControlPanelCreated(DescribeControlPanelRequest.builder().applyMutation(describeControlPanelRequest)
                .build(), WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link Route53RecoveryControlConfigAsyncClient#describeControlPanel} API until the desired condition
     * {@code ControlPanelDeleted} is met, or until it is determined that the resource will never enter into the desired
     * state
     *
     * @param describeControlPanelRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeControlPanelResponse>> waitUntilControlPanelDeleted(
            DescribeControlPanelRequest describeControlPanelRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link Route53RecoveryControlConfigAsyncClient#describeControlPanel} API until the desired condition
     * {@code ControlPanelDeleted} is met, or until it is determined that the resource will never enter into the desired
     * state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeControlPanelRequest#builder()}
     *
     * @param describeControlPanelRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeControlPanelResponse>> waitUntilControlPanelDeleted(
            Consumer<DescribeControlPanelRequest.Builder> describeControlPanelRequest) {
        return waitUntilControlPanelDeleted(DescribeControlPanelRequest.builder().applyMutation(describeControlPanelRequest)
                .build());
    }

    /**
     * Polls {@link Route53RecoveryControlConfigAsyncClient#describeControlPanel} API until the desired condition
     * {@code ControlPanelDeleted} is met, or until it is determined that the resource will never enter into the desired
     * state
     *
     * @param describeControlPanelRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeControlPanelResponse>> waitUntilControlPanelDeleted(
            DescribeControlPanelRequest describeControlPanelRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link Route53RecoveryControlConfigAsyncClient#describeControlPanel} API until the desired condition
     * {@code ControlPanelDeleted} is met, or until it is determined that the resource will never enter into the desired
     * state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeControlPanelRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeControlPanelResponse>> waitUntilControlPanelDeleted(
            Consumer<DescribeControlPanelRequest.Builder> describeControlPanelRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilControlPanelDeleted(DescribeControlPanelRequest.builder().applyMutation(describeControlPanelRequest)
                .build(), WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link Route53RecoveryControlConfigAsyncClient#describeRoutingControl} API until the desired condition
     * {@code RoutingControlCreated} is met, or until it is determined that the resource will never enter into the
     * desired state
     *
     * @param describeRoutingControlRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeRoutingControlResponse>> waitUntilRoutingControlCreated(
            DescribeRoutingControlRequest describeRoutingControlRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link Route53RecoveryControlConfigAsyncClient#describeRoutingControl} API until the desired condition
     * {@code RoutingControlCreated} is met, or until it is determined that the resource will never enter into the
     * desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeRoutingControlRequest#builder()}
     *
     * @param describeRoutingControlRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeRoutingControlResponse>> waitUntilRoutingControlCreated(
            Consumer<DescribeRoutingControlRequest.Builder> describeRoutingControlRequest) {
        return waitUntilRoutingControlCreated(DescribeRoutingControlRequest.builder()
                .applyMutation(describeRoutingControlRequest).build());
    }

    /**
     * Polls {@link Route53RecoveryControlConfigAsyncClient#describeRoutingControl} API until the desired condition
     * {@code RoutingControlCreated} is met, or until it is determined that the resource will never enter into the
     * desired state
     *
     * @param describeRoutingControlRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeRoutingControlResponse>> waitUntilRoutingControlCreated(
            DescribeRoutingControlRequest describeRoutingControlRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link Route53RecoveryControlConfigAsyncClient#describeRoutingControl} API until the desired condition
     * {@code RoutingControlCreated} is met, or until it is determined that the resource will never enter into the
     * desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeRoutingControlRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeRoutingControlResponse>> waitUntilRoutingControlCreated(
            Consumer<DescribeRoutingControlRequest.Builder> describeRoutingControlRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilRoutingControlCreated(DescribeRoutingControlRequest.builder()
                .applyMutation(describeRoutingControlRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link Route53RecoveryControlConfigAsyncClient#describeRoutingControl} API until the desired condition
     * {@code RoutingControlDeleted} is met, or until it is determined that the resource will never enter into the
     * desired state
     *
     * @param describeRoutingControlRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeRoutingControlResponse>> waitUntilRoutingControlDeleted(
            DescribeRoutingControlRequest describeRoutingControlRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link Route53RecoveryControlConfigAsyncClient#describeRoutingControl} API until the desired condition
     * {@code RoutingControlDeleted} is met, or until it is determined that the resource will never enter into the
     * desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeRoutingControlRequest#builder()}
     *
     * @param describeRoutingControlRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeRoutingControlResponse>> waitUntilRoutingControlDeleted(
            Consumer<DescribeRoutingControlRequest.Builder> describeRoutingControlRequest) {
        return waitUntilRoutingControlDeleted(DescribeRoutingControlRequest.builder()
                .applyMutation(describeRoutingControlRequest).build());
    }

    /**
     * Polls {@link Route53RecoveryControlConfigAsyncClient#describeRoutingControl} API until the desired condition
     * {@code RoutingControlDeleted} is met, or until it is determined that the resource will never enter into the
     * desired state
     *
     * @param describeRoutingControlRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeRoutingControlResponse>> waitUntilRoutingControlDeleted(
            DescribeRoutingControlRequest describeRoutingControlRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link Route53RecoveryControlConfigAsyncClient#describeRoutingControl} API until the desired condition
     * {@code RoutingControlDeleted} is met, or until it is determined that the resource will never enter into the
     * desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeRoutingControlRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeRoutingControlResponse>> waitUntilRoutingControlDeleted(
            Consumer<DescribeRoutingControlRequest.Builder> describeRoutingControlRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilRoutingControlDeleted(DescribeRoutingControlRequest.builder()
                .applyMutation(describeRoutingControlRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Create a builder that can be used to configure and create a {@link Route53RecoveryControlConfigAsyncWaiter}.
     *
     * @return a builder
     */
    static Builder builder() {
        return DefaultRoute53RecoveryControlConfigAsyncWaiter.builder();
    }

    /**
     * Create an instance of {@link Route53RecoveryControlConfigAsyncWaiter} with the default configuration.
     * <p>
     * <b>A default {@link Route53RecoveryControlConfigAsyncClient} will be created to poll resources. It is recommended
     * to share a single instance of the waiter created via this method. If it is not desirable to share a waiter
     * instance, invoke {@link #close()} to release the resources once the waiter is not needed.</b>
     *
     * @return an instance of {@link Route53RecoveryControlConfigAsyncWaiter}
     */
    static Route53RecoveryControlConfigAsyncWaiter create() {
        return DefaultRoute53RecoveryControlConfigAsyncWaiter.builder().build();
    }

    interface Builder {
        /**
         * Sets a custom {@link ScheduledExecutorService} that will be used to schedule async polling attempts
         * <p>
         * This executorService must be closed by the caller when it is ready to be disposed. The SDK will not close the
         * executorService when the waiter is closed
         *
         * @param executorService
         *        the executorService to set
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder scheduledExecutorService(ScheduledExecutorService executorService);

        /**
         * Defines overrides to the default SDK waiter configuration that should be used for waiters created from this
         * builder
         *
         * @param overrideConfiguration
         *        the override configuration to set
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration);

        /**
         * This is a convenient method to pass the override configuration without the need to create an instance
         * manually via {@link WaiterOverrideConfiguration#builder()}
         *
         * @param overrideConfiguration
         *        The consumer that will configure the overrideConfiguration
         * @return a reference to this object so that method calls can be chained together.
         * @see #overrideConfiguration(WaiterOverrideConfiguration)
         */
        default Builder overrideConfiguration(Consumer<WaiterOverrideConfiguration.Builder> overrideConfiguration) {
            WaiterOverrideConfiguration.Builder builder = WaiterOverrideConfiguration.builder();
            overrideConfiguration.accept(builder);
            return overrideConfiguration(builder.build());
        }

        /**
         * Sets a custom {@link Route53RecoveryControlConfigAsyncClient} that will be used to poll the resource
         * <p>
         * This SDK client must be closed by the caller when it is ready to be disposed. The SDK will not close the
         * client when the waiter is closed
         *
         * @param client
         *        the client to send the request
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder client(Route53RecoveryControlConfigAsyncClient client);

        /**
         * Builds an instance of {@link Route53RecoveryControlConfigAsyncWaiter} based on the configurations supplied to
         * this builder
         *
         * @return An initialized {@link Route53RecoveryControlConfigAsyncWaiter}
         */
        Route53RecoveryControlConfigAsyncWaiter build();
    }
}
