/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.route53recoverycontrolconfig.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.waiters.Waiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.core.waiters.WaiterState;
import software.amazon.awssdk.retries.api.BackoffStrategy;
import software.amazon.awssdk.services.route53recoverycontrolconfig.Route53RecoveryControlConfigClient;
import software.amazon.awssdk.services.route53recoverycontrolconfig.jmespath.internal.JmesPathRuntime;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.DescribeClusterRequest;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.DescribeClusterResponse;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.DescribeControlPanelRequest;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.DescribeControlPanelResponse;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.DescribeRoutingControlRequest;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.DescribeRoutingControlResponse;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.Route53RecoveryControlConfigRequest;
import software.amazon.awssdk.services.route53recoverycontrolconfig.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultRoute53RecoveryControlConfigWaiter implements Route53RecoveryControlConfigWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private final Route53RecoveryControlConfigClient client;

    private final AttributeMap managedResources;

    private final Waiter<DescribeClusterResponse> clusterCreatedWaiter;

    private final Waiter<DescribeClusterResponse> clusterDeletedWaiter;

    private final Waiter<DescribeControlPanelResponse> controlPanelCreatedWaiter;

    private final Waiter<DescribeControlPanelResponse> controlPanelDeletedWaiter;

    private final Waiter<DescribeRoutingControlResponse> routingControlCreatedWaiter;

    private final Waiter<DescribeRoutingControlResponse> routingControlDeletedWaiter;

    private DefaultRoute53RecoveryControlConfigWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = Route53RecoveryControlConfigClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        managedResources = attributeMapBuilder.build();
        this.clusterCreatedWaiter = Waiter.builder(DescribeClusterResponse.class).acceptors(clusterCreatedWaiterAcceptors())
                .overrideConfiguration(clusterCreatedWaiterConfig(builder.overrideConfiguration)).build();
        this.clusterDeletedWaiter = Waiter.builder(DescribeClusterResponse.class).acceptors(clusterDeletedWaiterAcceptors())
                .overrideConfiguration(clusterDeletedWaiterConfig(builder.overrideConfiguration)).build();
        this.controlPanelCreatedWaiter = Waiter.builder(DescribeControlPanelResponse.class)
                .acceptors(controlPanelCreatedWaiterAcceptors())
                .overrideConfiguration(controlPanelCreatedWaiterConfig(builder.overrideConfiguration)).build();
        this.controlPanelDeletedWaiter = Waiter.builder(DescribeControlPanelResponse.class)
                .acceptors(controlPanelDeletedWaiterAcceptors())
                .overrideConfiguration(controlPanelDeletedWaiterConfig(builder.overrideConfiguration)).build();
        this.routingControlCreatedWaiter = Waiter.builder(DescribeRoutingControlResponse.class)
                .acceptors(routingControlCreatedWaiterAcceptors())
                .overrideConfiguration(routingControlCreatedWaiterConfig(builder.overrideConfiguration)).build();
        this.routingControlDeletedWaiter = Waiter.builder(DescribeRoutingControlResponse.class)
                .acceptors(routingControlDeletedWaiterAcceptors())
                .overrideConfiguration(routingControlDeletedWaiterConfig(builder.overrideConfiguration)).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public WaiterResponse<DescribeClusterResponse> waitUntilClusterCreated(DescribeClusterRequest describeClusterRequest) {
        return clusterCreatedWaiter.run(() -> client.describeCluster(applyWaitersUserAgent(describeClusterRequest)));
    }

    @Override
    public WaiterResponse<DescribeClusterResponse> waitUntilClusterCreated(DescribeClusterRequest describeClusterRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return clusterCreatedWaiter.run(() -> client.describeCluster(applyWaitersUserAgent(describeClusterRequest)),
                clusterCreatedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeClusterResponse> waitUntilClusterDeleted(DescribeClusterRequest describeClusterRequest) {
        return clusterDeletedWaiter.run(() -> client.describeCluster(applyWaitersUserAgent(describeClusterRequest)));
    }

    @Override
    public WaiterResponse<DescribeClusterResponse> waitUntilClusterDeleted(DescribeClusterRequest describeClusterRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return clusterDeletedWaiter.run(() -> client.describeCluster(applyWaitersUserAgent(describeClusterRequest)),
                clusterDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeControlPanelResponse> waitUntilControlPanelCreated(
            DescribeControlPanelRequest describeControlPanelRequest) {
        return controlPanelCreatedWaiter.run(() -> client
                .describeControlPanel(applyWaitersUserAgent(describeControlPanelRequest)));
    }

    @Override
    public WaiterResponse<DescribeControlPanelResponse> waitUntilControlPanelCreated(
            DescribeControlPanelRequest describeControlPanelRequest, WaiterOverrideConfiguration overrideConfig) {
        return controlPanelCreatedWaiter.run(
                () -> client.describeControlPanel(applyWaitersUserAgent(describeControlPanelRequest)),
                controlPanelCreatedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeControlPanelResponse> waitUntilControlPanelDeleted(
            DescribeControlPanelRequest describeControlPanelRequest) {
        return controlPanelDeletedWaiter.run(() -> client
                .describeControlPanel(applyWaitersUserAgent(describeControlPanelRequest)));
    }

    @Override
    public WaiterResponse<DescribeControlPanelResponse> waitUntilControlPanelDeleted(
            DescribeControlPanelRequest describeControlPanelRequest, WaiterOverrideConfiguration overrideConfig) {
        return controlPanelDeletedWaiter.run(
                () -> client.describeControlPanel(applyWaitersUserAgent(describeControlPanelRequest)),
                controlPanelDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeRoutingControlResponse> waitUntilRoutingControlCreated(
            DescribeRoutingControlRequest describeRoutingControlRequest) {
        return routingControlCreatedWaiter.run(() -> client
                .describeRoutingControl(applyWaitersUserAgent(describeRoutingControlRequest)));
    }

    @Override
    public WaiterResponse<DescribeRoutingControlResponse> waitUntilRoutingControlCreated(
            DescribeRoutingControlRequest describeRoutingControlRequest, WaiterOverrideConfiguration overrideConfig) {
        return routingControlCreatedWaiter.run(
                () -> client.describeRoutingControl(applyWaitersUserAgent(describeRoutingControlRequest)),
                routingControlCreatedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeRoutingControlResponse> waitUntilRoutingControlDeleted(
            DescribeRoutingControlRequest describeRoutingControlRequest) {
        return routingControlDeletedWaiter.run(() -> client
                .describeRoutingControl(applyWaitersUserAgent(describeRoutingControlRequest)));
    }

    @Override
    public WaiterResponse<DescribeRoutingControlResponse> waitUntilRoutingControlDeleted(
            DescribeRoutingControlRequest describeRoutingControlRequest, WaiterOverrideConfiguration overrideConfig) {
        return routingControlDeletedWaiter.run(
                () -> client.describeRoutingControl(applyWaitersUserAgent(describeRoutingControlRequest)),
                routingControlDeletedWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super DescribeClusterResponse>> clusterCreatedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeClusterResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("Cluster").field("Status").value(), "DEPLOYED");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("Cluster").field("Status").value(), "PENDING");
        }));
        result.add(new WaitersRuntime.ResponseStatusAcceptor(500, WaiterState.RETRY));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeClusterResponse>> clusterDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeClusterResponse>> result = new ArrayList<>();
        result.add(new WaitersRuntime.ResponseStatusAcceptor(404, WaiterState.SUCCESS));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("Cluster").field("Status").value(), "PENDING_DELETION");
        }));
        result.add(new WaitersRuntime.ResponseStatusAcceptor(500, WaiterState.RETRY));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeControlPanelResponse>> controlPanelCreatedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeControlPanelResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("ControlPanel").field("Status").value(), "DEPLOYED");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("ControlPanel").field("Status").value(), "PENDING");
        }));
        result.add(new WaitersRuntime.ResponseStatusAcceptor(500, WaiterState.RETRY));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeControlPanelResponse>> controlPanelDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeControlPanelResponse>> result = new ArrayList<>();
        result.add(new WaitersRuntime.ResponseStatusAcceptor(404, WaiterState.SUCCESS));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("ControlPanel").field("Status").value(), "PENDING_DELETION");
        }));
        result.add(new WaitersRuntime.ResponseStatusAcceptor(500, WaiterState.RETRY));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeRoutingControlResponse>> routingControlCreatedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeRoutingControlResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("RoutingControl").field("Status").value(), "DEPLOYED");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("RoutingControl").field("Status").value(), "PENDING");
        }));
        result.add(new WaitersRuntime.ResponseStatusAcceptor(500, WaiterState.RETRY));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeRoutingControlResponse>> routingControlDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeRoutingControlResponse>> result = new ArrayList<>();
        result.add(new WaitersRuntime.ResponseStatusAcceptor(404, WaiterState.SUCCESS));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("RoutingControl").field("Status").value(), "PENDING_DELETION");
        }));
        result.add(new WaitersRuntime.ResponseStatusAcceptor(500, WaiterState.RETRY));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration clusterCreatedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(26);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration clusterDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(26);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration controlPanelCreatedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(26);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration controlPanelDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(26);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration routingControlCreatedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(26);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration routingControlDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(26);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static Route53RecoveryControlConfigWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends Route53RecoveryControlConfigRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version("waiter").name("hll").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements Route53RecoveryControlConfigWaiter.Builder {
        private Route53RecoveryControlConfigClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private DefaultBuilder() {
        }

        @Override
        public Route53RecoveryControlConfigWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public Route53RecoveryControlConfigWaiter.Builder client(Route53RecoveryControlConfigClient client) {
            this.client = client;
            return this;
        }

        public Route53RecoveryControlConfigWaiter build() {
            return new DefaultRoute53RecoveryControlConfigWaiter(this);
        }
    }
}
