/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.route53recoveryreadiness;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.route53recoveryreadiness.internal.Route53RecoveryReadinessServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.route53recoveryreadiness.model.AccessDeniedException;
import software.amazon.awssdk.services.route53recoveryreadiness.model.ConflictException;
import software.amazon.awssdk.services.route53recoveryreadiness.model.CreateCellRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.CreateCellResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.CreateCrossAccountAuthorizationRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.CreateCrossAccountAuthorizationResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.CreateReadinessCheckRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.CreateReadinessCheckResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.CreateRecoveryGroupRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.CreateRecoveryGroupResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.CreateResourceSetRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.CreateResourceSetResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.DeleteCellRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.DeleteCellResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.DeleteCrossAccountAuthorizationRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.DeleteCrossAccountAuthorizationResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.DeleteReadinessCheckRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.DeleteReadinessCheckResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.DeleteRecoveryGroupRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.DeleteRecoveryGroupResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.DeleteResourceSetRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.DeleteResourceSetResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.GetArchitectureRecommendationsRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.GetArchitectureRecommendationsResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.GetCellReadinessSummaryRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.GetCellReadinessSummaryResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.GetCellRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.GetCellResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.GetReadinessCheckRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.GetReadinessCheckResourceStatusRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.GetReadinessCheckResourceStatusResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.GetReadinessCheckResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.GetReadinessCheckStatusRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.GetReadinessCheckStatusResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.GetRecoveryGroupReadinessSummaryRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.GetRecoveryGroupReadinessSummaryResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.GetRecoveryGroupRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.GetRecoveryGroupResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.GetResourceSetRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.GetResourceSetResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.InternalServerException;
import software.amazon.awssdk.services.route53recoveryreadiness.model.ListCellsRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.ListCellsResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.ListCrossAccountAuthorizationsRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.ListCrossAccountAuthorizationsResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.ListReadinessChecksRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.ListReadinessChecksResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.ListRecoveryGroupsRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.ListRecoveryGroupsResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.ListResourceSetsRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.ListResourceSetsResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.ListRulesRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.ListRulesResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.ListTagsForResourcesRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.ListTagsForResourcesResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.ResourceNotFoundException;
import software.amazon.awssdk.services.route53recoveryreadiness.model.Route53RecoveryReadinessException;
import software.amazon.awssdk.services.route53recoveryreadiness.model.TagResourceRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.TagResourceResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.ThrottlingException;
import software.amazon.awssdk.services.route53recoveryreadiness.model.UntagResourceRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.UntagResourceResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.UpdateCellRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.UpdateCellResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.UpdateReadinessCheckRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.UpdateReadinessCheckResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.UpdateRecoveryGroupRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.UpdateRecoveryGroupResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.UpdateResourceSetRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.UpdateResourceSetResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.ValidationException;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.CreateCellRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.CreateCrossAccountAuthorizationRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.CreateReadinessCheckRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.CreateRecoveryGroupRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.CreateResourceSetRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.DeleteCellRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.DeleteCrossAccountAuthorizationRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.DeleteReadinessCheckRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.DeleteRecoveryGroupRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.DeleteResourceSetRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.GetArchitectureRecommendationsRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.GetCellReadinessSummaryRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.GetCellRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.GetReadinessCheckRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.GetReadinessCheckResourceStatusRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.GetReadinessCheckStatusRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.GetRecoveryGroupReadinessSummaryRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.GetRecoveryGroupRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.GetResourceSetRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.ListCellsRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.ListCrossAccountAuthorizationsRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.ListReadinessChecksRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.ListRecoveryGroupsRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.ListResourceSetsRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.ListRulesRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.ListTagsForResourcesRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.UpdateCellRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.UpdateReadinessCheckRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.UpdateRecoveryGroupRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.UpdateResourceSetRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link Route53RecoveryReadinessClient}.
 *
 * @see Route53RecoveryReadinessClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultRoute53RecoveryReadinessClient implements Route53RecoveryReadinessClient {
    private static final Logger log = Logger.loggerFor(DefaultRoute53RecoveryReadinessClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultRoute53RecoveryReadinessClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a cell in an account.
     * </p>
     *
     * @param createCellRequest
     * @return Result of the CreateCell operation returned by the service.
     * @throws ThrottlingException
     *         429 response - Limit exceeded exception or too many requests exception.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string, an input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - Internal service error or temporary service error. Retry the request.
     * @throws ConflictException
     *         409 response - Conflict exception. You might be using a predefined variable.
     * @throws AccessDeniedException
     *         403 response - Access denied exception. You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.CreateCell
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/CreateCell"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateCellResponse createCell(CreateCellRequest createCellRequest) throws ThrottlingException, ValidationException,
            InternalServerException, ConflictException, AccessDeniedException, AwsServiceException, SdkClientException,
            Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateCellResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateCellResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createCellRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createCellRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateCell");

            return clientHandler.execute(new ClientExecutionParams<CreateCellRequest, CreateCellResponse>()
                    .withOperationName("CreateCell").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(createCellRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateCellRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a cross-account readiness authorization. This lets you authorize another account to work with Route 53
     * Application Recovery Controller, for example, to check the readiness status of resources in a separate account.
     * </p>
     *
     * @param createCrossAccountAuthorizationRequest
     * @return Result of the CreateCrossAccountAuthorization operation returned by the service.
     * @throws ThrottlingException
     *         429 response - Limit exceeded exception or too many requests exception.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string, an input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - Internal service error or temporary service error. Retry the request.
     * @throws ConflictException
     *         409 response - Conflict exception. You might be using a predefined variable.
     * @throws AccessDeniedException
     *         403 response - Access denied exception. You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.CreateCrossAccountAuthorization
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/CreateCrossAccountAuthorization"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateCrossAccountAuthorizationResponse createCrossAccountAuthorization(
            CreateCrossAccountAuthorizationRequest createCrossAccountAuthorizationRequest) throws ThrottlingException,
            ValidationException, InternalServerException, ConflictException, AccessDeniedException, AwsServiceException,
            SdkClientException, Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateCrossAccountAuthorizationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateCrossAccountAuthorizationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createCrossAccountAuthorizationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createCrossAccountAuthorizationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateCrossAccountAuthorization");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateCrossAccountAuthorizationRequest, CreateCrossAccountAuthorizationResponse>()
                            .withOperationName("CreateCrossAccountAuthorization").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createCrossAccountAuthorizationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateCrossAccountAuthorizationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a readiness check in an account. A readiness check monitors a resource set in your application, such as a
     * set of Amazon Aurora instances, that Application Recovery Controller is auditing recovery readiness for. The
     * audits run once every minute on every resource that's associated with a readiness check.
     * </p>
     *
     * @param createReadinessCheckRequest
     * @return Result of the CreateReadinessCheck operation returned by the service.
     * @throws ThrottlingException
     *         429 response - Limit exceeded exception or too many requests exception.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string, an input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - Internal service error or temporary service error. Retry the request.
     * @throws ConflictException
     *         409 response - Conflict exception. You might be using a predefined variable.
     * @throws AccessDeniedException
     *         403 response - Access denied exception. You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.CreateReadinessCheck
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/CreateReadinessCheck"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateReadinessCheckResponse createReadinessCheck(CreateReadinessCheckRequest createReadinessCheckRequest)
            throws ThrottlingException, ValidationException, InternalServerException, ConflictException, AccessDeniedException,
            AwsServiceException, SdkClientException, Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateReadinessCheckResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateReadinessCheckResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createReadinessCheckRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createReadinessCheckRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateReadinessCheck");

            return clientHandler.execute(new ClientExecutionParams<CreateReadinessCheckRequest, CreateReadinessCheckResponse>()
                    .withOperationName("CreateReadinessCheck").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createReadinessCheckRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateReadinessCheckRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a recovery group in an account. A recovery group corresponds to an application and includes a list of the
     * cells that make up the application.
     * </p>
     *
     * @param createRecoveryGroupRequest
     * @return Result of the CreateRecoveryGroup operation returned by the service.
     * @throws ThrottlingException
     *         429 response - Limit exceeded exception or too many requests exception.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string, an input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - Internal service error or temporary service error. Retry the request.
     * @throws ConflictException
     *         409 response - Conflict exception. You might be using a predefined variable.
     * @throws AccessDeniedException
     *         403 response - Access denied exception. You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.CreateRecoveryGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/CreateRecoveryGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateRecoveryGroupResponse createRecoveryGroup(CreateRecoveryGroupRequest createRecoveryGroupRequest)
            throws ThrottlingException, ValidationException, InternalServerException, ConflictException, AccessDeniedException,
            AwsServiceException, SdkClientException, Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateRecoveryGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateRecoveryGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createRecoveryGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createRecoveryGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateRecoveryGroup");

            return clientHandler.execute(new ClientExecutionParams<CreateRecoveryGroupRequest, CreateRecoveryGroupResponse>()
                    .withOperationName("CreateRecoveryGroup").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createRecoveryGroupRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateRecoveryGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a resource set. A resource set is a set of resources of one type that span multiple cells. You can
     * associate a resource set with a readiness check to monitor the resources for failover readiness.
     * </p>
     *
     * @param createResourceSetRequest
     * @return Result of the CreateResourceSet operation returned by the service.
     * @throws ThrottlingException
     *         429 response - Limit exceeded exception or too many requests exception.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string, an input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - Internal service error or temporary service error. Retry the request.
     * @throws ConflictException
     *         409 response - Conflict exception. You might be using a predefined variable.
     * @throws AccessDeniedException
     *         403 response - Access denied exception. You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.CreateResourceSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/CreateResourceSet"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateResourceSetResponse createResourceSet(CreateResourceSetRequest createResourceSetRequest)
            throws ThrottlingException, ValidationException, InternalServerException, ConflictException, AccessDeniedException,
            AwsServiceException, SdkClientException, Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateResourceSetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateResourceSetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createResourceSetRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createResourceSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateResourceSet");

            return clientHandler.execute(new ClientExecutionParams<CreateResourceSetRequest, CreateResourceSetResponse>()
                    .withOperationName("CreateResourceSet").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createResourceSetRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateResourceSetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Delete a cell. When successful, the response code is 204, with no response body.
     * </p>
     *
     * @param deleteCellRequest
     * @return Result of the DeleteCell operation returned by the service.
     * @throws ResourceNotFoundException
     *         404 response - Malformed query string. The query string contains a syntax error or resource not found.
     * @throws ThrottlingException
     *         429 response - Limit exceeded exception or too many requests exception.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string, an input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - Internal service error or temporary service error. Retry the request.
     * @throws AccessDeniedException
     *         403 response - Access denied exception. You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.DeleteCell
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/DeleteCell"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteCellResponse deleteCell(DeleteCellRequest deleteCellRequest) throws ResourceNotFoundException,
            ThrottlingException, ValidationException, InternalServerException, AccessDeniedException, AwsServiceException,
            SdkClientException, Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteCellResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteCellResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteCellRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteCellRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteCell");

            return clientHandler.execute(new ClientExecutionParams<DeleteCellRequest, DeleteCellResponse>()
                    .withOperationName("DeleteCell").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(deleteCellRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteCellRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes cross account readiness authorization.
     * </p>
     *
     * @param deleteCrossAccountAuthorizationRequest
     * @return Result of the DeleteCrossAccountAuthorization operation returned by the service.
     * @throws ThrottlingException
     *         429 response - Limit exceeded exception or too many requests exception.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string, an input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - Internal service error or temporary service error. Retry the request.
     * @throws AccessDeniedException
     *         403 response - Access denied exception. You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.DeleteCrossAccountAuthorization
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/DeleteCrossAccountAuthorization"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteCrossAccountAuthorizationResponse deleteCrossAccountAuthorization(
            DeleteCrossAccountAuthorizationRequest deleteCrossAccountAuthorizationRequest) throws ThrottlingException,
            ValidationException, InternalServerException, AccessDeniedException, AwsServiceException, SdkClientException,
            Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteCrossAccountAuthorizationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteCrossAccountAuthorizationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteCrossAccountAuthorizationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteCrossAccountAuthorizationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteCrossAccountAuthorization");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteCrossAccountAuthorizationRequest, DeleteCrossAccountAuthorizationResponse>()
                            .withOperationName("DeleteCrossAccountAuthorization").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deleteCrossAccountAuthorizationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteCrossAccountAuthorizationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a readiness check.
     * </p>
     *
     * @param deleteReadinessCheckRequest
     * @return Result of the DeleteReadinessCheck operation returned by the service.
     * @throws ResourceNotFoundException
     *         404 response - Malformed query string. The query string contains a syntax error or resource not found.
     * @throws ThrottlingException
     *         429 response - Limit exceeded exception or too many requests exception.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string, an input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - Internal service error or temporary service error. Retry the request.
     * @throws AccessDeniedException
     *         403 response - Access denied exception. You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.DeleteReadinessCheck
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/DeleteReadinessCheck"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteReadinessCheckResponse deleteReadinessCheck(DeleteReadinessCheckRequest deleteReadinessCheckRequest)
            throws ResourceNotFoundException, ThrottlingException, ValidationException, InternalServerException,
            AccessDeniedException, AwsServiceException, SdkClientException, Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteReadinessCheckResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteReadinessCheckResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteReadinessCheckRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteReadinessCheckRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteReadinessCheck");

            return clientHandler.execute(new ClientExecutionParams<DeleteReadinessCheckRequest, DeleteReadinessCheckResponse>()
                    .withOperationName("DeleteReadinessCheck").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteReadinessCheckRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteReadinessCheckRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a recovery group.
     * </p>
     *
     * @param deleteRecoveryGroupRequest
     * @return Result of the DeleteRecoveryGroup operation returned by the service.
     * @throws ResourceNotFoundException
     *         404 response - Malformed query string. The query string contains a syntax error or resource not found.
     * @throws ThrottlingException
     *         429 response - Limit exceeded exception or too many requests exception.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string, an input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - Internal service error or temporary service error. Retry the request.
     * @throws AccessDeniedException
     *         403 response - Access denied exception. You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.DeleteRecoveryGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/DeleteRecoveryGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteRecoveryGroupResponse deleteRecoveryGroup(DeleteRecoveryGroupRequest deleteRecoveryGroupRequest)
            throws ResourceNotFoundException, ThrottlingException, ValidationException, InternalServerException,
            AccessDeniedException, AwsServiceException, SdkClientException, Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteRecoveryGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteRecoveryGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteRecoveryGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteRecoveryGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRecoveryGroup");

            return clientHandler.execute(new ClientExecutionParams<DeleteRecoveryGroupRequest, DeleteRecoveryGroupResponse>()
                    .withOperationName("DeleteRecoveryGroup").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteRecoveryGroupRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteRecoveryGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a resource set.
     * </p>
     *
     * @param deleteResourceSetRequest
     * @return Result of the DeleteResourceSet operation returned by the service.
     * @throws ResourceNotFoundException
     *         404 response - Malformed query string. The query string contains a syntax error or resource not found.
     * @throws ThrottlingException
     *         429 response - Limit exceeded exception or too many requests exception.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string, an input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - Internal service error or temporary service error. Retry the request.
     * @throws AccessDeniedException
     *         403 response - Access denied exception. You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.DeleteResourceSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/DeleteResourceSet"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteResourceSetResponse deleteResourceSet(DeleteResourceSetRequest deleteResourceSetRequest)
            throws ResourceNotFoundException, ThrottlingException, ValidationException, InternalServerException,
            AccessDeniedException, AwsServiceException, SdkClientException, Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteResourceSetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteResourceSetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteResourceSetRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteResourceSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteResourceSet");

            return clientHandler.execute(new ClientExecutionParams<DeleteResourceSetRequest, DeleteResourceSetResponse>()
                    .withOperationName("DeleteResourceSet").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteResourceSetRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteResourceSetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets recommendations about architecture designs for improving resiliency for an application, based on a recovery
     * group.
     * </p>
     *
     * @param getArchitectureRecommendationsRequest
     * @return Result of the GetArchitectureRecommendations operation returned by the service.
     * @throws ResourceNotFoundException
     *         404 response - Malformed query string. The query string contains a syntax error or resource not found.
     * @throws ThrottlingException
     *         429 response - Limit exceeded exception or too many requests exception.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string, an input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - Internal service error or temporary service error. Retry the request.
     * @throws AccessDeniedException
     *         403 response - Access denied exception. You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.GetArchitectureRecommendations
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/GetArchitectureRecommendations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetArchitectureRecommendationsResponse getArchitectureRecommendations(
            GetArchitectureRecommendationsRequest getArchitectureRecommendationsRequest) throws ResourceNotFoundException,
            ThrottlingException, ValidationException, InternalServerException, AccessDeniedException, AwsServiceException,
            SdkClientException, Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetArchitectureRecommendationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetArchitectureRecommendationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getArchitectureRecommendationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getArchitectureRecommendationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetArchitectureRecommendations");

            return clientHandler
                    .execute(new ClientExecutionParams<GetArchitectureRecommendationsRequest, GetArchitectureRecommendationsResponse>()
                            .withOperationName("GetArchitectureRecommendations").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getArchitectureRecommendationsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetArchitectureRecommendationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about a cell including cell name, cell Amazon Resource Name (ARN), ARNs of nested cells for this
     * cell, and a list of those cell ARNs with their associated recovery group ARNs.
     * </p>
     *
     * @param getCellRequest
     * @return Result of the GetCell operation returned by the service.
     * @throws ResourceNotFoundException
     *         404 response - Malformed query string. The query string contains a syntax error or resource not found.
     * @throws ThrottlingException
     *         429 response - Limit exceeded exception or too many requests exception.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string, an input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - Internal service error or temporary service error. Retry the request.
     * @throws AccessDeniedException
     *         403 response - Access denied exception. You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.GetCell
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/GetCell"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetCellResponse getCell(GetCellRequest getCellRequest) throws ResourceNotFoundException, ThrottlingException,
            ValidationException, InternalServerException, AccessDeniedException, AwsServiceException, SdkClientException,
            Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetCellResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetCellResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getCellRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getCellRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetCell");

            return clientHandler.execute(new ClientExecutionParams<GetCellRequest, GetCellResponse>()
                    .withOperationName("GetCell").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getCellRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetCellRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets readiness for a cell. Aggregates the readiness of all the resources that are associated with the cell into a
     * single value.
     * </p>
     *
     * @param getCellReadinessSummaryRequest
     * @return Result of the GetCellReadinessSummary operation returned by the service.
     * @throws ResourceNotFoundException
     *         404 response - Malformed query string. The query string contains a syntax error or resource not found.
     * @throws ThrottlingException
     *         429 response - Limit exceeded exception or too many requests exception.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string, an input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - Internal service error or temporary service error. Retry the request.
     * @throws AccessDeniedException
     *         403 response - Access denied exception. You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.GetCellReadinessSummary
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/GetCellReadinessSummary"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetCellReadinessSummaryResponse getCellReadinessSummary(GetCellReadinessSummaryRequest getCellReadinessSummaryRequest)
            throws ResourceNotFoundException, ThrottlingException, ValidationException, InternalServerException,
            AccessDeniedException, AwsServiceException, SdkClientException, Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetCellReadinessSummaryResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetCellReadinessSummaryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getCellReadinessSummaryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getCellReadinessSummaryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetCellReadinessSummary");

            return clientHandler
                    .execute(new ClientExecutionParams<GetCellReadinessSummaryRequest, GetCellReadinessSummaryResponse>()
                            .withOperationName("GetCellReadinessSummary").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getCellReadinessSummaryRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetCellReadinessSummaryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets details about a readiness check.
     * </p>
     *
     * @param getReadinessCheckRequest
     * @return Result of the GetReadinessCheck operation returned by the service.
     * @throws ResourceNotFoundException
     *         404 response - Malformed query string. The query string contains a syntax error or resource not found.
     * @throws ThrottlingException
     *         429 response - Limit exceeded exception or too many requests exception.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string, an input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - Internal service error or temporary service error. Retry the request.
     * @throws AccessDeniedException
     *         403 response - Access denied exception. You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.GetReadinessCheck
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/GetReadinessCheck"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetReadinessCheckResponse getReadinessCheck(GetReadinessCheckRequest getReadinessCheckRequest)
            throws ResourceNotFoundException, ThrottlingException, ValidationException, InternalServerException,
            AccessDeniedException, AwsServiceException, SdkClientException, Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetReadinessCheckResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetReadinessCheckResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getReadinessCheckRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getReadinessCheckRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetReadinessCheck");

            return clientHandler.execute(new ClientExecutionParams<GetReadinessCheckRequest, GetReadinessCheckResponse>()
                    .withOperationName("GetReadinessCheck").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getReadinessCheckRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetReadinessCheckRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets individual readiness status for a readiness check. To see the overall readiness status for a recovery group,
     * that considers the readiness status for all the readiness checks in the recovery group, use
     * GetRecoveryGroupReadinessSummary.
     * </p>
     *
     * @param getReadinessCheckResourceStatusRequest
     * @return Result of the GetReadinessCheckResourceStatus operation returned by the service.
     * @throws ResourceNotFoundException
     *         404 response - Malformed query string. The query string contains a syntax error or resource not found.
     * @throws ThrottlingException
     *         429 response - Limit exceeded exception or too many requests exception.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string, an input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - Internal service error or temporary service error. Retry the request.
     * @throws AccessDeniedException
     *         403 response - Access denied exception. You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.GetReadinessCheckResourceStatus
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/GetReadinessCheckResourceStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetReadinessCheckResourceStatusResponse getReadinessCheckResourceStatus(
            GetReadinessCheckResourceStatusRequest getReadinessCheckResourceStatusRequest) throws ResourceNotFoundException,
            ThrottlingException, ValidationException, InternalServerException, AccessDeniedException, AwsServiceException,
            SdkClientException, Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetReadinessCheckResourceStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetReadinessCheckResourceStatusResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getReadinessCheckResourceStatusRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getReadinessCheckResourceStatusRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetReadinessCheckResourceStatus");

            return clientHandler
                    .execute(new ClientExecutionParams<GetReadinessCheckResourceStatusRequest, GetReadinessCheckResourceStatusResponse>()
                            .withOperationName("GetReadinessCheckResourceStatus").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getReadinessCheckResourceStatusRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetReadinessCheckResourceStatusRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the readiness status for an individual readiness check. To see the overall readiness status for a recovery
     * group, that considers the readiness status for all the readiness checks in a recovery group, use
     * GetRecoveryGroupReadinessSummary.
     * </p>
     *
     * @param getReadinessCheckStatusRequest
     * @return Result of the GetReadinessCheckStatus operation returned by the service.
     * @throws ResourceNotFoundException
     *         404 response - Malformed query string. The query string contains a syntax error or resource not found.
     * @throws ThrottlingException
     *         429 response - Limit exceeded exception or too many requests exception.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string, an input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - Internal service error or temporary service error. Retry the request.
     * @throws AccessDeniedException
     *         403 response - Access denied exception. You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.GetReadinessCheckStatus
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/GetReadinessCheckStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetReadinessCheckStatusResponse getReadinessCheckStatus(GetReadinessCheckStatusRequest getReadinessCheckStatusRequest)
            throws ResourceNotFoundException, ThrottlingException, ValidationException, InternalServerException,
            AccessDeniedException, AwsServiceException, SdkClientException, Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetReadinessCheckStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetReadinessCheckStatusResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getReadinessCheckStatusRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getReadinessCheckStatusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetReadinessCheckStatus");

            return clientHandler
                    .execute(new ClientExecutionParams<GetReadinessCheckStatusRequest, GetReadinessCheckStatusResponse>()
                            .withOperationName("GetReadinessCheckStatus").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getReadinessCheckStatusRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetReadinessCheckStatusRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets details about a recovery group, including a list of the cells that are included in it.
     * </p>
     *
     * @param getRecoveryGroupRequest
     * @return Result of the GetRecoveryGroup operation returned by the service.
     * @throws ResourceNotFoundException
     *         404 response - Malformed query string. The query string contains a syntax error or resource not found.
     * @throws ThrottlingException
     *         429 response - Limit exceeded exception or too many requests exception.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string, an input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - Internal service error or temporary service error. Retry the request.
     * @throws AccessDeniedException
     *         403 response - Access denied exception. You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.GetRecoveryGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/GetRecoveryGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetRecoveryGroupResponse getRecoveryGroup(GetRecoveryGroupRequest getRecoveryGroupRequest)
            throws ResourceNotFoundException, ThrottlingException, ValidationException, InternalServerException,
            AccessDeniedException, AwsServiceException, SdkClientException, Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetRecoveryGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetRecoveryGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getRecoveryGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRecoveryGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRecoveryGroup");

            return clientHandler.execute(new ClientExecutionParams<GetRecoveryGroupRequest, GetRecoveryGroupResponse>()
                    .withOperationName("GetRecoveryGroup").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getRecoveryGroupRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetRecoveryGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Displays a summary of information about a recovery group's readiness status. Includes the readiness checks for
     * resources in the recovery group and the readiness status of each one.
     * </p>
     *
     * @param getRecoveryGroupReadinessSummaryRequest
     * @return Result of the GetRecoveryGroupReadinessSummary operation returned by the service.
     * @throws ResourceNotFoundException
     *         404 response - Malformed query string. The query string contains a syntax error or resource not found.
     * @throws ThrottlingException
     *         429 response - Limit exceeded exception or too many requests exception.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string, an input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - Internal service error or temporary service error. Retry the request.
     * @throws AccessDeniedException
     *         403 response - Access denied exception. You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.GetRecoveryGroupReadinessSummary
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/GetRecoveryGroupReadinessSummary"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetRecoveryGroupReadinessSummaryResponse getRecoveryGroupReadinessSummary(
            GetRecoveryGroupReadinessSummaryRequest getRecoveryGroupReadinessSummaryRequest) throws ResourceNotFoundException,
            ThrottlingException, ValidationException, InternalServerException, AccessDeniedException, AwsServiceException,
            SdkClientException, Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetRecoveryGroupReadinessSummaryResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetRecoveryGroupReadinessSummaryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getRecoveryGroupReadinessSummaryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getRecoveryGroupReadinessSummaryRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRecoveryGroupReadinessSummary");

            return clientHandler
                    .execute(new ClientExecutionParams<GetRecoveryGroupReadinessSummaryRequest, GetRecoveryGroupReadinessSummaryResponse>()
                            .withOperationName("GetRecoveryGroupReadinessSummary").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getRecoveryGroupReadinessSummaryRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetRecoveryGroupReadinessSummaryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Displays the details about a resource set, including a list of the resources in the set.
     * </p>
     *
     * @param getResourceSetRequest
     * @return Result of the GetResourceSet operation returned by the service.
     * @throws ResourceNotFoundException
     *         404 response - Malformed query string. The query string contains a syntax error or resource not found.
     * @throws ThrottlingException
     *         429 response - Limit exceeded exception or too many requests exception.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string, an input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - Internal service error or temporary service error. Retry the request.
     * @throws AccessDeniedException
     *         403 response - Access denied exception. You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.GetResourceSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/GetResourceSet"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetResourceSetResponse getResourceSet(GetResourceSetRequest getResourceSetRequest) throws ResourceNotFoundException,
            ThrottlingException, ValidationException, InternalServerException, AccessDeniedException, AwsServiceException,
            SdkClientException, Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetResourceSetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetResourceSetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getResourceSetRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourceSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResourceSet");

            return clientHandler.execute(new ClientExecutionParams<GetResourceSetRequest, GetResourceSetResponse>()
                    .withOperationName("GetResourceSet").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getResourceSetRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetResourceSetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the cells for an account.
     * </p>
     *
     * @param listCellsRequest
     * @return Result of the ListCells operation returned by the service.
     * @throws ThrottlingException
     *         429 response - Limit exceeded exception or too many requests exception.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string, an input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - Internal service error or temporary service error. Retry the request.
     * @throws AccessDeniedException
     *         403 response - Access denied exception. You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.ListCells
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/ListCells"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListCellsResponse listCells(ListCellsRequest listCellsRequest) throws ThrottlingException, ValidationException,
            InternalServerException, AccessDeniedException, AwsServiceException, SdkClientException,
            Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListCellsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListCellsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listCellsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listCellsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListCells");

            return clientHandler.execute(new ClientExecutionParams<ListCellsRequest, ListCellsResponse>()
                    .withOperationName("ListCells").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listCellsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListCellsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the cross-account readiness authorizations that are in place for an account.
     * </p>
     *
     * @param listCrossAccountAuthorizationsRequest
     * @return Result of the ListCrossAccountAuthorizations operation returned by the service.
     * @throws ThrottlingException
     *         429 response - Limit exceeded exception or too many requests exception.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string, an input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - Internal service error or temporary service error. Retry the request.
     * @throws AccessDeniedException
     *         403 response - Access denied exception. You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.ListCrossAccountAuthorizations
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/ListCrossAccountAuthorizations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListCrossAccountAuthorizationsResponse listCrossAccountAuthorizations(
            ListCrossAccountAuthorizationsRequest listCrossAccountAuthorizationsRequest) throws ThrottlingException,
            ValidationException, InternalServerException, AccessDeniedException, AwsServiceException, SdkClientException,
            Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListCrossAccountAuthorizationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListCrossAccountAuthorizationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listCrossAccountAuthorizationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listCrossAccountAuthorizationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListCrossAccountAuthorizations");

            return clientHandler
                    .execute(new ClientExecutionParams<ListCrossAccountAuthorizationsRequest, ListCrossAccountAuthorizationsResponse>()
                            .withOperationName("ListCrossAccountAuthorizations").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listCrossAccountAuthorizationsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListCrossAccountAuthorizationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the readiness checks for an account.
     * </p>
     *
     * @param listReadinessChecksRequest
     * @return Result of the ListReadinessChecks operation returned by the service.
     * @throws ThrottlingException
     *         429 response - Limit exceeded exception or too many requests exception.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string, an input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - Internal service error or temporary service error. Retry the request.
     * @throws AccessDeniedException
     *         403 response - Access denied exception. You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.ListReadinessChecks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/ListReadinessChecks"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListReadinessChecksResponse listReadinessChecks(ListReadinessChecksRequest listReadinessChecksRequest)
            throws ThrottlingException, ValidationException, InternalServerException, AccessDeniedException, AwsServiceException,
            SdkClientException, Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListReadinessChecksResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListReadinessChecksResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listReadinessChecksRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listReadinessChecksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListReadinessChecks");

            return clientHandler.execute(new ClientExecutionParams<ListReadinessChecksRequest, ListReadinessChecksResponse>()
                    .withOperationName("ListReadinessChecks").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listReadinessChecksRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListReadinessChecksRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the recovery groups in an account.
     * </p>
     *
     * @param listRecoveryGroupsRequest
     * @return Result of the ListRecoveryGroups operation returned by the service.
     * @throws ThrottlingException
     *         429 response - Limit exceeded exception or too many requests exception.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string, an input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - Internal service error or temporary service error. Retry the request.
     * @throws AccessDeniedException
     *         403 response - Access denied exception. You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.ListRecoveryGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/ListRecoveryGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListRecoveryGroupsResponse listRecoveryGroups(ListRecoveryGroupsRequest listRecoveryGroupsRequest)
            throws ThrottlingException, ValidationException, InternalServerException, AccessDeniedException, AwsServiceException,
            SdkClientException, Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListRecoveryGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListRecoveryGroupsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listRecoveryGroupsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRecoveryGroupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRecoveryGroups");

            return clientHandler.execute(new ClientExecutionParams<ListRecoveryGroupsRequest, ListRecoveryGroupsResponse>()
                    .withOperationName("ListRecoveryGroups").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listRecoveryGroupsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListRecoveryGroupsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the resource sets in an account.
     * </p>
     *
     * @param listResourceSetsRequest
     * @return Result of the ListResourceSets operation returned by the service.
     * @throws ThrottlingException
     *         429 response - Limit exceeded exception or too many requests exception.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string, an input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - Internal service error or temporary service error. Retry the request.
     * @throws AccessDeniedException
     *         403 response - Access denied exception. You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.ListResourceSets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/ListResourceSets"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListResourceSetsResponse listResourceSets(ListResourceSetsRequest listResourceSetsRequest) throws ThrottlingException,
            ValidationException, InternalServerException, AccessDeniedException, AwsServiceException, SdkClientException,
            Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListResourceSetsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListResourceSetsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listResourceSetsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listResourceSetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListResourceSets");

            return clientHandler.execute(new ClientExecutionParams<ListResourceSetsRequest, ListResourceSetsResponse>()
                    .withOperationName("ListResourceSets").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listResourceSetsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListResourceSetsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all readiness rules, or lists the readiness rules for a specific resource type.
     * </p>
     *
     * @param listRulesRequest
     * @return Result of the ListRules operation returned by the service.
     * @throws ThrottlingException
     *         429 response - Limit exceeded exception or too many requests exception.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string, an input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - Internal service error or temporary service error. Retry the request.
     * @throws AccessDeniedException
     *         403 response - Access denied exception. You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.ListRules
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/ListRules"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListRulesResponse listRules(ListRulesRequest listRulesRequest) throws ThrottlingException, ValidationException,
            InternalServerException, AccessDeniedException, AwsServiceException, SdkClientException,
            Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListRulesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListRulesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listRulesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRulesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRules");

            return clientHandler.execute(new ClientExecutionParams<ListRulesRequest, ListRulesResponse>()
                    .withOperationName("ListRules").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listRulesRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListRulesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the tags for a resource.
     * </p>
     *
     * @param listTagsForResourcesRequest
     * @return Result of the ListTagsForResources operation returned by the service.
     * @throws ResourceNotFoundException
     *         404 response - Malformed query string. The query string contains a syntax error or resource not found.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string, an input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - Internal service error or temporary service error. Retry the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.ListTagsForResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/ListTagsForResources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourcesResponse listTagsForResources(ListTagsForResourcesRequest listTagsForResourcesRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourcesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourcesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResources");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourcesRequest, ListTagsForResourcesResponse>()
                    .withOperationName("ListTagsForResources").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourcesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourcesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds a tag to a resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         404 response - Malformed query string. The query string contains a syntax error or resource not found.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string, an input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - Internal service error or temporary service error. Retry the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/TagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, AwsServiceException, SdkClientException,
            Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes a tag from a resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         404 response - Malformed query string. The query string contains a syntax error or resource not found.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string, an input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - Internal service error or temporary service error. Retry the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, AwsServiceException, SdkClientException,
            Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a cell to replace the list of nested cells with a new list of nested cells.
     * </p>
     *
     * @param updateCellRequest
     * @return Result of the UpdateCell operation returned by the service.
     * @throws ResourceNotFoundException
     *         404 response - Malformed query string. The query string contains a syntax error or resource not found.
     * @throws ThrottlingException
     *         429 response - Limit exceeded exception or too many requests exception.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string, an input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - Internal service error or temporary service error. Retry the request.
     * @throws AccessDeniedException
     *         403 response - Access denied exception. You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.UpdateCell
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/UpdateCell"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateCellResponse updateCell(UpdateCellRequest updateCellRequest) throws ResourceNotFoundException,
            ThrottlingException, ValidationException, InternalServerException, AccessDeniedException, AwsServiceException,
            SdkClientException, Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateCellResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateCellResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateCellRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateCellRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateCell");

            return clientHandler.execute(new ClientExecutionParams<UpdateCellRequest, UpdateCellResponse>()
                    .withOperationName("UpdateCell").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(updateCellRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateCellRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a readiness check.
     * </p>
     *
     * @param updateReadinessCheckRequest
     *        Name of a readiness check to describe.
     * @return Result of the UpdateReadinessCheck operation returned by the service.
     * @throws ResourceNotFoundException
     *         404 response - Malformed query string. The query string contains a syntax error or resource not found.
     * @throws ThrottlingException
     *         429 response - Limit exceeded exception or too many requests exception.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string, an input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - Internal service error or temporary service error. Retry the request.
     * @throws AccessDeniedException
     *         403 response - Access denied exception. You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.UpdateReadinessCheck
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/UpdateReadinessCheck"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateReadinessCheckResponse updateReadinessCheck(UpdateReadinessCheckRequest updateReadinessCheckRequest)
            throws ResourceNotFoundException, ThrottlingException, ValidationException, InternalServerException,
            AccessDeniedException, AwsServiceException, SdkClientException, Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateReadinessCheckResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateReadinessCheckResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateReadinessCheckRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateReadinessCheckRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateReadinessCheck");

            return clientHandler.execute(new ClientExecutionParams<UpdateReadinessCheckRequest, UpdateReadinessCheckResponse>()
                    .withOperationName("UpdateReadinessCheck").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateReadinessCheckRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateReadinessCheckRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a recovery group.
     * </p>
     *
     * @param updateRecoveryGroupRequest
     *        Name of a recovery group.
     * @return Result of the UpdateRecoveryGroup operation returned by the service.
     * @throws ResourceNotFoundException
     *         404 response - Malformed query string. The query string contains a syntax error or resource not found.
     * @throws ThrottlingException
     *         429 response - Limit exceeded exception or too many requests exception.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string, an input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - Internal service error or temporary service error. Retry the request.
     * @throws AccessDeniedException
     *         403 response - Access denied exception. You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.UpdateRecoveryGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/UpdateRecoveryGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateRecoveryGroupResponse updateRecoveryGroup(UpdateRecoveryGroupRequest updateRecoveryGroupRequest)
            throws ResourceNotFoundException, ThrottlingException, ValidationException, InternalServerException,
            AccessDeniedException, AwsServiceException, SdkClientException, Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateRecoveryGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateRecoveryGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateRecoveryGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateRecoveryGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateRecoveryGroup");

            return clientHandler.execute(new ClientExecutionParams<UpdateRecoveryGroupRequest, UpdateRecoveryGroupResponse>()
                    .withOperationName("UpdateRecoveryGroup").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateRecoveryGroupRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateRecoveryGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a resource set.
     * </p>
     *
     * @param updateResourceSetRequest
     *        Name of a resource set.
     * @return Result of the UpdateResourceSet operation returned by the service.
     * @throws ResourceNotFoundException
     *         404 response - Malformed query string. The query string contains a syntax error or resource not found.
     * @throws ThrottlingException
     *         429 response - Limit exceeded exception or too many requests exception.
     * @throws ValidationException
     *         400 response - Multiple causes. For example, you might have a malformed query string, an input parameter
     *         might be out of range, or you used parameters together incorrectly.
     * @throws InternalServerException
     *         500 response - Internal service error or temporary service error. Retry the request.
     * @throws AccessDeniedException
     *         403 response - Access denied exception. You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.UpdateResourceSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/UpdateResourceSet"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateResourceSetResponse updateResourceSet(UpdateResourceSetRequest updateResourceSetRequest)
            throws ResourceNotFoundException, ThrottlingException, ValidationException, InternalServerException,
            AccessDeniedException, AwsServiceException, SdkClientException, Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateResourceSetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateResourceSetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateResourceSetRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateResourceSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateResourceSet");

            return clientHandler.execute(new ClientExecutionParams<UpdateResourceSetRequest, UpdateResourceSetResponse>()
                    .withOperationName("UpdateResourceSet").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateResourceSetRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateResourceSetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        Route53RecoveryReadinessServiceClientConfigurationBuilder serviceConfigBuilder = new Route53RecoveryReadinessServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(Route53RecoveryReadinessException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    @Override
    public final Route53RecoveryReadinessServiceClientConfiguration serviceClientConfiguration() {
        return new Route53RecoveryReadinessServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
