/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.route53recoveryreadiness;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.route53recoveryreadiness.internal.Route53RecoveryReadinessServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.route53recoveryreadiness.model.AccessDeniedException;
import software.amazon.awssdk.services.route53recoveryreadiness.model.ConflictException;
import software.amazon.awssdk.services.route53recoveryreadiness.model.CreateCellRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.CreateCellResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.CreateCrossAccountAuthorizationRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.CreateCrossAccountAuthorizationResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.CreateReadinessCheckRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.CreateReadinessCheckResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.CreateRecoveryGroupRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.CreateRecoveryGroupResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.CreateResourceSetRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.CreateResourceSetResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.DeleteCellRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.DeleteCellResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.DeleteCrossAccountAuthorizationRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.DeleteCrossAccountAuthorizationResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.DeleteReadinessCheckRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.DeleteReadinessCheckResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.DeleteRecoveryGroupRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.DeleteRecoveryGroupResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.DeleteResourceSetRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.DeleteResourceSetResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.GetArchitectureRecommendationsRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.GetArchitectureRecommendationsResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.GetCellReadinessSummaryRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.GetCellReadinessSummaryResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.GetCellRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.GetCellResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.GetReadinessCheckRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.GetReadinessCheckResourceStatusRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.GetReadinessCheckResourceStatusResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.GetReadinessCheckResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.GetReadinessCheckStatusRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.GetReadinessCheckStatusResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.GetRecoveryGroupReadinessSummaryRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.GetRecoveryGroupReadinessSummaryResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.GetRecoveryGroupRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.GetRecoveryGroupResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.GetResourceSetRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.GetResourceSetResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.InternalServerException;
import software.amazon.awssdk.services.route53recoveryreadiness.model.ListCellsRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.ListCellsResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.ListCrossAccountAuthorizationsRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.ListCrossAccountAuthorizationsResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.ListReadinessChecksRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.ListReadinessChecksResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.ListRecoveryGroupsRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.ListRecoveryGroupsResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.ListResourceSetsRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.ListResourceSetsResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.ListRulesRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.ListRulesResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.ListTagsForResourcesRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.ListTagsForResourcesResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.ResourceNotFoundException;
import software.amazon.awssdk.services.route53recoveryreadiness.model.Route53RecoveryReadinessException;
import software.amazon.awssdk.services.route53recoveryreadiness.model.TagResourceRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.TagResourceResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.ThrottlingException;
import software.amazon.awssdk.services.route53recoveryreadiness.model.UntagResourceRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.UntagResourceResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.UpdateCellRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.UpdateCellResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.UpdateReadinessCheckRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.UpdateReadinessCheckResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.UpdateRecoveryGroupRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.UpdateRecoveryGroupResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.UpdateResourceSetRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.UpdateResourceSetResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.ValidationException;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.CreateCellRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.CreateCrossAccountAuthorizationRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.CreateReadinessCheckRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.CreateRecoveryGroupRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.CreateResourceSetRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.DeleteCellRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.DeleteCrossAccountAuthorizationRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.DeleteReadinessCheckRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.DeleteRecoveryGroupRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.DeleteResourceSetRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.GetArchitectureRecommendationsRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.GetCellReadinessSummaryRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.GetCellRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.GetReadinessCheckRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.GetReadinessCheckResourceStatusRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.GetReadinessCheckStatusRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.GetRecoveryGroupReadinessSummaryRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.GetRecoveryGroupRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.GetResourceSetRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.ListCellsRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.ListCrossAccountAuthorizationsRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.ListReadinessChecksRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.ListRecoveryGroupsRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.ListResourceSetsRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.ListRulesRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.ListTagsForResourcesRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.UpdateCellRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.UpdateReadinessCheckRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.UpdateRecoveryGroupRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.UpdateResourceSetRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link Route53RecoveryReadinessAsyncClient}.
 *
 * @see Route53RecoveryReadinessAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultRoute53RecoveryReadinessAsyncClient implements Route53RecoveryReadinessAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultRoute53RecoveryReadinessAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultRoute53RecoveryReadinessAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a cell in an account.
     * </p>
     *
     * @param createCellRequest
     * @return A Java Future containing the result of the CreateCell operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException 429 response - Limit exceeded exception or too many requests exception.</li>
     *         <li>ValidationException 400 response - Multiple causes. For example, you might have a malformed query
     *         string, an input parameter might be out of range, or you used parameters together incorrectly.</li>
     *         <li>InternalServerException 500 response - Internal service error or temporary service error. Retry the
     *         request.</li>
     *         <li>ConflictException 409 response - Conflict exception. You might be using a predefined variable.</li>
     *         <li>AccessDeniedException 403 response - Access denied exception. You do not have sufficient access to
     *         perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryReadinessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryReadinessAsyncClient.CreateCell
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/CreateCell"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateCellResponse> createCell(CreateCellRequest createCellRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createCellRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createCellRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateCell");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateCellResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateCellResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateCellResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateCellRequest, CreateCellResponse>().withOperationName("CreateCell")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateCellRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createCellRequest));
            CompletableFuture<CreateCellResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a cross-account readiness authorization. This lets you authorize another account to work with Route 53
     * Application Recovery Controller, for example, to check the readiness status of resources in a separate account.
     * </p>
     *
     * @param createCrossAccountAuthorizationRequest
     * @return A Java Future containing the result of the CreateCrossAccountAuthorization operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException 429 response - Limit exceeded exception or too many requests exception.</li>
     *         <li>ValidationException 400 response - Multiple causes. For example, you might have a malformed query
     *         string, an input parameter might be out of range, or you used parameters together incorrectly.</li>
     *         <li>InternalServerException 500 response - Internal service error or temporary service error. Retry the
     *         request.</li>
     *         <li>ConflictException 409 response - Conflict exception. You might be using a predefined variable.</li>
     *         <li>AccessDeniedException 403 response - Access denied exception. You do not have sufficient access to
     *         perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryReadinessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryReadinessAsyncClient.CreateCrossAccountAuthorization
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/CreateCrossAccountAuthorization"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateCrossAccountAuthorizationResponse> createCrossAccountAuthorization(
            CreateCrossAccountAuthorizationRequest createCrossAccountAuthorizationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createCrossAccountAuthorizationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createCrossAccountAuthorizationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateCrossAccountAuthorization");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateCrossAccountAuthorizationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateCrossAccountAuthorizationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateCrossAccountAuthorizationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateCrossAccountAuthorizationRequest, CreateCrossAccountAuthorizationResponse>()
                            .withOperationName("CreateCrossAccountAuthorization").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateCrossAccountAuthorizationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createCrossAccountAuthorizationRequest));
            CompletableFuture<CreateCrossAccountAuthorizationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a readiness check in an account. A readiness check monitors a resource set in your application, such as a
     * set of Amazon Aurora instances, that Application Recovery Controller is auditing recovery readiness for. The
     * audits run once every minute on every resource that's associated with a readiness check.
     * </p>
     *
     * @param createReadinessCheckRequest
     * @return A Java Future containing the result of the CreateReadinessCheck operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException 429 response - Limit exceeded exception or too many requests exception.</li>
     *         <li>ValidationException 400 response - Multiple causes. For example, you might have a malformed query
     *         string, an input parameter might be out of range, or you used parameters together incorrectly.</li>
     *         <li>InternalServerException 500 response - Internal service error or temporary service error. Retry the
     *         request.</li>
     *         <li>ConflictException 409 response - Conflict exception. You might be using a predefined variable.</li>
     *         <li>AccessDeniedException 403 response - Access denied exception. You do not have sufficient access to
     *         perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryReadinessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryReadinessAsyncClient.CreateReadinessCheck
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/CreateReadinessCheck"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateReadinessCheckResponse> createReadinessCheck(
            CreateReadinessCheckRequest createReadinessCheckRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createReadinessCheckRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createReadinessCheckRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateReadinessCheck");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateReadinessCheckResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateReadinessCheckResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateReadinessCheckResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateReadinessCheckRequest, CreateReadinessCheckResponse>()
                            .withOperationName("CreateReadinessCheck").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateReadinessCheckRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createReadinessCheckRequest));
            CompletableFuture<CreateReadinessCheckResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a recovery group in an account. A recovery group corresponds to an application and includes a list of the
     * cells that make up the application.
     * </p>
     *
     * @param createRecoveryGroupRequest
     * @return A Java Future containing the result of the CreateRecoveryGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException 429 response - Limit exceeded exception or too many requests exception.</li>
     *         <li>ValidationException 400 response - Multiple causes. For example, you might have a malformed query
     *         string, an input parameter might be out of range, or you used parameters together incorrectly.</li>
     *         <li>InternalServerException 500 response - Internal service error or temporary service error. Retry the
     *         request.</li>
     *         <li>ConflictException 409 response - Conflict exception. You might be using a predefined variable.</li>
     *         <li>AccessDeniedException 403 response - Access denied exception. You do not have sufficient access to
     *         perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryReadinessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryReadinessAsyncClient.CreateRecoveryGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/CreateRecoveryGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateRecoveryGroupResponse> createRecoveryGroup(
            CreateRecoveryGroupRequest createRecoveryGroupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createRecoveryGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createRecoveryGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateRecoveryGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateRecoveryGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateRecoveryGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateRecoveryGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateRecoveryGroupRequest, CreateRecoveryGroupResponse>()
                            .withOperationName("CreateRecoveryGroup").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateRecoveryGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createRecoveryGroupRequest));
            CompletableFuture<CreateRecoveryGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a resource set. A resource set is a set of resources of one type that span multiple cells. You can
     * associate a resource set with a readiness check to monitor the resources for failover readiness.
     * </p>
     *
     * @param createResourceSetRequest
     * @return A Java Future containing the result of the CreateResourceSet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException 429 response - Limit exceeded exception or too many requests exception.</li>
     *         <li>ValidationException 400 response - Multiple causes. For example, you might have a malformed query
     *         string, an input parameter might be out of range, or you used parameters together incorrectly.</li>
     *         <li>InternalServerException 500 response - Internal service error or temporary service error. Retry the
     *         request.</li>
     *         <li>ConflictException 409 response - Conflict exception. You might be using a predefined variable.</li>
     *         <li>AccessDeniedException 403 response - Access denied exception. You do not have sufficient access to
     *         perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryReadinessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryReadinessAsyncClient.CreateResourceSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/CreateResourceSet"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateResourceSetResponse> createResourceSet(CreateResourceSetRequest createResourceSetRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createResourceSetRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createResourceSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateResourceSet");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateResourceSetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateResourceSetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateResourceSetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateResourceSetRequest, CreateResourceSetResponse>()
                            .withOperationName("CreateResourceSet").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateResourceSetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createResourceSetRequest));
            CompletableFuture<CreateResourceSetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Delete a cell. When successful, the response code is 204, with no response body.
     * </p>
     *
     * @param deleteCellRequest
     * @return A Java Future containing the result of the DeleteCell operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException 404 response - Malformed query string. The query string contains a syntax
     *         error or resource not found.</li>
     *         <li>ThrottlingException 429 response - Limit exceeded exception or too many requests exception.</li>
     *         <li>ValidationException 400 response - Multiple causes. For example, you might have a malformed query
     *         string, an input parameter might be out of range, or you used parameters together incorrectly.</li>
     *         <li>InternalServerException 500 response - Internal service error or temporary service error. Retry the
     *         request.</li>
     *         <li>AccessDeniedException 403 response - Access denied exception. You do not have sufficient access to
     *         perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryReadinessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryReadinessAsyncClient.DeleteCell
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/DeleteCell"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteCellResponse> deleteCell(DeleteCellRequest deleteCellRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteCellRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteCellRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteCell");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteCellResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteCellResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteCellResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteCellRequest, DeleteCellResponse>().withOperationName("DeleteCell")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteCellRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteCellRequest));
            CompletableFuture<DeleteCellResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes cross account readiness authorization.
     * </p>
     *
     * @param deleteCrossAccountAuthorizationRequest
     * @return A Java Future containing the result of the DeleteCrossAccountAuthorization operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException 429 response - Limit exceeded exception or too many requests exception.</li>
     *         <li>ValidationException 400 response - Multiple causes. For example, you might have a malformed query
     *         string, an input parameter might be out of range, or you used parameters together incorrectly.</li>
     *         <li>InternalServerException 500 response - Internal service error or temporary service error. Retry the
     *         request.</li>
     *         <li>AccessDeniedException 403 response - Access denied exception. You do not have sufficient access to
     *         perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryReadinessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryReadinessAsyncClient.DeleteCrossAccountAuthorization
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/DeleteCrossAccountAuthorization"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteCrossAccountAuthorizationResponse> deleteCrossAccountAuthorization(
            DeleteCrossAccountAuthorizationRequest deleteCrossAccountAuthorizationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteCrossAccountAuthorizationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteCrossAccountAuthorizationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteCrossAccountAuthorization");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteCrossAccountAuthorizationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteCrossAccountAuthorizationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteCrossAccountAuthorizationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteCrossAccountAuthorizationRequest, DeleteCrossAccountAuthorizationResponse>()
                            .withOperationName("DeleteCrossAccountAuthorization").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteCrossAccountAuthorizationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteCrossAccountAuthorizationRequest));
            CompletableFuture<DeleteCrossAccountAuthorizationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a readiness check.
     * </p>
     *
     * @param deleteReadinessCheckRequest
     * @return A Java Future containing the result of the DeleteReadinessCheck operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException 404 response - Malformed query string. The query string contains a syntax
     *         error or resource not found.</li>
     *         <li>ThrottlingException 429 response - Limit exceeded exception or too many requests exception.</li>
     *         <li>ValidationException 400 response - Multiple causes. For example, you might have a malformed query
     *         string, an input parameter might be out of range, or you used parameters together incorrectly.</li>
     *         <li>InternalServerException 500 response - Internal service error or temporary service error. Retry the
     *         request.</li>
     *         <li>AccessDeniedException 403 response - Access denied exception. You do not have sufficient access to
     *         perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryReadinessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryReadinessAsyncClient.DeleteReadinessCheck
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/DeleteReadinessCheck"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteReadinessCheckResponse> deleteReadinessCheck(
            DeleteReadinessCheckRequest deleteReadinessCheckRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteReadinessCheckRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteReadinessCheckRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteReadinessCheck");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteReadinessCheckResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteReadinessCheckResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteReadinessCheckResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteReadinessCheckRequest, DeleteReadinessCheckResponse>()
                            .withOperationName("DeleteReadinessCheck").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteReadinessCheckRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteReadinessCheckRequest));
            CompletableFuture<DeleteReadinessCheckResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a recovery group.
     * </p>
     *
     * @param deleteRecoveryGroupRequest
     * @return A Java Future containing the result of the DeleteRecoveryGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException 404 response - Malformed query string. The query string contains a syntax
     *         error or resource not found.</li>
     *         <li>ThrottlingException 429 response - Limit exceeded exception or too many requests exception.</li>
     *         <li>ValidationException 400 response - Multiple causes. For example, you might have a malformed query
     *         string, an input parameter might be out of range, or you used parameters together incorrectly.</li>
     *         <li>InternalServerException 500 response - Internal service error or temporary service error. Retry the
     *         request.</li>
     *         <li>AccessDeniedException 403 response - Access denied exception. You do not have sufficient access to
     *         perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryReadinessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryReadinessAsyncClient.DeleteRecoveryGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/DeleteRecoveryGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteRecoveryGroupResponse> deleteRecoveryGroup(
            DeleteRecoveryGroupRequest deleteRecoveryGroupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteRecoveryGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteRecoveryGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRecoveryGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteRecoveryGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteRecoveryGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteRecoveryGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteRecoveryGroupRequest, DeleteRecoveryGroupResponse>()
                            .withOperationName("DeleteRecoveryGroup").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteRecoveryGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteRecoveryGroupRequest));
            CompletableFuture<DeleteRecoveryGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a resource set.
     * </p>
     *
     * @param deleteResourceSetRequest
     * @return A Java Future containing the result of the DeleteResourceSet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException 404 response - Malformed query string. The query string contains a syntax
     *         error or resource not found.</li>
     *         <li>ThrottlingException 429 response - Limit exceeded exception or too many requests exception.</li>
     *         <li>ValidationException 400 response - Multiple causes. For example, you might have a malformed query
     *         string, an input parameter might be out of range, or you used parameters together incorrectly.</li>
     *         <li>InternalServerException 500 response - Internal service error or temporary service error. Retry the
     *         request.</li>
     *         <li>AccessDeniedException 403 response - Access denied exception. You do not have sufficient access to
     *         perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryReadinessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryReadinessAsyncClient.DeleteResourceSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/DeleteResourceSet"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteResourceSetResponse> deleteResourceSet(DeleteResourceSetRequest deleteResourceSetRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteResourceSetRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteResourceSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteResourceSet");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteResourceSetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteResourceSetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteResourceSetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteResourceSetRequest, DeleteResourceSetResponse>()
                            .withOperationName("DeleteResourceSet").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteResourceSetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteResourceSetRequest));
            CompletableFuture<DeleteResourceSetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets recommendations about architecture designs for improving resiliency for an application, based on a recovery
     * group.
     * </p>
     *
     * @param getArchitectureRecommendationsRequest
     * @return A Java Future containing the result of the GetArchitectureRecommendations operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException 404 response - Malformed query string. The query string contains a syntax
     *         error or resource not found.</li>
     *         <li>ThrottlingException 429 response - Limit exceeded exception or too many requests exception.</li>
     *         <li>ValidationException 400 response - Multiple causes. For example, you might have a malformed query
     *         string, an input parameter might be out of range, or you used parameters together incorrectly.</li>
     *         <li>InternalServerException 500 response - Internal service error or temporary service error. Retry the
     *         request.</li>
     *         <li>AccessDeniedException 403 response - Access denied exception. You do not have sufficient access to
     *         perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryReadinessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryReadinessAsyncClient.GetArchitectureRecommendations
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/GetArchitectureRecommendations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetArchitectureRecommendationsResponse> getArchitectureRecommendations(
            GetArchitectureRecommendationsRequest getArchitectureRecommendationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getArchitectureRecommendationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getArchitectureRecommendationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetArchitectureRecommendations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetArchitectureRecommendationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetArchitectureRecommendationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetArchitectureRecommendationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetArchitectureRecommendationsRequest, GetArchitectureRecommendationsResponse>()
                            .withOperationName("GetArchitectureRecommendations").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetArchitectureRecommendationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getArchitectureRecommendationsRequest));
            CompletableFuture<GetArchitectureRecommendationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about a cell including cell name, cell Amazon Resource Name (ARN), ARNs of nested cells for this
     * cell, and a list of those cell ARNs with their associated recovery group ARNs.
     * </p>
     *
     * @param getCellRequest
     * @return A Java Future containing the result of the GetCell operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException 404 response - Malformed query string. The query string contains a syntax
     *         error or resource not found.</li>
     *         <li>ThrottlingException 429 response - Limit exceeded exception or too many requests exception.</li>
     *         <li>ValidationException 400 response - Multiple causes. For example, you might have a malformed query
     *         string, an input parameter might be out of range, or you used parameters together incorrectly.</li>
     *         <li>InternalServerException 500 response - Internal service error or temporary service error. Retry the
     *         request.</li>
     *         <li>AccessDeniedException 403 response - Access denied exception. You do not have sufficient access to
     *         perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryReadinessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryReadinessAsyncClient.GetCell
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/GetCell"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetCellResponse> getCell(GetCellRequest getCellRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getCellRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getCellRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetCell");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetCellResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetCellResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetCellResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetCellRequest, GetCellResponse>().withOperationName("GetCell")
                            .withProtocolMetadata(protocolMetadata).withMarshaller(new GetCellRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getCellRequest));
            CompletableFuture<GetCellResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets readiness for a cell. Aggregates the readiness of all the resources that are associated with the cell into a
     * single value.
     * </p>
     *
     * @param getCellReadinessSummaryRequest
     * @return A Java Future containing the result of the GetCellReadinessSummary operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException 404 response - Malformed query string. The query string contains a syntax
     *         error or resource not found.</li>
     *         <li>ThrottlingException 429 response - Limit exceeded exception or too many requests exception.</li>
     *         <li>ValidationException 400 response - Multiple causes. For example, you might have a malformed query
     *         string, an input parameter might be out of range, or you used parameters together incorrectly.</li>
     *         <li>InternalServerException 500 response - Internal service error or temporary service error. Retry the
     *         request.</li>
     *         <li>AccessDeniedException 403 response - Access denied exception. You do not have sufficient access to
     *         perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryReadinessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryReadinessAsyncClient.GetCellReadinessSummary
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/GetCellReadinessSummary"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetCellReadinessSummaryResponse> getCellReadinessSummary(
            GetCellReadinessSummaryRequest getCellReadinessSummaryRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getCellReadinessSummaryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getCellReadinessSummaryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetCellReadinessSummary");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetCellReadinessSummaryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetCellReadinessSummaryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetCellReadinessSummaryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetCellReadinessSummaryRequest, GetCellReadinessSummaryResponse>()
                            .withOperationName("GetCellReadinessSummary").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetCellReadinessSummaryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getCellReadinessSummaryRequest));
            CompletableFuture<GetCellReadinessSummaryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets details about a readiness check.
     * </p>
     *
     * @param getReadinessCheckRequest
     * @return A Java Future containing the result of the GetReadinessCheck operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException 404 response - Malformed query string. The query string contains a syntax
     *         error or resource not found.</li>
     *         <li>ThrottlingException 429 response - Limit exceeded exception or too many requests exception.</li>
     *         <li>ValidationException 400 response - Multiple causes. For example, you might have a malformed query
     *         string, an input parameter might be out of range, or you used parameters together incorrectly.</li>
     *         <li>InternalServerException 500 response - Internal service error or temporary service error. Retry the
     *         request.</li>
     *         <li>AccessDeniedException 403 response - Access denied exception. You do not have sufficient access to
     *         perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryReadinessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryReadinessAsyncClient.GetReadinessCheck
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/GetReadinessCheck"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetReadinessCheckResponse> getReadinessCheck(GetReadinessCheckRequest getReadinessCheckRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getReadinessCheckRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getReadinessCheckRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetReadinessCheck");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetReadinessCheckResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetReadinessCheckResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetReadinessCheckResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetReadinessCheckRequest, GetReadinessCheckResponse>()
                            .withOperationName("GetReadinessCheck").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetReadinessCheckRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getReadinessCheckRequest));
            CompletableFuture<GetReadinessCheckResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets individual readiness status for a readiness check. To see the overall readiness status for a recovery group,
     * that considers the readiness status for all the readiness checks in the recovery group, use
     * GetRecoveryGroupReadinessSummary.
     * </p>
     *
     * @param getReadinessCheckResourceStatusRequest
     * @return A Java Future containing the result of the GetReadinessCheckResourceStatus operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException 404 response - Malformed query string. The query string contains a syntax
     *         error or resource not found.</li>
     *         <li>ThrottlingException 429 response - Limit exceeded exception or too many requests exception.</li>
     *         <li>ValidationException 400 response - Multiple causes. For example, you might have a malformed query
     *         string, an input parameter might be out of range, or you used parameters together incorrectly.</li>
     *         <li>InternalServerException 500 response - Internal service error or temporary service error. Retry the
     *         request.</li>
     *         <li>AccessDeniedException 403 response - Access denied exception. You do not have sufficient access to
     *         perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryReadinessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryReadinessAsyncClient.GetReadinessCheckResourceStatus
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/GetReadinessCheckResourceStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetReadinessCheckResourceStatusResponse> getReadinessCheckResourceStatus(
            GetReadinessCheckResourceStatusRequest getReadinessCheckResourceStatusRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getReadinessCheckResourceStatusRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getReadinessCheckResourceStatusRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetReadinessCheckResourceStatus");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetReadinessCheckResourceStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetReadinessCheckResourceStatusResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetReadinessCheckResourceStatusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetReadinessCheckResourceStatusRequest, GetReadinessCheckResourceStatusResponse>()
                            .withOperationName("GetReadinessCheckResourceStatus").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetReadinessCheckResourceStatusRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getReadinessCheckResourceStatusRequest));
            CompletableFuture<GetReadinessCheckResourceStatusResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the readiness status for an individual readiness check. To see the overall readiness status for a recovery
     * group, that considers the readiness status for all the readiness checks in a recovery group, use
     * GetRecoveryGroupReadinessSummary.
     * </p>
     *
     * @param getReadinessCheckStatusRequest
     * @return A Java Future containing the result of the GetReadinessCheckStatus operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException 404 response - Malformed query string. The query string contains a syntax
     *         error or resource not found.</li>
     *         <li>ThrottlingException 429 response - Limit exceeded exception or too many requests exception.</li>
     *         <li>ValidationException 400 response - Multiple causes. For example, you might have a malformed query
     *         string, an input parameter might be out of range, or you used parameters together incorrectly.</li>
     *         <li>InternalServerException 500 response - Internal service error or temporary service error. Retry the
     *         request.</li>
     *         <li>AccessDeniedException 403 response - Access denied exception. You do not have sufficient access to
     *         perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryReadinessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryReadinessAsyncClient.GetReadinessCheckStatus
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/GetReadinessCheckStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetReadinessCheckStatusResponse> getReadinessCheckStatus(
            GetReadinessCheckStatusRequest getReadinessCheckStatusRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getReadinessCheckStatusRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getReadinessCheckStatusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetReadinessCheckStatus");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetReadinessCheckStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetReadinessCheckStatusResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetReadinessCheckStatusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetReadinessCheckStatusRequest, GetReadinessCheckStatusResponse>()
                            .withOperationName("GetReadinessCheckStatus").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetReadinessCheckStatusRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getReadinessCheckStatusRequest));
            CompletableFuture<GetReadinessCheckStatusResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets details about a recovery group, including a list of the cells that are included in it.
     * </p>
     *
     * @param getRecoveryGroupRequest
     * @return A Java Future containing the result of the GetRecoveryGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException 404 response - Malformed query string. The query string contains a syntax
     *         error or resource not found.</li>
     *         <li>ThrottlingException 429 response - Limit exceeded exception or too many requests exception.</li>
     *         <li>ValidationException 400 response - Multiple causes. For example, you might have a malformed query
     *         string, an input parameter might be out of range, or you used parameters together incorrectly.</li>
     *         <li>InternalServerException 500 response - Internal service error or temporary service error. Retry the
     *         request.</li>
     *         <li>AccessDeniedException 403 response - Access denied exception. You do not have sufficient access to
     *         perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryReadinessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryReadinessAsyncClient.GetRecoveryGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/GetRecoveryGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetRecoveryGroupResponse> getRecoveryGroup(GetRecoveryGroupRequest getRecoveryGroupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getRecoveryGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRecoveryGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRecoveryGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetRecoveryGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetRecoveryGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetRecoveryGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetRecoveryGroupRequest, GetRecoveryGroupResponse>()
                            .withOperationName("GetRecoveryGroup").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetRecoveryGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getRecoveryGroupRequest));
            CompletableFuture<GetRecoveryGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Displays a summary of information about a recovery group's readiness status. Includes the readiness checks for
     * resources in the recovery group and the readiness status of each one.
     * </p>
     *
     * @param getRecoveryGroupReadinessSummaryRequest
     * @return A Java Future containing the result of the GetRecoveryGroupReadinessSummary operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException 404 response - Malformed query string. The query string contains a syntax
     *         error or resource not found.</li>
     *         <li>ThrottlingException 429 response - Limit exceeded exception or too many requests exception.</li>
     *         <li>ValidationException 400 response - Multiple causes. For example, you might have a malformed query
     *         string, an input parameter might be out of range, or you used parameters together incorrectly.</li>
     *         <li>InternalServerException 500 response - Internal service error or temporary service error. Retry the
     *         request.</li>
     *         <li>AccessDeniedException 403 response - Access denied exception. You do not have sufficient access to
     *         perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryReadinessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryReadinessAsyncClient.GetRecoveryGroupReadinessSummary
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/GetRecoveryGroupReadinessSummary"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetRecoveryGroupReadinessSummaryResponse> getRecoveryGroupReadinessSummary(
            GetRecoveryGroupReadinessSummaryRequest getRecoveryGroupReadinessSummaryRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getRecoveryGroupReadinessSummaryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getRecoveryGroupReadinessSummaryRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRecoveryGroupReadinessSummary");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetRecoveryGroupReadinessSummaryResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, GetRecoveryGroupReadinessSummaryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetRecoveryGroupReadinessSummaryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetRecoveryGroupReadinessSummaryRequest, GetRecoveryGroupReadinessSummaryResponse>()
                            .withOperationName("GetRecoveryGroupReadinessSummary").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetRecoveryGroupReadinessSummaryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getRecoveryGroupReadinessSummaryRequest));
            CompletableFuture<GetRecoveryGroupReadinessSummaryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Displays the details about a resource set, including a list of the resources in the set.
     * </p>
     *
     * @param getResourceSetRequest
     * @return A Java Future containing the result of the GetResourceSet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException 404 response - Malformed query string. The query string contains a syntax
     *         error or resource not found.</li>
     *         <li>ThrottlingException 429 response - Limit exceeded exception or too many requests exception.</li>
     *         <li>ValidationException 400 response - Multiple causes. For example, you might have a malformed query
     *         string, an input parameter might be out of range, or you used parameters together incorrectly.</li>
     *         <li>InternalServerException 500 response - Internal service error or temporary service error. Retry the
     *         request.</li>
     *         <li>AccessDeniedException 403 response - Access denied exception. You do not have sufficient access to
     *         perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryReadinessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryReadinessAsyncClient.GetResourceSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/GetResourceSet"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetResourceSetResponse> getResourceSet(GetResourceSetRequest getResourceSetRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getResourceSetRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourceSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResourceSet");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetResourceSetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetResourceSetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetResourceSetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetResourceSetRequest, GetResourceSetResponse>()
                            .withOperationName("GetResourceSet").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetResourceSetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getResourceSetRequest));
            CompletableFuture<GetResourceSetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the cells for an account.
     * </p>
     *
     * @param listCellsRequest
     * @return A Java Future containing the result of the ListCells operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException 429 response - Limit exceeded exception or too many requests exception.</li>
     *         <li>ValidationException 400 response - Multiple causes. For example, you might have a malformed query
     *         string, an input parameter might be out of range, or you used parameters together incorrectly.</li>
     *         <li>InternalServerException 500 response - Internal service error or temporary service error. Retry the
     *         request.</li>
     *         <li>AccessDeniedException 403 response - Access denied exception. You do not have sufficient access to
     *         perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryReadinessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryReadinessAsyncClient.ListCells
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/ListCells"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListCellsResponse> listCells(ListCellsRequest listCellsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listCellsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listCellsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListCells");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListCellsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListCellsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListCellsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListCellsRequest, ListCellsResponse>().withOperationName("ListCells")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListCellsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(listCellsRequest));
            CompletableFuture<ListCellsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the cross-account readiness authorizations that are in place for an account.
     * </p>
     *
     * @param listCrossAccountAuthorizationsRequest
     * @return A Java Future containing the result of the ListCrossAccountAuthorizations operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException 429 response - Limit exceeded exception or too many requests exception.</li>
     *         <li>ValidationException 400 response - Multiple causes. For example, you might have a malformed query
     *         string, an input parameter might be out of range, or you used parameters together incorrectly.</li>
     *         <li>InternalServerException 500 response - Internal service error or temporary service error. Retry the
     *         request.</li>
     *         <li>AccessDeniedException 403 response - Access denied exception. You do not have sufficient access to
     *         perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryReadinessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryReadinessAsyncClient.ListCrossAccountAuthorizations
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/ListCrossAccountAuthorizations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListCrossAccountAuthorizationsResponse> listCrossAccountAuthorizations(
            ListCrossAccountAuthorizationsRequest listCrossAccountAuthorizationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listCrossAccountAuthorizationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listCrossAccountAuthorizationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListCrossAccountAuthorizations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListCrossAccountAuthorizationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListCrossAccountAuthorizationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListCrossAccountAuthorizationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListCrossAccountAuthorizationsRequest, ListCrossAccountAuthorizationsResponse>()
                            .withOperationName("ListCrossAccountAuthorizations").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListCrossAccountAuthorizationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listCrossAccountAuthorizationsRequest));
            CompletableFuture<ListCrossAccountAuthorizationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the readiness checks for an account.
     * </p>
     *
     * @param listReadinessChecksRequest
     * @return A Java Future containing the result of the ListReadinessChecks operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException 429 response - Limit exceeded exception or too many requests exception.</li>
     *         <li>ValidationException 400 response - Multiple causes. For example, you might have a malformed query
     *         string, an input parameter might be out of range, or you used parameters together incorrectly.</li>
     *         <li>InternalServerException 500 response - Internal service error or temporary service error. Retry the
     *         request.</li>
     *         <li>AccessDeniedException 403 response - Access denied exception. You do not have sufficient access to
     *         perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryReadinessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryReadinessAsyncClient.ListReadinessChecks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/ListReadinessChecks"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListReadinessChecksResponse> listReadinessChecks(
            ListReadinessChecksRequest listReadinessChecksRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listReadinessChecksRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listReadinessChecksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListReadinessChecks");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListReadinessChecksResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListReadinessChecksResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListReadinessChecksResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListReadinessChecksRequest, ListReadinessChecksResponse>()
                            .withOperationName("ListReadinessChecks").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListReadinessChecksRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listReadinessChecksRequest));
            CompletableFuture<ListReadinessChecksResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the recovery groups in an account.
     * </p>
     *
     * @param listRecoveryGroupsRequest
     * @return A Java Future containing the result of the ListRecoveryGroups operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException 429 response - Limit exceeded exception or too many requests exception.</li>
     *         <li>ValidationException 400 response - Multiple causes. For example, you might have a malformed query
     *         string, an input parameter might be out of range, or you used parameters together incorrectly.</li>
     *         <li>InternalServerException 500 response - Internal service error or temporary service error. Retry the
     *         request.</li>
     *         <li>AccessDeniedException 403 response - Access denied exception. You do not have sufficient access to
     *         perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryReadinessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryReadinessAsyncClient.ListRecoveryGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/ListRecoveryGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListRecoveryGroupsResponse> listRecoveryGroups(ListRecoveryGroupsRequest listRecoveryGroupsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listRecoveryGroupsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRecoveryGroupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRecoveryGroups");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListRecoveryGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListRecoveryGroupsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListRecoveryGroupsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListRecoveryGroupsRequest, ListRecoveryGroupsResponse>()
                            .withOperationName("ListRecoveryGroups").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListRecoveryGroupsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listRecoveryGroupsRequest));
            CompletableFuture<ListRecoveryGroupsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the resource sets in an account.
     * </p>
     *
     * @param listResourceSetsRequest
     * @return A Java Future containing the result of the ListResourceSets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException 429 response - Limit exceeded exception or too many requests exception.</li>
     *         <li>ValidationException 400 response - Multiple causes. For example, you might have a malformed query
     *         string, an input parameter might be out of range, or you used parameters together incorrectly.</li>
     *         <li>InternalServerException 500 response - Internal service error or temporary service error. Retry the
     *         request.</li>
     *         <li>AccessDeniedException 403 response - Access denied exception. You do not have sufficient access to
     *         perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryReadinessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryReadinessAsyncClient.ListResourceSets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/ListResourceSets"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListResourceSetsResponse> listResourceSets(ListResourceSetsRequest listResourceSetsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listResourceSetsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listResourceSetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListResourceSets");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListResourceSetsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListResourceSetsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListResourceSetsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListResourceSetsRequest, ListResourceSetsResponse>()
                            .withOperationName("ListResourceSets").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListResourceSetsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listResourceSetsRequest));
            CompletableFuture<ListResourceSetsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all readiness rules, or lists the readiness rules for a specific resource type.
     * </p>
     *
     * @param listRulesRequest
     * @return A Java Future containing the result of the ListRules operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException 429 response - Limit exceeded exception or too many requests exception.</li>
     *         <li>ValidationException 400 response - Multiple causes. For example, you might have a malformed query
     *         string, an input parameter might be out of range, or you used parameters together incorrectly.</li>
     *         <li>InternalServerException 500 response - Internal service error or temporary service error. Retry the
     *         request.</li>
     *         <li>AccessDeniedException 403 response - Access denied exception. You do not have sufficient access to
     *         perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryReadinessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryReadinessAsyncClient.ListRules
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/ListRules"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListRulesResponse> listRules(ListRulesRequest listRulesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listRulesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRulesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRules");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListRulesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListRulesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListRulesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListRulesRequest, ListRulesResponse>().withOperationName("ListRules")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListRulesRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(listRulesRequest));
            CompletableFuture<ListRulesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the tags for a resource.
     * </p>
     *
     * @param listTagsForResourcesRequest
     * @return A Java Future containing the result of the ListTagsForResources operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException 404 response - Malformed query string. The query string contains a syntax
     *         error or resource not found.</li>
     *         <li>ValidationException 400 response - Multiple causes. For example, you might have a malformed query
     *         string, an input parameter might be out of range, or you used parameters together incorrectly.</li>
     *         <li>InternalServerException 500 response - Internal service error or temporary service error. Retry the
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryReadinessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryReadinessAsyncClient.ListTagsForResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/ListTagsForResources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourcesResponse> listTagsForResources(
            ListTagsForResourcesRequest listTagsForResourcesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourcesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResources");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourcesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourcesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourcesRequest, ListTagsForResourcesResponse>()
                            .withOperationName("ListTagsForResources").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourcesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourcesRequest));
            CompletableFuture<ListTagsForResourcesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds a tag to a resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException 404 response - Malformed query string. The query string contains a syntax
     *         error or resource not found.</li>
     *         <li>ValidationException 400 response - Multiple causes. For example, you might have a malformed query
     *         string, an input parameter might be out of range, or you used parameters together incorrectly.</li>
     *         <li>InternalServerException 500 response - Internal service error or temporary service error. Retry the
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryReadinessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryReadinessAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/TagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes a tag from a resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException 404 response - Malformed query string. The query string contains a syntax
     *         error or resource not found.</li>
     *         <li>ValidationException 400 response - Multiple causes. For example, you might have a malformed query
     *         string, an input parameter might be out of range, or you used parameters together incorrectly.</li>
     *         <li>InternalServerException 500 response - Internal service error or temporary service error. Retry the
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryReadinessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryReadinessAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a cell to replace the list of nested cells with a new list of nested cells.
     * </p>
     *
     * @param updateCellRequest
     * @return A Java Future containing the result of the UpdateCell operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException 404 response - Malformed query string. The query string contains a syntax
     *         error or resource not found.</li>
     *         <li>ThrottlingException 429 response - Limit exceeded exception or too many requests exception.</li>
     *         <li>ValidationException 400 response - Multiple causes. For example, you might have a malformed query
     *         string, an input parameter might be out of range, or you used parameters together incorrectly.</li>
     *         <li>InternalServerException 500 response - Internal service error or temporary service error. Retry the
     *         request.</li>
     *         <li>AccessDeniedException 403 response - Access denied exception. You do not have sufficient access to
     *         perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryReadinessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryReadinessAsyncClient.UpdateCell
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/UpdateCell"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateCellResponse> updateCell(UpdateCellRequest updateCellRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateCellRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateCellRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateCell");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateCellResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateCellResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateCellResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateCellRequest, UpdateCellResponse>().withOperationName("UpdateCell")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateCellRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateCellRequest));
            CompletableFuture<UpdateCellResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a readiness check.
     * </p>
     *
     * @param updateReadinessCheckRequest
     *        Name of a readiness check to describe.
     * @return A Java Future containing the result of the UpdateReadinessCheck operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException 404 response - Malformed query string. The query string contains a syntax
     *         error or resource not found.</li>
     *         <li>ThrottlingException 429 response - Limit exceeded exception or too many requests exception.</li>
     *         <li>ValidationException 400 response - Multiple causes. For example, you might have a malformed query
     *         string, an input parameter might be out of range, or you used parameters together incorrectly.</li>
     *         <li>InternalServerException 500 response - Internal service error or temporary service error. Retry the
     *         request.</li>
     *         <li>AccessDeniedException 403 response - Access denied exception. You do not have sufficient access to
     *         perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryReadinessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryReadinessAsyncClient.UpdateReadinessCheck
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/UpdateReadinessCheck"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateReadinessCheckResponse> updateReadinessCheck(
            UpdateReadinessCheckRequest updateReadinessCheckRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateReadinessCheckRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateReadinessCheckRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateReadinessCheck");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateReadinessCheckResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateReadinessCheckResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateReadinessCheckResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateReadinessCheckRequest, UpdateReadinessCheckResponse>()
                            .withOperationName("UpdateReadinessCheck").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateReadinessCheckRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateReadinessCheckRequest));
            CompletableFuture<UpdateReadinessCheckResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a recovery group.
     * </p>
     *
     * @param updateRecoveryGroupRequest
     *        Name of a recovery group.
     * @return A Java Future containing the result of the UpdateRecoveryGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException 404 response - Malformed query string. The query string contains a syntax
     *         error or resource not found.</li>
     *         <li>ThrottlingException 429 response - Limit exceeded exception or too many requests exception.</li>
     *         <li>ValidationException 400 response - Multiple causes. For example, you might have a malformed query
     *         string, an input parameter might be out of range, or you used parameters together incorrectly.</li>
     *         <li>InternalServerException 500 response - Internal service error or temporary service error. Retry the
     *         request.</li>
     *         <li>AccessDeniedException 403 response - Access denied exception. You do not have sufficient access to
     *         perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryReadinessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryReadinessAsyncClient.UpdateRecoveryGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/UpdateRecoveryGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateRecoveryGroupResponse> updateRecoveryGroup(
            UpdateRecoveryGroupRequest updateRecoveryGroupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateRecoveryGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateRecoveryGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateRecoveryGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateRecoveryGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateRecoveryGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateRecoveryGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateRecoveryGroupRequest, UpdateRecoveryGroupResponse>()
                            .withOperationName("UpdateRecoveryGroup").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateRecoveryGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateRecoveryGroupRequest));
            CompletableFuture<UpdateRecoveryGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a resource set.
     * </p>
     *
     * @param updateResourceSetRequest
     *        Name of a resource set.
     * @return A Java Future containing the result of the UpdateResourceSet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException 404 response - Malformed query string. The query string contains a syntax
     *         error or resource not found.</li>
     *         <li>ThrottlingException 429 response - Limit exceeded exception or too many requests exception.</li>
     *         <li>ValidationException 400 response - Multiple causes. For example, you might have a malformed query
     *         string, an input parameter might be out of range, or you used parameters together incorrectly.</li>
     *         <li>InternalServerException 500 response - Internal service error or temporary service error. Retry the
     *         request.</li>
     *         <li>AccessDeniedException 403 response - Access denied exception. You do not have sufficient access to
     *         perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53RecoveryReadinessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53RecoveryReadinessAsyncClient.UpdateResourceSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/UpdateResourceSet"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateResourceSetResponse> updateResourceSet(UpdateResourceSetRequest updateResourceSetRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateResourceSetRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateResourceSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateResourceSet");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateResourceSetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateResourceSetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateResourceSetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateResourceSetRequest, UpdateResourceSetResponse>()
                            .withOperationName("UpdateResourceSet").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateResourceSetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateResourceSetRequest));
            CompletableFuture<UpdateResourceSetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final Route53RecoveryReadinessServiceClientConfiguration serviceClientConfiguration() {
        return new Route53RecoveryReadinessServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(Route53RecoveryReadinessException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        Route53RecoveryReadinessServiceClientConfigurationBuilder serviceConfigBuilder = new Route53RecoveryReadinessServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
