/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.route53resolver.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateResolverEndpointRequest extends Route53ResolverRequest implements
        ToCopyableBuilder<CreateResolverEndpointRequest.Builder, CreateResolverEndpointRequest> {
    private static final SdkField<String> CREATOR_REQUEST_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CreatorRequestId").getter(getter(CreateResolverEndpointRequest::creatorRequestId))
            .setter(setter(Builder::creatorRequestId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatorRequestId").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(CreateResolverEndpointRequest::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<List<String>> SECURITY_GROUP_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SecurityGroupIds")
            .getter(getter(CreateResolverEndpointRequest::securityGroupIds))
            .setter(setter(Builder::securityGroupIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecurityGroupIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> DIRECTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Direction").getter(getter(CreateResolverEndpointRequest::directionAsString))
            .setter(setter(Builder::direction))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Direction").build()).build();

    private static final SdkField<List<IpAddressRequest>> IP_ADDRESSES_FIELD = SdkField
            .<List<IpAddressRequest>> builder(MarshallingType.LIST)
            .memberName("IpAddresses")
            .getter(getter(CreateResolverEndpointRequest::ipAddresses))
            .setter(setter(Builder::ipAddresses))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpAddresses").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<IpAddressRequest> builder(MarshallingType.SDK_POJO)
                                            .constructor(IpAddressRequest::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> OUTPOST_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OutpostArn").getter(getter(CreateResolverEndpointRequest::outpostArn))
            .setter(setter(Builder::outpostArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OutpostArn").build()).build();

    private static final SdkField<String> PREFERRED_INSTANCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PreferredInstanceType").getter(getter(CreateResolverEndpointRequest::preferredInstanceType))
            .setter(setter(Builder::preferredInstanceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PreferredInstanceType").build())
            .build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(CreateResolverEndpointRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> RESOLVER_ENDPOINT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResolverEndpointType").getter(getter(CreateResolverEndpointRequest::resolverEndpointTypeAsString))
            .setter(setter(Builder::resolverEndpointType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResolverEndpointType").build())
            .build();

    private static final SdkField<List<String>> PROTOCOLS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Protocols")
            .getter(getter(CreateResolverEndpointRequest::protocolsAsStrings))
            .setter(setter(Builder::protocolsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Protocols").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CREATOR_REQUEST_ID_FIELD,
            NAME_FIELD, SECURITY_GROUP_IDS_FIELD, DIRECTION_FIELD, IP_ADDRESSES_FIELD, OUTPOST_ARN_FIELD,
            PREFERRED_INSTANCE_TYPE_FIELD, TAGS_FIELD, RESOLVER_ENDPOINT_TYPE_FIELD, PROTOCOLS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private final String creatorRequestId;

    private final String name;

    private final List<String> securityGroupIds;

    private final String direction;

    private final List<IpAddressRequest> ipAddresses;

    private final String outpostArn;

    private final String preferredInstanceType;

    private final List<Tag> tags;

    private final String resolverEndpointType;

    private final List<String> protocols;

    private CreateResolverEndpointRequest(BuilderImpl builder) {
        super(builder);
        this.creatorRequestId = builder.creatorRequestId;
        this.name = builder.name;
        this.securityGroupIds = builder.securityGroupIds;
        this.direction = builder.direction;
        this.ipAddresses = builder.ipAddresses;
        this.outpostArn = builder.outpostArn;
        this.preferredInstanceType = builder.preferredInstanceType;
        this.tags = builder.tags;
        this.resolverEndpointType = builder.resolverEndpointType;
        this.protocols = builder.protocols;
    }

    /**
     * <p>
     * A unique string that identifies the request and that allows failed requests to be retried without the risk of
     * running the operation twice. <code>CreatorRequestId</code> can be any unique string, for example, a date/time
     * stamp.
     * </p>
     * 
     * @return A unique string that identifies the request and that allows failed requests to be retried without the
     *         risk of running the operation twice. <code>CreatorRequestId</code> can be any unique string, for example,
     *         a date/time stamp.
     */
    public final String creatorRequestId() {
        return creatorRequestId;
    }

    /**
     * <p>
     * A friendly name that lets you easily find a configuration in the Resolver dashboard in the Route 53 console.
     * </p>
     * 
     * @return A friendly name that lets you easily find a configuration in the Resolver dashboard in the Route 53
     *         console.
     */
    public final String name() {
        return name;
    }

    /**
     * For responses, this returns true if the service returned a value for the SecurityGroupIds property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSecurityGroupIds() {
        return securityGroupIds != null && !(securityGroupIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The ID of one or more security groups that you want to use to control access to this VPC. The security group that
     * you specify must include one or more inbound rules (for inbound Resolver endpoints) or outbound rules (for
     * outbound Resolver endpoints). Inbound and outbound rules must allow TCP and UDP access. For inbound access, open
     * port 53. For outbound access, open the port that you're using for DNS queries on your network.
     * </p>
     * <p>
     * Some security group rules will cause your connection to be tracked. For outbound resolver endpoint, it can
     * potentially impact the maximum queries per second from outbound endpoint to your target name server. For inbound
     * resolver endpoint, it can bring down the overall maximum queries per second per IP address to as low as 1500. To
     * avoid connection tracking caused by security group, see <a href=
     * "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/security-group-connection-tracking.html#untracked-connectionsl"
     * >Untracked connections</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSecurityGroupIds} method.
     * </p>
     * 
     * @return The ID of one or more security groups that you want to use to control access to this VPC. The security
     *         group that you specify must include one or more inbound rules (for inbound Resolver endpoints) or
     *         outbound rules (for outbound Resolver endpoints). Inbound and outbound rules must allow TCP and UDP
     *         access. For inbound access, open port 53. For outbound access, open the port that you're using for DNS
     *         queries on your network.</p>
     *         <p>
     *         Some security group rules will cause your connection to be tracked. For outbound resolver endpoint, it
     *         can potentially impact the maximum queries per second from outbound endpoint to your target name server.
     *         For inbound resolver endpoint, it can bring down the overall maximum queries per second per IP address to
     *         as low as 1500. To avoid connection tracking caused by security group, see <a href=
     *         "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/security-group-connection-tracking.html#untracked-connectionsl"
     *         >Untracked connections</a>.
     */
    public final List<String> securityGroupIds() {
        return securityGroupIds;
    }

    /**
     * <p>
     * Specify the applicable value:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>INBOUND</code>: Resolver forwards DNS queries to the DNS service for a VPC from your network.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>OUTBOUND</code>: Resolver forwards DNS queries from the DNS service for a VPC to your network.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>INBOUND_DELEGATION</code>: Resolver delegates queries to Route 53 private hosted zones from your network.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #direction} will
     * return {@link ResolverEndpointDirection#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #directionAsString}.
     * </p>
     * 
     * @return Specify the applicable value:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>INBOUND</code>: Resolver forwards DNS queries to the DNS service for a VPC from your network.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>OUTBOUND</code>: Resolver forwards DNS queries from the DNS service for a VPC to your network.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>INBOUND_DELEGATION</code>: Resolver delegates queries to Route 53 private hosted zones from your
     *         network.
     *         </p>
     *         </li>
     * @see ResolverEndpointDirection
     */
    public final ResolverEndpointDirection direction() {
        return ResolverEndpointDirection.fromValue(direction);
    }

    /**
     * <p>
     * Specify the applicable value:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>INBOUND</code>: Resolver forwards DNS queries to the DNS service for a VPC from your network.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>OUTBOUND</code>: Resolver forwards DNS queries from the DNS service for a VPC to your network.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>INBOUND_DELEGATION</code>: Resolver delegates queries to Route 53 private hosted zones from your network.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #direction} will
     * return {@link ResolverEndpointDirection#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #directionAsString}.
     * </p>
     * 
     * @return Specify the applicable value:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>INBOUND</code>: Resolver forwards DNS queries to the DNS service for a VPC from your network.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>OUTBOUND</code>: Resolver forwards DNS queries from the DNS service for a VPC to your network.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>INBOUND_DELEGATION</code>: Resolver delegates queries to Route 53 private hosted zones from your
     *         network.
     *         </p>
     *         </li>
     * @see ResolverEndpointDirection
     */
    public final String directionAsString() {
        return direction;
    }

    /**
     * For responses, this returns true if the service returned a value for the IpAddresses property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasIpAddresses() {
        return ipAddresses != null && !(ipAddresses instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The subnets and IP addresses in your VPC that DNS queries originate from (for outbound endpoints) or that you
     * forward DNS queries to (for inbound endpoints). The subnet ID uniquely identifies a VPC.
     * </p>
     * <note>
     * <p>
     * Even though the minimum is 1, Route 53 requires that you create at least two.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasIpAddresses} method.
     * </p>
     * 
     * @return The subnets and IP addresses in your VPC that DNS queries originate from (for outbound endpoints) or that
     *         you forward DNS queries to (for inbound endpoints). The subnet ID uniquely identifies a VPC. </p> <note>
     *         <p>
     *         Even though the minimum is 1, Route 53 requires that you create at least two.
     *         </p>
     */
    public final List<IpAddressRequest> ipAddresses() {
        return ipAddresses;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the Outpost. If you specify this, you must also specify a value for the
     * <code>PreferredInstanceType</code>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the Outpost. If you specify this, you must also specify a value for the
     *         <code>PreferredInstanceType</code>.
     */
    public final String outpostArn() {
        return outpostArn;
    }

    /**
     * <p>
     * The instance type. If you specify this, you must also specify a value for the <code>OutpostArn</code>.
     * </p>
     * 
     * @return The instance type. If you specify this, you must also specify a value for the <code>OutpostArn</code>.
     */
    public final String preferredInstanceType() {
        return preferredInstanceType;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of the tag keys and values that you want to associate with the endpoint.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return A list of the tag keys and values that you want to associate with the endpoint.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * For the endpoint type you can choose either IPv4, IPv6, or dual-stack. A dual-stack endpoint means that it will
     * resolve via both IPv4 and IPv6. This endpoint type is applied to all IP addresses.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #resolverEndpointType} will return {@link ResolverEndpointType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #resolverEndpointTypeAsString}.
     * </p>
     * 
     * @return For the endpoint type you can choose either IPv4, IPv6, or dual-stack. A dual-stack endpoint means that
     *         it will resolve via both IPv4 and IPv6. This endpoint type is applied to all IP addresses.
     * @see ResolverEndpointType
     */
    public final ResolverEndpointType resolverEndpointType() {
        return ResolverEndpointType.fromValue(resolverEndpointType);
    }

    /**
     * <p>
     * For the endpoint type you can choose either IPv4, IPv6, or dual-stack. A dual-stack endpoint means that it will
     * resolve via both IPv4 and IPv6. This endpoint type is applied to all IP addresses.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #resolverEndpointType} will return {@link ResolverEndpointType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #resolverEndpointTypeAsString}.
     * </p>
     * 
     * @return For the endpoint type you can choose either IPv4, IPv6, or dual-stack. A dual-stack endpoint means that
     *         it will resolve via both IPv4 and IPv6. This endpoint type is applied to all IP addresses.
     * @see ResolverEndpointType
     */
    public final String resolverEndpointTypeAsString() {
        return resolverEndpointType;
    }

    /**
     * <p>
     * The protocols you want to use for the endpoint. DoH-FIPS is applicable for default inbound endpoints only.
     * </p>
     * <p>
     * For a default inbound endpoint you can apply the protocols as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Do53 and DoH in combination.
     * </p>
     * </li>
     * <li>
     * <p>
     * Do53 and DoH-FIPS in combination.
     * </p>
     * </li>
     * <li>
     * <p>
     * Do53 alone.
     * </p>
     * </li>
     * <li>
     * <p>
     * DoH alone.
     * </p>
     * </li>
     * <li>
     * <p>
     * DoH-FIPS alone.
     * </p>
     * </li>
     * <li>
     * <p>
     * None, which is treated as Do53.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For a delegation inbound endpoint you can use Do53 only.
     * </p>
     * <p>
     * For an outbound endpoint you can apply the protocols as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Do53 and DoH in combination.
     * </p>
     * </li>
     * <li>
     * <p>
     * Do53 alone.
     * </p>
     * </li>
     * <li>
     * <p>
     * DoH alone.
     * </p>
     * </li>
     * <li>
     * <p>
     * None, which is treated as Do53.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasProtocols} method.
     * </p>
     * 
     * @return The protocols you want to use for the endpoint. DoH-FIPS is applicable for default inbound endpoints
     *         only. </p>
     *         <p>
     *         For a default inbound endpoint you can apply the protocols as follows:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Do53 and DoH in combination.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Do53 and DoH-FIPS in combination.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Do53 alone.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         DoH alone.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         DoH-FIPS alone.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         None, which is treated as Do53.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For a delegation inbound endpoint you can use Do53 only.
     *         </p>
     *         <p>
     *         For an outbound endpoint you can apply the protocols as follows:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Do53 and DoH in combination.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Do53 alone.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         DoH alone.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         None, which is treated as Do53.
     *         </p>
     *         </li>
     */
    public final List<Protocol> protocols() {
        return ProtocolListCopier.copyStringToEnum(protocols);
    }

    /**
     * For responses, this returns true if the service returned a value for the Protocols property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasProtocols() {
        return protocols != null && !(protocols instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The protocols you want to use for the endpoint. DoH-FIPS is applicable for default inbound endpoints only.
     * </p>
     * <p>
     * For a default inbound endpoint you can apply the protocols as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Do53 and DoH in combination.
     * </p>
     * </li>
     * <li>
     * <p>
     * Do53 and DoH-FIPS in combination.
     * </p>
     * </li>
     * <li>
     * <p>
     * Do53 alone.
     * </p>
     * </li>
     * <li>
     * <p>
     * DoH alone.
     * </p>
     * </li>
     * <li>
     * <p>
     * DoH-FIPS alone.
     * </p>
     * </li>
     * <li>
     * <p>
     * None, which is treated as Do53.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For a delegation inbound endpoint you can use Do53 only.
     * </p>
     * <p>
     * For an outbound endpoint you can apply the protocols as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Do53 and DoH in combination.
     * </p>
     * </li>
     * <li>
     * <p>
     * Do53 alone.
     * </p>
     * </li>
     * <li>
     * <p>
     * DoH alone.
     * </p>
     * </li>
     * <li>
     * <p>
     * None, which is treated as Do53.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasProtocols} method.
     * </p>
     * 
     * @return The protocols you want to use for the endpoint. DoH-FIPS is applicable for default inbound endpoints
     *         only. </p>
     *         <p>
     *         For a default inbound endpoint you can apply the protocols as follows:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Do53 and DoH in combination.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Do53 and DoH-FIPS in combination.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Do53 alone.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         DoH alone.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         DoH-FIPS alone.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         None, which is treated as Do53.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For a delegation inbound endpoint you can use Do53 only.
     *         </p>
     *         <p>
     *         For an outbound endpoint you can apply the protocols as follows:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Do53 and DoH in combination.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Do53 alone.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         DoH alone.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         None, which is treated as Do53.
     *         </p>
     *         </li>
     */
    public final List<String> protocolsAsStrings() {
        return protocols;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(creatorRequestId());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(hasSecurityGroupIds() ? securityGroupIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(directionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasIpAddresses() ? ipAddresses() : null);
        hashCode = 31 * hashCode + Objects.hashCode(outpostArn());
        hashCode = 31 * hashCode + Objects.hashCode(preferredInstanceType());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(resolverEndpointTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasProtocols() ? protocolsAsStrings() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateResolverEndpointRequest)) {
            return false;
        }
        CreateResolverEndpointRequest other = (CreateResolverEndpointRequest) obj;
        return Objects.equals(creatorRequestId(), other.creatorRequestId()) && Objects.equals(name(), other.name())
                && hasSecurityGroupIds() == other.hasSecurityGroupIds()
                && Objects.equals(securityGroupIds(), other.securityGroupIds())
                && Objects.equals(directionAsString(), other.directionAsString()) && hasIpAddresses() == other.hasIpAddresses()
                && Objects.equals(ipAddresses(), other.ipAddresses()) && Objects.equals(outpostArn(), other.outpostArn())
                && Objects.equals(preferredInstanceType(), other.preferredInstanceType()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags())
                && Objects.equals(resolverEndpointTypeAsString(), other.resolverEndpointTypeAsString())
                && hasProtocols() == other.hasProtocols() && Objects.equals(protocolsAsStrings(), other.protocolsAsStrings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateResolverEndpointRequest").add("CreatorRequestId", creatorRequestId()).add("Name", name())
                .add("SecurityGroupIds", hasSecurityGroupIds() ? securityGroupIds() : null).add("Direction", directionAsString())
                .add("IpAddresses", hasIpAddresses() ? ipAddresses() : null).add("OutpostArn", outpostArn())
                .add("PreferredInstanceType", preferredInstanceType()).add("Tags", hasTags() ? tags() : null)
                .add("ResolverEndpointType", resolverEndpointTypeAsString())
                .add("Protocols", hasProtocols() ? protocolsAsStrings() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "CreatorRequestId":
            return Optional.ofNullable(clazz.cast(creatorRequestId()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "SecurityGroupIds":
            return Optional.ofNullable(clazz.cast(securityGroupIds()));
        case "Direction":
            return Optional.ofNullable(clazz.cast(directionAsString()));
        case "IpAddresses":
            return Optional.ofNullable(clazz.cast(ipAddresses()));
        case "OutpostArn":
            return Optional.ofNullable(clazz.cast(outpostArn()));
        case "PreferredInstanceType":
            return Optional.ofNullable(clazz.cast(preferredInstanceType()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "ResolverEndpointType":
            return Optional.ofNullable(clazz.cast(resolverEndpointTypeAsString()));
        case "Protocols":
            return Optional.ofNullable(clazz.cast(protocolsAsStrings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("CreatorRequestId", CREATOR_REQUEST_ID_FIELD);
        map.put("Name", NAME_FIELD);
        map.put("SecurityGroupIds", SECURITY_GROUP_IDS_FIELD);
        map.put("Direction", DIRECTION_FIELD);
        map.put("IpAddresses", IP_ADDRESSES_FIELD);
        map.put("OutpostArn", OUTPOST_ARN_FIELD);
        map.put("PreferredInstanceType", PREFERRED_INSTANCE_TYPE_FIELD);
        map.put("Tags", TAGS_FIELD);
        map.put("ResolverEndpointType", RESOLVER_ENDPOINT_TYPE_FIELD);
        map.put("Protocols", PROTOCOLS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<CreateResolverEndpointRequest, T> g) {
        return obj -> g.apply((CreateResolverEndpointRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends Route53ResolverRequest.Builder, SdkPojo,
            CopyableBuilder<Builder, CreateResolverEndpointRequest> {
        /**
         * <p>
         * A unique string that identifies the request and that allows failed requests to be retried without the risk of
         * running the operation twice. <code>CreatorRequestId</code> can be any unique string, for example, a date/time
         * stamp.
         * </p>
         * 
         * @param creatorRequestId
         *        A unique string that identifies the request and that allows failed requests to be retried without the
         *        risk of running the operation twice. <code>CreatorRequestId</code> can be any unique string, for
         *        example, a date/time stamp.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creatorRequestId(String creatorRequestId);

        /**
         * <p>
         * A friendly name that lets you easily find a configuration in the Resolver dashboard in the Route 53 console.
         * </p>
         * 
         * @param name
         *        A friendly name that lets you easily find a configuration in the Resolver dashboard in the Route 53
         *        console.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The ID of one or more security groups that you want to use to control access to this VPC. The security group
         * that you specify must include one or more inbound rules (for inbound Resolver endpoints) or outbound rules
         * (for outbound Resolver endpoints). Inbound and outbound rules must allow TCP and UDP access. For inbound
         * access, open port 53. For outbound access, open the port that you're using for DNS queries on your network.
         * </p>
         * <p>
         * Some security group rules will cause your connection to be tracked. For outbound resolver endpoint, it can
         * potentially impact the maximum queries per second from outbound endpoint to your target name server. For
         * inbound resolver endpoint, it can bring down the overall maximum queries per second per IP address to as low
         * as 1500. To avoid connection tracking caused by security group, see <a href=
         * "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/security-group-connection-tracking.html#untracked-connectionsl"
         * >Untracked connections</a>.
         * </p>
         * 
         * @param securityGroupIds
         *        The ID of one or more security groups that you want to use to control access to this VPC. The security
         *        group that you specify must include one or more inbound rules (for inbound Resolver endpoints) or
         *        outbound rules (for outbound Resolver endpoints). Inbound and outbound rules must allow TCP and UDP
         *        access. For inbound access, open port 53. For outbound access, open the port that you're using for DNS
         *        queries on your network.</p>
         *        <p>
         *        Some security group rules will cause your connection to be tracked. For outbound resolver endpoint, it
         *        can potentially impact the maximum queries per second from outbound endpoint to your target name
         *        server. For inbound resolver endpoint, it can bring down the overall maximum queries per second per IP
         *        address to as low as 1500. To avoid connection tracking caused by security group, see <a href=
         *        "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/security-group-connection-tracking.html#untracked-connectionsl"
         *        >Untracked connections</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroupIds(Collection<String> securityGroupIds);

        /**
         * <p>
         * The ID of one or more security groups that you want to use to control access to this VPC. The security group
         * that you specify must include one or more inbound rules (for inbound Resolver endpoints) or outbound rules
         * (for outbound Resolver endpoints). Inbound and outbound rules must allow TCP and UDP access. For inbound
         * access, open port 53. For outbound access, open the port that you're using for DNS queries on your network.
         * </p>
         * <p>
         * Some security group rules will cause your connection to be tracked. For outbound resolver endpoint, it can
         * potentially impact the maximum queries per second from outbound endpoint to your target name server. For
         * inbound resolver endpoint, it can bring down the overall maximum queries per second per IP address to as low
         * as 1500. To avoid connection tracking caused by security group, see <a href=
         * "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/security-group-connection-tracking.html#untracked-connectionsl"
         * >Untracked connections</a>.
         * </p>
         * 
         * @param securityGroupIds
         *        The ID of one or more security groups that you want to use to control access to this VPC. The security
         *        group that you specify must include one or more inbound rules (for inbound Resolver endpoints) or
         *        outbound rules (for outbound Resolver endpoints). Inbound and outbound rules must allow TCP and UDP
         *        access. For inbound access, open port 53. For outbound access, open the port that you're using for DNS
         *        queries on your network.</p>
         *        <p>
         *        Some security group rules will cause your connection to be tracked. For outbound resolver endpoint, it
         *        can potentially impact the maximum queries per second from outbound endpoint to your target name
         *        server. For inbound resolver endpoint, it can bring down the overall maximum queries per second per IP
         *        address to as low as 1500. To avoid connection tracking caused by security group, see <a href=
         *        "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/security-group-connection-tracking.html#untracked-connectionsl"
         *        >Untracked connections</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroupIds(String... securityGroupIds);

        /**
         * <p>
         * Specify the applicable value:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>INBOUND</code>: Resolver forwards DNS queries to the DNS service for a VPC from your network.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>OUTBOUND</code>: Resolver forwards DNS queries from the DNS service for a VPC to your network.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>INBOUND_DELEGATION</code>: Resolver delegates queries to Route 53 private hosted zones from your
         * network.
         * </p>
         * </li>
         * </ul>
         * 
         * @param direction
         *        Specify the applicable value:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>INBOUND</code>: Resolver forwards DNS queries to the DNS service for a VPC from your network.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>OUTBOUND</code>: Resolver forwards DNS queries from the DNS service for a VPC to your network.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>INBOUND_DELEGATION</code>: Resolver delegates queries to Route 53 private hosted zones from your
         *        network.
         *        </p>
         *        </li>
         * @see ResolverEndpointDirection
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResolverEndpointDirection
         */
        Builder direction(String direction);

        /**
         * <p>
         * Specify the applicable value:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>INBOUND</code>: Resolver forwards DNS queries to the DNS service for a VPC from your network.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>OUTBOUND</code>: Resolver forwards DNS queries from the DNS service for a VPC to your network.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>INBOUND_DELEGATION</code>: Resolver delegates queries to Route 53 private hosted zones from your
         * network.
         * </p>
         * </li>
         * </ul>
         * 
         * @param direction
         *        Specify the applicable value:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>INBOUND</code>: Resolver forwards DNS queries to the DNS service for a VPC from your network.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>OUTBOUND</code>: Resolver forwards DNS queries from the DNS service for a VPC to your network.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>INBOUND_DELEGATION</code>: Resolver delegates queries to Route 53 private hosted zones from your
         *        network.
         *        </p>
         *        </li>
         * @see ResolverEndpointDirection
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResolverEndpointDirection
         */
        Builder direction(ResolverEndpointDirection direction);

        /**
         * <p>
         * The subnets and IP addresses in your VPC that DNS queries originate from (for outbound endpoints) or that you
         * forward DNS queries to (for inbound endpoints). The subnet ID uniquely identifies a VPC.
         * </p>
         * <note>
         * <p>
         * Even though the minimum is 1, Route 53 requires that you create at least two.
         * </p>
         * </note>
         * 
         * @param ipAddresses
         *        The subnets and IP addresses in your VPC that DNS queries originate from (for outbound endpoints) or
         *        that you forward DNS queries to (for inbound endpoints). The subnet ID uniquely identifies a VPC. </p>
         *        <note>
         *        <p>
         *        Even though the minimum is 1, Route 53 requires that you create at least two.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipAddresses(Collection<IpAddressRequest> ipAddresses);

        /**
         * <p>
         * The subnets and IP addresses in your VPC that DNS queries originate from (for outbound endpoints) or that you
         * forward DNS queries to (for inbound endpoints). The subnet ID uniquely identifies a VPC.
         * </p>
         * <note>
         * <p>
         * Even though the minimum is 1, Route 53 requires that you create at least two.
         * </p>
         * </note>
         * 
         * @param ipAddresses
         *        The subnets and IP addresses in your VPC that DNS queries originate from (for outbound endpoints) or
         *        that you forward DNS queries to (for inbound endpoints). The subnet ID uniquely identifies a VPC. </p>
         *        <note>
         *        <p>
         *        Even though the minimum is 1, Route 53 requires that you create at least two.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipAddresses(IpAddressRequest... ipAddresses);

        /**
         * <p>
         * The subnets and IP addresses in your VPC that DNS queries originate from (for outbound endpoints) or that you
         * forward DNS queries to (for inbound endpoints). The subnet ID uniquely identifies a VPC.
         * </p>
         * <note>
         * <p>
         * Even though the minimum is 1, Route 53 requires that you create at least two.
         * </p>
         * </note> This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.route53resolver.model.IpAddressRequest.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.route53resolver.model.IpAddressRequest#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.route53resolver.model.IpAddressRequest.Builder#build()} is called
         * immediately and its result is passed to {@link #ipAddresses(List<IpAddressRequest>)}.
         * 
         * @param ipAddresses
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.route53resolver.model.IpAddressRequest.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #ipAddresses(java.util.Collection<IpAddressRequest>)
         */
        Builder ipAddresses(Consumer<IpAddressRequest.Builder>... ipAddresses);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the Outpost. If you specify this, you must also specify a value for the
         * <code>PreferredInstanceType</code>.
         * </p>
         * 
         * @param outpostArn
         *        The Amazon Resource Name (ARN) of the Outpost. If you specify this, you must also specify a value for
         *        the <code>PreferredInstanceType</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outpostArn(String outpostArn);

        /**
         * <p>
         * The instance type. If you specify this, you must also specify a value for the <code>OutpostArn</code>.
         * </p>
         * 
         * @param preferredInstanceType
         *        The instance type. If you specify this, you must also specify a value for the <code>OutpostArn</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder preferredInstanceType(String preferredInstanceType);

        /**
         * <p>
         * A list of the tag keys and values that you want to associate with the endpoint.
         * </p>
         * 
         * @param tags
         *        A list of the tag keys and values that you want to associate with the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * A list of the tag keys and values that you want to associate with the endpoint.
         * </p>
         * 
         * @param tags
         *        A list of the tag keys and values that you want to associate with the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * A list of the tag keys and values that you want to associate with the endpoint.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.route53resolver.model.Tag.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.route53resolver.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.route53resolver.model.Tag.Builder#build()} is called immediately and
         * its result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.route53resolver.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * For the endpoint type you can choose either IPv4, IPv6, or dual-stack. A dual-stack endpoint means that it
         * will resolve via both IPv4 and IPv6. This endpoint type is applied to all IP addresses.
         * </p>
         * 
         * @param resolverEndpointType
         *        For the endpoint type you can choose either IPv4, IPv6, or dual-stack. A dual-stack endpoint means
         *        that it will resolve via both IPv4 and IPv6. This endpoint type is applied to all IP addresses.
         * @see ResolverEndpointType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResolverEndpointType
         */
        Builder resolverEndpointType(String resolverEndpointType);

        /**
         * <p>
         * For the endpoint type you can choose either IPv4, IPv6, or dual-stack. A dual-stack endpoint means that it
         * will resolve via both IPv4 and IPv6. This endpoint type is applied to all IP addresses.
         * </p>
         * 
         * @param resolverEndpointType
         *        For the endpoint type you can choose either IPv4, IPv6, or dual-stack. A dual-stack endpoint means
         *        that it will resolve via both IPv4 and IPv6. This endpoint type is applied to all IP addresses.
         * @see ResolverEndpointType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResolverEndpointType
         */
        Builder resolverEndpointType(ResolverEndpointType resolverEndpointType);

        /**
         * <p>
         * The protocols you want to use for the endpoint. DoH-FIPS is applicable for default inbound endpoints only.
         * </p>
         * <p>
         * For a default inbound endpoint you can apply the protocols as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Do53 and DoH in combination.
         * </p>
         * </li>
         * <li>
         * <p>
         * Do53 and DoH-FIPS in combination.
         * </p>
         * </li>
         * <li>
         * <p>
         * Do53 alone.
         * </p>
         * </li>
         * <li>
         * <p>
         * DoH alone.
         * </p>
         * </li>
         * <li>
         * <p>
         * DoH-FIPS alone.
         * </p>
         * </li>
         * <li>
         * <p>
         * None, which is treated as Do53.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For a delegation inbound endpoint you can use Do53 only.
         * </p>
         * <p>
         * For an outbound endpoint you can apply the protocols as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Do53 and DoH in combination.
         * </p>
         * </li>
         * <li>
         * <p>
         * Do53 alone.
         * </p>
         * </li>
         * <li>
         * <p>
         * DoH alone.
         * </p>
         * </li>
         * <li>
         * <p>
         * None, which is treated as Do53.
         * </p>
         * </li>
         * </ul>
         * 
         * @param protocols
         *        The protocols you want to use for the endpoint. DoH-FIPS is applicable for default inbound endpoints
         *        only. </p>
         *        <p>
         *        For a default inbound endpoint you can apply the protocols as follows:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Do53 and DoH in combination.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Do53 and DoH-FIPS in combination.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Do53 alone.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        DoH alone.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        DoH-FIPS alone.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        None, which is treated as Do53.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For a delegation inbound endpoint you can use Do53 only.
         *        </p>
         *        <p>
         *        For an outbound endpoint you can apply the protocols as follows:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Do53 and DoH in combination.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Do53 alone.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        DoH alone.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        None, which is treated as Do53.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder protocolsWithStrings(Collection<String> protocols);

        /**
         * <p>
         * The protocols you want to use for the endpoint. DoH-FIPS is applicable for default inbound endpoints only.
         * </p>
         * <p>
         * For a default inbound endpoint you can apply the protocols as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Do53 and DoH in combination.
         * </p>
         * </li>
         * <li>
         * <p>
         * Do53 and DoH-FIPS in combination.
         * </p>
         * </li>
         * <li>
         * <p>
         * Do53 alone.
         * </p>
         * </li>
         * <li>
         * <p>
         * DoH alone.
         * </p>
         * </li>
         * <li>
         * <p>
         * DoH-FIPS alone.
         * </p>
         * </li>
         * <li>
         * <p>
         * None, which is treated as Do53.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For a delegation inbound endpoint you can use Do53 only.
         * </p>
         * <p>
         * For an outbound endpoint you can apply the protocols as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Do53 and DoH in combination.
         * </p>
         * </li>
         * <li>
         * <p>
         * Do53 alone.
         * </p>
         * </li>
         * <li>
         * <p>
         * DoH alone.
         * </p>
         * </li>
         * <li>
         * <p>
         * None, which is treated as Do53.
         * </p>
         * </li>
         * </ul>
         * 
         * @param protocols
         *        The protocols you want to use for the endpoint. DoH-FIPS is applicable for default inbound endpoints
         *        only. </p>
         *        <p>
         *        For a default inbound endpoint you can apply the protocols as follows:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Do53 and DoH in combination.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Do53 and DoH-FIPS in combination.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Do53 alone.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        DoH alone.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        DoH-FIPS alone.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        None, which is treated as Do53.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For a delegation inbound endpoint you can use Do53 only.
         *        </p>
         *        <p>
         *        For an outbound endpoint you can apply the protocols as follows:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Do53 and DoH in combination.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Do53 alone.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        DoH alone.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        None, which is treated as Do53.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder protocolsWithStrings(String... protocols);

        /**
         * <p>
         * The protocols you want to use for the endpoint. DoH-FIPS is applicable for default inbound endpoints only.
         * </p>
         * <p>
         * For a default inbound endpoint you can apply the protocols as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Do53 and DoH in combination.
         * </p>
         * </li>
         * <li>
         * <p>
         * Do53 and DoH-FIPS in combination.
         * </p>
         * </li>
         * <li>
         * <p>
         * Do53 alone.
         * </p>
         * </li>
         * <li>
         * <p>
         * DoH alone.
         * </p>
         * </li>
         * <li>
         * <p>
         * DoH-FIPS alone.
         * </p>
         * </li>
         * <li>
         * <p>
         * None, which is treated as Do53.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For a delegation inbound endpoint you can use Do53 only.
         * </p>
         * <p>
         * For an outbound endpoint you can apply the protocols as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Do53 and DoH in combination.
         * </p>
         * </li>
         * <li>
         * <p>
         * Do53 alone.
         * </p>
         * </li>
         * <li>
         * <p>
         * DoH alone.
         * </p>
         * </li>
         * <li>
         * <p>
         * None, which is treated as Do53.
         * </p>
         * </li>
         * </ul>
         * 
         * @param protocols
         *        The protocols you want to use for the endpoint. DoH-FIPS is applicable for default inbound endpoints
         *        only. </p>
         *        <p>
         *        For a default inbound endpoint you can apply the protocols as follows:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Do53 and DoH in combination.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Do53 and DoH-FIPS in combination.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Do53 alone.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        DoH alone.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        DoH-FIPS alone.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        None, which is treated as Do53.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For a delegation inbound endpoint you can use Do53 only.
         *        </p>
         *        <p>
         *        For an outbound endpoint you can apply the protocols as follows:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Do53 and DoH in combination.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Do53 alone.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        DoH alone.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        None, which is treated as Do53.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder protocols(Collection<Protocol> protocols);

        /**
         * <p>
         * The protocols you want to use for the endpoint. DoH-FIPS is applicable for default inbound endpoints only.
         * </p>
         * <p>
         * For a default inbound endpoint you can apply the protocols as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Do53 and DoH in combination.
         * </p>
         * </li>
         * <li>
         * <p>
         * Do53 and DoH-FIPS in combination.
         * </p>
         * </li>
         * <li>
         * <p>
         * Do53 alone.
         * </p>
         * </li>
         * <li>
         * <p>
         * DoH alone.
         * </p>
         * </li>
         * <li>
         * <p>
         * DoH-FIPS alone.
         * </p>
         * </li>
         * <li>
         * <p>
         * None, which is treated as Do53.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For a delegation inbound endpoint you can use Do53 only.
         * </p>
         * <p>
         * For an outbound endpoint you can apply the protocols as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Do53 and DoH in combination.
         * </p>
         * </li>
         * <li>
         * <p>
         * Do53 alone.
         * </p>
         * </li>
         * <li>
         * <p>
         * DoH alone.
         * </p>
         * </li>
         * <li>
         * <p>
         * None, which is treated as Do53.
         * </p>
         * </li>
         * </ul>
         * 
         * @param protocols
         *        The protocols you want to use for the endpoint. DoH-FIPS is applicable for default inbound endpoints
         *        only. </p>
         *        <p>
         *        For a default inbound endpoint you can apply the protocols as follows:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Do53 and DoH in combination.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Do53 and DoH-FIPS in combination.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Do53 alone.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        DoH alone.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        DoH-FIPS alone.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        None, which is treated as Do53.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For a delegation inbound endpoint you can use Do53 only.
         *        </p>
         *        <p>
         *        For an outbound endpoint you can apply the protocols as follows:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Do53 and DoH in combination.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Do53 alone.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        DoH alone.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        None, which is treated as Do53.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder protocols(Protocol... protocols);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends Route53ResolverRequest.BuilderImpl implements Builder {
        private String creatorRequestId;

        private String name;

        private List<String> securityGroupIds = DefaultSdkAutoConstructList.getInstance();

        private String direction;

        private List<IpAddressRequest> ipAddresses = DefaultSdkAutoConstructList.getInstance();

        private String outpostArn;

        private String preferredInstanceType;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private String resolverEndpointType;

        private List<String> protocols = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CreateResolverEndpointRequest model) {
            super(model);
            creatorRequestId(model.creatorRequestId);
            name(model.name);
            securityGroupIds(model.securityGroupIds);
            direction(model.direction);
            ipAddresses(model.ipAddresses);
            outpostArn(model.outpostArn);
            preferredInstanceType(model.preferredInstanceType);
            tags(model.tags);
            resolverEndpointType(model.resolverEndpointType);
            protocolsWithStrings(model.protocols);
        }

        public final String getCreatorRequestId() {
            return creatorRequestId;
        }

        public final void setCreatorRequestId(String creatorRequestId) {
            this.creatorRequestId = creatorRequestId;
        }

        @Override
        public final Builder creatorRequestId(String creatorRequestId) {
            this.creatorRequestId = creatorRequestId;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final Collection<String> getSecurityGroupIds() {
            if (securityGroupIds instanceof SdkAutoConstructList) {
                return null;
            }
            return securityGroupIds;
        }

        public final void setSecurityGroupIds(Collection<String> securityGroupIds) {
            this.securityGroupIds = SecurityGroupIdsCopier.copy(securityGroupIds);
        }

        @Override
        public final Builder securityGroupIds(Collection<String> securityGroupIds) {
            this.securityGroupIds = SecurityGroupIdsCopier.copy(securityGroupIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder securityGroupIds(String... securityGroupIds) {
            securityGroupIds(Arrays.asList(securityGroupIds));
            return this;
        }

        public final String getDirection() {
            return direction;
        }

        public final void setDirection(String direction) {
            this.direction = direction;
        }

        @Override
        public final Builder direction(String direction) {
            this.direction = direction;
            return this;
        }

        @Override
        public final Builder direction(ResolverEndpointDirection direction) {
            this.direction(direction == null ? null : direction.toString());
            return this;
        }

        public final List<IpAddressRequest.Builder> getIpAddresses() {
            List<IpAddressRequest.Builder> result = IpAddressesRequestCopier.copyToBuilder(this.ipAddresses);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setIpAddresses(Collection<IpAddressRequest.BuilderImpl> ipAddresses) {
            this.ipAddresses = IpAddressesRequestCopier.copyFromBuilder(ipAddresses);
        }

        @Override
        public final Builder ipAddresses(Collection<IpAddressRequest> ipAddresses) {
            this.ipAddresses = IpAddressesRequestCopier.copy(ipAddresses);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ipAddresses(IpAddressRequest... ipAddresses) {
            ipAddresses(Arrays.asList(ipAddresses));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ipAddresses(Consumer<IpAddressRequest.Builder>... ipAddresses) {
            ipAddresses(Stream.of(ipAddresses).map(c -> IpAddressRequest.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getOutpostArn() {
            return outpostArn;
        }

        public final void setOutpostArn(String outpostArn) {
            this.outpostArn = outpostArn;
        }

        @Override
        public final Builder outpostArn(String outpostArn) {
            this.outpostArn = outpostArn;
            return this;
        }

        public final String getPreferredInstanceType() {
            return preferredInstanceType;
        }

        public final void setPreferredInstanceType(String preferredInstanceType) {
            this.preferredInstanceType = preferredInstanceType;
        }

        @Override
        public final Builder preferredInstanceType(String preferredInstanceType) {
            this.preferredInstanceType = preferredInstanceType;
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getResolverEndpointType() {
            return resolverEndpointType;
        }

        public final void setResolverEndpointType(String resolverEndpointType) {
            this.resolverEndpointType = resolverEndpointType;
        }

        @Override
        public final Builder resolverEndpointType(String resolverEndpointType) {
            this.resolverEndpointType = resolverEndpointType;
            return this;
        }

        @Override
        public final Builder resolverEndpointType(ResolverEndpointType resolverEndpointType) {
            this.resolverEndpointType(resolverEndpointType == null ? null : resolverEndpointType.toString());
            return this;
        }

        public final Collection<String> getProtocols() {
            if (protocols instanceof SdkAutoConstructList) {
                return null;
            }
            return protocols;
        }

        public final void setProtocols(Collection<String> protocols) {
            this.protocols = ProtocolListCopier.copy(protocols);
        }

        @Override
        public final Builder protocolsWithStrings(Collection<String> protocols) {
            this.protocols = ProtocolListCopier.copy(protocols);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder protocolsWithStrings(String... protocols) {
            protocolsWithStrings(Arrays.asList(protocols));
            return this;
        }

        @Override
        public final Builder protocols(Collection<Protocol> protocols) {
            this.protocols = ProtocolListCopier.copyEnumToString(protocols);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder protocols(Protocol... protocols) {
            protocols(Arrays.asList(protocols));
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateResolverEndpointRequest build() {
            return new CreateResolverEndpointRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
