/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rum;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.core.SdkClient;
import software.amazon.awssdk.services.rum.model.CreateAppMonitorRequest;
import software.amazon.awssdk.services.rum.model.CreateAppMonitorResponse;
import software.amazon.awssdk.services.rum.model.DeleteAppMonitorRequest;
import software.amazon.awssdk.services.rum.model.DeleteAppMonitorResponse;
import software.amazon.awssdk.services.rum.model.GetAppMonitorDataRequest;
import software.amazon.awssdk.services.rum.model.GetAppMonitorDataResponse;
import software.amazon.awssdk.services.rum.model.GetAppMonitorRequest;
import software.amazon.awssdk.services.rum.model.GetAppMonitorResponse;
import software.amazon.awssdk.services.rum.model.ListAppMonitorsRequest;
import software.amazon.awssdk.services.rum.model.ListAppMonitorsResponse;
import software.amazon.awssdk.services.rum.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.rum.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.rum.model.PutRumEventsRequest;
import software.amazon.awssdk.services.rum.model.PutRumEventsResponse;
import software.amazon.awssdk.services.rum.model.TagResourceRequest;
import software.amazon.awssdk.services.rum.model.TagResourceResponse;
import software.amazon.awssdk.services.rum.model.UntagResourceRequest;
import software.amazon.awssdk.services.rum.model.UntagResourceResponse;
import software.amazon.awssdk.services.rum.model.UpdateAppMonitorRequest;
import software.amazon.awssdk.services.rum.model.UpdateAppMonitorResponse;
import software.amazon.awssdk.services.rum.paginators.GetAppMonitorDataPublisher;
import software.amazon.awssdk.services.rum.paginators.ListAppMonitorsPublisher;

/**
 * Service client for accessing CloudWatch RUM asynchronously. This can be created using the static {@link #builder()}
 * method.
 *
 * <p>
 * With Amazon CloudWatch RUM, you can perform real-user monitoring to collect client-side data about your web
 * application performance from actual user sessions in real time. The data collected includes page load times,
 * client-side errors, and user behavior. When you view this data, you can see it all aggregated together and also see
 * breakdowns by the browsers and devices that your customers use.
 * </p>
 * 
 * <pre>
 * <code> &lt;p&gt;You can use the collected data to quickly identify and debug client-side performance issues. CloudWatch RUM helps you visualize anomalies in your application performance and find relevant debugging data such as error messages, stack traces, and user sessions. You can also use RUM to understand the range of end-user impact including the number of users, geolocations, and browsers used.&lt;/p&gt; </code>
 * </pre>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface RumAsyncClient extends SdkClient {
    String SERVICE_NAME = "rum";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "rum";

    /**
     * Create a {@link RumAsyncClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static RumAsyncClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link RumAsyncClient}.
     */
    static RumAsyncClientBuilder builder() {
        return new DefaultRumAsyncClientBuilder();
    }

    /**
     * <p>
     * Creates a Amazon CloudWatch RUM app monitor, which collects telemetry data from your application and sends that
     * data to RUM. The data includes performance and reliability information such as page load time, client-side
     * errors, and user behavior.
     * </p>
     * <p>
     * You use this operation only to create a new app monitor. To update an existing app monitor, use <a
     * href="https://docs.aws.amazon.com/cloudwatchrum/latest/APIReference/API_UpdateAppMonitor.html"
     * >UpdateAppMonitor</a> instead.
     * </p>
     * <p>
     * After you create an app monitor, sign in to the CloudWatch RUM console to get the JavaScript code snippet to add
     * to your web application. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-RUM-find-code-snippet.html">How
     * do I find a code snippet that I've already generated?</a>
     * </p>
     *
     * @param createAppMonitorRequest
     * @return A Java Future containing the result of the CreateAppMonitor operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException This operation attempted to create a resource that already exists.</li>
     *         <li>ServiceQuotaExceededException This request exceeds a service quota.</li>
     *         <li>InternalServerException Internal service exception.</li>
     *         <li>ValidationException One of the arguments for the request is not valid.</li>
     *         <li>ThrottlingException The request was throttled because of quota limits.</li>
     *         <li>AccessDeniedException You don't have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RumException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RumAsyncClient.CreateAppMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/CreateAppMonitor" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<CreateAppMonitorResponse> createAppMonitor(CreateAppMonitorRequest createAppMonitorRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates a Amazon CloudWatch RUM app monitor, which collects telemetry data from your application and sends that
     * data to RUM. The data includes performance and reliability information such as page load time, client-side
     * errors, and user behavior.
     * </p>
     * <p>
     * You use this operation only to create a new app monitor. To update an existing app monitor, use <a
     * href="https://docs.aws.amazon.com/cloudwatchrum/latest/APIReference/API_UpdateAppMonitor.html"
     * >UpdateAppMonitor</a> instead.
     * </p>
     * <p>
     * After you create an app monitor, sign in to the CloudWatch RUM console to get the JavaScript code snippet to add
     * to your web application. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-RUM-find-code-snippet.html">How
     * do I find a code snippet that I've already generated?</a>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateAppMonitorRequest.Builder} avoiding the need
     * to create one manually via {@link CreateAppMonitorRequest#builder()}
     * </p>
     *
     * @param createAppMonitorRequest
     *        A {@link Consumer} that will call methods on {@link CreateAppMonitorRequest.Builder} to create a request.
     * @return A Java Future containing the result of the CreateAppMonitor operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException This operation attempted to create a resource that already exists.</li>
     *         <li>ServiceQuotaExceededException This request exceeds a service quota.</li>
     *         <li>InternalServerException Internal service exception.</li>
     *         <li>ValidationException One of the arguments for the request is not valid.</li>
     *         <li>ThrottlingException The request was throttled because of quota limits.</li>
     *         <li>AccessDeniedException You don't have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RumException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RumAsyncClient.CreateAppMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/CreateAppMonitor" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<CreateAppMonitorResponse> createAppMonitor(
            Consumer<CreateAppMonitorRequest.Builder> createAppMonitorRequest) {
        return createAppMonitor(CreateAppMonitorRequest.builder().applyMutation(createAppMonitorRequest).build());
    }

    /**
     * <p>
     * Deletes an existing app monitor. This immediately stops the collection of data.
     * </p>
     *
     * @param deleteAppMonitorRequest
     * @return A Java Future containing the result of the DeleteAppMonitor operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException This operation attempted to create a resource that already exists.</li>
     *         <li>ResourceNotFoundException Resource not found.</li>
     *         <li>InternalServerException Internal service exception.</li>
     *         <li>ValidationException One of the arguments for the request is not valid.</li>
     *         <li>ThrottlingException The request was throttled because of quota limits.</li>
     *         <li>AccessDeniedException You don't have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RumException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RumAsyncClient.DeleteAppMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/DeleteAppMonitor" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DeleteAppMonitorResponse> deleteAppMonitor(DeleteAppMonitorRequest deleteAppMonitorRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes an existing app monitor. This immediately stops the collection of data.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteAppMonitorRequest.Builder} avoiding the need
     * to create one manually via {@link DeleteAppMonitorRequest#builder()}
     * </p>
     *
     * @param deleteAppMonitorRequest
     *        A {@link Consumer} that will call methods on {@link DeleteAppMonitorRequest.Builder} to create a request.
     * @return A Java Future containing the result of the DeleteAppMonitor operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException This operation attempted to create a resource that already exists.</li>
     *         <li>ResourceNotFoundException Resource not found.</li>
     *         <li>InternalServerException Internal service exception.</li>
     *         <li>ValidationException One of the arguments for the request is not valid.</li>
     *         <li>ThrottlingException The request was throttled because of quota limits.</li>
     *         <li>AccessDeniedException You don't have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RumException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RumAsyncClient.DeleteAppMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/DeleteAppMonitor" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DeleteAppMonitorResponse> deleteAppMonitor(
            Consumer<DeleteAppMonitorRequest.Builder> deleteAppMonitorRequest) {
        return deleteAppMonitor(DeleteAppMonitorRequest.builder().applyMutation(deleteAppMonitorRequest).build());
    }

    /**
     * <p>
     * Retrieves the complete configuration information for one app monitor.
     * </p>
     *
     * @param getAppMonitorRequest
     * @return A Java Future containing the result of the GetAppMonitor operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Resource not found.</li>
     *         <li>InternalServerException Internal service exception.</li>
     *         <li>ValidationException One of the arguments for the request is not valid.</li>
     *         <li>ThrottlingException The request was throttled because of quota limits.</li>
     *         <li>AccessDeniedException You don't have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RumException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RumAsyncClient.GetAppMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/GetAppMonitor" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<GetAppMonitorResponse> getAppMonitor(GetAppMonitorRequest getAppMonitorRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieves the complete configuration information for one app monitor.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetAppMonitorRequest.Builder} avoiding the need to
     * create one manually via {@link GetAppMonitorRequest#builder()}
     * </p>
     *
     * @param getAppMonitorRequest
     *        A {@link Consumer} that will call methods on {@link GetAppMonitorRequest.Builder} to create a request.
     * @return A Java Future containing the result of the GetAppMonitor operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Resource not found.</li>
     *         <li>InternalServerException Internal service exception.</li>
     *         <li>ValidationException One of the arguments for the request is not valid.</li>
     *         <li>ThrottlingException The request was throttled because of quota limits.</li>
     *         <li>AccessDeniedException You don't have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RumException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RumAsyncClient.GetAppMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/GetAppMonitor" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<GetAppMonitorResponse> getAppMonitor(Consumer<GetAppMonitorRequest.Builder> getAppMonitorRequest) {
        return getAppMonitor(GetAppMonitorRequest.builder().applyMutation(getAppMonitorRequest).build());
    }

    /**
     * <p>
     * Retrieves the raw performance events that RUM has collected from your web application, so that you can do your
     * own processing or analysis of this data.
     * </p>
     *
     * @param getAppMonitorDataRequest
     * @return A Java Future containing the result of the GetAppMonitorData operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Resource not found.</li>
     *         <li>InternalServerException Internal service exception.</li>
     *         <li>ValidationException One of the arguments for the request is not valid.</li>
     *         <li>ThrottlingException The request was throttled because of quota limits.</li>
     *         <li>AccessDeniedException You don't have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RumException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RumAsyncClient.GetAppMonitorData
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/GetAppMonitorData" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<GetAppMonitorDataResponse> getAppMonitorData(GetAppMonitorDataRequest getAppMonitorDataRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieves the raw performance events that RUM has collected from your web application, so that you can do your
     * own processing or analysis of this data.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetAppMonitorDataRequest.Builder} avoiding the need
     * to create one manually via {@link GetAppMonitorDataRequest#builder()}
     * </p>
     *
     * @param getAppMonitorDataRequest
     *        A {@link Consumer} that will call methods on {@link GetAppMonitorDataRequest.Builder} to create a request.
     * @return A Java Future containing the result of the GetAppMonitorData operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Resource not found.</li>
     *         <li>InternalServerException Internal service exception.</li>
     *         <li>ValidationException One of the arguments for the request is not valid.</li>
     *         <li>ThrottlingException The request was throttled because of quota limits.</li>
     *         <li>AccessDeniedException You don't have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RumException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RumAsyncClient.GetAppMonitorData
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/GetAppMonitorData" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<GetAppMonitorDataResponse> getAppMonitorData(
            Consumer<GetAppMonitorDataRequest.Builder> getAppMonitorDataRequest) {
        return getAppMonitorData(GetAppMonitorDataRequest.builder().applyMutation(getAppMonitorDataRequest).build());
    }

    /**
     * <p>
     * Retrieves the raw performance events that RUM has collected from your web application, so that you can do your
     * own processing or analysis of this data.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getAppMonitorData(software.amazon.awssdk.services.rum.model.GetAppMonitorDataRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rum.paginators.GetAppMonitorDataPublisher publisher = client.getAppMonitorDataPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rum.paginators.GetAppMonitorDataPublisher publisher = client.getAppMonitorDataPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.rum.model.GetAppMonitorDataResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.rum.model.GetAppMonitorDataResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getAppMonitorData(software.amazon.awssdk.services.rum.model.GetAppMonitorDataRequest)} operation.</b>
     * </p>
     *
     * @param getAppMonitorDataRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Resource not found.</li>
     *         <li>InternalServerException Internal service exception.</li>
     *         <li>ValidationException One of the arguments for the request is not valid.</li>
     *         <li>ThrottlingException The request was throttled because of quota limits.</li>
     *         <li>AccessDeniedException You don't have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RumException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RumAsyncClient.GetAppMonitorData
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/GetAppMonitorData" target="_top">AWS API
     *      Documentation</a>
     */
    default GetAppMonitorDataPublisher getAppMonitorDataPaginator(GetAppMonitorDataRequest getAppMonitorDataRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieves the raw performance events that RUM has collected from your web application, so that you can do your
     * own processing or analysis of this data.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getAppMonitorData(software.amazon.awssdk.services.rum.model.GetAppMonitorDataRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rum.paginators.GetAppMonitorDataPublisher publisher = client.getAppMonitorDataPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rum.paginators.GetAppMonitorDataPublisher publisher = client.getAppMonitorDataPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.rum.model.GetAppMonitorDataResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.rum.model.GetAppMonitorDataResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getAppMonitorData(software.amazon.awssdk.services.rum.model.GetAppMonitorDataRequest)} operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link GetAppMonitorDataRequest.Builder} avoiding the need
     * to create one manually via {@link GetAppMonitorDataRequest#builder()}
     * </p>
     *
     * @param getAppMonitorDataRequest
     *        A {@link Consumer} that will call methods on {@link GetAppMonitorDataRequest.Builder} to create a request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Resource not found.</li>
     *         <li>InternalServerException Internal service exception.</li>
     *         <li>ValidationException One of the arguments for the request is not valid.</li>
     *         <li>ThrottlingException The request was throttled because of quota limits.</li>
     *         <li>AccessDeniedException You don't have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RumException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RumAsyncClient.GetAppMonitorData
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/GetAppMonitorData" target="_top">AWS API
     *      Documentation</a>
     */
    default GetAppMonitorDataPublisher getAppMonitorDataPaginator(
            Consumer<GetAppMonitorDataRequest.Builder> getAppMonitorDataRequest) {
        return getAppMonitorDataPaginator(GetAppMonitorDataRequest.builder().applyMutation(getAppMonitorDataRequest).build());
    }

    /**
     * <p>
     * Returns a list of the Amazon CloudWatch RUM app monitors in the account.
     * </p>
     *
     * @param listAppMonitorsRequest
     * @return A Java Future containing the result of the ListAppMonitors operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Internal service exception.</li>
     *         <li>ValidationException One of the arguments for the request is not valid.</li>
     *         <li>ThrottlingException The request was throttled because of quota limits.</li>
     *         <li>AccessDeniedException You don't have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RumException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RumAsyncClient.ListAppMonitors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/ListAppMonitors" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListAppMonitorsResponse> listAppMonitors(ListAppMonitorsRequest listAppMonitorsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns a list of the Amazon CloudWatch RUM app monitors in the account.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListAppMonitorsRequest.Builder} avoiding the need
     * to create one manually via {@link ListAppMonitorsRequest#builder()}
     * </p>
     *
     * @param listAppMonitorsRequest
     *        A {@link Consumer} that will call methods on {@link ListAppMonitorsRequest.Builder} to create a request.
     * @return A Java Future containing the result of the ListAppMonitors operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Internal service exception.</li>
     *         <li>ValidationException One of the arguments for the request is not valid.</li>
     *         <li>ThrottlingException The request was throttled because of quota limits.</li>
     *         <li>AccessDeniedException You don't have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RumException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RumAsyncClient.ListAppMonitors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/ListAppMonitors" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListAppMonitorsResponse> listAppMonitors(
            Consumer<ListAppMonitorsRequest.Builder> listAppMonitorsRequest) {
        return listAppMonitors(ListAppMonitorsRequest.builder().applyMutation(listAppMonitorsRequest).build());
    }

    /**
     * <p>
     * Returns a list of the Amazon CloudWatch RUM app monitors in the account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listAppMonitors(software.amazon.awssdk.services.rum.model.ListAppMonitorsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rum.paginators.ListAppMonitorsPublisher publisher = client.listAppMonitorsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rum.paginators.ListAppMonitorsPublisher publisher = client.listAppMonitorsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.rum.model.ListAppMonitorsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.rum.model.ListAppMonitorsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAppMonitors(software.amazon.awssdk.services.rum.model.ListAppMonitorsRequest)} operation.</b>
     * </p>
     *
     * @param listAppMonitorsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Internal service exception.</li>
     *         <li>ValidationException One of the arguments for the request is not valid.</li>
     *         <li>ThrottlingException The request was throttled because of quota limits.</li>
     *         <li>AccessDeniedException You don't have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RumException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RumAsyncClient.ListAppMonitors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/ListAppMonitors" target="_top">AWS API
     *      Documentation</a>
     */
    default ListAppMonitorsPublisher listAppMonitorsPaginator(ListAppMonitorsRequest listAppMonitorsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns a list of the Amazon CloudWatch RUM app monitors in the account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listAppMonitors(software.amazon.awssdk.services.rum.model.ListAppMonitorsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rum.paginators.ListAppMonitorsPublisher publisher = client.listAppMonitorsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rum.paginators.ListAppMonitorsPublisher publisher = client.listAppMonitorsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.rum.model.ListAppMonitorsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.rum.model.ListAppMonitorsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAppMonitors(software.amazon.awssdk.services.rum.model.ListAppMonitorsRequest)} operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListAppMonitorsRequest.Builder} avoiding the need
     * to create one manually via {@link ListAppMonitorsRequest#builder()}
     * </p>
     *
     * @param listAppMonitorsRequest
     *        A {@link Consumer} that will call methods on {@link ListAppMonitorsRequest.Builder} to create a request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Internal service exception.</li>
     *         <li>ValidationException One of the arguments for the request is not valid.</li>
     *         <li>ThrottlingException The request was throttled because of quota limits.</li>
     *         <li>AccessDeniedException You don't have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RumException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RumAsyncClient.ListAppMonitors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/ListAppMonitors" target="_top">AWS API
     *      Documentation</a>
     */
    default ListAppMonitorsPublisher listAppMonitorsPaginator(Consumer<ListAppMonitorsRequest.Builder> listAppMonitorsRequest) {
        return listAppMonitorsPaginator(ListAppMonitorsRequest.builder().applyMutation(listAppMonitorsRequest).build());
    }

    /**
     * <p>
     * Displays the tags associated with a CloudWatch RUM resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Resource not found.</li>
     *         <li>InternalServerException Internal service exception.</li>
     *         <li>ValidationException One of the arguments for the request is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RumException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RumAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Displays the tags associated with a CloudWatch RUM resource.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListTagsForResourceRequest.Builder} avoiding the
     * need to create one manually via {@link ListTagsForResourceRequest#builder()}
     * </p>
     *
     * @param listTagsForResourceRequest
     *        A {@link Consumer} that will call methods on {@link ListTagsForResourceRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Resource not found.</li>
     *         <li>InternalServerException Internal service exception.</li>
     *         <li>ValidationException One of the arguments for the request is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RumException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RumAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            Consumer<ListTagsForResourceRequest.Builder> listTagsForResourceRequest) {
        return listTagsForResource(ListTagsForResourceRequest.builder().applyMutation(listTagsForResourceRequest).build());
    }

    /**
     * <p>
     * Sends telemetry events about your application performance and user behavior to CloudWatch RUM. The code snippet
     * that RUM generates for you to add to your application includes <code>PutRumEvents</code> operations to send this
     * data to RUM.
     * </p>
     * <p>
     * Each <code>PutRumEvents</code> operation can send a batch of events from one user session.
     * </p>
     *
     * @param putRumEventsRequest
     * @return A Java Future containing the result of the PutRumEvents operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Resource not found.</li>
     *         <li>InternalServerException Internal service exception.</li>
     *         <li>ValidationException One of the arguments for the request is not valid.</li>
     *         <li>ThrottlingException The request was throttled because of quota limits.</li>
     *         <li>AccessDeniedException You don't have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RumException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RumAsyncClient.PutRumEvents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/PutRumEvents" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<PutRumEventsResponse> putRumEvents(PutRumEventsRequest putRumEventsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Sends telemetry events about your application performance and user behavior to CloudWatch RUM. The code snippet
     * that RUM generates for you to add to your application includes <code>PutRumEvents</code> operations to send this
     * data to RUM.
     * </p>
     * <p>
     * Each <code>PutRumEvents</code> operation can send a batch of events from one user session.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link PutRumEventsRequest.Builder} avoiding the need to
     * create one manually via {@link PutRumEventsRequest#builder()}
     * </p>
     *
     * @param putRumEventsRequest
     *        A {@link Consumer} that will call methods on {@link PutRumEventsRequest.Builder} to create a request.
     * @return A Java Future containing the result of the PutRumEvents operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Resource not found.</li>
     *         <li>InternalServerException Internal service exception.</li>
     *         <li>ValidationException One of the arguments for the request is not valid.</li>
     *         <li>ThrottlingException The request was throttled because of quota limits.</li>
     *         <li>AccessDeniedException You don't have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RumException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RumAsyncClient.PutRumEvents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/PutRumEvents" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<PutRumEventsResponse> putRumEvents(Consumer<PutRumEventsRequest.Builder> putRumEventsRequest) {
        return putRumEvents(PutRumEventsRequest.builder().applyMutation(putRumEventsRequest).build());
    }

    /**
     * <p>
     * Assigns one or more tags (key-value pairs) to the specified CloudWatch RUM resource. Currently, the only
     * resources that can be tagged app monitors.
     * </p>
     * <p>
     * Tags can help you organize and categorize your resources. You can also use them to scope user permissions by
     * granting a user permission to access or change only resources with certain tag values.
     * </p>
     * <p>
     * Tags don't have any semantic meaning to Amazon Web Services and are interpreted strictly as strings of
     * characters.
     * </p>
     * <p>
     * You can use the <code>TagResource</code> action with a resource that already has tags. If you specify a new tag
     * key for the resource, this tag is appended to the list of tags associated with the alarm. If you specify a tag
     * key that is already associated with the resource, the new tag value that you specify replaces the previous value
     * for that tag.
     * </p>
     * <p>
     * You can associate as many as 50 tags with a resource.
     * </p>
     * <p>
     * For more information, see <a href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html">Tagging Amazon
     * Web Services resources</a>.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Resource not found.</li>
     *         <li>InternalServerException Internal service exception.</li>
     *         <li>ValidationException One of the arguments for the request is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RumException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RumAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Assigns one or more tags (key-value pairs) to the specified CloudWatch RUM resource. Currently, the only
     * resources that can be tagged app monitors.
     * </p>
     * <p>
     * Tags can help you organize and categorize your resources. You can also use them to scope user permissions by
     * granting a user permission to access or change only resources with certain tag values.
     * </p>
     * <p>
     * Tags don't have any semantic meaning to Amazon Web Services and are interpreted strictly as strings of
     * characters.
     * </p>
     * <p>
     * You can use the <code>TagResource</code> action with a resource that already has tags. If you specify a new tag
     * key for the resource, this tag is appended to the list of tags associated with the alarm. If you specify a tag
     * key that is already associated with the resource, the new tag value that you specify replaces the previous value
     * for that tag.
     * </p>
     * <p>
     * You can associate as many as 50 tags with a resource.
     * </p>
     * <p>
     * For more information, see <a href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html">Tagging Amazon
     * Web Services resources</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link TagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link TagResourceRequest#builder()}
     * </p>
     *
     * @param tagResourceRequest
     *        A {@link Consumer} that will call methods on {@link TagResourceRequest.Builder} to create a request.
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Resource not found.</li>
     *         <li>InternalServerException Internal service exception.</li>
     *         <li>ValidationException One of the arguments for the request is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RumException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RumAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<TagResourceResponse> tagResource(Consumer<TagResourceRequest.Builder> tagResourceRequest) {
        return tagResource(TagResourceRequest.builder().applyMutation(tagResourceRequest).build());
    }

    /**
     * <p>
     * Removes one or more tags from the specified resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Resource not found.</li>
     *         <li>InternalServerException Internal service exception.</li>
     *         <li>ValidationException One of the arguments for the request is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RumException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RumAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Removes one or more tags from the specified resource.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UntagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link UntagResourceRequest#builder()}
     * </p>
     *
     * @param untagResourceRequest
     *        A {@link Consumer} that will call methods on {@link UntagResourceRequest.Builder} to create a request.
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Resource not found.</li>
     *         <li>InternalServerException Internal service exception.</li>
     *         <li>ValidationException One of the arguments for the request is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RumException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RumAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<UntagResourceResponse> untagResource(Consumer<UntagResourceRequest.Builder> untagResourceRequest) {
        return untagResource(UntagResourceRequest.builder().applyMutation(untagResourceRequest).build());
    }

    /**
     * <p>
     * Updates the configuration of an existing app monitor. When you use this operation, only the parts of the app
     * monitor configuration that you specify in this operation are changed. For any parameters that you omit, the
     * existing values are kept.
     * </p>
     * <p>
     * You can't use this operation to change the tags of an existing app monitor. To change the tags of an existing app
     * monitor, use <a
     * href="https://docs.aws.amazon.com/cloudwatchrum/latest/APIReference/API_TagResource.html">TagResource</a>.
     * </p>
     * <p>
     * To create a new app monitor, use <a
     * href="https://docs.aws.amazon.com/cloudwatchrum/latest/APIReference/API_CreateAppMonitor.html"
     * >CreateAppMonitor</a>.
     * </p>
     * <p>
     * After you update an app monitor, sign in to the CloudWatch RUM console to get the updated JavaScript code snippet
     * to add to your web application. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-RUM-find-code-snippet.html">How
     * do I find a code snippet that I've already generated?</a>
     * </p>
     *
     * @param updateAppMonitorRequest
     * @return A Java Future containing the result of the UpdateAppMonitor operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException This operation attempted to create a resource that already exists.</li>
     *         <li>ResourceNotFoundException Resource not found.</li>
     *         <li>InternalServerException Internal service exception.</li>
     *         <li>ValidationException One of the arguments for the request is not valid.</li>
     *         <li>ThrottlingException The request was throttled because of quota limits.</li>
     *         <li>AccessDeniedException You don't have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RumException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RumAsyncClient.UpdateAppMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/UpdateAppMonitor" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<UpdateAppMonitorResponse> updateAppMonitor(UpdateAppMonitorRequest updateAppMonitorRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Updates the configuration of an existing app monitor. When you use this operation, only the parts of the app
     * monitor configuration that you specify in this operation are changed. For any parameters that you omit, the
     * existing values are kept.
     * </p>
     * <p>
     * You can't use this operation to change the tags of an existing app monitor. To change the tags of an existing app
     * monitor, use <a
     * href="https://docs.aws.amazon.com/cloudwatchrum/latest/APIReference/API_TagResource.html">TagResource</a>.
     * </p>
     * <p>
     * To create a new app monitor, use <a
     * href="https://docs.aws.amazon.com/cloudwatchrum/latest/APIReference/API_CreateAppMonitor.html"
     * >CreateAppMonitor</a>.
     * </p>
     * <p>
     * After you update an app monitor, sign in to the CloudWatch RUM console to get the updated JavaScript code snippet
     * to add to your web application. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-RUM-find-code-snippet.html">How
     * do I find a code snippet that I've already generated?</a>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateAppMonitorRequest.Builder} avoiding the need
     * to create one manually via {@link UpdateAppMonitorRequest#builder()}
     * </p>
     *
     * @param updateAppMonitorRequest
     *        A {@link Consumer} that will call methods on {@link UpdateAppMonitorRequest.Builder} to create a request.
     * @return A Java Future containing the result of the UpdateAppMonitor operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException This operation attempted to create a resource that already exists.</li>
     *         <li>ResourceNotFoundException Resource not found.</li>
     *         <li>InternalServerException Internal service exception.</li>
     *         <li>ValidationException One of the arguments for the request is not valid.</li>
     *         <li>ThrottlingException The request was throttled because of quota limits.</li>
     *         <li>AccessDeniedException You don't have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RumException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RumAsyncClient.UpdateAppMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/UpdateAppMonitor" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<UpdateAppMonitorResponse> updateAppMonitor(
            Consumer<UpdateAppMonitorRequest.Builder> updateAppMonitorRequest) {
        return updateAppMonitor(UpdateAppMonitorRequest.builder().applyMutation(updateAppMonitorRequest).build());
    }
}
