/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rum;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.SdkClient;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.regions.ServiceMetadata;
import software.amazon.awssdk.services.rum.model.AccessDeniedException;
import software.amazon.awssdk.services.rum.model.ConflictException;
import software.amazon.awssdk.services.rum.model.CreateAppMonitorRequest;
import software.amazon.awssdk.services.rum.model.CreateAppMonitorResponse;
import software.amazon.awssdk.services.rum.model.DeleteAppMonitorRequest;
import software.amazon.awssdk.services.rum.model.DeleteAppMonitorResponse;
import software.amazon.awssdk.services.rum.model.GetAppMonitorDataRequest;
import software.amazon.awssdk.services.rum.model.GetAppMonitorDataResponse;
import software.amazon.awssdk.services.rum.model.GetAppMonitorRequest;
import software.amazon.awssdk.services.rum.model.GetAppMonitorResponse;
import software.amazon.awssdk.services.rum.model.InternalServerException;
import software.amazon.awssdk.services.rum.model.ListAppMonitorsRequest;
import software.amazon.awssdk.services.rum.model.ListAppMonitorsResponse;
import software.amazon.awssdk.services.rum.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.rum.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.rum.model.PutRumEventsRequest;
import software.amazon.awssdk.services.rum.model.PutRumEventsResponse;
import software.amazon.awssdk.services.rum.model.ResourceNotFoundException;
import software.amazon.awssdk.services.rum.model.RumException;
import software.amazon.awssdk.services.rum.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.rum.model.TagResourceRequest;
import software.amazon.awssdk.services.rum.model.TagResourceResponse;
import software.amazon.awssdk.services.rum.model.ThrottlingException;
import software.amazon.awssdk.services.rum.model.UntagResourceRequest;
import software.amazon.awssdk.services.rum.model.UntagResourceResponse;
import software.amazon.awssdk.services.rum.model.UpdateAppMonitorRequest;
import software.amazon.awssdk.services.rum.model.UpdateAppMonitorResponse;
import software.amazon.awssdk.services.rum.model.ValidationException;
import software.amazon.awssdk.services.rum.paginators.GetAppMonitorDataIterable;
import software.amazon.awssdk.services.rum.paginators.ListAppMonitorsIterable;

/**
 * Service client for accessing CloudWatch RUM. This can be created using the static {@link #builder()} method.
 *
 * <p>
 * With Amazon CloudWatch RUM, you can perform real-user monitoring to collect client-side data about your web
 * application performance from actual user sessions in real time. The data collected includes page load times,
 * client-side errors, and user behavior. When you view this data, you can see it all aggregated together and also see
 * breakdowns by the browsers and devices that your customers use.
 * </p>
 * 
 * <pre>
 * <code> &lt;p&gt;You can use the collected data to quickly identify and debug client-side performance issues. CloudWatch RUM helps you visualize anomalies in your application performance and find relevant debugging data such as error messages, stack traces, and user sessions. You can also use RUM to understand the range of end-user impact including the number of users, geolocations, and browsers used.&lt;/p&gt; </code>
 * </pre>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface RumClient extends SdkClient {
    String SERVICE_NAME = "rum";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "rum";

    /**
     * Create a {@link RumClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static RumClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link RumClient}.
     */
    static RumClientBuilder builder() {
        return new DefaultRumClientBuilder();
    }

    /**
     * <p>
     * Creates a Amazon CloudWatch RUM app monitor, which collects telemetry data from your application and sends that
     * data to RUM. The data includes performance and reliability information such as page load time, client-side
     * errors, and user behavior.
     * </p>
     * <p>
     * You use this operation only to create a new app monitor. To update an existing app monitor, use <a
     * href="https://docs.aws.amazon.com/cloudwatchrum/latest/APIReference/API_UpdateAppMonitor.html"
     * >UpdateAppMonitor</a> instead.
     * </p>
     * <p>
     * After you create an app monitor, sign in to the CloudWatch RUM console to get the JavaScript code snippet to add
     * to your web application. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-RUM-find-code-snippet.html">How
     * do I find a code snippet that I've already generated?</a>
     * </p>
     *
     * @param createAppMonitorRequest
     * @return Result of the CreateAppMonitor operation returned by the service.
     * @throws ConflictException
     *         This operation attempted to create a resource that already exists.
     * @throws ServiceQuotaExceededException
     *         This request exceeds a service quota.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.CreateAppMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/CreateAppMonitor" target="_top">AWS API
     *      Documentation</a>
     */
    default CreateAppMonitorResponse createAppMonitor(CreateAppMonitorRequest createAppMonitorRequest) throws ConflictException,
            ServiceQuotaExceededException, InternalServerException, ValidationException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, RumException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates a Amazon CloudWatch RUM app monitor, which collects telemetry data from your application and sends that
     * data to RUM. The data includes performance and reliability information such as page load time, client-side
     * errors, and user behavior.
     * </p>
     * <p>
     * You use this operation only to create a new app monitor. To update an existing app monitor, use <a
     * href="https://docs.aws.amazon.com/cloudwatchrum/latest/APIReference/API_UpdateAppMonitor.html"
     * >UpdateAppMonitor</a> instead.
     * </p>
     * <p>
     * After you create an app monitor, sign in to the CloudWatch RUM console to get the JavaScript code snippet to add
     * to your web application. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-RUM-find-code-snippet.html">How
     * do I find a code snippet that I've already generated?</a>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateAppMonitorRequest.Builder} avoiding the need
     * to create one manually via {@link CreateAppMonitorRequest#builder()}
     * </p>
     *
     * @param createAppMonitorRequest
     *        A {@link Consumer} that will call methods on {@link CreateAppMonitorRequest.Builder} to create a request.
     * @return Result of the CreateAppMonitor operation returned by the service.
     * @throws ConflictException
     *         This operation attempted to create a resource that already exists.
     * @throws ServiceQuotaExceededException
     *         This request exceeds a service quota.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.CreateAppMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/CreateAppMonitor" target="_top">AWS API
     *      Documentation</a>
     */
    default CreateAppMonitorResponse createAppMonitor(Consumer<CreateAppMonitorRequest.Builder> createAppMonitorRequest)
            throws ConflictException, ServiceQuotaExceededException, InternalServerException, ValidationException,
            ThrottlingException, AccessDeniedException, AwsServiceException, SdkClientException, RumException {
        return createAppMonitor(CreateAppMonitorRequest.builder().applyMutation(createAppMonitorRequest).build());
    }

    /**
     * <p>
     * Deletes an existing app monitor. This immediately stops the collection of data.
     * </p>
     *
     * @param deleteAppMonitorRequest
     * @return Result of the DeleteAppMonitor operation returned by the service.
     * @throws ConflictException
     *         This operation attempted to create a resource that already exists.
     * @throws ResourceNotFoundException
     *         Resource not found.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.DeleteAppMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/DeleteAppMonitor" target="_top">AWS API
     *      Documentation</a>
     */
    default DeleteAppMonitorResponse deleteAppMonitor(DeleteAppMonitorRequest deleteAppMonitorRequest) throws ConflictException,
            ResourceNotFoundException, InternalServerException, ValidationException, ThrottlingException, AccessDeniedException,
            AwsServiceException, SdkClientException, RumException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes an existing app monitor. This immediately stops the collection of data.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteAppMonitorRequest.Builder} avoiding the need
     * to create one manually via {@link DeleteAppMonitorRequest#builder()}
     * </p>
     *
     * @param deleteAppMonitorRequest
     *        A {@link Consumer} that will call methods on {@link DeleteAppMonitorRequest.Builder} to create a request.
     * @return Result of the DeleteAppMonitor operation returned by the service.
     * @throws ConflictException
     *         This operation attempted to create a resource that already exists.
     * @throws ResourceNotFoundException
     *         Resource not found.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.DeleteAppMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/DeleteAppMonitor" target="_top">AWS API
     *      Documentation</a>
     */
    default DeleteAppMonitorResponse deleteAppMonitor(Consumer<DeleteAppMonitorRequest.Builder> deleteAppMonitorRequest)
            throws ConflictException, ResourceNotFoundException, InternalServerException, ValidationException,
            ThrottlingException, AccessDeniedException, AwsServiceException, SdkClientException, RumException {
        return deleteAppMonitor(DeleteAppMonitorRequest.builder().applyMutation(deleteAppMonitorRequest).build());
    }

    /**
     * <p>
     * Retrieves the complete configuration information for one app monitor.
     * </p>
     *
     * @param getAppMonitorRequest
     * @return Result of the GetAppMonitor operation returned by the service.
     * @throws ResourceNotFoundException
     *         Resource not found.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.GetAppMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/GetAppMonitor" target="_top">AWS API
     *      Documentation</a>
     */
    default GetAppMonitorResponse getAppMonitor(GetAppMonitorRequest getAppMonitorRequest) throws ResourceNotFoundException,
            InternalServerException, ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, RumException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieves the complete configuration information for one app monitor.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetAppMonitorRequest.Builder} avoiding the need to
     * create one manually via {@link GetAppMonitorRequest#builder()}
     * </p>
     *
     * @param getAppMonitorRequest
     *        A {@link Consumer} that will call methods on {@link GetAppMonitorRequest.Builder} to create a request.
     * @return Result of the GetAppMonitor operation returned by the service.
     * @throws ResourceNotFoundException
     *         Resource not found.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.GetAppMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/GetAppMonitor" target="_top">AWS API
     *      Documentation</a>
     */
    default GetAppMonitorResponse getAppMonitor(Consumer<GetAppMonitorRequest.Builder> getAppMonitorRequest)
            throws ResourceNotFoundException, InternalServerException, ValidationException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, RumException {
        return getAppMonitor(GetAppMonitorRequest.builder().applyMutation(getAppMonitorRequest).build());
    }

    /**
     * <p>
     * Retrieves the raw performance events that RUM has collected from your web application, so that you can do your
     * own processing or analysis of this data.
     * </p>
     *
     * @param getAppMonitorDataRequest
     * @return Result of the GetAppMonitorData operation returned by the service.
     * @throws ResourceNotFoundException
     *         Resource not found.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.GetAppMonitorData
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/GetAppMonitorData" target="_top">AWS API
     *      Documentation</a>
     */
    default GetAppMonitorDataResponse getAppMonitorData(GetAppMonitorDataRequest getAppMonitorDataRequest)
            throws ResourceNotFoundException, InternalServerException, ValidationException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, RumException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieves the raw performance events that RUM has collected from your web application, so that you can do your
     * own processing or analysis of this data.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetAppMonitorDataRequest.Builder} avoiding the need
     * to create one manually via {@link GetAppMonitorDataRequest#builder()}
     * </p>
     *
     * @param getAppMonitorDataRequest
     *        A {@link Consumer} that will call methods on {@link GetAppMonitorDataRequest.Builder} to create a request.
     * @return Result of the GetAppMonitorData operation returned by the service.
     * @throws ResourceNotFoundException
     *         Resource not found.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.GetAppMonitorData
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/GetAppMonitorData" target="_top">AWS API
     *      Documentation</a>
     */
    default GetAppMonitorDataResponse getAppMonitorData(Consumer<GetAppMonitorDataRequest.Builder> getAppMonitorDataRequest)
            throws ResourceNotFoundException, InternalServerException, ValidationException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, RumException {
        return getAppMonitorData(GetAppMonitorDataRequest.builder().applyMutation(getAppMonitorDataRequest).build());
    }

    /**
     * <p>
     * Retrieves the raw performance events that RUM has collected from your web application, so that you can do your
     * own processing or analysis of this data.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getAppMonitorData(software.amazon.awssdk.services.rum.model.GetAppMonitorDataRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rum.paginators.GetAppMonitorDataIterable responses = client.getAppMonitorDataPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.rum.paginators.GetAppMonitorDataIterable responses = client
     *             .getAppMonitorDataPaginator(request);
     *     for (software.amazon.awssdk.services.rum.model.GetAppMonitorDataResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rum.paginators.GetAppMonitorDataIterable responses = client.getAppMonitorDataPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getAppMonitorData(software.amazon.awssdk.services.rum.model.GetAppMonitorDataRequest)} operation.</b>
     * </p>
     *
     * @param getAppMonitorDataRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         Resource not found.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.GetAppMonitorData
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/GetAppMonitorData" target="_top">AWS API
     *      Documentation</a>
     */
    default GetAppMonitorDataIterable getAppMonitorDataPaginator(GetAppMonitorDataRequest getAppMonitorDataRequest)
            throws ResourceNotFoundException, InternalServerException, ValidationException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, RumException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieves the raw performance events that RUM has collected from your web application, so that you can do your
     * own processing or analysis of this data.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getAppMonitorData(software.amazon.awssdk.services.rum.model.GetAppMonitorDataRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rum.paginators.GetAppMonitorDataIterable responses = client.getAppMonitorDataPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.rum.paginators.GetAppMonitorDataIterable responses = client
     *             .getAppMonitorDataPaginator(request);
     *     for (software.amazon.awssdk.services.rum.model.GetAppMonitorDataResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rum.paginators.GetAppMonitorDataIterable responses = client.getAppMonitorDataPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getAppMonitorData(software.amazon.awssdk.services.rum.model.GetAppMonitorDataRequest)} operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link GetAppMonitorDataRequest.Builder} avoiding the need
     * to create one manually via {@link GetAppMonitorDataRequest#builder()}
     * </p>
     *
     * @param getAppMonitorDataRequest
     *        A {@link Consumer} that will call methods on {@link GetAppMonitorDataRequest.Builder} to create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         Resource not found.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.GetAppMonitorData
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/GetAppMonitorData" target="_top">AWS API
     *      Documentation</a>
     */
    default GetAppMonitorDataIterable getAppMonitorDataPaginator(
            Consumer<GetAppMonitorDataRequest.Builder> getAppMonitorDataRequest) throws ResourceNotFoundException,
            InternalServerException, ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, RumException {
        return getAppMonitorDataPaginator(GetAppMonitorDataRequest.builder().applyMutation(getAppMonitorDataRequest).build());
    }

    /**
     * <p>
     * Returns a list of the Amazon CloudWatch RUM app monitors in the account.
     * </p>
     *
     * @param listAppMonitorsRequest
     * @return Result of the ListAppMonitors operation returned by the service.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.ListAppMonitors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/ListAppMonitors" target="_top">AWS API
     *      Documentation</a>
     */
    default ListAppMonitorsResponse listAppMonitors(ListAppMonitorsRequest listAppMonitorsRequest)
            throws InternalServerException, ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, RumException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns a list of the Amazon CloudWatch RUM app monitors in the account.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListAppMonitorsRequest.Builder} avoiding the need
     * to create one manually via {@link ListAppMonitorsRequest#builder()}
     * </p>
     *
     * @param listAppMonitorsRequest
     *        A {@link Consumer} that will call methods on {@link ListAppMonitorsRequest.Builder} to create a request.
     * @return Result of the ListAppMonitors operation returned by the service.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.ListAppMonitors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/ListAppMonitors" target="_top">AWS API
     *      Documentation</a>
     */
    default ListAppMonitorsResponse listAppMonitors(Consumer<ListAppMonitorsRequest.Builder> listAppMonitorsRequest)
            throws InternalServerException, ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, RumException {
        return listAppMonitors(ListAppMonitorsRequest.builder().applyMutation(listAppMonitorsRequest).build());
    }

    /**
     * <p>
     * Returns a list of the Amazon CloudWatch RUM app monitors in the account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listAppMonitors(software.amazon.awssdk.services.rum.model.ListAppMonitorsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rum.paginators.ListAppMonitorsIterable responses = client.listAppMonitorsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.rum.paginators.ListAppMonitorsIterable responses = client.listAppMonitorsPaginator(request);
     *     for (software.amazon.awssdk.services.rum.model.ListAppMonitorsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rum.paginators.ListAppMonitorsIterable responses = client.listAppMonitorsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAppMonitors(software.amazon.awssdk.services.rum.model.ListAppMonitorsRequest)} operation.</b>
     * </p>
     *
     * @param listAppMonitorsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.ListAppMonitors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/ListAppMonitors" target="_top">AWS API
     *      Documentation</a>
     */
    default ListAppMonitorsIterable listAppMonitorsPaginator(ListAppMonitorsRequest listAppMonitorsRequest)
            throws InternalServerException, ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, RumException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns a list of the Amazon CloudWatch RUM app monitors in the account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listAppMonitors(software.amazon.awssdk.services.rum.model.ListAppMonitorsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rum.paginators.ListAppMonitorsIterable responses = client.listAppMonitorsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.rum.paginators.ListAppMonitorsIterable responses = client.listAppMonitorsPaginator(request);
     *     for (software.amazon.awssdk.services.rum.model.ListAppMonitorsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rum.paginators.ListAppMonitorsIterable responses = client.listAppMonitorsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAppMonitors(software.amazon.awssdk.services.rum.model.ListAppMonitorsRequest)} operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListAppMonitorsRequest.Builder} avoiding the need
     * to create one manually via {@link ListAppMonitorsRequest#builder()}
     * </p>
     *
     * @param listAppMonitorsRequest
     *        A {@link Consumer} that will call methods on {@link ListAppMonitorsRequest.Builder} to create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.ListAppMonitors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/ListAppMonitors" target="_top">AWS API
     *      Documentation</a>
     */
    default ListAppMonitorsIterable listAppMonitorsPaginator(Consumer<ListAppMonitorsRequest.Builder> listAppMonitorsRequest)
            throws InternalServerException, ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, RumException {
        return listAppMonitorsPaginator(ListAppMonitorsRequest.builder().applyMutation(listAppMonitorsRequest).build());
    }

    /**
     * <p>
     * Displays the tags associated with a CloudWatch RUM resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         Resource not found.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    default ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, InternalServerException, ValidationException, AwsServiceException,
            SdkClientException, RumException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Displays the tags associated with a CloudWatch RUM resource.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListTagsForResourceRequest.Builder} avoiding the
     * need to create one manually via {@link ListTagsForResourceRequest#builder()}
     * </p>
     *
     * @param listTagsForResourceRequest
     *        A {@link Consumer} that will call methods on {@link ListTagsForResourceRequest.Builder} to create a
     *        request.
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         Resource not found.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    default ListTagsForResourceResponse listTagsForResource(
            Consumer<ListTagsForResourceRequest.Builder> listTagsForResourceRequest) throws ResourceNotFoundException,
            InternalServerException, ValidationException, AwsServiceException, SdkClientException, RumException {
        return listTagsForResource(ListTagsForResourceRequest.builder().applyMutation(listTagsForResourceRequest).build());
    }

    /**
     * <p>
     * Sends telemetry events about your application performance and user behavior to CloudWatch RUM. The code snippet
     * that RUM generates for you to add to your application includes <code>PutRumEvents</code> operations to send this
     * data to RUM.
     * </p>
     * <p>
     * Each <code>PutRumEvents</code> operation can send a batch of events from one user session.
     * </p>
     *
     * @param putRumEventsRequest
     * @return Result of the PutRumEvents operation returned by the service.
     * @throws ResourceNotFoundException
     *         Resource not found.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.PutRumEvents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/PutRumEvents" target="_top">AWS API
     *      Documentation</a>
     */
    default PutRumEventsResponse putRumEvents(PutRumEventsRequest putRumEventsRequest) throws ResourceNotFoundException,
            InternalServerException, ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, RumException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Sends telemetry events about your application performance and user behavior to CloudWatch RUM. The code snippet
     * that RUM generates for you to add to your application includes <code>PutRumEvents</code> operations to send this
     * data to RUM.
     * </p>
     * <p>
     * Each <code>PutRumEvents</code> operation can send a batch of events from one user session.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link PutRumEventsRequest.Builder} avoiding the need to
     * create one manually via {@link PutRumEventsRequest#builder()}
     * </p>
     *
     * @param putRumEventsRequest
     *        A {@link Consumer} that will call methods on {@link PutRumEventsRequest.Builder} to create a request.
     * @return Result of the PutRumEvents operation returned by the service.
     * @throws ResourceNotFoundException
     *         Resource not found.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.PutRumEvents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/PutRumEvents" target="_top">AWS API
     *      Documentation</a>
     */
    default PutRumEventsResponse putRumEvents(Consumer<PutRumEventsRequest.Builder> putRumEventsRequest)
            throws ResourceNotFoundException, InternalServerException, ValidationException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, RumException {
        return putRumEvents(PutRumEventsRequest.builder().applyMutation(putRumEventsRequest).build());
    }

    /**
     * <p>
     * Assigns one or more tags (key-value pairs) to the specified CloudWatch RUM resource. Currently, the only
     * resources that can be tagged app monitors.
     * </p>
     * <p>
     * Tags can help you organize and categorize your resources. You can also use them to scope user permissions by
     * granting a user permission to access or change only resources with certain tag values.
     * </p>
     * <p>
     * Tags don't have any semantic meaning to Amazon Web Services and are interpreted strictly as strings of
     * characters.
     * </p>
     * <p>
     * You can use the <code>TagResource</code> action with a resource that already has tags. If you specify a new tag
     * key for the resource, this tag is appended to the list of tags associated with the alarm. If you specify a tag
     * key that is already associated with the resource, the new tag value that you specify replaces the previous value
     * for that tag.
     * </p>
     * <p>
     * You can associate as many as 50 tags with a resource.
     * </p>
     * <p>
     * For more information, see <a href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html">Tagging Amazon
     * Web Services resources</a>.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         Resource not found.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ResourceNotFoundException,
            InternalServerException, ValidationException, AwsServiceException, SdkClientException, RumException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Assigns one or more tags (key-value pairs) to the specified CloudWatch RUM resource. Currently, the only
     * resources that can be tagged app monitors.
     * </p>
     * <p>
     * Tags can help you organize and categorize your resources. You can also use them to scope user permissions by
     * granting a user permission to access or change only resources with certain tag values.
     * </p>
     * <p>
     * Tags don't have any semantic meaning to Amazon Web Services and are interpreted strictly as strings of
     * characters.
     * </p>
     * <p>
     * You can use the <code>TagResource</code> action with a resource that already has tags. If you specify a new tag
     * key for the resource, this tag is appended to the list of tags associated with the alarm. If you specify a tag
     * key that is already associated with the resource, the new tag value that you specify replaces the previous value
     * for that tag.
     * </p>
     * <p>
     * You can associate as many as 50 tags with a resource.
     * </p>
     * <p>
     * For more information, see <a href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html">Tagging Amazon
     * Web Services resources</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link TagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link TagResourceRequest#builder()}
     * </p>
     *
     * @param tagResourceRequest
     *        A {@link Consumer} that will call methods on {@link TagResourceRequest.Builder} to create a request.
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         Resource not found.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default TagResourceResponse tagResource(Consumer<TagResourceRequest.Builder> tagResourceRequest)
            throws ResourceNotFoundException, InternalServerException, ValidationException, AwsServiceException,
            SdkClientException, RumException {
        return tagResource(TagResourceRequest.builder().applyMutation(tagResourceRequest).build());
    }

    /**
     * <p>
     * Removes one or more tags from the specified resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         Resource not found.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            InternalServerException, ValidationException, AwsServiceException, SdkClientException, RumException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Removes one or more tags from the specified resource.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UntagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link UntagResourceRequest#builder()}
     * </p>
     *
     * @param untagResourceRequest
     *        A {@link Consumer} that will call methods on {@link UntagResourceRequest.Builder} to create a request.
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         Resource not found.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default UntagResourceResponse untagResource(Consumer<UntagResourceRequest.Builder> untagResourceRequest)
            throws ResourceNotFoundException, InternalServerException, ValidationException, AwsServiceException,
            SdkClientException, RumException {
        return untagResource(UntagResourceRequest.builder().applyMutation(untagResourceRequest).build());
    }

    /**
     * <p>
     * Updates the configuration of an existing app monitor. When you use this operation, only the parts of the app
     * monitor configuration that you specify in this operation are changed. For any parameters that you omit, the
     * existing values are kept.
     * </p>
     * <p>
     * You can't use this operation to change the tags of an existing app monitor. To change the tags of an existing app
     * monitor, use <a
     * href="https://docs.aws.amazon.com/cloudwatchrum/latest/APIReference/API_TagResource.html">TagResource</a>.
     * </p>
     * <p>
     * To create a new app monitor, use <a
     * href="https://docs.aws.amazon.com/cloudwatchrum/latest/APIReference/API_CreateAppMonitor.html"
     * >CreateAppMonitor</a>.
     * </p>
     * <p>
     * After you update an app monitor, sign in to the CloudWatch RUM console to get the updated JavaScript code snippet
     * to add to your web application. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-RUM-find-code-snippet.html">How
     * do I find a code snippet that I've already generated?</a>
     * </p>
     *
     * @param updateAppMonitorRequest
     * @return Result of the UpdateAppMonitor operation returned by the service.
     * @throws ConflictException
     *         This operation attempted to create a resource that already exists.
     * @throws ResourceNotFoundException
     *         Resource not found.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.UpdateAppMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/UpdateAppMonitor" target="_top">AWS API
     *      Documentation</a>
     */
    default UpdateAppMonitorResponse updateAppMonitor(UpdateAppMonitorRequest updateAppMonitorRequest) throws ConflictException,
            ResourceNotFoundException, InternalServerException, ValidationException, ThrottlingException, AccessDeniedException,
            AwsServiceException, SdkClientException, RumException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Updates the configuration of an existing app monitor. When you use this operation, only the parts of the app
     * monitor configuration that you specify in this operation are changed. For any parameters that you omit, the
     * existing values are kept.
     * </p>
     * <p>
     * You can't use this operation to change the tags of an existing app monitor. To change the tags of an existing app
     * monitor, use <a
     * href="https://docs.aws.amazon.com/cloudwatchrum/latest/APIReference/API_TagResource.html">TagResource</a>.
     * </p>
     * <p>
     * To create a new app monitor, use <a
     * href="https://docs.aws.amazon.com/cloudwatchrum/latest/APIReference/API_CreateAppMonitor.html"
     * >CreateAppMonitor</a>.
     * </p>
     * <p>
     * After you update an app monitor, sign in to the CloudWatch RUM console to get the updated JavaScript code snippet
     * to add to your web application. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-RUM-find-code-snippet.html">How
     * do I find a code snippet that I've already generated?</a>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateAppMonitorRequest.Builder} avoiding the need
     * to create one manually via {@link UpdateAppMonitorRequest#builder()}
     * </p>
     *
     * @param updateAppMonitorRequest
     *        A {@link Consumer} that will call methods on {@link UpdateAppMonitorRequest.Builder} to create a request.
     * @return Result of the UpdateAppMonitor operation returned by the service.
     * @throws ConflictException
     *         This operation attempted to create a resource that already exists.
     * @throws ResourceNotFoundException
     *         Resource not found.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.UpdateAppMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/UpdateAppMonitor" target="_top">AWS API
     *      Documentation</a>
     */
    default UpdateAppMonitorResponse updateAppMonitor(Consumer<UpdateAppMonitorRequest.Builder> updateAppMonitorRequest)
            throws ConflictException, ResourceNotFoundException, InternalServerException, ValidationException,
            ThrottlingException, AccessDeniedException, AwsServiceException, SdkClientException, RumException {
        return updateAppMonitor(UpdateAppMonitorRequest.builder().applyMutation(updateAppMonitorRequest).build());
    }

    static ServiceMetadata serviceMetadata() {
        return ServiceMetadata.of(SERVICE_METADATA_ID);
    }
}
