/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rum;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.rum.model.AccessDeniedException;
import software.amazon.awssdk.services.rum.model.ConflictException;
import software.amazon.awssdk.services.rum.model.CreateAppMonitorRequest;
import software.amazon.awssdk.services.rum.model.CreateAppMonitorResponse;
import software.amazon.awssdk.services.rum.model.DeleteAppMonitorRequest;
import software.amazon.awssdk.services.rum.model.DeleteAppMonitorResponse;
import software.amazon.awssdk.services.rum.model.GetAppMonitorDataRequest;
import software.amazon.awssdk.services.rum.model.GetAppMonitorDataResponse;
import software.amazon.awssdk.services.rum.model.GetAppMonitorRequest;
import software.amazon.awssdk.services.rum.model.GetAppMonitorResponse;
import software.amazon.awssdk.services.rum.model.InternalServerException;
import software.amazon.awssdk.services.rum.model.ListAppMonitorsRequest;
import software.amazon.awssdk.services.rum.model.ListAppMonitorsResponse;
import software.amazon.awssdk.services.rum.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.rum.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.rum.model.PutRumEventsRequest;
import software.amazon.awssdk.services.rum.model.PutRumEventsResponse;
import software.amazon.awssdk.services.rum.model.ResourceNotFoundException;
import software.amazon.awssdk.services.rum.model.RumException;
import software.amazon.awssdk.services.rum.model.RumRequest;
import software.amazon.awssdk.services.rum.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.rum.model.TagResourceRequest;
import software.amazon.awssdk.services.rum.model.TagResourceResponse;
import software.amazon.awssdk.services.rum.model.ThrottlingException;
import software.amazon.awssdk.services.rum.model.UntagResourceRequest;
import software.amazon.awssdk.services.rum.model.UntagResourceResponse;
import software.amazon.awssdk.services.rum.model.UpdateAppMonitorRequest;
import software.amazon.awssdk.services.rum.model.UpdateAppMonitorResponse;
import software.amazon.awssdk.services.rum.model.ValidationException;
import software.amazon.awssdk.services.rum.paginators.GetAppMonitorDataIterable;
import software.amazon.awssdk.services.rum.paginators.ListAppMonitorsIterable;
import software.amazon.awssdk.services.rum.transform.CreateAppMonitorRequestMarshaller;
import software.amazon.awssdk.services.rum.transform.DeleteAppMonitorRequestMarshaller;
import software.amazon.awssdk.services.rum.transform.GetAppMonitorDataRequestMarshaller;
import software.amazon.awssdk.services.rum.transform.GetAppMonitorRequestMarshaller;
import software.amazon.awssdk.services.rum.transform.ListAppMonitorsRequestMarshaller;
import software.amazon.awssdk.services.rum.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.rum.transform.PutRumEventsRequestMarshaller;
import software.amazon.awssdk.services.rum.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.rum.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.rum.transform.UpdateAppMonitorRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link RumClient}.
 *
 * @see RumClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultRumClient implements RumClient {
    private static final Logger log = Logger.loggerFor(DefaultRumClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultRumClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates a Amazon CloudWatch RUM app monitor, which collects telemetry data from your application and sends that
     * data to RUM. The data includes performance and reliability information such as page load time, client-side
     * errors, and user behavior.
     * </p>
     * <p>
     * You use this operation only to create a new app monitor. To update an existing app monitor, use <a
     * href="https://docs.aws.amazon.com/cloudwatchrum/latest/APIReference/API_UpdateAppMonitor.html"
     * >UpdateAppMonitor</a> instead.
     * </p>
     * <p>
     * After you create an app monitor, sign in to the CloudWatch RUM console to get the JavaScript code snippet to add
     * to your web application. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-RUM-find-code-snippet.html">How
     * do I find a code snippet that I've already generated?</a>
     * </p>
     *
     * @param createAppMonitorRequest
     * @return Result of the CreateAppMonitor operation returned by the service.
     * @throws ConflictException
     *         This operation attempted to create a resource that already exists.
     * @throws ServiceQuotaExceededException
     *         This request exceeds a service quota.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.CreateAppMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/CreateAppMonitor" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateAppMonitorResponse createAppMonitor(CreateAppMonitorRequest createAppMonitorRequest) throws ConflictException,
            ServiceQuotaExceededException, InternalServerException, ValidationException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, RumException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateAppMonitorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateAppMonitorResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAppMonitorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RUM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAppMonitor");

            return clientHandler.execute(new ClientExecutionParams<CreateAppMonitorRequest, CreateAppMonitorResponse>()
                    .withOperationName("CreateAppMonitor").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createAppMonitorRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateAppMonitorRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an existing app monitor. This immediately stops the collection of data.
     * </p>
     *
     * @param deleteAppMonitorRequest
     * @return Result of the DeleteAppMonitor operation returned by the service.
     * @throws ConflictException
     *         This operation attempted to create a resource that already exists.
     * @throws ResourceNotFoundException
     *         Resource not found.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.DeleteAppMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/DeleteAppMonitor" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteAppMonitorResponse deleteAppMonitor(DeleteAppMonitorRequest deleteAppMonitorRequest) throws ConflictException,
            ResourceNotFoundException, InternalServerException, ValidationException, ThrottlingException, AccessDeniedException,
            AwsServiceException, SdkClientException, RumException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteAppMonitorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteAppMonitorResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAppMonitorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RUM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAppMonitor");

            return clientHandler.execute(new ClientExecutionParams<DeleteAppMonitorRequest, DeleteAppMonitorResponse>()
                    .withOperationName("DeleteAppMonitor").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteAppMonitorRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteAppMonitorRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the complete configuration information for one app monitor.
     * </p>
     *
     * @param getAppMonitorRequest
     * @return Result of the GetAppMonitor operation returned by the service.
     * @throws ResourceNotFoundException
     *         Resource not found.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.GetAppMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/GetAppMonitor" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetAppMonitorResponse getAppMonitor(GetAppMonitorRequest getAppMonitorRequest) throws ResourceNotFoundException,
            InternalServerException, ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, RumException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetAppMonitorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetAppMonitorResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAppMonitorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RUM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAppMonitor");

            return clientHandler.execute(new ClientExecutionParams<GetAppMonitorRequest, GetAppMonitorResponse>()
                    .withOperationName("GetAppMonitor").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getAppMonitorRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetAppMonitorRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the raw performance events that RUM has collected from your web application, so that you can do your
     * own processing or analysis of this data.
     * </p>
     *
     * @param getAppMonitorDataRequest
     * @return Result of the GetAppMonitorData operation returned by the service.
     * @throws ResourceNotFoundException
     *         Resource not found.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.GetAppMonitorData
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/GetAppMonitorData" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetAppMonitorDataResponse getAppMonitorData(GetAppMonitorDataRequest getAppMonitorDataRequest)
            throws ResourceNotFoundException, InternalServerException, ValidationException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, RumException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetAppMonitorDataResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetAppMonitorDataResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAppMonitorDataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RUM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAppMonitorData");

            return clientHandler.execute(new ClientExecutionParams<GetAppMonitorDataRequest, GetAppMonitorDataResponse>()
                    .withOperationName("GetAppMonitorData").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getAppMonitorDataRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetAppMonitorDataRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the raw performance events that RUM has collected from your web application, so that you can do your
     * own processing or analysis of this data.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getAppMonitorData(software.amazon.awssdk.services.rum.model.GetAppMonitorDataRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rum.paginators.GetAppMonitorDataIterable responses = client.getAppMonitorDataPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.rum.paginators.GetAppMonitorDataIterable responses = client
     *             .getAppMonitorDataPaginator(request);
     *     for (software.amazon.awssdk.services.rum.model.GetAppMonitorDataResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rum.paginators.GetAppMonitorDataIterable responses = client.getAppMonitorDataPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getAppMonitorData(software.amazon.awssdk.services.rum.model.GetAppMonitorDataRequest)} operation.</b>
     * </p>
     *
     * @param getAppMonitorDataRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         Resource not found.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.GetAppMonitorData
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/GetAppMonitorData" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetAppMonitorDataIterable getAppMonitorDataPaginator(GetAppMonitorDataRequest getAppMonitorDataRequest)
            throws ResourceNotFoundException, InternalServerException, ValidationException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, RumException {
        return new GetAppMonitorDataIterable(this, applyPaginatorUserAgent(getAppMonitorDataRequest));
    }

    /**
     * <p>
     * Returns a list of the Amazon CloudWatch RUM app monitors in the account.
     * </p>
     *
     * @param listAppMonitorsRequest
     * @return Result of the ListAppMonitors operation returned by the service.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.ListAppMonitors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/ListAppMonitors" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListAppMonitorsResponse listAppMonitors(ListAppMonitorsRequest listAppMonitorsRequest) throws InternalServerException,
            ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException, SdkClientException,
            RumException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAppMonitorsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListAppMonitorsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAppMonitorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RUM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAppMonitors");

            return clientHandler.execute(new ClientExecutionParams<ListAppMonitorsRequest, ListAppMonitorsResponse>()
                    .withOperationName("ListAppMonitors").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listAppMonitorsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListAppMonitorsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of the Amazon CloudWatch RUM app monitors in the account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listAppMonitors(software.amazon.awssdk.services.rum.model.ListAppMonitorsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rum.paginators.ListAppMonitorsIterable responses = client.listAppMonitorsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.rum.paginators.ListAppMonitorsIterable responses = client.listAppMonitorsPaginator(request);
     *     for (software.amazon.awssdk.services.rum.model.ListAppMonitorsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rum.paginators.ListAppMonitorsIterable responses = client.listAppMonitorsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAppMonitors(software.amazon.awssdk.services.rum.model.ListAppMonitorsRequest)} operation.</b>
     * </p>
     *
     * @param listAppMonitorsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.ListAppMonitors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/ListAppMonitors" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListAppMonitorsIterable listAppMonitorsPaginator(ListAppMonitorsRequest listAppMonitorsRequest)
            throws InternalServerException, ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, RumException {
        return new ListAppMonitorsIterable(this, applyPaginatorUserAgent(listAppMonitorsRequest));
    }

    /**
     * <p>
     * Displays the tags associated with a CloudWatch RUM resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         Resource not found.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, InternalServerException, ValidationException, AwsServiceException,
            SdkClientException, RumException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RUM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Sends telemetry events about your application performance and user behavior to CloudWatch RUM. The code snippet
     * that RUM generates for you to add to your application includes <code>PutRumEvents</code> operations to send this
     * data to RUM.
     * </p>
     * <p>
     * Each <code>PutRumEvents</code> operation can send a batch of events from one user session.
     * </p>
     *
     * @param putRumEventsRequest
     * @return Result of the PutRumEvents operation returned by the service.
     * @throws ResourceNotFoundException
     *         Resource not found.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.PutRumEvents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/PutRumEvents" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutRumEventsResponse putRumEvents(PutRumEventsRequest putRumEventsRequest) throws ResourceNotFoundException,
            InternalServerException, ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, RumException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutRumEventsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutRumEventsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putRumEventsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RUM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutRumEvents");
            String hostPrefix = "dataplane.";
            String resolvedHostExpression = "dataplane.";

            return clientHandler.execute(new ClientExecutionParams<PutRumEventsRequest, PutRumEventsResponse>()
                    .withOperationName("PutRumEvents").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                    .withInput(putRumEventsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutRumEventsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Assigns one or more tags (key-value pairs) to the specified CloudWatch RUM resource. Currently, the only
     * resources that can be tagged app monitors.
     * </p>
     * <p>
     * Tags can help you organize and categorize your resources. You can also use them to scope user permissions by
     * granting a user permission to access or change only resources with certain tag values.
     * </p>
     * <p>
     * Tags don't have any semantic meaning to Amazon Web Services and are interpreted strictly as strings of
     * characters.
     * </p>
     * <p>
     * You can use the <code>TagResource</code> action with a resource that already has tags. If you specify a new tag
     * key for the resource, this tag is appended to the list of tags associated with the alarm. If you specify a tag
     * key that is already associated with the resource, the new tag value that you specify replaces the previous value
     * for that tag.
     * </p>
     * <p>
     * You can associate as many as 50 tags with a resource.
     * </p>
     * <p>
     * For more information, see <a href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html">Tagging Amazon
     * Web Services resources</a>.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         Resource not found.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ResourceNotFoundException,
            InternalServerException, ValidationException, AwsServiceException, SdkClientException, RumException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RUM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes one or more tags from the specified resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         Resource not found.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            InternalServerException, ValidationException, AwsServiceException, SdkClientException, RumException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RUM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the configuration of an existing app monitor. When you use this operation, only the parts of the app
     * monitor configuration that you specify in this operation are changed. For any parameters that you omit, the
     * existing values are kept.
     * </p>
     * <p>
     * You can't use this operation to change the tags of an existing app monitor. To change the tags of an existing app
     * monitor, use <a
     * href="https://docs.aws.amazon.com/cloudwatchrum/latest/APIReference/API_TagResource.html">TagResource</a>.
     * </p>
     * <p>
     * To create a new app monitor, use <a
     * href="https://docs.aws.amazon.com/cloudwatchrum/latest/APIReference/API_CreateAppMonitor.html"
     * >CreateAppMonitor</a>.
     * </p>
     * <p>
     * After you update an app monitor, sign in to the CloudWatch RUM console to get the updated JavaScript code snippet
     * to add to your web application. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-RUM-find-code-snippet.html">How
     * do I find a code snippet that I've already generated?</a>
     * </p>
     *
     * @param updateAppMonitorRequest
     * @return Result of the UpdateAppMonitor operation returned by the service.
     * @throws ConflictException
     *         This operation attempted to create a resource that already exists.
     * @throws ResourceNotFoundException
     *         Resource not found.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.UpdateAppMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/UpdateAppMonitor" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateAppMonitorResponse updateAppMonitor(UpdateAppMonitorRequest updateAppMonitorRequest) throws ConflictException,
            ResourceNotFoundException, InternalServerException, ValidationException, ThrottlingException, AccessDeniedException,
            AwsServiceException, SdkClientException, RumException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateAppMonitorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateAppMonitorResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateAppMonitorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RUM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateAppMonitor");

            return clientHandler.execute(new ClientExecutionParams<UpdateAppMonitorRequest, UpdateAppMonitorResponse>()
                    .withOperationName("UpdateAppMonitor").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateAppMonitorRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateAppMonitorRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(RumException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends RumRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
