/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3.transform;

import static java.nio.charset.StandardCharsets.UTF_8;

import java.io.StringWriter;

import javax.annotation.Generated;

import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.Request;
import software.amazon.awssdk.core.DefaultRequest;
import software.amazon.awssdk.core.http.HttpMethodName;
import software.amazon.awssdk.services.s3.model.*;
import software.amazon.awssdk.core.runtime.transform.Marshaller;

import software.amazon.awssdk.utils.StringInputStream;
import software.amazon.awssdk.core.util.StringConversion;

import software.amazon.awssdk.utils.Md5Utils;
import software.amazon.awssdk.core.util.xml.XmlWriter;

/**
 * RestoreObjectRequest Marshaller
 */

@Generated("software.amazon.awssdk:aws-java-sdk-code-generator")
public class RestoreObjectRequestMarshaller implements Marshaller<Request<RestoreObjectRequest>, RestoreObjectRequest> {

    public Request<RestoreObjectRequest> marshall(RestoreObjectRequest restoreObjectRequest) {

        if (restoreObjectRequest == null) {
            throw SdkClientException.builder().message("Invalid argument passed to marshall(...)").build();
        }

        Request<RestoreObjectRequest> request = new DefaultRequest<RestoreObjectRequest>(restoreObjectRequest, "S3Client");

        request.setHttpMethod(HttpMethodName.POST);

        if (restoreObjectRequest.requestPayerAsString() != null) {
            request.addHeader("x-amz-request-payer", StringConversion.fromString(restoreObjectRequest.requestPayerAsString()));
        }

        String uriResourcePath = "/{Bucket}/{Key+}?restore";

        uriResourcePath = software.amazon.awssdk.core.util.UriResourcePathUtils.addStaticQueryParametersToRequest(request,
                uriResourcePath);

        uriResourcePath = software.amazon.awssdk.core.runtime.transform.PathMarshaller.NON_GREEDY.marshall(uriResourcePath,
                "Bucket", restoreObjectRequest.bucket());
        uriResourcePath = software.amazon.awssdk.core.runtime.transform.PathMarshaller.GREEDY.marshall(uriResourcePath, "Key",
                restoreObjectRequest.key());
        request.setResourcePath(uriResourcePath);

        if (restoreObjectRequest.versionId() != null) {
            request.addParameter("versionId", StringConversion.fromString(restoreObjectRequest.versionId()));
        }

        try {
            StringWriter stringWriter = null;
            RestoreRequest restoreRequest = restoreObjectRequest.restoreRequest();
            if (restoreRequest != null) {
                stringWriter = new StringWriter();
                XmlWriter xmlWriter = new XmlWriter(stringWriter, "http://s3.amazonaws.com/doc/2006-03-01/");
                xmlWriter.startElement("RestoreRequest");

                if (restoreRequest.days() != null) {
                    xmlWriter.startElement("Days").value(restoreRequest.days()).endElement();
                }

                GlacierJobParameters glacierJobParameters = restoreRequest.glacierJobParameters();
                if (glacierJobParameters != null) {
                    xmlWriter.startElement("GlacierJobParameters");

                    if (glacierJobParameters.tierAsString() != null) {
                        xmlWriter.startElement("Tier").value(glacierJobParameters.tierAsString()).endElement();
                    }
                    xmlWriter.endElement();
                }
                xmlWriter.endElement();
            }

            if (stringWriter != null) {
                if (!request.getHeaders().containsKey("Content-MD5")) {
                    request.addHeader("Content-MD5", Md5Utils.md5AsBase64(stringWriter.getBuffer().toString().getBytes(UTF_8)));
                }
                request.setContent(new StringInputStream(stringWriter.getBuffer().toString()));
                request.addHeader("Content-Length", Integer.toString(stringWriter.getBuffer().toString().getBytes(UTF_8).length));
            }
            if (!request.getHeaders().containsKey("Content-Type")) {
                request.addHeader("Content-Type", "application/xml");
            }
        } catch (Throwable t) {
            throw SdkClientException.builder().message("Unable to marshall request to XML: " + t.getMessage()).cause(t).build();
        }

        return request;
    }

}