/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Container for the transition rule that describes when noncurrent objects transition to the STANDARD_IA, ONEZONE_IA,
 * INTELLIGENT_TIERING or GLACIER storage class. If your bucket is versioning-enabled (or versioning is suspended), you
 * can set this action to request that Amazon S3 transition noncurrent object versions to the STANDARD_IA, ONEZONE_IA,
 * INTELLIGENT_TIERING or GLACIER storage class at a specific period in the object's lifetime.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class NoncurrentVersionTransition implements SdkPojo, Serializable,
        ToCopyableBuilder<NoncurrentVersionTransition.Builder, NoncurrentVersionTransition> {
    private static final SdkField<Integer> NONCURRENT_DAYS_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(NoncurrentVersionTransition::noncurrentDays))
            .setter(setter(Builder::noncurrentDays))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NoncurrentDays")
                    .unmarshallLocationName("NoncurrentDays").build()).build();

    private static final SdkField<String> STORAGE_CLASS_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(NoncurrentVersionTransition::storageClassAsString))
            .setter(setter(Builder::storageClass))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StorageClass")
                    .unmarshallLocationName("StorageClass").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NONCURRENT_DAYS_FIELD,
            STORAGE_CLASS_FIELD));

    private static final long serialVersionUID = 1L;

    private final Integer noncurrentDays;

    private final String storageClass;

    private NoncurrentVersionTransition(BuilderImpl builder) {
        this.noncurrentDays = builder.noncurrentDays;
        this.storageClass = builder.storageClass;
    }

    /**
     * <p>
     * Specifies the number of days an object is noncurrent before Amazon S3 can perform the associated action. For
     * information about the noncurrent days calculations, see <a
     * href="http://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">How Amazon S3 Calculates When an
     * Object Became Noncurrent</a> in the Amazon Simple Storage Service Developer Guide.
     * </p>
     * 
     * @return Specifies the number of days an object is noncurrent before Amazon S3 can perform the associated action.
     *         For information about the noncurrent days calculations, see <a
     *         href="http://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">How Amazon S3 Calculates
     *         When an Object Became Noncurrent</a> in the Amazon Simple Storage Service Developer Guide.
     */
    public Integer noncurrentDays() {
        return noncurrentDays;
    }

    /**
     * <p>
     * The class of storage used to store the object.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #storageClass} will
     * return {@link TransitionStorageClass#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #storageClassAsString}.
     * </p>
     * 
     * @return The class of storage used to store the object.
     * @see TransitionStorageClass
     */
    public TransitionStorageClass storageClass() {
        return TransitionStorageClass.fromValue(storageClass);
    }

    /**
     * <p>
     * The class of storage used to store the object.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #storageClass} will
     * return {@link TransitionStorageClass#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #storageClassAsString}.
     * </p>
     * 
     * @return The class of storage used to store the object.
     * @see TransitionStorageClass
     */
    public String storageClassAsString() {
        return storageClass;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(noncurrentDays());
        hashCode = 31 * hashCode + Objects.hashCode(storageClassAsString());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof NoncurrentVersionTransition)) {
            return false;
        }
        NoncurrentVersionTransition other = (NoncurrentVersionTransition) obj;
        return Objects.equals(noncurrentDays(), other.noncurrentDays())
                && Objects.equals(storageClassAsString(), other.storageClassAsString());
    }

    @Override
    public String toString() {
        return ToString.builder("NoncurrentVersionTransition").add("NoncurrentDays", noncurrentDays())
                .add("StorageClass", storageClassAsString()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "NoncurrentDays":
            return Optional.ofNullable(clazz.cast(noncurrentDays()));
        case "StorageClass":
            return Optional.ofNullable(clazz.cast(storageClassAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<NoncurrentVersionTransition, T> g) {
        return obj -> g.apply((NoncurrentVersionTransition) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, NoncurrentVersionTransition> {
        /**
         * <p>
         * Specifies the number of days an object is noncurrent before Amazon S3 can perform the associated action. For
         * information about the noncurrent days calculations, see <a
         * href="http://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">How Amazon S3 Calculates When an
         * Object Became Noncurrent</a> in the Amazon Simple Storage Service Developer Guide.
         * </p>
         * 
         * @param noncurrentDays
         *        Specifies the number of days an object is noncurrent before Amazon S3 can perform the associated
         *        action. For information about the noncurrent days calculations, see <a
         *        href="http://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">How Amazon S3 Calculates
         *        When an Object Became Noncurrent</a> in the Amazon Simple Storage Service Developer Guide.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder noncurrentDays(Integer noncurrentDays);

        /**
         * <p>
         * The class of storage used to store the object.
         * </p>
         * 
         * @param storageClass
         *        The class of storage used to store the object.
         * @see TransitionStorageClass
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TransitionStorageClass
         */
        Builder storageClass(String storageClass);

        /**
         * <p>
         * The class of storage used to store the object.
         * </p>
         * 
         * @param storageClass
         *        The class of storage used to store the object.
         * @see TransitionStorageClass
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TransitionStorageClass
         */
        Builder storageClass(TransitionStorageClass storageClass);
    }

    static final class BuilderImpl implements Builder {
        private Integer noncurrentDays;

        private String storageClass;

        private BuilderImpl() {
        }

        private BuilderImpl(NoncurrentVersionTransition model) {
            noncurrentDays(model.noncurrentDays);
            storageClass(model.storageClass);
        }

        public final Integer getNoncurrentDays() {
            return noncurrentDays;
        }

        @Override
        public final Builder noncurrentDays(Integer noncurrentDays) {
            this.noncurrentDays = noncurrentDays;
            return this;
        }

        public final void setNoncurrentDays(Integer noncurrentDays) {
            this.noncurrentDays = noncurrentDays;
        }

        public final String getStorageClass() {
            return storageClass;
        }

        @Override
        public final Builder storageClass(String storageClass) {
            this.storageClass = storageClass;
            return this;
        }

        @Override
        public final Builder storageClass(TransitionStorageClass storageClass) {
            this.storageClass(storageClass.toString());
            return this;
        }

        public final void setStorageClass(String storageClass) {
            this.storageClass = storageClass;
        }

        @Override
        public NoncurrentVersionTransition build() {
            return new NoncurrentVersionTransition(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
