/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.PayloadTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class PutObjectLockConfigurationRequest extends S3Request implements
        ToCopyableBuilder<PutObjectLockConfigurationRequest.Builder, PutObjectLockConfigurationRequest> {
    private static final SdkField<String> BUCKET_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(PutObjectLockConfigurationRequest::bucket))
            .setter(setter(Builder::bucket))
            .traits(LocationTrait.builder().location(MarshallLocation.PATH).locationName("Bucket")
                    .unmarshallLocationName("Bucket").build()).build();

    private static final SdkField<ObjectLockConfiguration> OBJECT_LOCK_CONFIGURATION_FIELD = SdkField
            .<ObjectLockConfiguration> builder(MarshallingType.SDK_POJO)
            .getter(getter(PutObjectLockConfigurationRequest::objectLockConfiguration))
            .setter(setter(Builder::objectLockConfiguration))
            .constructor(ObjectLockConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ObjectLockConfiguration")
                    .unmarshallLocationName("ObjectLockConfiguration").build(), PayloadTrait.create()).build();

    private static final SdkField<String> REQUEST_PAYER_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(PutObjectLockConfigurationRequest::requestPayerAsString))
            .setter(setter(Builder::requestPayer))
            .traits(LocationTrait.builder().location(MarshallLocation.HEADER).locationName("x-amz-request-payer")
                    .unmarshallLocationName("x-amz-request-payer").build()).build();

    private static final SdkField<String> TOKEN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(PutObjectLockConfigurationRequest::token))
            .setter(setter(Builder::token))
            .traits(LocationTrait.builder().location(MarshallLocation.HEADER).locationName("x-amz-bucket-object-lock-token")
                    .unmarshallLocationName("x-amz-bucket-object-lock-token").build()).build();

    private static final SdkField<String> CONTENT_MD5_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(PutObjectLockConfigurationRequest::contentMD5))
            .setter(setter(Builder::contentMD5))
            .traits(LocationTrait.builder().location(MarshallLocation.HEADER).locationName("Content-MD5")
                    .unmarshallLocationName("Content-MD5").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(BUCKET_FIELD,
            OBJECT_LOCK_CONFIGURATION_FIELD, REQUEST_PAYER_FIELD, TOKEN_FIELD, CONTENT_MD5_FIELD));

    private final String bucket;

    private final ObjectLockConfiguration objectLockConfiguration;

    private final String requestPayer;

    private final String token;

    private final String contentMD5;

    private PutObjectLockConfigurationRequest(BuilderImpl builder) {
        super(builder);
        this.bucket = builder.bucket;
        this.objectLockConfiguration = builder.objectLockConfiguration;
        this.requestPayer = builder.requestPayer;
        this.token = builder.token;
        this.contentMD5 = builder.contentMD5;
    }

    /**
     * <p>
     * The bucket whose Object Lock configuration you want to create or replace.
     * </p>
     * 
     * @return The bucket whose Object Lock configuration you want to create or replace.
     */
    public String bucket() {
        return bucket;
    }

    /**
     * <p>
     * The Object Lock configuration that you want to apply to the specified bucket.
     * </p>
     * 
     * @return The Object Lock configuration that you want to apply to the specified bucket.
     */
    public ObjectLockConfiguration objectLockConfiguration() {
        return objectLockConfiguration;
    }

    /**
     * Returns the value of the RequestPayer property for this object.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #requestPayer} will
     * return {@link RequestPayer#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #requestPayerAsString}.
     * </p>
     * 
     * @return The value of the RequestPayer property for this object.
     * @see RequestPayer
     */
    public RequestPayer requestPayer() {
        return RequestPayer.fromValue(requestPayer);
    }

    /**
     * Returns the value of the RequestPayer property for this object.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #requestPayer} will
     * return {@link RequestPayer#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #requestPayerAsString}.
     * </p>
     * 
     * @return The value of the RequestPayer property for this object.
     * @see RequestPayer
     */
    public String requestPayerAsString() {
        return requestPayer;
    }

    /**
     * <p>
     * A token to allow Object Lock to be enabled for an existing bucket.
     * </p>
     * 
     * @return A token to allow Object Lock to be enabled for an existing bucket.
     */
    public String token() {
        return token;
    }

    /**
     * <p>
     * The MD5 hash for the request body.
     * </p>
     * 
     * @return The MD5 hash for the request body.
     */
    public String contentMD5() {
        return contentMD5;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(bucket());
        hashCode = 31 * hashCode + Objects.hashCode(objectLockConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(requestPayerAsString());
        hashCode = 31 * hashCode + Objects.hashCode(token());
        hashCode = 31 * hashCode + Objects.hashCode(contentMD5());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PutObjectLockConfigurationRequest)) {
            return false;
        }
        PutObjectLockConfigurationRequest other = (PutObjectLockConfigurationRequest) obj;
        return Objects.equals(bucket(), other.bucket())
                && Objects.equals(objectLockConfiguration(), other.objectLockConfiguration())
                && Objects.equals(requestPayerAsString(), other.requestPayerAsString()) && Objects.equals(token(), other.token())
                && Objects.equals(contentMD5(), other.contentMD5());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("PutObjectLockConfigurationRequest").add("Bucket", bucket())
                .add("ObjectLockConfiguration", objectLockConfiguration()).add("RequestPayer", requestPayerAsString())
                .add("Token", token()).add("ContentMD5", contentMD5()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Bucket":
            return Optional.ofNullable(clazz.cast(bucket()));
        case "ObjectLockConfiguration":
            return Optional.ofNullable(clazz.cast(objectLockConfiguration()));
        case "RequestPayer":
            return Optional.ofNullable(clazz.cast(requestPayerAsString()));
        case "Token":
            return Optional.ofNullable(clazz.cast(token()));
        case "ContentMD5":
            return Optional.ofNullable(clazz.cast(contentMD5()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PutObjectLockConfigurationRequest, T> g) {
        return obj -> g.apply((PutObjectLockConfigurationRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends S3Request.Builder, SdkPojo, CopyableBuilder<Builder, PutObjectLockConfigurationRequest> {
        /**
         * <p>
         * The bucket whose Object Lock configuration you want to create or replace.
         * </p>
         * 
         * @param bucket
         *        The bucket whose Object Lock configuration you want to create or replace.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bucket(String bucket);

        /**
         * <p>
         * The Object Lock configuration that you want to apply to the specified bucket.
         * </p>
         * 
         * @param objectLockConfiguration
         *        The Object Lock configuration that you want to apply to the specified bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder objectLockConfiguration(ObjectLockConfiguration objectLockConfiguration);

        /**
         * <p>
         * The Object Lock configuration that you want to apply to the specified bucket.
         * </p>
         * This is a convenience that creates an instance of the {@link ObjectLockConfiguration.Builder} avoiding the
         * need to create one manually via {@link ObjectLockConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link ObjectLockConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #objectLockConfiguration(ObjectLockConfiguration)}.
         * 
         * @param objectLockConfiguration
         *        a consumer that will call methods on {@link ObjectLockConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #objectLockConfiguration(ObjectLockConfiguration)
         */
        default Builder objectLockConfiguration(Consumer<ObjectLockConfiguration.Builder> objectLockConfiguration) {
            return objectLockConfiguration(ObjectLockConfiguration.builder().applyMutation(objectLockConfiguration).build());
        }

        /**
         * Sets the value of the RequestPayer property for this object.
         *
         * @param requestPayer
         *        The new value for the RequestPayer property for this object.
         * @see RequestPayer
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RequestPayer
         */
        Builder requestPayer(String requestPayer);

        /**
         * Sets the value of the RequestPayer property for this object.
         *
         * @param requestPayer
         *        The new value for the RequestPayer property for this object.
         * @see RequestPayer
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RequestPayer
         */
        Builder requestPayer(RequestPayer requestPayer);

        /**
         * <p>
         * A token to allow Object Lock to be enabled for an existing bucket.
         * </p>
         * 
         * @param token
         *        A token to allow Object Lock to be enabled for an existing bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder token(String token);

        /**
         * <p>
         * The MD5 hash for the request body.
         * </p>
         * 
         * @param contentMD5
         *        The MD5 hash for the request body.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder contentMD5(String contentMD5);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends S3Request.BuilderImpl implements Builder {
        private String bucket;

        private ObjectLockConfiguration objectLockConfiguration;

        private String requestPayer;

        private String token;

        private String contentMD5;

        private BuilderImpl() {
        }

        private BuilderImpl(PutObjectLockConfigurationRequest model) {
            super(model);
            bucket(model.bucket);
            objectLockConfiguration(model.objectLockConfiguration);
            requestPayer(model.requestPayer);
            token(model.token);
            contentMD5(model.contentMD5);
        }

        public final String getBucket() {
            return bucket;
        }

        @Override
        public final Builder bucket(String bucket) {
            this.bucket = bucket;
            return this;
        }

        public final void setBucket(String bucket) {
            this.bucket = bucket;
        }

        public final ObjectLockConfiguration.Builder getObjectLockConfiguration() {
            return objectLockConfiguration != null ? objectLockConfiguration.toBuilder() : null;
        }

        @Override
        public final Builder objectLockConfiguration(ObjectLockConfiguration objectLockConfiguration) {
            this.objectLockConfiguration = objectLockConfiguration;
            return this;
        }

        public final void setObjectLockConfiguration(ObjectLockConfiguration.BuilderImpl objectLockConfiguration) {
            this.objectLockConfiguration = objectLockConfiguration != null ? objectLockConfiguration.build() : null;
        }

        public final String getRequestPayerAsString() {
            return requestPayer;
        }

        @Override
        public final Builder requestPayer(String requestPayer) {
            this.requestPayer = requestPayer;
            return this;
        }

        @Override
        public final Builder requestPayer(RequestPayer requestPayer) {
            this.requestPayer(requestPayer == null ? null : requestPayer.toString());
            return this;
        }

        public final void setRequestPayer(String requestPayer) {
            this.requestPayer = requestPayer;
        }

        public final String getToken() {
            return token;
        }

        @Override
        public final Builder token(String token) {
            this.token = token;
            return this;
        }

        public final void setToken(String token) {
            this.token = token;
        }

        public final String getContentMD5() {
            return contentMD5;
        }

        @Override
        public final Builder contentMD5(String contentMD5) {
            this.contentMD5 = contentMD5;
            return this;
        }

        public final void setContentMD5(String contentMD5) {
            this.contentMD5 = contentMD5;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public PutObjectLockConfigurationRequest build() {
            return new PutObjectLockConfigurationRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
